"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediastore = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediastore](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediastore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediastore extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediastore](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediastore.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediastore';
        this.actionList = {
            "CreateContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html",
                "description": "Grants permission to create containers.",
                "accessLevel": "Write"
            },
            "DeleteContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html",
                "description": "Grants permission to delete any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html",
                "description": "Grants permission to delete the access policy of any container in the current account.",
                "accessLevel": "Permissions management"
            },
            "DeleteCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html",
                "description": "Grants permission to delete the CORS policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html",
                "description": "Grants permission to delete the lifecycle policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html",
                "description": "Grants permission to delete the metric policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html",
                "description": "Grants permission to delete objects.",
                "accessLevel": "Write"
            },
            "DescribeContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html",
                "description": "Grants permission to retrieve details on any container in the current account.",
                "accessLevel": "List"
            },
            "DescribeObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html",
                "description": "Grants permission to retrieve object metadata.",
                "accessLevel": "List"
            },
            "GetContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html",
                "description": "Grants permission to retrieve the access policy of any container in the current account.",
                "accessLevel": "Read"
            },
            "GetCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html",
                "description": "Grants permission to retrieve the CORS policy of any container in the current account.",
                "accessLevel": "Read"
            },
            "GetLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html",
                "description": "Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.",
                "accessLevel": "Read"
            },
            "GetMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html",
                "description": "Grants permission to retrieve the metric policy that is assigned to any container in the current account.",
                "accessLevel": "Read"
            },
            "GetObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html",
                "description": "Grants permission to retrieve objects.",
                "accessLevel": "Read"
            },
            "ListContainers": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html",
                "description": "Grants permission to retrieve a list of containers in the current account.",
                "accessLevel": "List"
            },
            "ListItems": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html",
                "description": "Grants permission to retrieve a list of objects and folders in the current account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags on any container in the current account.",
                "accessLevel": "Read"
            },
            "PutContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html",
                "description": "Grants permission to create or replace the access policy of any container in the current account.",
                "accessLevel": "Permissions management"
            },
            "PutCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html",
                "description": "Grants permission to add or modify the CORS policy of any container in the current account.",
                "accessLevel": "Write"
            },
            "PutLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html",
                "description": "Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.",
                "accessLevel": "Write"
            },
            "PutMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html",
                "description": "Grants permission to add or modify the metric policy that is assigned to any container in the current account.",
                "accessLevel": "Write"
            },
            "PutObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html",
                "description": "Grants permission to upload objects.",
                "accessLevel": "Write"
            },
            "StartAccessLogging": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html",
                "description": "Grants permission to enable access logging on any container in the current account.",
                "accessLevel": "Write"
            },
            "StopAccessLogging": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html",
                "description": "Grants permission to disable access logging on any container in the current account.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html",
                "description": "Grants permission to add tags to any container in the current account.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html",
                "description": "Grants permission to remove tags from any container in the current account.",
                "accessLevel": "Tagging"
            }
        };
        this.resourceTypes = {
            "container": {
                "name": "container",
                "url": "https://docs.aws.amazon.com/mediastore/latest/ug/containers.html",
                "arn": "arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create containers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html
     */
    toCreateContainer() {
        this.add('mediastore:CreateContainer');
        return this;
    }
    /**
     * Grants permission to delete any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html
     */
    toDeleteContainer() {
        this.add('mediastore:DeleteContainer');
        return this;
    }
    /**
     * Grants permission to delete the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html
     */
    toDeleteContainerPolicy() {
        this.add('mediastore:DeleteContainerPolicy');
        return this;
    }
    /**
     * Grants permission to delete the CORS policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html
     */
    toDeleteCorsPolicy() {
        this.add('mediastore:DeleteCorsPolicy');
        return this;
    }
    /**
     * Grants permission to delete the lifecycle policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html
     */
    toDeleteLifecyclePolicy() {
        this.add('mediastore:DeleteLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to delete the metric policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html
     */
    toDeleteMetricPolicy() {
        this.add('mediastore:DeleteMetricPolicy');
        return this;
    }
    /**
     * Grants permission to delete objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html
     */
    toDeleteObject() {
        this.add('mediastore:DeleteObject');
        return this;
    }
    /**
     * Grants permission to retrieve details on any container in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html
     */
    toDescribeContainer() {
        this.add('mediastore:DescribeContainer');
        return this;
    }
    /**
     * Grants permission to retrieve object metadata.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html
     */
    toDescribeObject() {
        this.add('mediastore:DescribeObject');
        return this;
    }
    /**
     * Grants permission to retrieve the access policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html
     */
    toGetContainerPolicy() {
        this.add('mediastore:GetContainerPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the CORS policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html
     */
    toGetCorsPolicy() {
        this.add('mediastore:GetCorsPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html
     */
    toGetLifecyclePolicy() {
        this.add('mediastore:GetLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html
     */
    toGetMetricPolicy() {
        this.add('mediastore:GetMetricPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html
     */
    toGetObject() {
        this.add('mediastore:GetObject');
        return this;
    }
    /**
     * Grants permission to retrieve a list of containers in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html
     */
    toListContainers() {
        this.add('mediastore:ListContainers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of objects and folders in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html
     */
    toListItems() {
        this.add('mediastore:ListItems');
        return this;
    }
    /**
     * Grants permission to list tags on any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('mediastore:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create or replace the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html
     */
    toPutContainerPolicy() {
        this.add('mediastore:PutContainerPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the CORS policy of any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html
     */
    toPutCorsPolicy() {
        this.add('mediastore:PutCorsPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html
     */
    toPutLifecyclePolicy() {
        this.add('mediastore:PutLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html
     */
    toPutMetricPolicy() {
        this.add('mediastore:PutMetricPolicy');
        return this;
    }
    /**
     * Grants permission to upload objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html
     */
    toPutObject() {
        this.add('mediastore:PutObject');
        return this;
    }
    /**
     * Grants permission to enable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html
     */
    toStartAccessLogging() {
        this.add('mediastore:StartAccessLogging');
        return this;
    }
    /**
     * Grants permission to disable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html
     */
    toStopAccessLogging() {
        this.add('mediastore:StopAccessLogging');
        return this;
    }
    /**
     * Grants permission to add tags to any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        this.add('mediastore:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('mediastore:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type container to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/containers.html
     *
     * @param containerName - Identifier for the containerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContainer(containerName, account, region, partition) {
        var arn = 'arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}';
        arn = arn.replace('${ContainerName}', containerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediastore = Mediastore;
//# sourceMappingURL=data:application/json;base64,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