"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glue = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glue extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glue';
        this.actionList = {
            "BatchCreatePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchCreatePartition",
                "description": "Grants permission to create one or more partitions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchDeleteConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-BatchDeleteConnection",
                "description": "Grants permission to delete one or more connections",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "BatchDeletePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchDeletePartition",
                "description": "Grants permission to delete one or more partitions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchDeleteTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-BatchDeleteTable",
                "description": "Grants permission to delete one or more tables",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchDeleteTableVersion": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion",
                "description": "Grants permission to delete one or more versions of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "BatchGetCrawlers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-BatchGetCrawlers",
                "description": "Grants permission to retrieve one or more crawlers",
                "accessLevel": "Read"
            },
            "BatchGetDevEndpoints": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-BatchGetDevEndpoints",
                "description": "Grants permission to retrieve one or more development endpoints",
                "accessLevel": "Read"
            },
            "BatchGetJobs": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-BatchGetJobs",
                "description": "Grants permission to retrieve one or more jobs",
                "accessLevel": "Read"
            },
            "BatchGetPartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchGetPartition",
                "description": "Grants permission to retrieve one or more partitions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchGetTriggers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-BatchGetTriggers",
                "description": "Grants permission to retrieve one or more triggers",
                "accessLevel": "Read"
            },
            "BatchGetWorkflows": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-BatchGetWorkflows",
                "description": "Grants permission to retrieve one or more workflows",
                "accessLevel": "Read"
            },
            "BatchStopJobRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-BatchStopStartJobRun",
                "description": "Grants permission to stop one or more job runs for a job",
                "accessLevel": "Write"
            },
            "CancelMLTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CancelMLTaskRun",
                "description": "Grants permission to stop a running ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "CreateClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-CreateClassifier",
                "description": "Grants permission to create a classifier",
                "accessLevel": "Write"
            },
            "CreateConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-CreateConnection",
                "description": "Grants permission to create a connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "CreateCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-CreateCrawler",
                "description": "Grants permission to create a crawler",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-CreateDatabase",
                "description": "Grants permission to create a database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "CreateDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-CreateDevEndpoint",
                "description": "Grants permission to create a development endpoint",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-CreateJob",
                "description": "Grants permission to create a job",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CreateMLTransform",
                "description": "Grants permission to create an ML Transform",
                "accessLevel": "Write"
            },
            "CreatePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-CreatePartition",
                "description": "Grants permission to create a partition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateScript": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-CreateScript",
                "description": "Grants permission to create a script",
                "accessLevel": "Write"
            },
            "CreateSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-CreateSecurityConfiguration",
                "description": "Grants permission to create a security configuration",
                "accessLevel": "Write"
            },
            "CreateTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-CreateTable",
                "description": "Grants permission to create a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-CreateTrigger",
                "description": "Grants permission to create a trigger",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-CreateUserDefinedFunction",
                "description": "Grants permission to create a function definition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "CreateWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-CreateWorkflow",
                "description": "Grants permission to create a workflow",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-DeleteClassifier",
                "description": "Grants permission to delete a classifier",
                "accessLevel": "Write"
            },
            "DeleteConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-DeleteConnection",
                "description": "Grants permission to delete a connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "DeleteCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-DeleteCrawler",
                "description": "Grants permission to delete a crawler",
                "accessLevel": "Write"
            },
            "DeleteDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-DeleteDatabase",
                "description": "Grants permission to delete a database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "DeleteDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-DeleteDevEndpoint",
                "description": "Grants permission to delete a development endpoint",
                "accessLevel": "Write"
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-DeleteJob",
                "description": "Grants permission to delete a job",
                "accessLevel": "Write"
            },
            "DeleteMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-DeleteMLTransform",
                "description": "Grants permission to delete an ML Transform",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "DeletePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-DeletePartition",
                "description": "Grants permission to delete a partition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteResourcePolicy",
                "description": "Grants permission to delete a resource policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "DeleteSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteSecurityConfiguration",
                "description": "Grants permission to delete a security configuration",
                "accessLevel": "Write"
            },
            "DeleteTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTable",
                "description": "Grants permission to delete a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "DeleteTableVersion": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion",
                "description": "Grants permission to delete a version of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "DeleteTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-DeleteTrigger",
                "description": "Grants permission to delete a trigger",
                "accessLevel": "Write"
            },
            "DeleteUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-DeleteUserDefinedFunction",
                "description": "Grants permission to delete a function definition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "DeleteWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-DeleteWorkflow",
                "description": "Grants permission to delete a workflow",
                "accessLevel": "Write"
            },
            "GetCatalogImportStatus": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-GetCatalogImportStatus",
                "description": "Grants permission to retrieve the catalog import status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "GetClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifier",
                "description": "Grants permission to retrieve a classifier",
                "accessLevel": "Read"
            },
            "GetClassifiers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifiers",
                "description": "Grants permission to list all classifiers",
                "accessLevel": "Read"
            },
            "GetConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnection",
                "description": "Grants permission to retrieve a connection",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "GetConnections": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnections",
                "description": "Grants permission to retrieve a list of connections",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "GetCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawler",
                "description": "Grants permission to retrieve a crawler",
                "accessLevel": "Read"
            },
            "GetCrawlerMetrics": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlerMetrics",
                "description": "Grants permission to retrieve metrics about crawlers",
                "accessLevel": "Read"
            },
            "GetCrawlers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlers",
                "description": "Grants permission to retrieve all crawlers",
                "accessLevel": "Read"
            },
            "GetDataCatalogEncryptionSettings": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetDataCatalogEncryptionSettings",
                "description": "Grants permission to retrieve catalog encryption settings",
                "accessLevel": "Read"
            },
            "GetDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabase",
                "description": "Grants permission to retrieve a database",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "GetDatabases": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabases",
                "description": "Grants permission to retrieve all databases",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "GetDataflowGraph": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetDataflowGraph",
                "description": "Grants permission to transform a script into a directed acyclic graph (DAG)",
                "accessLevel": "Read"
            },
            "GetDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoint",
                "description": "Grants permission to retrieve a development endpoint",
                "accessLevel": "Read"
            },
            "GetDevEndpoints": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoints",
                "description": "Grants permission to retrieve all development endpoints",
                "accessLevel": "Read"
            },
            "GetJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJob",
                "description": "Grants permission to retrieve a job",
                "accessLevel": "Read"
            },
            "GetJobBookmark": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-job-GetJobBookmark",
                "description": "Grants permission to retrieve a job bookmark",
                "accessLevel": "Read"
            },
            "GetJobRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRun",
                "description": "Grants permission to retrieve a job run",
                "accessLevel": "Read"
            },
            "GetJobRuns": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRuns",
                "description": "Grants permission to retrieve all job runs of a job",
                "accessLevel": "Read"
            },
            "GetJobs": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJobs",
                "description": "Grants permission to retrieve all current jobs",
                "accessLevel": "Read"
            },
            "GetMLTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRun",
                "description": "Grants permission to retrieve an ML Task Run",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "GetMLTaskRuns": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRuns",
                "description": "Grants permission to retrieve all ML Task Runs",
                "accessLevel": "List",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "GetMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransform",
                "description": "Grants permission to retrieve an ML Transform",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "GetMLTransforms": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransforms",
                "description": "Grants permission to retrieve all ML Transforms",
                "accessLevel": "List"
            },
            "GetMapping": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetMapping",
                "description": "Grants permission to create a mapping",
                "accessLevel": "Write"
            },
            "GetPartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartition",
                "description": "Grants permission to retrieve a partition",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetPartitions": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartitions",
                "description": "Grants permission to retrieve the partitions of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetPlan": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetPlan",
                "description": "Grants permission to retrieve a mapping for a script",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetResourcePolicy",
                "description": "Grants permission to retrieve a resource policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "GetSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfiguration",
                "description": "Grants permission to retrieve a security configuration",
                "accessLevel": "Read"
            },
            "GetSecurityConfigurations": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfigurations",
                "description": "Grants permission to retrieve one or more security configurations",
                "accessLevel": "Read"
            },
            "GetTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTable",
                "description": "Grants permission to retrieve a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetTableVersion": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersion",
                "description": "Grants permission to retrieve a version of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "GetTableVersions": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersions",
                "description": "Grants permission to retrieve a list of versions of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "GetTables": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTables",
                "description": "Grants permission to retrieve the tables in a database",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetTags": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource",
                "description": "Grants permission to retrieve all tags associated with a resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "crawler": {
                        "required": false
                    },
                    "devendpoint": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "trigger": {
                        "required": false
                    },
                    "workflow": {
                        "required": false
                    }
                }
            },
            "GetTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTrigger",
                "description": "Grants permission to retrieve a trigger",
                "accessLevel": "Read"
            },
            "GetTriggers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTriggers",
                "description": "Grants permission to retrieve the triggers associated with a job",
                "accessLevel": "Read"
            },
            "GetUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunction",
                "description": "Grants permission to retrieve a function definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "GetUserDefinedFunctions": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunctions",
                "description": "Grants permission to retrieve multiple function definitions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "GetWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflow",
                "description": "Grants permission to retrieve a workflow",
                "accessLevel": "Read"
            },
            "GetWorkflowRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRun",
                "description": "Grants permission to retrieve a workflow run",
                "accessLevel": "Read"
            },
            "GetWorkflowRunProperties": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRunProperties",
                "description": "Grants permission to retrieve workflow run properties",
                "accessLevel": "Read"
            },
            "GetWorkflowRuns": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRuns",
                "description": "Grants permission to retrieve all runs of a workflow",
                "accessLevel": "Read"
            },
            "ImportCatalogToGlue": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-ImportCatalogToGlue",
                "description": "Grants permission to import an Athena data catalog into AWS Glue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "ListCrawlers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-ListCrawlers",
                "description": "Grants permission to retrieve all crawlers",
                "accessLevel": "List"
            },
            "ListDevEndpoints": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-ListDevEndpoints",
                "description": "Grants permission to retrieve all development endpoints",
                "accessLevel": "List"
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-ListJobs",
                "description": "Grants permission to retrieve all current jobs",
                "accessLevel": "List"
            },
            "ListMLTransforms": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-ListMLTransforms",
                "description": "Grants permission to retrieve all ML Transforms",
                "accessLevel": "List"
            },
            "ListTriggers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-ListTriggers",
                "description": "Grants permission to retrieve all triggers",
                "accessLevel": "List"
            },
            "ListWorkflows": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-ListWorkflows",
                "description": "Grants permission to retrieve all workflows",
                "accessLevel": "List"
            },
            "PutDataCatalogEncryptionSettings": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutDataCatalogEncryptionSettings",
                "description": "Grants permission to update catalog encryption settings",
                "accessLevel": "Write"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutResourcePolicy",
                "description": "Grants permission to update a resource policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "PutWorkflowRunProperties": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-PutWorkflowRunProperties",
                "description": "Grants permission to update workflow run properties",
                "accessLevel": "Write"
            },
            "ResetJobBookmark": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-ResetJobBookmark",
                "description": "Grants permission to reset a job bookmark",
                "accessLevel": "Write"
            },
            "SearchTables": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-SearchTables",
                "description": "Grants permission to retrieve the tables in the catalog",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "StartCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler",
                "description": "Grants permission to start a crawler",
                "accessLevel": "Write"
            },
            "StartCrawlerSchedule": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StartCrawlerSchedule",
                "description": "Grants permission to change the schedule state of a crawler to SCHEDULED",
                "accessLevel": "Write"
            },
            "StartExportLabelsTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartExportLabelsTaskRun",
                "description": "Grants permission to start an Export Labels ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartImportLabelsTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartImportLabelsTaskRun",
                "description": "Grants permission to start an Import Labels ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartJobRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun",
                "description": "Grants permission to start running a job",
                "accessLevel": "Write"
            },
            "StartMLEvaluationTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLEvaluationTaskRun",
                "description": "Grants permission to start an Evaluation ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartMLLabelingSetGenerationTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLLabelingSetGenerationTaskRun",
                "description": "Grants permission to start a Labeling Set Generation ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StartTrigger",
                "description": "Grants permission to start a trigger",
                "accessLevel": "Write"
            },
            "StartWorkflowRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-StartWorkflowRun",
                "description": "Grants permission to start running a workflow",
                "accessLevel": "Write"
            },
            "StopCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StopCrawler",
                "description": "Grants permission to stop a running crawler",
                "accessLevel": "Write"
            },
            "StopCrawlerSchedule": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StopCrawlerSchedule",
                "description": "Grants permission to set the schedule state of a crawler to NOT_SCHEDULED",
                "accessLevel": "Write"
            },
            "StopTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StopTrigger",
                "description": "Grants permission to stop a trigger",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-TagResource",
                "description": "Grants permission to add tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "crawler": {
                        "required": false
                    },
                    "devendpoint": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "trigger": {
                        "required": false
                    },
                    "workflow": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource",
                "description": "Grants permission to remove tags associated with a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "crawler": {
                        "required": false
                    },
                    "devendpoint": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "trigger": {
                        "required": false
                    },
                    "workflow": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-UpdateClassifier",
                "description": "Grants permission to update a classifier",
                "accessLevel": "Write"
            },
            "UpdateConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-UpdateConnection",
                "description": "Grants permission to update a connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "UpdateCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-UpdateCrawler",
                "description": "Grants permission to update a crawler",
                "accessLevel": "Write"
            },
            "UpdateCrawlerSchedule": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-UpdateCrawlerSchedule",
                "description": "Grants permission to update the schedule of a crawler",
                "accessLevel": "Write"
            },
            "UpdateDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-UpdateDatabase",
                "description": "Grants permission to update a database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "UpdateDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-UpdateDevEndpoint",
                "description": "Grants permission to update a development endpoint",
                "accessLevel": "Write"
            },
            "UpdateJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-UpdateJob",
                "description": "Grants permission to update a job",
                "accessLevel": "Write"
            },
            "UpdateMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-UpdateMLTransform",
                "description": "Grants permission to update an ML Transform",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "UpdatePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-UpdatePartition",
                "description": "Grants permission to update a partition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-UpdateTable",
                "description": "Grants permission to update a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-UpdateTrigger",
                "description": "Grants permission to update a trigger",
                "accessLevel": "Write"
            },
            "UpdateUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-UpdateUserDefinedFunction",
                "description": "Grants permission to update a function definition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "UpdateWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-UpdateWorkflow",
                "description": "Grants permission to update a workflow",
                "accessLevel": "Write"
            },
            "UseMLTransforms": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html",
                "description": "Grants permission to use an ML Transform from within a Glue ETL Script",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "catalog": {
                "name": "catalog",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:catalog",
                "conditionKeys": []
            },
            "database": {
                "name": "database",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:database/${DatabaseName}",
                "conditionKeys": []
            },
            "table": {
                "name": "table",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:table/${DatabaseName}/${TableName}",
                "conditionKeys": []
            },
            "tableversion": {
                "name": "tableversion",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:tableVersion/${DatabaseName}/${TableName}/${TableVersionName}",
                "conditionKeys": []
            },
            "connection": {
                "name": "connection",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:connection/${ConnectionName}",
                "conditionKeys": []
            },
            "userdefinedfunction": {
                "name": "userdefinedfunction",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:userDefinedFunction/${DatabaseName}/${UserDefinedFunctionName}",
                "conditionKeys": []
            },
            "devendpoint": {
                "name": "devendpoint",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:devendpoint/${DevEndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:job/${JobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "trigger": {
                "name": "trigger",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:trigger/${TriggerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "crawler": {
                "name": "crawler",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:crawler/${CrawlerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "workflow": {
                "name": "workflow",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:workflow/${WorkflowName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "mlTransform": {
                "name": "mlTransform",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:mlTransform/${TransformId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchCreatePartition
     */
    toBatchCreatePartition() {
        this.add('glue:BatchCreatePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-BatchDeleteConnection
     */
    toBatchDeleteConnection() {
        this.add('glue:BatchDeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchDeletePartition
     */
    toBatchDeletePartition() {
        this.add('glue:BatchDeletePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more tables
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-BatchDeleteTable
     */
    toBatchDeleteTable() {
        this.add('glue:BatchDeleteTable');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    toBatchDeleteTableVersion() {
        this.add('glue:BatchDeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve one or more crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-BatchGetCrawlers
     */
    toBatchGetCrawlers() {
        this.add('glue:BatchGetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-BatchGetDevEndpoints
     */
    toBatchGetDevEndpoints() {
        this.add('glue:BatchGetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve one or more jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-BatchGetJobs
     */
    toBatchGetJobs() {
        this.add('glue:BatchGetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve one or more partitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchGetPartition
     */
    toBatchGetPartition() {
        this.add('glue:BatchGetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve one or more triggers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-BatchGetTriggers
     */
    toBatchGetTriggers() {
        this.add('glue:BatchGetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more workflows
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-BatchGetWorkflows
     */
    toBatchGetWorkflows() {
        this.add('glue:BatchGetWorkflows');
        return this;
    }
    /**
     * Grants permission to stop one or more job runs for a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-BatchStopStartJobRun
     */
    toBatchStopJobRun() {
        this.add('glue:BatchStopJobRun');
        return this;
    }
    /**
     * Grants permission to stop a running ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CancelMLTaskRun
     */
    toCancelMLTaskRun() {
        this.add('glue:CancelMLTaskRun');
        return this;
    }
    /**
     * Grants permission to create a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-CreateClassifier
     */
    toCreateClassifier() {
        this.add('glue:CreateClassifier');
        return this;
    }
    /**
     * Grants permission to create a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-CreateConnection
     */
    toCreateConnection() {
        this.add('glue:CreateConnection');
        return this;
    }
    /**
     * Grants permission to create a crawler
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-CreateCrawler
     */
    toCreateCrawler() {
        this.add('glue:CreateCrawler');
        return this;
    }
    /**
     * Grants permission to create a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-CreateDatabase
     */
    toCreateDatabase() {
        this.add('glue:CreateDatabase');
        return this;
    }
    /**
     * Grants permission to create a development endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-CreateDevEndpoint
     */
    toCreateDevEndpoint() {
        this.add('glue:CreateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to create a job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-CreateJob
     */
    toCreateJob() {
        this.add('glue:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CreateMLTransform
     */
    toCreateMLTransform() {
        this.add('glue:CreateMLTransform');
        return this;
    }
    /**
     * Grants permission to create a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-CreatePartition
     */
    toCreatePartition() {
        this.add('glue:CreatePartition');
        return this;
    }
    /**
     * Grants permission to create a script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-CreateScript
     */
    toCreateScript() {
        this.add('glue:CreateScript');
        return this;
    }
    /**
     * Grants permission to create a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-CreateSecurityConfiguration
     */
    toCreateSecurityConfiguration() {
        this.add('glue:CreateSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to create a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-CreateTable
     */
    toCreateTable() {
        this.add('glue:CreateTable');
        return this;
    }
    /**
     * Grants permission to create a trigger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-CreateTrigger
     */
    toCreateTrigger() {
        this.add('glue:CreateTrigger');
        return this;
    }
    /**
     * Grants permission to create a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-CreateUserDefinedFunction
     */
    toCreateUserDefinedFunction() {
        this.add('glue:CreateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to create a workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-CreateWorkflow
     */
    toCreateWorkflow() {
        this.add('glue:CreateWorkflow');
        return this;
    }
    /**
     * Grants permission to delete a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-DeleteClassifier
     */
    toDeleteClassifier() {
        this.add('glue:DeleteClassifier');
        return this;
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-DeleteConnection
     */
    toDeleteConnection() {
        this.add('glue:DeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-DeleteCrawler
     */
    toDeleteCrawler() {
        this.add('glue:DeleteCrawler');
        return this;
    }
    /**
     * Grants permission to delete a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-DeleteDatabase
     */
    toDeleteDatabase() {
        this.add('glue:DeleteDatabase');
        return this;
    }
    /**
     * Grants permission to delete a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-DeleteDevEndpoint
     */
    toDeleteDevEndpoint() {
        this.add('glue:DeleteDevEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-DeleteJob
     */
    toDeleteJob() {
        this.add('glue:DeleteJob');
        return this;
    }
    /**
     * Grants permission to delete an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-DeleteMLTransform
     */
    toDeleteMLTransform() {
        this.add('glue:DeleteMLTransform');
        return this;
    }
    /**
     * Grants permission to delete a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-DeletePartition
     */
    toDeletePartition() {
        this.add('glue:DeletePartition');
        return this;
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteResourcePolicy
     */
    toDeleteResourcePolicy() {
        this.add('glue:DeleteResourcePolicy');
        return this;
    }
    /**
     * Grants permission to delete a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteSecurityConfiguration
     */
    toDeleteSecurityConfiguration() {
        this.add('glue:DeleteSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTable
     */
    toDeleteTable() {
        this.add('glue:DeleteTable');
        return this;
    }
    /**
     * Grants permission to delete a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    toDeleteTableVersion() {
        this.add('glue:DeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to delete a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-DeleteTrigger
     */
    toDeleteTrigger() {
        this.add('glue:DeleteTrigger');
        return this;
    }
    /**
     * Grants permission to delete a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-DeleteUserDefinedFunction
     */
    toDeleteUserDefinedFunction() {
        this.add('glue:DeleteUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-DeleteWorkflow
     */
    toDeleteWorkflow() {
        this.add('glue:DeleteWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve the catalog import status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-GetCatalogImportStatus
     */
    toGetCatalogImportStatus() {
        this.add('glue:GetCatalogImportStatus');
        return this;
    }
    /**
     * Grants permission to retrieve a classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifier
     */
    toGetClassifier() {
        this.add('glue:GetClassifier');
        return this;
    }
    /**
     * Grants permission to list all classifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifiers
     */
    toGetClassifiers() {
        this.add('glue:GetClassifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnection
     */
    toGetConnection() {
        this.add('glue:GetConnection');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnections
     */
    toGetConnections() {
        this.add('glue:GetConnections');
        return this;
    }
    /**
     * Grants permission to retrieve a crawler
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawler
     */
    toGetCrawler() {
        this.add('glue:GetCrawler');
        return this;
    }
    /**
     * Grants permission to retrieve metrics about crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlerMetrics
     */
    toGetCrawlerMetrics() {
        this.add('glue:GetCrawlerMetrics');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlers
     */
    toGetCrawlers() {
        this.add('glue:GetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve catalog encryption settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetDataCatalogEncryptionSettings
     */
    toGetDataCatalogEncryptionSettings() {
        this.add('glue:GetDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabase
     */
    toGetDatabase() {
        this.add('glue:GetDatabase');
        return this;
    }
    /**
     * Grants permission to retrieve all databases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabases
     */
    toGetDatabases() {
        this.add('glue:GetDatabases');
        return this;
    }
    /**
     * Grants permission to transform a script into a directed acyclic graph (DAG)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetDataflowGraph
     */
    toGetDataflowGraph() {
        this.add('glue:GetDataflowGraph');
        return this;
    }
    /**
     * Grants permission to retrieve a development endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoint
     */
    toGetDevEndpoint() {
        this.add('glue:GetDevEndpoint');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoints
     */
    toGetDevEndpoints() {
        this.add('glue:GetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJob
     */
    toGetJob() {
        this.add('glue:GetJob');
        return this;
    }
    /**
     * Grants permission to retrieve a job bookmark
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-job-GetJobBookmark
     */
    toGetJobBookmark() {
        this.add('glue:GetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve a job run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRun
     */
    toGetJobRun() {
        this.add('glue:GetJobRun');
        return this;
    }
    /**
     * Grants permission to retrieve all job runs of a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRuns
     */
    toGetJobRuns() {
        this.add('glue:GetJobRuns');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJobs
     */
    toGetJobs() {
        this.add('glue:GetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Task Run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRun
     */
    toGetMLTaskRun() {
        this.add('glue:GetMLTaskRun');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Task Runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRuns
     */
    toGetMLTaskRuns() {
        this.add('glue:GetMLTaskRuns');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Transform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransform
     */
    toGetMLTransform() {
        this.add('glue:GetMLTransform');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransforms
     */
    toGetMLTransforms() {
        this.add('glue:GetMLTransforms');
        return this;
    }
    /**
     * Grants permission to create a mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetMapping
     */
    toGetMapping() {
        this.add('glue:GetMapping');
        return this;
    }
    /**
     * Grants permission to retrieve a partition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartition
     */
    toGetPartition() {
        this.add('glue:GetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve the partitions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartitions
     */
    toGetPartitions() {
        this.add('glue:GetPartitions');
        return this;
    }
    /**
     * Grants permission to retrieve a mapping for a script
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetPlan
     */
    toGetPlan() {
        this.add('glue:GetPlan');
        return this;
    }
    /**
     * Grants permission to retrieve a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetResourcePolicy
     */
    toGetResourcePolicy() {
        this.add('glue:GetResourcePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a security configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfiguration
     */
    toGetSecurityConfiguration() {
        this.add('glue:GetSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve one or more security configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfigurations
     */
    toGetSecurityConfigurations() {
        this.add('glue:GetSecurityConfigurations');
        return this;
    }
    /**
     * Grants permission to retrieve a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTable
     */
    toGetTable() {
        this.add('glue:GetTable');
        return this;
    }
    /**
     * Grants permission to retrieve a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersion
     */
    toGetTableVersion() {
        this.add('glue:GetTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersions
     */
    toGetTableVersions() {
        this.add('glue:GetTableVersions');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTables
     */
    toGetTables() {
        this.add('glue:GetTables');
        return this;
    }
    /**
     * Grants permission to retrieve all tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    toGetTags() {
        this.add('glue:GetTags');
        return this;
    }
    /**
     * Grants permission to retrieve a trigger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTrigger
     */
    toGetTrigger() {
        this.add('glue:GetTrigger');
        return this;
    }
    /**
     * Grants permission to retrieve the triggers associated with a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTriggers
     */
    toGetTriggers() {
        this.add('glue:GetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve a function definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunction
     */
    toGetUserDefinedFunction() {
        this.add('glue:GetUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to retrieve multiple function definitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunctions
     */
    toGetUserDefinedFunctions() {
        this.add('glue:GetUserDefinedFunctions');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflow
     */
    toGetWorkflow() {
        this.add('glue:GetWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRun
     */
    toGetWorkflowRun() {
        this.add('glue:GetWorkflowRun');
        return this;
    }
    /**
     * Grants permission to retrieve workflow run properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRunProperties
     */
    toGetWorkflowRunProperties() {
        this.add('glue:GetWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to retrieve all runs of a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRuns
     */
    toGetWorkflowRuns() {
        this.add('glue:GetWorkflowRuns');
        return this;
    }
    /**
     * Grants permission to import an Athena data catalog into AWS Glue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-ImportCatalogToGlue
     */
    toImportCatalogToGlue() {
        this.add('glue:ImportCatalogToGlue');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-ListCrawlers
     */
    toListCrawlers() {
        this.add('glue:ListCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-ListDevEndpoints
     */
    toListDevEndpoints() {
        this.add('glue:ListDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-ListJobs
     */
    toListJobs() {
        this.add('glue:ListJobs');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-ListMLTransforms
     */
    toListMLTransforms() {
        this.add('glue:ListMLTransforms');
        return this;
    }
    /**
     * Grants permission to retrieve all triggers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-ListTriggers
     */
    toListTriggers() {
        this.add('glue:ListTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve all workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-ListWorkflows
     */
    toListWorkflows() {
        this.add('glue:ListWorkflows');
        return this;
    }
    /**
     * Grants permission to update catalog encryption settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutDataCatalogEncryptionSettings
     */
    toPutDataCatalogEncryptionSettings() {
        this.add('glue:PutDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to update a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutResourcePolicy
     */
    toPutResourcePolicy() {
        this.add('glue:PutResourcePolicy');
        return this;
    }
    /**
     * Grants permission to update workflow run properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-PutWorkflowRunProperties
     */
    toPutWorkflowRunProperties() {
        this.add('glue:PutWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to reset a job bookmark
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-ResetJobBookmark
     */
    toResetJobBookmark() {
        this.add('glue:ResetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in the catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-SearchTables
     */
    toSearchTables() {
        this.add('glue:SearchTables');
        return this;
    }
    /**
     * Grants permission to start a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler
     */
    toStartCrawler() {
        this.add('glue:StartCrawler');
        return this;
    }
    /**
     * Grants permission to change the schedule state of a crawler to SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StartCrawlerSchedule
     */
    toStartCrawlerSchedule() {
        this.add('glue:StartCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to start an Export Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartExportLabelsTaskRun
     */
    toStartExportLabelsTaskRun() {
        this.add('glue:StartExportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start an Import Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartImportLabelsTaskRun
     */
    toStartImportLabelsTaskRun() {
        this.add('glue:StartImportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start running a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun
     */
    toStartJobRun() {
        this.add('glue:StartJobRun');
        return this;
    }
    /**
     * Grants permission to start an Evaluation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLEvaluationTaskRun
     */
    toStartMLEvaluationTaskRun() {
        this.add('glue:StartMLEvaluationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a Labeling Set Generation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLLabelingSetGenerationTaskRun
     */
    toStartMLLabelingSetGenerationTaskRun() {
        this.add('glue:StartMLLabelingSetGenerationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StartTrigger
     */
    toStartTrigger() {
        this.add('glue:StartTrigger');
        return this;
    }
    /**
     * Grants permission to start running a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-StartWorkflowRun
     */
    toStartWorkflowRun() {
        this.add('glue:StartWorkflowRun');
        return this;
    }
    /**
     * Grants permission to stop a running crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StopCrawler
     */
    toStopCrawler() {
        this.add('glue:StopCrawler');
        return this;
    }
    /**
     * Grants permission to set the schedule state of a crawler to NOT_SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StopCrawlerSchedule
     */
    toStopCrawlerSchedule() {
        this.add('glue:StopCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to stop a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StopTrigger
     */
    toStopTrigger() {
        this.add('glue:StopTrigger');
        return this;
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-TagResource
     */
    toTagResource() {
        this.add('glue:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags associated with a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    toUntagResource() {
        this.add('glue:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-UpdateClassifier
     */
    toUpdateClassifier() {
        this.add('glue:UpdateClassifier');
        return this;
    }
    /**
     * Grants permission to update a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-UpdateConnection
     */
    toUpdateConnection() {
        this.add('glue:UpdateConnection');
        return this;
    }
    /**
     * Grants permission to update a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-UpdateCrawler
     */
    toUpdateCrawler() {
        this.add('glue:UpdateCrawler');
        return this;
    }
    /**
     * Grants permission to update the schedule of a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-UpdateCrawlerSchedule
     */
    toUpdateCrawlerSchedule() {
        this.add('glue:UpdateCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to update a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-UpdateDatabase
     */
    toUpdateDatabase() {
        this.add('glue:UpdateDatabase');
        return this;
    }
    /**
     * Grants permission to update a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-UpdateDevEndpoint
     */
    toUpdateDevEndpoint() {
        this.add('glue:UpdateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to update a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-UpdateJob
     */
    toUpdateJob() {
        this.add('glue:UpdateJob');
        return this;
    }
    /**
     * Grants permission to update an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-UpdateMLTransform
     */
    toUpdateMLTransform() {
        this.add('glue:UpdateMLTransform');
        return this;
    }
    /**
     * Grants permission to update a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-UpdatePartition
     */
    toUpdatePartition() {
        this.add('glue:UpdatePartition');
        return this;
    }
    /**
     * Grants permission to update a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-UpdateTable
     */
    toUpdateTable() {
        this.add('glue:UpdateTable');
        return this;
    }
    /**
     * Grants permission to update a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-UpdateTrigger
     */
    toUpdateTrigger() {
        this.add('glue:UpdateTrigger');
        return this;
    }
    /**
     * Grants permission to update a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-UpdateUserDefinedFunction
     */
    toUpdateUserDefinedFunction() {
        this.add('glue:UpdateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to update a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-UpdateWorkflow
     */
    toUpdateWorkflow() {
        this.add('glue:UpdateWorkflow');
        return this;
    }
    /**
     * Grants permission to use an ML Transform from within a Glue ETL Script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html
     */
    toUseMLTransforms() {
        this.add('glue:UseMLTransforms');
        return this;
    }
    /**
     * Adds a resource of type catalog to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCatalog(account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:catalog';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatabase(databaseName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:database/${DatabaseName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(databaseName, tableName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:table/${DatabaseName}/${TableName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tableversion to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param tableVersionName - Identifier for the tableVersionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTableversion(databaseName, tableName, tableVersionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:tableVersion/${DatabaseName}/${TableName}/${TableVersionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${TableVersionName}', tableVersionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param connectionName - Identifier for the connectionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnection(connectionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:connection/${ConnectionName}';
        arn = arn.replace('${ConnectionName}', connectionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type userdefinedfunction to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param userDefinedFunctionName - Identifier for the userDefinedFunctionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUserdefinedfunction(databaseName, userDefinedFunctionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:userDefinedFunction/${DatabaseName}/${UserDefinedFunctionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${UserDefinedFunctionName}', userDefinedFunctionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type devendpoint to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param devEndpointName - Identifier for the devEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevendpoint(devEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:devendpoint/${DevEndpointName}';
        arn = arn.replace('${DevEndpointName}', devEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:job/${JobName}';
        arn = arn.replace('${JobName}', jobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trigger to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param triggerName - Identifier for the triggerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrigger(triggerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:trigger/${TriggerName}';
        arn = arn.replace('${TriggerName}', triggerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type crawler to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param crawlerName - Identifier for the crawlerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCrawler(crawlerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:crawler/${CrawlerName}';
        arn = arn.replace('${CrawlerName}', crawlerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflow(workflowName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:workflow/${WorkflowName}';
        arn = arn.replace('${WorkflowName}', workflowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mlTransform to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param transformId - Identifier for the transformId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMlTransform(transformId, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:mlTransform/${TransformId}';
        arn = arn.replace('${TransformId}', transformId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Glue = Glue;
//# sourceMappingURL=data:application/json;base64,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