"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iot = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iot extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iot';
        this.actionList = {
            "AcceptCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AcceptCertificateTransfer.html",
                "description": "Accepts a pending certificate transfer.",
                "accessLevel": "Write"
            },
            "AddThingToBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToBillingGroup.html",
                "description": "Adds a thing to the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "AddThingToThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToThingGroup.html",
                "description": "Adds a thing to the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "AssociateTargetsWithJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AssociateTargetsWithJob.html",
                "description": "Associates a group with a continuous job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPolicy.html",
                "description": "Attaches a policy to the specified target.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "AttachPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPrincipalPolicy.html",
                "description": "Attaches the specified policy to the specified principal (certificate or other credential).",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "AttachSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachSecurityProfile.html",
                "description": "Associates a Device Defender security profile with a thing group or with this account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "AttachThingPrincipal": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachThingPrincipal.html",
                "description": "Attaches the specified principal to the specified thing.",
                "accessLevel": "Write"
            },
            "CancelAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditMitigationActionsTask.html",
                "description": "Cancels a mitigation action task that is in progress.",
                "accessLevel": "Write"
            },
            "CancelAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditTask.html",
                "description": "Cancels an audit that is in progress. The audit can be either scheduled or on-demand.",
                "accessLevel": "Write"
            },
            "CancelCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelCertificateTransfer.html",
                "description": "Cancels a pending transfer for the specified certificate.",
                "accessLevel": "Write"
            },
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJob.html",
                "description": "Cancels a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "CancelJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJobExecution.html",
                "description": "Cancels a job execution on a particular device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "ClearDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ClearDefaultAuthorizer.html",
                "description": "Clears the default authorizer.",
                "accessLevel": "Write"
            },
            "CloseTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CloseTunnel.html",
                "description": "Closes a tunnel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "tunnel": {
                        "required": true
                    }
                },
                "conditions": [
                    "iot:Delete"
                ]
            },
            "Connect": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Connect as the specified client",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client": {
                        "required": true
                    }
                }
            },
            "CreateAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateAuthorizer.html",
                "description": "Creates an authorizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "CreateBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateBillingGroup.html",
                "description": "Creates a billing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCertificateFromCsr": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html",
                "description": "Creates an X.509 certificate using the specified certificate signing request.",
                "accessLevel": "Write"
            },
            "CreateDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDimension.html",
                "description": "Defines a dimension that can be used to to limit the scope of a metric used in a security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDynamicThingGroup.html",
                "description": "Creates a Dynamic Thing Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateJob.html",
                "description": "Creates a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeysAndCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html",
                "description": "Creates a 2048 bit RSA key pair and issues an X.509 certificate using the issued public key.",
                "accessLevel": "Write"
            },
            "CreateMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateMitigationAction.html",
                "description": "Defines an action that can be applied to audit findings by using StartAuditMitigationActionsTask.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateOTAUpdate.html",
                "description": "Creates an OTA update job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicy.html",
                "description": "Creates an AWS IoT policy.",
                "accessLevel": "Write"
            },
            "CreatePolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicyVersion.html",
                "description": "Creates a new version of the specified AWS IoT policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningClaim": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningClaim.html",
                "description": "Creates a provisioning claim.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplate.html",
                "description": "Creates a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplateVersion.html",
                "description": "Creates a new version of a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateRoleAlias.html",
                "description": "Creates a role alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "CreateScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateScheduledAudit.html",
                "description": "Creates a scheduled audit that is run at a specified time interval.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateSecurityProfile.html",
                "description": "Creates a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateStream.html",
                "description": "Creates a new AWS IoT stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThing.html",
                "description": "Creates a thing in the thing registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "billinggroup": {
                        "required": false
                    }
                }
            },
            "CreateThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingGroup.html",
                "description": "Creates a thing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingType.html",
                "description": "Creates a new thing type.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateTopicRule.html",
                "description": "Creates a rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAccountAuditConfiguration.html",
                "description": "Deletes the audit configuration associated with the account.",
                "accessLevel": "Write"
            },
            "DeleteAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAuthorizer.html",
                "description": "Deletes the specified authorizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "DeleteBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteBillingGroup.html",
                "description": "Deletes the specified billing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCACertificate.html",
                "description": "Deletes a registered CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCertificate.html",
                "description": "Deletes the specified certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "DeleteDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDimension.html",
                "description": "Removes the specified dimension from your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "DeleteDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDynamicThingGroup.html",
                "description": "Deletes the specified Dynamic Thing Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJob.html",
                "description": "Deletes a job and its related job executions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "DeleteJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJobExecution.html",
                "description": "Deletes a job execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteMitigationAction.html",
                "description": "Deletes a defined mitigation action from your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "DeleteOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteOTAUpdate.html",
                "description": "Deletes an OTA update job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicy.html",
                "description": "Deletes the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeletePolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicyVersion.html",
                "description": "Deletes the specified version of the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeleteProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplate.html",
                "description": "Deletes a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DeleteProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplateVersion.html",
                "description": "Deletes a fleet provisioning template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DeleteRegistrationCode": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRegistrationCode.html",
                "description": "Deletes a CA certificate registration code.",
                "accessLevel": "Write"
            },
            "DeleteRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRoleAlias.html",
                "description": "Deletes the specified role alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "DeleteScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteScheduledAudit.html",
                "description": "Deletes a scheduled audit.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "DeleteSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteSecurityProfile.html",
                "description": "Deletes a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "DeleteStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteStream.html",
                "description": "Deletes a specified stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeleteThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html",
                "description": "Deletes the specified thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingGroup.html",
                "description": "Deletes the specified thing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Deletes the specified thing shadow.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingType.html",
                "description": "Deletes the specified thing type.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DeleteTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteTopicRule.html",
                "description": "Deletes the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DeleteV2LoggingLevel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteV2LoggingLevel.html",
                "description": "Deletes the specified v2 logging level.",
                "accessLevel": "Write"
            },
            "DeprecateThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeprecateThingType.html",
                "description": "Deprecates the specified thing type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAccountAuditConfiguration.html",
                "description": "Gets information about audit configurations for the account.",
                "accessLevel": "Read"
            },
            "DescribeAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditMitigationActionsTask.html",
                "description": "Gets information about an audit mitigation task that is used to apply mitigation actions to a set of audit findings.",
                "accessLevel": "Read"
            },
            "DescribeAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditTask.html",
                "description": "Gets information about a Device Defender audit.",
                "accessLevel": "Read"
            },
            "DescribeAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuthorizer.html",
                "description": "Describes an authorizer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "DescribeBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeBillingGroup.html",
                "description": "Gets information about the specified billing group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "DescribeCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCACertificate.html",
                "description": "Describes a registered CA certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCertificate.html",
                "description": "Gets information about the specified certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "DescribeDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDefaultAuthorizer.html",
                "description": "Describes the default authorizer.",
                "accessLevel": "Read"
            },
            "DescribeDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDimension.html",
                "description": "Provides details about a dimension that is defined in your AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEndpoint.html",
                "description": "Returns a unique endpoint specific to the AWS account making the call.",
                "accessLevel": "Read"
            },
            "DescribeEventConfigurations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEventConfigurations.html",
                "description": "Returns account event configurations.",
                "accessLevel": "Read"
            },
            "DescribeIndex": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeIndex.html",
                "description": "Gets information about the specified index.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJob.html",
                "description": "Describes a job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "DescribeJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJobExecution.html",
                "description": "Describes a job execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "DescribeMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeMitigationAction.html",
                "description": "Gets information about a mitigation action.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplate.html",
                "description": "Returns information about a fleet provisioning template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplateVersion.html",
                "description": "Returns information about a fleet provisioning template version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DescribeRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeRoleAlias.html",
                "description": "Describes a role alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "DescribeScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeScheduledAudit.html",
                "description": "Gets information about a scheduled audit.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "DescribeSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeSecurityProfile.html",
                "description": "Gets information about a Device Defender security profile.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    }
                }
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeStream.html",
                "description": "Gets information about the specified stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThing.html",
                "description": "Gets information about the specified thing.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DescribeThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingGroup.html",
                "description": "Gets information about the specified thing group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "DescribeThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingRegistrationTask.html",
                "description": "Gets information about the bulk thing registration task.",
                "accessLevel": "Read"
            },
            "DescribeThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingType.html",
                "description": "Gets information about the specified thing type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DescribeTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeTunnel.html",
                "description": "Describes a tunnel.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "tunnel": {
                        "required": true
                    }
                }
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPolicy.html",
                "description": "Detaches a policy from the specified target.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "DetachPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPrincipalPolicy.html",
                "description": "Removes the specified policy from the specified certificate.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "DetachSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachSecurityProfile.html",
                "description": "Disassociates a Device Defender security profile from a thing group or from this account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "DetachThingPrincipal": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachThingPrincipal.html",
                "description": "Detaches the specified principal from the specified thing.",
                "accessLevel": "Write"
            },
            "DisableTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DisableTopicRule.html",
                "description": "Disables the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "EnableTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_EnableTopicRule.html",
                "description": "Enables the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetCardinality": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetCardinality.html",
                "description": "Get cardinality for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetEffectivePolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetEffectivePolicies.html",
                "description": "Gets effective policies.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "GetIndexingConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetIndexingConfiguration.html",
                "description": "Gets current fleet indexing configuration",
                "accessLevel": "Read"
            },
            "GetJobDocument": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetJobDocument.html",
                "description": "Gets a job document.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "GetLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetLoggingOptions.html",
                "description": "Gets the logging options.",
                "accessLevel": "Read"
            },
            "GetOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetOTAUpdate.html",
                "description": "Gets the information about the OTA update job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                }
            },
            "GetPendingJobExecutions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPendingJobExecutions.html",
                "description": "Gets the list of all jobs for a thing that are not in a terminal state.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetPercentiles": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPercentiles.html",
                "description": "Get percentiles for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicy.html",
                "description": "Gets information about the specified policy with the policy document of the default version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetPolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicyVersion.html",
                "description": "Gets information about the specified policy version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetRegistrationCode": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetRegistrationCode.html",
                "description": "Gets a registration code used to register a CA certificate with AWS IoT.",
                "accessLevel": "Read"
            },
            "GetStatistics": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetStatistics.html",
                "description": "Get statistics for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Gets the thing shadow.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetTopicRule.html",
                "description": "Gets information about the specified rule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetV2LoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetV2LoggingOptions.html",
                "description": "Gets v2 logging options.",
                "accessLevel": "Read"
            },
            "ListActiveViolations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListActiveViolations.html",
                "description": "Lists the active violations for a given Device Defender security profile or Thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "ListAttachedPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAttachedPolicies.html",
                "description": "Lists the policies attached to the specified thing group.",
                "accessLevel": "List"
            },
            "ListAuditFindings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditFindings.html",
                "description": "Lists the findings (results) of a Device Defender audit or of the audits performed during a specified time period.",
                "accessLevel": "List"
            },
            "ListAuditMitigationActionsExecutions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsExecutions.html",
                "description": "Gets the status of audit mitigation action tasks that were executed.",
                "accessLevel": "List"
            },
            "ListAuditMitigationActionsTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsTasks.html",
                "description": "Gets a list of audit mitigation action tasks that match the specified filters.",
                "accessLevel": "List"
            },
            "ListAuditTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditTasks.html",
                "description": "Lists the Device Defender audits that have been performed during a given time period.",
                "accessLevel": "List"
            },
            "ListAuthorizers": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuthorizers.html",
                "description": "Lists the authorizers registered in your account.",
                "accessLevel": "List"
            },
            "ListBillingGroups": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListBillingGroups.html",
                "description": "Lists all billing groups.",
                "accessLevel": "List"
            },
            "ListCACertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCACertificates.html",
                "description": "Lists the CA certificates registered for your AWS account.",
                "accessLevel": "List"
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificates.html",
                "description": "Lists your certificates.",
                "accessLevel": "List"
            },
            "ListCertificatesByCA": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificatesByCA.html",
                "description": "List the device certificates signed by the specified CA certificate.",
                "accessLevel": "List"
            },
            "ListDimensions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListDimensions.html",
                "description": "Lists the dimensions that are defined for your AWS account.",
                "accessLevel": "List"
            },
            "ListIndices": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListIndices.html",
                "description": "Lists all indices for fleet index",
                "accessLevel": "List"
            },
            "ListJobExecutionsForJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForJob.html",
                "description": "Lists the job executions for a job.",
                "accessLevel": "List",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "ListJobExecutionsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForThing.html",
                "description": "Lists the job executions for the specified thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobs.html",
                "description": "Lists jobs.",
                "accessLevel": "List"
            },
            "ListMitigationActions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListMitigationActions.html",
                "description": "Gets a list of all mitigation actions that match the specified filter criteria.",
                "accessLevel": "List"
            },
            "ListNamedShadowsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListNamedShadowsForThing.html",
                "description": "Lists all named shadows for a given thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListOTAUpdates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListOTAUpdates.html",
                "description": "Lists OTA update jobs in the account.",
                "accessLevel": "List"
            },
            "ListOutgoingCertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListOutgoingCertificates.html",
                "description": "Lists certificates that are being transfered but not yet accepted.",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicies.html",
                "description": "Lists your policies.",
                "accessLevel": "List"
            },
            "ListPolicyPrincipals": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyPrincipals.html",
                "description": "Lists the principals associated with the specified policy.",
                "accessLevel": "List"
            },
            "ListPolicyVersions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyVersions.html",
                "description": "Lists the versions of the specified policy, and identifies the default version.",
                "accessLevel": "List"
            },
            "ListPrincipalPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalPolicies.html",
                "description": "Lists the policies attached to the specified principal. If you use an Amazon Cognito identity, the ID needs to be in Amazon Cognito Identity format.",
                "accessLevel": "List"
            },
            "ListPrincipalThings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalThings.html",
                "description": "Lists the things associated with the specified principal.",
                "accessLevel": "List"
            },
            "ListProvisioningTemplateVersions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplateVersions.html",
                "description": "A list of fleet provisioning template versions.",
                "accessLevel": "List",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "ListProvisioningTemplates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplates.html",
                "description": "Lists the fleet provisioning templates in your AWS account.",
                "accessLevel": "List"
            },
            "ListRoleAliases": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListRoleAliases.html",
                "description": "Lists role aliases.",
                "accessLevel": "List"
            },
            "ListScheduledAudits": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListScheduledAudits.html",
                "description": "Lists all of your scheduled audits.",
                "accessLevel": "List"
            },
            "ListSecurityProfiles": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfiles.html",
                "description": "Lists the Device Defender security profiles you have created.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dimension": {
                        "required": false
                    }
                }
            },
            "ListSecurityProfilesForTarget": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfilesForTarget.html",
                "description": "Lists the Device Defender security profiles attached to a target.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListStreams.html",
                "description": "Lists the streams in your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTagsForResource.html",
                "description": "Lists all tags for a given resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                }
            },
            "ListTargetsForPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForPolicy.html",
                "description": "List targets for the specified policy.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListTargetsForSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForSecurityProfile.html",
                "description": "Lists the targets associated with a given Device Defender security profile.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    }
                }
            },
            "ListThingGroups": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroups.html",
                "description": "Lists all thing groups.",
                "accessLevel": "List"
            },
            "ListThingGroupsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroupsForThing.html",
                "description": "List thing groups to which the specified thing belongs.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListThingPrincipals": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingPrincipals.html",
                "description": "Lists the principals associated with the specified thing.",
                "accessLevel": "List"
            },
            "ListThingRegistrationTaskReports": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTaskReports.html",
                "description": "Lists information about bulk thing registration tasks.",
                "accessLevel": "List"
            },
            "ListThingRegistrationTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTasks.html",
                "description": "Lists bulk thing registration tasks.",
                "accessLevel": "List"
            },
            "ListThingTypes": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingTypes.html",
                "description": "Lists all thing types.",
                "accessLevel": "List"
            },
            "ListThings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThings.html",
                "description": "Lists all things.",
                "accessLevel": "List"
            },
            "ListThingsInBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInBillingGroup.html",
                "description": "Lists all things in the specified billing group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "ListThingsInThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInThingGroup.html",
                "description": "Lists all things in the specified thing group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "ListTopicRules": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTopicRules.html",
                "description": "Lists the rules for the specific topic.",
                "accessLevel": "List"
            },
            "ListTunnels": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTunnels.html",
                "description": "Lists tunnels.",
                "accessLevel": "List"
            },
            "ListV2LoggingLevels": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListV2LoggingLevels.html",
                "description": "Lists the v2 logging levels.",
                "accessLevel": "List"
            },
            "ListViolationEvents": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListViolationEvents.html",
                "description": "Lists the Device Defender security profile violations discovered during the given time period.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "OpenTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_OpenTunnel.html",
                "description": "Opens a tunnel.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iot:ThingGroupArn",
                    "iot:TunnelDestinationService"
                ]
            },
            "Publish": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Publish to the specified topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "Receive": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Receive from the specified topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "RegisterCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCACertificate.html",
                "description": "Registers a CA certificate with AWS IoT.",
                "accessLevel": "Write"
            },
            "RegisterCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html",
                "description": "Registers a device certificate with AWS IoT.",
                "accessLevel": "Write"
            },
            "RegisterCertificateWithoutCA": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html",
                "description": "Registers a device certificate with AWS IoT without a registered CA (certificate authority).",
                "accessLevel": "Write"
            },
            "RegisterThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterThing.html",
                "description": "Registers your thing.",
                "accessLevel": "Write"
            },
            "RejectCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RejectCertificateTransfer.html",
                "description": "Rejects a pending certificate transfer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "RemoveThingFromBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromBillingGroup.html",
                "description": "Removes thing from the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "RemoveThingFromThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromThingGroup.html",
                "description": "Removes thing from the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "ReplaceTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ReplaceTopicRule.html",
                "description": "Replaces the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "SearchIndex": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SearchIndex.html",
                "description": "Search IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "SetDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultAuthorizer.html",
                "description": "Sets the default authorizer. This will be used if a websocket connection is made without specifying an authorizer.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "SetDefaultPolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultPolicyVersion.html",
                "description": "Sets the specified version of the specified policy as the policy's default (operative) version.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "SetLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetLoggingOptions.html",
                "description": "Sets the logging options.",
                "accessLevel": "Write"
            },
            "SetV2LoggingLevel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingLevel.html",
                "description": "Sets the v2 logging level.",
                "accessLevel": "Write"
            },
            "SetV2LoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingOptions.html",
                "description": "Sets the v2 logging options.",
                "accessLevel": "Write"
            },
            "StartAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartAuditMitigationActionsTask.html",
                "description": "Starts a task that applies a set of mitigation actions to the specified target.",
                "accessLevel": "Write"
            },
            "StartNextPendingJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartNextPendingJobExecution.html",
                "description": "Gets and starts the next pending job execution for a thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "StartOnDemandAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartOnDemandAuditTask.html",
                "description": "Starts an on-demand Device Defender audit.",
                "accessLevel": "Write"
            },
            "StartThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartThingRegistrationTask.html",
                "description": "Starts a bulk thing registration task.",
                "accessLevel": "Write"
            },
            "StopThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StopThingRegistrationTask.html",
                "description": "Stops a bulk thing registration task.",
                "accessLevel": "Write"
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Subscribe to the specified TopicFilter.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topicfilter": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "",
                "description": "Tag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestAuthorization": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TestAuthorization.html",
                "description": "Test the policies evaluation for group policies",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "TestInvokeAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TestInvokeAuthorizer.html",
                "description": "Invoke the specified custom authorizer for testing purposes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "TransferCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TransferCertificate.html",
                "description": "Transfers the specified certificate to the specified AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "",
                "description": "Untag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateAccountAuditConfiguration.html",
                "description": "Configures or reconfigures the Device Defender audit settings for this account.",
                "accessLevel": "Write"
            },
            "UpdateAuthorizer": {
                "url": "",
                "description": "Updates an authorizer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "UpdateBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateBillingGroup.html",
                "description": "Updates information associated with the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCACertificate.html",
                "description": "Updates a registered CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "UpdateCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCertificate.html",
                "description": "Updates the status of the specified certificate. This operation is idempotent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "UpdateDimension": {
                "url": "",
                "description": "Updates the definition for a dimension.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "UpdateDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateDynamicThingGroup.html",
                "description": "Updates a Dynamic Thing Group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateEventConfigurations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateEventConfigurations.html",
                "description": "Updates event configurations.",
                "accessLevel": "Write"
            },
            "UpdateIndexingConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateIndexingConfiguration.html",
                "description": "Updates fleet indexing configuration",
                "accessLevel": "Write"
            },
            "UpdateJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJob.html",
                "description": "Updates a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "UpdateJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJobExecution.html",
                "description": "Updates a job execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "UpdateMitigationAction": {
                "url": "",
                "description": "Updates the definition for the specified mitigation action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "UpdateProvisioningTemplate": {
                "url": "",
                "description": "Updates a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "UpdateRoleAlias": {
                "url": "",
                "description": "Updates the role alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "UpdateScheduledAudit": {
                "url": "",
                "description": "Updates a scheduled audit, including what checks are performed and how often the audit takes place.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "UpdateSecurityProfile": {
                "url": "",
                "description": "Updates a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "UpdateStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateStream.html",
                "description": "Updates the data for a stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "UpdateThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThing.html",
                "description": "Updates information associated with the specified thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "UpdateThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroup.html",
                "description": "Updates information associated with the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateThingGroupsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroupsForThing.html",
                "description": "Updates the thing groups to which the thing belongs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "UpdateThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Updates the thing shadow.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ValidateSecurityProfileBehaviors": {
                "url": "",
                "description": "Validates a Device Defender security profile behaviors specification.",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "client": {
                "name": "client",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:client/${ClientId}",
                "conditionKeys": []
            },
            "index": {
                "name": "index",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-indexing.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:index/${IndexName}",
                "conditionKeys": []
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-jobs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:job/${JobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "tunnel": {
                "name": "tunnel",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-tunnels.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:tunnel/${TunnelId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "thing": {
                "name": "thing",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}",
                "conditionKeys": []
            },
            "thinggroup": {
                "name": "thinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "billinggroup": {
                "name": "billinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/billing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:billinggroup/${BillingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dynamicthinggroup": {
                "name": "dynamicthinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/dynamic-thing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "thingtype": {
                "name": "thingtype",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-types.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thingtype/${ThingTypeName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "topic": {
                "name": "topic",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:topic/${TopicName}",
                "conditionKeys": []
            },
            "topicfilter": {
                "name": "topicfilter",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/topics.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:topicfilter/${TopicFilter}",
                "conditionKeys": []
            },
            "rolealias": {
                "name": "rolealias",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/authorizing-direct-aws.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:rolealias/${RoleAlias}",
                "conditionKeys": []
            },
            "authorizer": {
                "name": "authorizer",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/custom-authorizer.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:authorizer/${AuthorizerName}",
                "conditionKeys": []
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-policies.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:policy/${PolicyName}",
                "conditionKeys": []
            },
            "cert": {
                "name": "cert",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:cert/${Certificate}",
                "conditionKeys": []
            },
            "cacert": {
                "name": "cacert",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:cacert/${CACertificate}",
                "conditionKeys": []
            },
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:stream/${StreamId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "otaupdate": {
                "name": "otaupdate",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:otaupdate/${OtaUpdateId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "scheduledaudit": {
                "name": "scheduledaudit",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-audit.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:scheduledaudit/${ScheduleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "mitigationaction": {
                "name": "mitigationaction",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-mitigation-actions.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:mitigationaction/${MitigationActionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "securityprofile": {
                "name": "securityprofile",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:securityprofile/${SecurityProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dimension": {
                "name": "dimension",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:dimension/${DimensionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:rule/${RuleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "provisioningtemplate": {
                "name": "provisioningtemplate",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/provision-template.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:provisioningtemplate/${ProvisioningTemplate}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Accepts a pending certificate transfer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AcceptCertificateTransfer.html
     */
    toAcceptCertificateTransfer() {
        this.add('iot:AcceptCertificateTransfer');
        return this;
    }
    /**
     * Adds a thing to the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToBillingGroup.html
     */
    toAddThingToBillingGroup() {
        this.add('iot:AddThingToBillingGroup');
        return this;
    }
    /**
     * Adds a thing to the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToThingGroup.html
     */
    toAddThingToThingGroup() {
        this.add('iot:AddThingToThingGroup');
        return this;
    }
    /**
     * Associates a group with a continuous job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AssociateTargetsWithJob.html
     */
    toAssociateTargetsWithJob() {
        this.add('iot:AssociateTargetsWithJob');
        return this;
    }
    /**
     * Attaches a policy to the specified target.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        this.add('iot:AttachPolicy');
        return this;
    }
    /**
     * Attaches the specified policy to the specified principal (certificate or other credential).
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPrincipalPolicy.html
     */
    toAttachPrincipalPolicy() {
        this.add('iot:AttachPrincipalPolicy');
        return this;
    }
    /**
     * Associates a Device Defender security profile with a thing group or with this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachSecurityProfile.html
     */
    toAttachSecurityProfile() {
        this.add('iot:AttachSecurityProfile');
        return this;
    }
    /**
     * Attaches the specified principal to the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachThingPrincipal.html
     */
    toAttachThingPrincipal() {
        this.add('iot:AttachThingPrincipal');
        return this;
    }
    /**
     * Cancels a mitigation action task that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditMitigationActionsTask.html
     */
    toCancelAuditMitigationActionsTask() {
        this.add('iot:CancelAuditMitigationActionsTask');
        return this;
    }
    /**
     * Cancels an audit that is in progress. The audit can be either scheduled or on-demand.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditTask.html
     */
    toCancelAuditTask() {
        this.add('iot:CancelAuditTask');
        return this;
    }
    /**
     * Cancels a pending transfer for the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelCertificateTransfer.html
     */
    toCancelCertificateTransfer() {
        this.add('iot:CancelCertificateTransfer');
        return this;
    }
    /**
     * Cancels a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJob.html
     */
    toCancelJob() {
        this.add('iot:CancelJob');
        return this;
    }
    /**
     * Cancels a job execution on a particular device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJobExecution.html
     */
    toCancelJobExecution() {
        this.add('iot:CancelJobExecution');
        return this;
    }
    /**
     * Clears the default authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ClearDefaultAuthorizer.html
     */
    toClearDefaultAuthorizer() {
        this.add('iot:ClearDefaultAuthorizer');
        return this;
    }
    /**
     * Closes a tunnel.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDelete()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CloseTunnel.html
     */
    toCloseTunnel() {
        this.add('iot:CloseTunnel');
        return this;
    }
    /**
     * Connect as the specified client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toConnect() {
        this.add('iot:Connect');
        return this;
    }
    /**
     * Creates an authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateAuthorizer.html
     */
    toCreateAuthorizer() {
        this.add('iot:CreateAuthorizer');
        return this;
    }
    /**
     * Creates a billing group.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateBillingGroup.html
     */
    toCreateBillingGroup() {
        this.add('iot:CreateBillingGroup');
        return this;
    }
    /**
     * Creates an X.509 certificate using the specified certificate signing request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html
     */
    toCreateCertificateFromCsr() {
        this.add('iot:CreateCertificateFromCsr');
        return this;
    }
    /**
     * Defines a dimension that can be used to to limit the scope of a metric used in a security profile.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDimension.html
     */
    toCreateDimension() {
        this.add('iot:CreateDimension');
        return this;
    }
    /**
     * Creates a Dynamic Thing Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDynamicThingGroup.html
     */
    toCreateDynamicThingGroup() {
        this.add('iot:CreateDynamicThingGroup');
        return this;
    }
    /**
     * Creates a job.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateJob.html
     */
    toCreateJob() {
        this.add('iot:CreateJob');
        return this;
    }
    /**
     * Creates a 2048 bit RSA key pair and issues an X.509 certificate using the issued public key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html
     */
    toCreateKeysAndCertificate() {
        this.add('iot:CreateKeysAndCertificate');
        return this;
    }
    /**
     * Defines an action that can be applied to audit findings by using StartAuditMitigationActionsTask.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateMitigationAction.html
     */
    toCreateMitigationAction() {
        this.add('iot:CreateMitigationAction');
        return this;
    }
    /**
     * Creates an OTA update job.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateOTAUpdate.html
     */
    toCreateOTAUpdate() {
        this.add('iot:CreateOTAUpdate');
        return this;
    }
    /**
     * Creates an AWS IoT policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        this.add('iot:CreatePolicy');
        return this;
    }
    /**
     * Creates a new version of the specified AWS IoT policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicyVersion.html
     */
    toCreatePolicyVersion() {
        this.add('iot:CreatePolicyVersion');
        return this;
    }
    /**
     * Creates a provisioning claim.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningClaim.html
     */
    toCreateProvisioningClaim() {
        this.add('iot:CreateProvisioningClaim');
        return this;
    }
    /**
     * Creates a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplate.html
     */
    toCreateProvisioningTemplate() {
        this.add('iot:CreateProvisioningTemplate');
        return this;
    }
    /**
     * Creates a new version of a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplateVersion.html
     */
    toCreateProvisioningTemplateVersion() {
        this.add('iot:CreateProvisioningTemplateVersion');
        return this;
    }
    /**
     * Creates a role alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateRoleAlias.html
     */
    toCreateRoleAlias() {
        this.add('iot:CreateRoleAlias');
        return this;
    }
    /**
     * Creates a scheduled audit that is run at a specified time interval.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateScheduledAudit.html
     */
    toCreateScheduledAudit() {
        this.add('iot:CreateScheduledAudit');
        return this;
    }
    /**
     * Creates a Device Defender security profile.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateSecurityProfile.html
     */
    toCreateSecurityProfile() {
        this.add('iot:CreateSecurityProfile');
        return this;
    }
    /**
     * Creates a new AWS IoT stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateStream.html
     */
    toCreateStream() {
        this.add('iot:CreateStream');
        return this;
    }
    /**
     * Creates a thing in the thing registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThing.html
     */
    toCreateThing() {
        this.add('iot:CreateThing');
        return this;
    }
    /**
     * Creates a thing group.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingGroup.html
     */
    toCreateThingGroup() {
        this.add('iot:CreateThingGroup');
        return this;
    }
    /**
     * Creates a new thing type.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingType.html
     */
    toCreateThingType() {
        this.add('iot:CreateThingType');
        return this;
    }
    /**
     * Creates a rule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateTopicRule.html
     */
    toCreateTopicRule() {
        this.add('iot:CreateTopicRule');
        return this;
    }
    /**
     * Deletes the audit configuration associated with the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAccountAuditConfiguration.html
     */
    toDeleteAccountAuditConfiguration() {
        this.add('iot:DeleteAccountAuditConfiguration');
        return this;
    }
    /**
     * Deletes the specified authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAuthorizer.html
     */
    toDeleteAuthorizer() {
        this.add('iot:DeleteAuthorizer');
        return this;
    }
    /**
     * Deletes the specified billing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteBillingGroup.html
     */
    toDeleteBillingGroup() {
        this.add('iot:DeleteBillingGroup');
        return this;
    }
    /**
     * Deletes a registered CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCACertificate.html
     */
    toDeleteCACertificate() {
        this.add('iot:DeleteCACertificate');
        return this;
    }
    /**
     * Deletes the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        this.add('iot:DeleteCertificate');
        return this;
    }
    /**
     * Removes the specified dimension from your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDimension.html
     */
    toDeleteDimension() {
        this.add('iot:DeleteDimension');
        return this;
    }
    /**
     * Deletes the specified Dynamic Thing Group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDynamicThingGroup.html
     */
    toDeleteDynamicThingGroup() {
        this.add('iot:DeleteDynamicThingGroup');
        return this;
    }
    /**
     * Deletes a job and its related job executions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJob.html
     */
    toDeleteJob() {
        this.add('iot:DeleteJob');
        return this;
    }
    /**
     * Deletes a job execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJobExecution.html
     */
    toDeleteJobExecution() {
        this.add('iot:DeleteJobExecution');
        return this;
    }
    /**
     * Deletes a defined mitigation action from your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteMitigationAction.html
     */
    toDeleteMitigationAction() {
        this.add('iot:DeleteMitigationAction');
        return this;
    }
    /**
     * Deletes an OTA update job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteOTAUpdate.html
     */
    toDeleteOTAUpdate() {
        this.add('iot:DeleteOTAUpdate');
        return this;
    }
    /**
     * Deletes the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.add('iot:DeletePolicy');
        return this;
    }
    /**
     * Deletes the specified version of the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicyVersion.html
     */
    toDeletePolicyVersion() {
        this.add('iot:DeletePolicyVersion');
        return this;
    }
    /**
     * Deletes a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplate.html
     */
    toDeleteProvisioningTemplate() {
        this.add('iot:DeleteProvisioningTemplate');
        return this;
    }
    /**
     * Deletes a fleet provisioning template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplateVersion.html
     */
    toDeleteProvisioningTemplateVersion() {
        this.add('iot:DeleteProvisioningTemplateVersion');
        return this;
    }
    /**
     * Deletes a CA certificate registration code.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRegistrationCode.html
     */
    toDeleteRegistrationCode() {
        this.add('iot:DeleteRegistrationCode');
        return this;
    }
    /**
     * Deletes the specified role alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRoleAlias.html
     */
    toDeleteRoleAlias() {
        this.add('iot:DeleteRoleAlias');
        return this;
    }
    /**
     * Deletes a scheduled audit.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteScheduledAudit.html
     */
    toDeleteScheduledAudit() {
        this.add('iot:DeleteScheduledAudit');
        return this;
    }
    /**
     * Deletes a Device Defender security profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteSecurityProfile.html
     */
    toDeleteSecurityProfile() {
        this.add('iot:DeleteSecurityProfile');
        return this;
    }
    /**
     * Deletes a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteStream.html
     */
    toDeleteStream() {
        this.add('iot:DeleteStream');
        return this;
    }
    /**
     * Deletes the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html
     */
    toDeleteThing() {
        this.add('iot:DeleteThing');
        return this;
    }
    /**
     * Deletes the specified thing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingGroup.html
     */
    toDeleteThingGroup() {
        this.add('iot:DeleteThingGroup');
        return this;
    }
    /**
     * Deletes the specified thing shadow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toDeleteThingShadow() {
        this.add('iot:DeleteThingShadow');
        return this;
    }
    /**
     * Deletes the specified thing type.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingType.html
     */
    toDeleteThingType() {
        this.add('iot:DeleteThingType');
        return this;
    }
    /**
     * Deletes the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteTopicRule.html
     */
    toDeleteTopicRule() {
        this.add('iot:DeleteTopicRule');
        return this;
    }
    /**
     * Deletes the specified v2 logging level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteV2LoggingLevel.html
     */
    toDeleteV2LoggingLevel() {
        this.add('iot:DeleteV2LoggingLevel');
        return this;
    }
    /**
     * Deprecates the specified thing type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeprecateThingType.html
     */
    toDeprecateThingType() {
        this.add('iot:DeprecateThingType');
        return this;
    }
    /**
     * Gets information about audit configurations for the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAccountAuditConfiguration.html
     */
    toDescribeAccountAuditConfiguration() {
        this.add('iot:DescribeAccountAuditConfiguration');
        return this;
    }
    /**
     * Gets information about an audit mitigation task that is used to apply mitigation actions to a set of audit findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditMitigationActionsTask.html
     */
    toDescribeAuditMitigationActionsTask() {
        this.add('iot:DescribeAuditMitigationActionsTask');
        return this;
    }
    /**
     * Gets information about a Device Defender audit.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditTask.html
     */
    toDescribeAuditTask() {
        this.add('iot:DescribeAuditTask');
        return this;
    }
    /**
     * Describes an authorizer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuthorizer.html
     */
    toDescribeAuthorizer() {
        this.add('iot:DescribeAuthorizer');
        return this;
    }
    /**
     * Gets information about the specified billing group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeBillingGroup.html
     */
    toDescribeBillingGroup() {
        this.add('iot:DescribeBillingGroup');
        return this;
    }
    /**
     * Describes a registered CA certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCACertificate.html
     */
    toDescribeCACertificate() {
        this.add('iot:DescribeCACertificate');
        return this;
    }
    /**
     * Gets information about the specified certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        this.add('iot:DescribeCertificate');
        return this;
    }
    /**
     * Describes the default authorizer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDefaultAuthorizer.html
     */
    toDescribeDefaultAuthorizer() {
        this.add('iot:DescribeDefaultAuthorizer');
        return this;
    }
    /**
     * Provides details about a dimension that is defined in your AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDimension.html
     */
    toDescribeDimension() {
        this.add('iot:DescribeDimension');
        return this;
    }
    /**
     * Returns a unique endpoint specific to the AWS account making the call.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        this.add('iot:DescribeEndpoint');
        return this;
    }
    /**
     * Returns account event configurations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEventConfigurations.html
     */
    toDescribeEventConfigurations() {
        this.add('iot:DescribeEventConfigurations');
        return this;
    }
    /**
     * Gets information about the specified index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeIndex.html
     */
    toDescribeIndex() {
        this.add('iot:DescribeIndex');
        return this;
    }
    /**
     * Describes a job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJob.html
     */
    toDescribeJob() {
        this.add('iot:DescribeJob');
        return this;
    }
    /**
     * Describes a job execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJobExecution.html
     */
    toDescribeJobExecution() {
        this.add('iot:DescribeJobExecution');
        return this;
    }
    /**
     * Gets information about a mitigation action.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeMitigationAction.html
     */
    toDescribeMitigationAction() {
        this.add('iot:DescribeMitigationAction');
        return this;
    }
    /**
     * Returns information about a fleet provisioning template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplate.html
     */
    toDescribeProvisioningTemplate() {
        this.add('iot:DescribeProvisioningTemplate');
        return this;
    }
    /**
     * Returns information about a fleet provisioning template version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplateVersion.html
     */
    toDescribeProvisioningTemplateVersion() {
        this.add('iot:DescribeProvisioningTemplateVersion');
        return this;
    }
    /**
     * Describes a role alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeRoleAlias.html
     */
    toDescribeRoleAlias() {
        this.add('iot:DescribeRoleAlias');
        return this;
    }
    /**
     * Gets information about a scheduled audit.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeScheduledAudit.html
     */
    toDescribeScheduledAudit() {
        this.add('iot:DescribeScheduledAudit');
        return this;
    }
    /**
     * Gets information about a Device Defender security profile.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeSecurityProfile.html
     */
    toDescribeSecurityProfile() {
        this.add('iot:DescribeSecurityProfile');
        return this;
    }
    /**
     * Gets information about the specified stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeStream.html
     */
    toDescribeStream() {
        this.add('iot:DescribeStream');
        return this;
    }
    /**
     * Gets information about the specified thing.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThing.html
     */
    toDescribeThing() {
        this.add('iot:DescribeThing');
        return this;
    }
    /**
     * Gets information about the specified thing group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingGroup.html
     */
    toDescribeThingGroup() {
        this.add('iot:DescribeThingGroup');
        return this;
    }
    /**
     * Gets information about the bulk thing registration task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingRegistrationTask.html
     */
    toDescribeThingRegistrationTask() {
        this.add('iot:DescribeThingRegistrationTask');
        return this;
    }
    /**
     * Gets information about the specified thing type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingType.html
     */
    toDescribeThingType() {
        this.add('iot:DescribeThingType');
        return this;
    }
    /**
     * Describes a tunnel.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeTunnel.html
     */
    toDescribeTunnel() {
        this.add('iot:DescribeTunnel');
        return this;
    }
    /**
     * Detaches a policy from the specified target.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        this.add('iot:DetachPolicy');
        return this;
    }
    /**
     * Removes the specified policy from the specified certificate.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPrincipalPolicy.html
     */
    toDetachPrincipalPolicy() {
        this.add('iot:DetachPrincipalPolicy');
        return this;
    }
    /**
     * Disassociates a Device Defender security profile from a thing group or from this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachSecurityProfile.html
     */
    toDetachSecurityProfile() {
        this.add('iot:DetachSecurityProfile');
        return this;
    }
    /**
     * Detaches the specified principal from the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachThingPrincipal.html
     */
    toDetachThingPrincipal() {
        this.add('iot:DetachThingPrincipal');
        return this;
    }
    /**
     * Disables the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DisableTopicRule.html
     */
    toDisableTopicRule() {
        this.add('iot:DisableTopicRule');
        return this;
    }
    /**
     * Enables the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_EnableTopicRule.html
     */
    toEnableTopicRule() {
        this.add('iot:EnableTopicRule');
        return this;
    }
    /**
     * Get cardinality for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetCardinality.html
     */
    toGetCardinality() {
        this.add('iot:GetCardinality');
        return this;
    }
    /**
     * Gets effective policies.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetEffectivePolicies.html
     */
    toGetEffectivePolicies() {
        this.add('iot:GetEffectivePolicies');
        return this;
    }
    /**
     * Gets current fleet indexing configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetIndexingConfiguration.html
     */
    toGetIndexingConfiguration() {
        this.add('iot:GetIndexingConfiguration');
        return this;
    }
    /**
     * Gets a job document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetJobDocument.html
     */
    toGetJobDocument() {
        this.add('iot:GetJobDocument');
        return this;
    }
    /**
     * Gets the logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetLoggingOptions.html
     */
    toGetLoggingOptions() {
        this.add('iot:GetLoggingOptions');
        return this;
    }
    /**
     * Gets the information about the OTA update job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetOTAUpdate.html
     */
    toGetOTAUpdate() {
        this.add('iot:GetOTAUpdate');
        return this;
    }
    /**
     * Gets the list of all jobs for a thing that are not in a terminal state.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPendingJobExecutions.html
     */
    toGetPendingJobExecutions() {
        this.add('iot:GetPendingJobExecutions');
        return this;
    }
    /**
     * Get percentiles for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPercentiles.html
     */
    toGetPercentiles() {
        this.add('iot:GetPercentiles');
        return this;
    }
    /**
     * Gets information about the specified policy with the policy document of the default version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicy.html
     */
    toGetPolicy() {
        this.add('iot:GetPolicy');
        return this;
    }
    /**
     * Gets information about the specified policy version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicyVersion.html
     */
    toGetPolicyVersion() {
        this.add('iot:GetPolicyVersion');
        return this;
    }
    /**
     * Gets a registration code used to register a CA certificate with AWS IoT.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetRegistrationCode.html
     */
    toGetRegistrationCode() {
        this.add('iot:GetRegistrationCode');
        return this;
    }
    /**
     * Get statistics for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetStatistics.html
     */
    toGetStatistics() {
        this.add('iot:GetStatistics');
        return this;
    }
    /**
     * Gets the thing shadow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toGetThingShadow() {
        this.add('iot:GetThingShadow');
        return this;
    }
    /**
     * Gets information about the specified rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetTopicRule.html
     */
    toGetTopicRule() {
        this.add('iot:GetTopicRule');
        return this;
    }
    /**
     * Gets v2 logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetV2LoggingOptions.html
     */
    toGetV2LoggingOptions() {
        this.add('iot:GetV2LoggingOptions');
        return this;
    }
    /**
     * Lists the active violations for a given Device Defender security profile or Thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListActiveViolations.html
     */
    toListActiveViolations() {
        this.add('iot:ListActiveViolations');
        return this;
    }
    /**
     * Lists the policies attached to the specified thing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAttachedPolicies.html
     */
    toListAttachedPolicies() {
        this.add('iot:ListAttachedPolicies');
        return this;
    }
    /**
     * Lists the findings (results) of a Device Defender audit or of the audits performed during a specified time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditFindings.html
     */
    toListAuditFindings() {
        this.add('iot:ListAuditFindings');
        return this;
    }
    /**
     * Gets the status of audit mitigation action tasks that were executed.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsExecutions.html
     */
    toListAuditMitigationActionsExecutions() {
        this.add('iot:ListAuditMitigationActionsExecutions');
        return this;
    }
    /**
     * Gets a list of audit mitigation action tasks that match the specified filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsTasks.html
     */
    toListAuditMitigationActionsTasks() {
        this.add('iot:ListAuditMitigationActionsTasks');
        return this;
    }
    /**
     * Lists the Device Defender audits that have been performed during a given time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditTasks.html
     */
    toListAuditTasks() {
        this.add('iot:ListAuditTasks');
        return this;
    }
    /**
     * Lists the authorizers registered in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuthorizers.html
     */
    toListAuthorizers() {
        this.add('iot:ListAuthorizers');
        return this;
    }
    /**
     * Lists all billing groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListBillingGroups.html
     */
    toListBillingGroups() {
        this.add('iot:ListBillingGroups');
        return this;
    }
    /**
     * Lists the CA certificates registered for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCACertificates.html
     */
    toListCACertificates() {
        this.add('iot:ListCACertificates');
        return this;
    }
    /**
     * Lists your certificates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificates.html
     */
    toListCertificates() {
        this.add('iot:ListCertificates');
        return this;
    }
    /**
     * List the device certificates signed by the specified CA certificate.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificatesByCA.html
     */
    toListCertificatesByCA() {
        this.add('iot:ListCertificatesByCA');
        return this;
    }
    /**
     * Lists the dimensions that are defined for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListDimensions.html
     */
    toListDimensions() {
        this.add('iot:ListDimensions');
        return this;
    }
    /**
     * Lists all indices for fleet index
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListIndices.html
     */
    toListIndices() {
        this.add('iot:ListIndices');
        return this;
    }
    /**
     * Lists the job executions for a job.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForJob.html
     */
    toListJobExecutionsForJob() {
        this.add('iot:ListJobExecutionsForJob');
        return this;
    }
    /**
     * Lists the job executions for the specified thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForThing.html
     */
    toListJobExecutionsForThing() {
        this.add('iot:ListJobExecutionsForThing');
        return this;
    }
    /**
     * Lists jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobs.html
     */
    toListJobs() {
        this.add('iot:ListJobs');
        return this;
    }
    /**
     * Gets a list of all mitigation actions that match the specified filter criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListMitigationActions.html
     */
    toListMitigationActions() {
        this.add('iot:ListMitigationActions');
        return this;
    }
    /**
     * Lists all named shadows for a given thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListNamedShadowsForThing.html
     */
    toListNamedShadowsForThing() {
        this.add('iot:ListNamedShadowsForThing');
        return this;
    }
    /**
     * Lists OTA update jobs in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListOTAUpdates.html
     */
    toListOTAUpdates() {
        this.add('iot:ListOTAUpdates');
        return this;
    }
    /**
     * Lists certificates that are being transfered but not yet accepted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListOutgoingCertificates.html
     */
    toListOutgoingCertificates() {
        this.add('iot:ListOutgoingCertificates');
        return this;
    }
    /**
     * Lists your policies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicies.html
     */
    toListPolicies() {
        this.add('iot:ListPolicies');
        return this;
    }
    /**
     * Lists the principals associated with the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyPrincipals.html
     */
    toListPolicyPrincipals() {
        this.add('iot:ListPolicyPrincipals');
        return this;
    }
    /**
     * Lists the versions of the specified policy, and identifies the default version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyVersions.html
     */
    toListPolicyVersions() {
        this.add('iot:ListPolicyVersions');
        return this;
    }
    /**
     * Lists the policies attached to the specified principal. If you use an Amazon Cognito identity, the ID needs to be in Amazon Cognito Identity format.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalPolicies.html
     */
    toListPrincipalPolicies() {
        this.add('iot:ListPrincipalPolicies');
        return this;
    }
    /**
     * Lists the things associated with the specified principal.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalThings.html
     */
    toListPrincipalThings() {
        this.add('iot:ListPrincipalThings');
        return this;
    }
    /**
     * A list of fleet provisioning template versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplateVersions.html
     */
    toListProvisioningTemplateVersions() {
        this.add('iot:ListProvisioningTemplateVersions');
        return this;
    }
    /**
     * Lists the fleet provisioning templates in your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplates.html
     */
    toListProvisioningTemplates() {
        this.add('iot:ListProvisioningTemplates');
        return this;
    }
    /**
     * Lists role aliases.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListRoleAliases.html
     */
    toListRoleAliases() {
        this.add('iot:ListRoleAliases');
        return this;
    }
    /**
     * Lists all of your scheduled audits.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListScheduledAudits.html
     */
    toListScheduledAudits() {
        this.add('iot:ListScheduledAudits');
        return this;
    }
    /**
     * Lists the Device Defender security profiles you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfiles.html
     */
    toListSecurityProfiles() {
        this.add('iot:ListSecurityProfiles');
        return this;
    }
    /**
     * Lists the Device Defender security profiles attached to a target.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfilesForTarget.html
     */
    toListSecurityProfilesForTarget() {
        this.add('iot:ListSecurityProfilesForTarget');
        return this;
    }
    /**
     * Lists the streams in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListStreams.html
     */
    toListStreams() {
        this.add('iot:ListStreams');
        return this;
    }
    /**
     * Lists all tags for a given resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('iot:ListTagsForResource');
        return this;
    }
    /**
     * List targets for the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForPolicy.html
     */
    toListTargetsForPolicy() {
        this.add('iot:ListTargetsForPolicy');
        return this;
    }
    /**
     * Lists the targets associated with a given Device Defender security profile.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForSecurityProfile.html
     */
    toListTargetsForSecurityProfile() {
        this.add('iot:ListTargetsForSecurityProfile');
        return this;
    }
    /**
     * Lists all thing groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroups.html
     */
    toListThingGroups() {
        this.add('iot:ListThingGroups');
        return this;
    }
    /**
     * List thing groups to which the specified thing belongs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroupsForThing.html
     */
    toListThingGroupsForThing() {
        this.add('iot:ListThingGroupsForThing');
        return this;
    }
    /**
     * Lists the principals associated with the specified thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingPrincipals.html
     */
    toListThingPrincipals() {
        this.add('iot:ListThingPrincipals');
        return this;
    }
    /**
     * Lists information about bulk thing registration tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTaskReports.html
     */
    toListThingRegistrationTaskReports() {
        this.add('iot:ListThingRegistrationTaskReports');
        return this;
    }
    /**
     * Lists bulk thing registration tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTasks.html
     */
    toListThingRegistrationTasks() {
        this.add('iot:ListThingRegistrationTasks');
        return this;
    }
    /**
     * Lists all thing types.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingTypes.html
     */
    toListThingTypes() {
        this.add('iot:ListThingTypes');
        return this;
    }
    /**
     * Lists all things.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThings.html
     */
    toListThings() {
        this.add('iot:ListThings');
        return this;
    }
    /**
     * Lists all things in the specified billing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInBillingGroup.html
     */
    toListThingsInBillingGroup() {
        this.add('iot:ListThingsInBillingGroup');
        return this;
    }
    /**
     * Lists all things in the specified thing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInThingGroup.html
     */
    toListThingsInThingGroup() {
        this.add('iot:ListThingsInThingGroup');
        return this;
    }
    /**
     * Lists the rules for the specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTopicRules.html
     */
    toListTopicRules() {
        this.add('iot:ListTopicRules');
        return this;
    }
    /**
     * Lists tunnels.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTunnels.html
     */
    toListTunnels() {
        this.add('iot:ListTunnels');
        return this;
    }
    /**
     * Lists the v2 logging levels.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListV2LoggingLevels.html
     */
    toListV2LoggingLevels() {
        this.add('iot:ListV2LoggingLevels');
        return this;
    }
    /**
     * Lists the Device Defender security profile violations discovered during the given time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListViolationEvents.html
     */
    toListViolationEvents() {
        this.add('iot:ListViolationEvents');
        return this;
    }
    /**
     * Opens a tunnel.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifThingGroupArn()
     * - .ifTunnelDestinationService()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_OpenTunnel.html
     */
    toOpenTunnel() {
        this.add('iot:OpenTunnel');
        return this;
    }
    /**
     * Publish to the specified topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toPublish() {
        this.add('iot:Publish');
        return this;
    }
    /**
     * Receive from the specified topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toReceive() {
        this.add('iot:Receive');
        return this;
    }
    /**
     * Registers a CA certificate with AWS IoT.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCACertificate.html
     */
    toRegisterCACertificate() {
        this.add('iot:RegisterCACertificate');
        return this;
    }
    /**
     * Registers a device certificate with AWS IoT.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html
     */
    toRegisterCertificate() {
        this.add('iot:RegisterCertificate');
        return this;
    }
    /**
     * Registers a device certificate with AWS IoT without a registered CA (certificate authority).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html
     */
    toRegisterCertificateWithoutCA() {
        this.add('iot:RegisterCertificateWithoutCA');
        return this;
    }
    /**
     * Registers your thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterThing.html
     */
    toRegisterThing() {
        this.add('iot:RegisterThing');
        return this;
    }
    /**
     * Rejects a pending certificate transfer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RejectCertificateTransfer.html
     */
    toRejectCertificateTransfer() {
        this.add('iot:RejectCertificateTransfer');
        return this;
    }
    /**
     * Removes thing from the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromBillingGroup.html
     */
    toRemoveThingFromBillingGroup() {
        this.add('iot:RemoveThingFromBillingGroup');
        return this;
    }
    /**
     * Removes thing from the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromThingGroup.html
     */
    toRemoveThingFromThingGroup() {
        this.add('iot:RemoveThingFromThingGroup');
        return this;
    }
    /**
     * Replaces the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ReplaceTopicRule.html
     */
    toReplaceTopicRule() {
        this.add('iot:ReplaceTopicRule');
        return this;
    }
    /**
     * Search IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SearchIndex.html
     */
    toSearchIndex() {
        this.add('iot:SearchIndex');
        return this;
    }
    /**
     * Sets the default authorizer. This will be used if a websocket connection is made without specifying an authorizer.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultAuthorizer.html
     */
    toSetDefaultAuthorizer() {
        this.add('iot:SetDefaultAuthorizer');
        return this;
    }
    /**
     * Sets the specified version of the specified policy as the policy's default (operative) version.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultPolicyVersion.html
     */
    toSetDefaultPolicyVersion() {
        this.add('iot:SetDefaultPolicyVersion');
        return this;
    }
    /**
     * Sets the logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetLoggingOptions.html
     */
    toSetLoggingOptions() {
        this.add('iot:SetLoggingOptions');
        return this;
    }
    /**
     * Sets the v2 logging level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingLevel.html
     */
    toSetV2LoggingLevel() {
        this.add('iot:SetV2LoggingLevel');
        return this;
    }
    /**
     * Sets the v2 logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingOptions.html
     */
    toSetV2LoggingOptions() {
        this.add('iot:SetV2LoggingOptions');
        return this;
    }
    /**
     * Starts a task that applies a set of mitigation actions to the specified target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartAuditMitigationActionsTask.html
     */
    toStartAuditMitigationActionsTask() {
        this.add('iot:StartAuditMitigationActionsTask');
        return this;
    }
    /**
     * Gets and starts the next pending job execution for a thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartNextPendingJobExecution.html
     */
    toStartNextPendingJobExecution() {
        this.add('iot:StartNextPendingJobExecution');
        return this;
    }
    /**
     * Starts an on-demand Device Defender audit.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartOnDemandAuditTask.html
     */
    toStartOnDemandAuditTask() {
        this.add('iot:StartOnDemandAuditTask');
        return this;
    }
    /**
     * Starts a bulk thing registration task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartThingRegistrationTask.html
     */
    toStartThingRegistrationTask() {
        this.add('iot:StartThingRegistrationTask');
        return this;
    }
    /**
     * Stops a bulk thing registration task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StopThingRegistrationTask.html
     */
    toStopThingRegistrationTask() {
        this.add('iot:StopThingRegistrationTask');
        return this;
    }
    /**
     * Subscribe to the specified TopicFilter.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toSubscribe() {
        this.add('iot:Subscribe');
        return this;
    }
    /**
     * Tag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     */
    toTagResource() {
        this.add('iot:TagResource');
        return this;
    }
    /**
     * Test the policies evaluation for group policies
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TestAuthorization.html
     */
    toTestAuthorization() {
        this.add('iot:TestAuthorization');
        return this;
    }
    /**
     * Invoke the specified custom authorizer for testing purposes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TestInvokeAuthorizer.html
     */
    toTestInvokeAuthorizer() {
        this.add('iot:TestInvokeAuthorizer');
        return this;
    }
    /**
     * Transfers the specified certificate to the specified AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TransferCertificate.html
     */
    toTransferCertificate() {
        this.add('iot:TransferCertificate');
        return this;
    }
    /**
     * Untag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     */
    toUntagResource() {
        this.add('iot:UntagResource');
        return this;
    }
    /**
     * Configures or reconfigures the Device Defender audit settings for this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateAccountAuditConfiguration.html
     */
    toUpdateAccountAuditConfiguration() {
        this.add('iot:UpdateAccountAuditConfiguration');
        return this;
    }
    /**
     * Updates an authorizer
     *
     * Access Level: Write
     */
    toUpdateAuthorizer() {
        this.add('iot:UpdateAuthorizer');
        return this;
    }
    /**
     * Updates information associated with the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateBillingGroup.html
     */
    toUpdateBillingGroup() {
        this.add('iot:UpdateBillingGroup');
        return this;
    }
    /**
     * Updates a registered CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCACertificate.html
     */
    toUpdateCACertificate() {
        this.add('iot:UpdateCACertificate');
        return this;
    }
    /**
     * Updates the status of the specified certificate. This operation is idempotent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCertificate.html
     */
    toUpdateCertificate() {
        this.add('iot:UpdateCertificate');
        return this;
    }
    /**
     * Updates the definition for a dimension.
     *
     * Access Level: Write
     */
    toUpdateDimension() {
        this.add('iot:UpdateDimension');
        return this;
    }
    /**
     * Updates a Dynamic Thing Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateDynamicThingGroup.html
     */
    toUpdateDynamicThingGroup() {
        this.add('iot:UpdateDynamicThingGroup');
        return this;
    }
    /**
     * Updates event configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateEventConfigurations.html
     */
    toUpdateEventConfigurations() {
        this.add('iot:UpdateEventConfigurations');
        return this;
    }
    /**
     * Updates fleet indexing configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateIndexingConfiguration.html
     */
    toUpdateIndexingConfiguration() {
        this.add('iot:UpdateIndexingConfiguration');
        return this;
    }
    /**
     * Updates a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJob.html
     */
    toUpdateJob() {
        this.add('iot:UpdateJob');
        return this;
    }
    /**
     * Updates a job execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJobExecution.html
     */
    toUpdateJobExecution() {
        this.add('iot:UpdateJobExecution');
        return this;
    }
    /**
     * Updates the definition for the specified mitigation action.
     *
     * Access Level: Write
     */
    toUpdateMitigationAction() {
        this.add('iot:UpdateMitigationAction');
        return this;
    }
    /**
     * Updates a fleet provisioning template.
     *
     * Access Level: Write
     */
    toUpdateProvisioningTemplate() {
        this.add('iot:UpdateProvisioningTemplate');
        return this;
    }
    /**
     * Updates the role alias
     *
     * Access Level: Write
     */
    toUpdateRoleAlias() {
        this.add('iot:UpdateRoleAlias');
        return this;
    }
    /**
     * Updates a scheduled audit, including what checks are performed and how often the audit takes place.
     *
     * Access Level: Write
     */
    toUpdateScheduledAudit() {
        this.add('iot:UpdateScheduledAudit');
        return this;
    }
    /**
     * Updates a Device Defender security profile.
     *
     * Access Level: Write
     */
    toUpdateSecurityProfile() {
        this.add('iot:UpdateSecurityProfile');
        return this;
    }
    /**
     * Updates the data for a stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateStream.html
     */
    toUpdateStream() {
        this.add('iot:UpdateStream');
        return this;
    }
    /**
     * Updates information associated with the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThing.html
     */
    toUpdateThing() {
        this.add('iot:UpdateThing');
        return this;
    }
    /**
     * Updates information associated with the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroup.html
     */
    toUpdateThingGroup() {
        this.add('iot:UpdateThingGroup');
        return this;
    }
    /**
     * Updates the thing groups to which the thing belongs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroupsForThing.html
     */
    toUpdateThingGroupsForThing() {
        this.add('iot:UpdateThingGroupsForThing');
        return this;
    }
    /**
     * Updates the thing shadow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    toUpdateThingShadow() {
        this.add('iot:UpdateThingShadow');
        return this;
    }
    /**
     * Validates a Device Defender security profile behaviors specification.
     *
     * Access Level: Read
     */
    toValidateSecurityProfileBehaviors() {
        this.add('iot:ValidateSecurityProfileBehaviors');
        return this;
    }
    /**
     * Adds a resource of type client to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html
     *
     * @param clientId - Identifier for the clientId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onClient(clientId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:client/${ClientId}';
        arn = arn.replace('${ClientId}', clientId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-indexing.html
     *
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIndex(indexName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:index/${IndexName}';
        arn = arn.replace('${IndexName}', indexName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tunnel to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-tunnels.html
     *
     * @param tunnelId - Identifier for the tunnelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTunnel(tunnelId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:tunnel/${TunnelId}';
        arn = arn.replace('${TunnelId}', tunnelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thing to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onThing(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-groups.html
     *
     * @param thingGroupName - Identifier for the thingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThinggroup(thingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}';
        arn = arn.replace('${ThingGroupName}', thingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type billinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/billing-groups.html
     *
     * @param billingGroupName - Identifier for the billingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBillinggroup(billingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:billinggroup/${BillingGroupName}';
        arn = arn.replace('${BillingGroupName}', billingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dynamicthinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/dynamic-thing-groups.html
     *
     * @param thingGroupName - Identifier for the thingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDynamicthinggroup(thingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}';
        arn = arn.replace('${ThingGroupName}', thingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thingtype to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-types.html
     *
     * @param thingTypeName - Identifier for the thingTypeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThingtype(thingTypeName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thingtype/${ThingTypeName}';
        arn = arn.replace('${ThingTypeName}', thingTypeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:topic/${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type topicfilter to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/topics.html
     *
     * @param topicFilter - Identifier for the topicFilter.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopicfilter(topicFilter, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:topicfilter/${TopicFilter}';
        arn = arn.replace('${TopicFilter}', topicFilter);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rolealias to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/authorizing-direct-aws.html
     *
     * @param roleAlias - Identifier for the roleAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRolealias(roleAlias, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:rolealias/${RoleAlias}';
        arn = arn.replace('${RoleAlias}', roleAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type authorizer to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/custom-authorizer.html
     *
     * @param authorizerName - Identifier for the authorizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAuthorizer(authorizerName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:authorizer/${AuthorizerName}';
        arn = arn.replace('${AuthorizerName}', authorizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-policies.html
     *
     * @param policyName - Identifier for the policyName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(policyName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:policy/${PolicyName}';
        arn = arn.replace('${PolicyName}', policyName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html
     *
     * @param certificate - Identifier for the certificate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCert(certificate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:cert/${Certificate}';
        arn = arn.replace('${Certificate}', certificate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cacert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html
     *
     * @param cACertificate - Identifier for the cACertificate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCacert(cACertificate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:cacert/${CACertificate}';
        arn = arn.replace('${CACertificate}', cACertificate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html
     *
     * @param streamId - Identifier for the streamId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(streamId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:stream/${StreamId}';
        arn = arn.replace('${StreamId}', streamId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type otaupdate to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html
     *
     * @param otaUpdateId - Identifier for the otaUpdateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOtaupdate(otaUpdateId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:otaupdate/${OtaUpdateId}';
        arn = arn.replace('${OtaUpdateId}', otaUpdateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type scheduledaudit to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-audit.html
     *
     * @param scheduleName - Identifier for the scheduleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScheduledaudit(scheduleName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:scheduledaudit/${ScheduleName}';
        arn = arn.replace('${ScheduleName}', scheduleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mitigationaction to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-mitigation-actions.html
     *
     * @param mitigationActionName - Identifier for the mitigationActionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMitigationaction(mitigationActionName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:mitigationaction/${MitigationActionName}';
        arn = arn.replace('${MitigationActionName}', mitigationActionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securityprofile to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html
     *
     * @param securityProfileName - Identifier for the securityProfileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecurityprofile(securityProfileName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:securityprofile/${SecurityProfileName}';
        arn = arn.replace('${SecurityProfileName}', securityProfileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dimension to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html
     *
     * @param dimensionName - Identifier for the dimensionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDimension(dimensionName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:dimension/${DimensionName}';
        arn = arn.replace('${DimensionName}', dimensionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type provisioningtemplate to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/provision-template.html
     *
     * @param provisioningTemplate - Identifier for the provisioningTemplate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProvisioningtemplate(provisioningTemplate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:provisioningtemplate/${ProvisioningTemplate}';
        arn = arn.replace('${ProvisioningTemplate}', provisioningTemplate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The flag indicating whether or not to also delete an IoT Tunnel immediately
     *
     * Applies to actions:
     * - .toCloseTunnel()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifDelete(value) {
        return this.if(`iot:Delete`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The list of all IoT Thing Group ARNs that the destination IoT Thing belongs to for an IoT Tunnel
     *
     * Applies to actions:
     * - .toOpenTunnel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifThingGroupArn(value, operator) {
        return this.if(`iot:ThingGroupArn`, value, operator || 'StringLike');
    }
    /**
     * The list of all destination services for an IoT Tunnel
     *
     * Applies to actions:
     * - .toOpenTunnel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTunnelDestinationService(value, operator) {
        return this.if(`iot:TunnelDestinationService`, value, operator || 'StringLike');
    }
}
exports.Iot = Iot;
//# sourceMappingURL=data:application/json;base64,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