"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotanalytics = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotanalytics](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotanalytics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotanalytics extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotanalytics](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotanalytics.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotanalytics';
        this.actionList = {
            "BatchPutMessage": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_BatchPutMessage.html",
                "description": "Puts a batch of messages into the specified channel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "CancelPipelineReprocessing": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CancelPipelineReprocessing.html",
                "description": "Cancels reprocessing for the specified pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "CreateChannel": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateChannel.html",
                "description": "Creates a channel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDataset": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDataset.html",
                "description": "Creates a dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDatasetContent": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDatasetContent.html",
                "description": "Generates content of the specified dataset (by executing the dataset actions).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "CreateDatastore": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDatastore.html",
                "description": "Creates a datastore.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datastore": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePipeline": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreatePipeline.html",
                "description": "Creates a pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteChannel": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteChannel.html",
                "description": "Deletes the specified channel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DeleteDataset": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDataset.html",
                "description": "Deletes the specified dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DeleteDatasetContent": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDatasetContent.html",
                "description": "Deletes the content of the specified dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DeleteDatastore": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDatastore.html",
                "description": "Deletes the specified datastore.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datastore": {
                        "required": true
                    }
                }
            },
            "DeletePipeline": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeletePipeline.html",
                "description": "Deletes the specified pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "DescribeChannel": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeChannel.html",
                "description": "Describes the specified channel.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DescribeDataset": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeDataset.html",
                "description": "Describes the specified dataset.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeDatastore": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeDatastore.html",
                "description": "Describes the specified datastore.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datastore": {
                        "required": true
                    }
                }
            },
            "DescribeLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeLoggingOptions.html",
                "description": "Describes logging options for the the account.",
                "accessLevel": "Read"
            },
            "DescribePipeline": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribePipeline.html",
                "description": "Describes the specified pipeline.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "GetDatasetContent": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_GetDatasetContent.html",
                "description": "Gets the content of the specified dataset.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "ListChannels": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListChannels.html",
                "description": "Lists the channels for the account.",
                "accessLevel": "List"
            },
            "ListDatasets": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListDatasets.html",
                "description": "Lists the datasets for the account.",
                "accessLevel": "List"
            },
            "ListDatastores": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListDatastores.html",
                "description": "Lists the datastores for the account.",
                "accessLevel": "List"
            },
            "ListPipelines": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListPipelines.html",
                "description": "Lists the pipelines for the account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists the tags (metadata) which you have assigned to the resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "dataset": {
                        "required": false
                    },
                    "datastore": {
                        "required": false
                    },
                    "pipeline": {
                        "required": false
                    }
                }
            },
            "PutLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_PutLoggingOptions.html",
                "description": "Puts logging options for the the account.",
                "accessLevel": "Write"
            },
            "RunPipelineActivity": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_RunPipelineActivity.html",
                "description": "Runs the specified pipeline activity.",
                "accessLevel": "Read"
            },
            "SampleChannelData": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_SampleChannelData.html",
                "description": "Samples the specified channel's data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "StartPipelineReprocessing": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_StartPipelineReprocessing.html",
                "description": "Starts reprocessing for the specified pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_TagResource.html",
                "description": "Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "dataset": {
                        "required": false
                    },
                    "datastore": {
                        "required": false
                    },
                    "pipeline": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UntagResource.html",
                "description": "Removes the given tags (metadata) from the resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "dataset": {
                        "required": false
                    },
                    "datastore": {
                        "required": false
                    },
                    "pipeline": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateChannel": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateChannel.html",
                "description": "Updates the specified channel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "UpdateDataset": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateDataset.html",
                "description": "Updates the specified dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UpdateDatastore": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateDatastore.html",
                "description": "Updates the specified datastore.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datastore": {
                        "required": true
                    }
                }
            },
            "UpdatePipeline": {
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdatePipeline.html",
                "description": "Updates the specified pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "channel": {
                "name": "channel",
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how",
                "arn": "arn:${Partition}:iotanalytics:${Region}:${Account}:channel/${ChannelName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iotanalytics:ResourceTag/${TagKey}"
                ]
            },
            "dataset": {
                "name": "dataset",
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how",
                "arn": "arn:${Partition}:iotanalytics:${Region}:${Account}:dataset/${DatasetName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iotanalytics:ResourceTag/${TagKey}"
                ]
            },
            "datastore": {
                "name": "datastore",
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how",
                "arn": "arn:${Partition}:iotanalytics:${Region}:${Account}:datastore/${DatastoreName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iotanalytics:ResourceTag/${TagKey}"
                ]
            },
            "pipeline": {
                "name": "pipeline",
                "url": "https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how",
                "arn": "arn:${Partition}:iotanalytics:${Region}:${Account}:pipeline/${PipelineName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iotanalytics:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Puts a batch of messages into the specified channel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_BatchPutMessage.html
     */
    toBatchPutMessage() {
        this.add('iotanalytics:BatchPutMessage');
        return this;
    }
    /**
     * Cancels reprocessing for the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CancelPipelineReprocessing.html
     */
    toCancelPipelineReprocessing() {
        this.add('iotanalytics:CancelPipelineReprocessing');
        return this;
    }
    /**
     * Creates a channel.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        this.add('iotanalytics:CreateChannel');
        return this;
    }
    /**
     * Creates a dataset.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDataset.html
     */
    toCreateDataset() {
        this.add('iotanalytics:CreateDataset');
        return this;
    }
    /**
     * Generates content of the specified dataset (by executing the dataset actions).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDatasetContent.html
     */
    toCreateDatasetContent() {
        this.add('iotanalytics:CreateDatasetContent');
        return this;
    }
    /**
     * Creates a datastore.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDatastore.html
     */
    toCreateDatastore() {
        this.add('iotanalytics:CreateDatastore');
        return this;
    }
    /**
     * Creates a pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        this.add('iotanalytics:CreatePipeline');
        return this;
    }
    /**
     * Deletes the specified channel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        this.add('iotanalytics:DeleteChannel');
        return this;
    }
    /**
     * Deletes the specified dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        this.add('iotanalytics:DeleteDataset');
        return this;
    }
    /**
     * Deletes the content of the specified dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDatasetContent.html
     */
    toDeleteDatasetContent() {
        this.add('iotanalytics:DeleteDatasetContent');
        return this;
    }
    /**
     * Deletes the specified datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDatastore.html
     */
    toDeleteDatastore() {
        this.add('iotanalytics:DeleteDatastore');
        return this;
    }
    /**
     * Deletes the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        this.add('iotanalytics:DeletePipeline');
        return this;
    }
    /**
     * Describes the specified channel.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeChannel.html
     */
    toDescribeChannel() {
        this.add('iotanalytics:DescribeChannel');
        return this;
    }
    /**
     * Describes the specified dataset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        this.add('iotanalytics:DescribeDataset');
        return this;
    }
    /**
     * Describes the specified datastore.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeDatastore.html
     */
    toDescribeDatastore() {
        this.add('iotanalytics:DescribeDatastore');
        return this;
    }
    /**
     * Describes logging options for the the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeLoggingOptions.html
     */
    toDescribeLoggingOptions() {
        this.add('iotanalytics:DescribeLoggingOptions');
        return this;
    }
    /**
     * Describes the specified pipeline.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribePipeline.html
     */
    toDescribePipeline() {
        this.add('iotanalytics:DescribePipeline');
        return this;
    }
    /**
     * Gets the content of the specified dataset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_GetDatasetContent.html
     */
    toGetDatasetContent() {
        this.add('iotanalytics:GetDatasetContent');
        return this;
    }
    /**
     * Lists the channels for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        this.add('iotanalytics:ListChannels');
        return this;
    }
    /**
     * Lists the datasets for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        this.add('iotanalytics:ListDatasets');
        return this;
    }
    /**
     * Lists the datastores for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListDatastores.html
     */
    toListDatastores() {
        this.add('iotanalytics:ListDatastores');
        return this;
    }
    /**
     * Lists the pipelines for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        this.add('iotanalytics:ListPipelines');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('iotanalytics:ListTagsForResource');
        return this;
    }
    /**
     * Puts logging options for the the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        this.add('iotanalytics:PutLoggingOptions');
        return this;
    }
    /**
     * Runs the specified pipeline activity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_RunPipelineActivity.html
     */
    toRunPipelineActivity() {
        this.add('iotanalytics:RunPipelineActivity');
        return this;
    }
    /**
     * Samples the specified channel's data.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_SampleChannelData.html
     */
    toSampleChannelData() {
        this.add('iotanalytics:SampleChannelData');
        return this;
    }
    /**
     * Starts reprocessing for the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_StartPipelineReprocessing.html
     */
    toStartPipelineReprocessing() {
        this.add('iotanalytics:StartPipelineReprocessing');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('iotanalytics:TagResource');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('iotanalytics:UntagResource');
        return this;
    }
    /**
     * Updates the specified channel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        this.add('iotanalytics:UpdateChannel');
        return this;
    }
    /**
     * Updates the specified dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateDataset.html
     */
    toUpdateDataset() {
        this.add('iotanalytics:UpdateDataset');
        return this;
    }
    /**
     * Updates the specified datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateDatastore.html
     */
    toUpdateDatastore() {
        this.add('iotanalytics:UpdateDatastore');
        return this;
    }
    /**
     * Updates the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdatePipeline.html
     */
    toUpdatePipeline() {
        this.add('iotanalytics:UpdatePipeline');
        return this;
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onChannel(channelName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:channel/${ChannelName}';
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onDataset(datasetName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:dataset/${DatasetName}';
        arn = arn.replace('${DatasetName}', datasetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type datastore to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param datastoreName - Identifier for the datastoreName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onDatastore(datastoreName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:datastore/${DatastoreName}';
        arn = arn.replace('${DatastoreName}', datastoreName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onPipeline(pipelineName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:pipeline/${PipelineName}';
        arn = arn.replace('${PipelineName}', pipelineName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The preface string for a tag key and value pair attached to an IoT Analytics resource.
     *
     * Applies to resource types:
     * - channel
     * - dataset
     * - datastore
     * - pipeline
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iotanalytics:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Iotanalytics = Iotanalytics;
//# sourceMappingURL=data:application/json;base64,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