"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Greengrass = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [greengrass](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotgreengrass.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Greengrass extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [greengrass](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotgreengrass.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'greengrass';
        this.actionList = {
            "AssociateRoleToGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/associateroletogroup-put.html",
                "description": "Grants permission to associate a role with a group. The role's permissions must allow Greengrass core Lambda functions and connectors to perform actions in other AWS services.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "AssociateServiceRoleToAccount": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/associateserviceroletoaccount-put.html",
                "description": "Grants permission to associate a role with your account. AWS IoT Greengrass uses this role to access your Lambda functions and AWS IoT resources.",
                "accessLevel": "Permissions management"
            },
            "CreateConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinition-post.html",
                "description": "Grants permission to create a connector definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateConnectorDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing connector definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "CreateCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinition-post.html",
                "description": "Grants permission to create a core definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCoreDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing core definition. Greengrass groups must each contain exactly one Greengrass core.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createdeployment-post.html",
                "description": "Grants permission to create a deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinition-post.html",
                "description": "Grants permission to create a device definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDeviceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing device definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "CreateFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinition-post.html",
                "description": "Grants permission to create a Lambda function definition to be used in a group that contains a list of Lambda functions and their configurations.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFunctionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing Lambda function definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/creategroup-post.html",
                "description": "Grants permission to create a group.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGroupCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupcertificateauthority-post.html",
                "description": "Grants permission to create a CA for the group, or rotate the existing CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateGroupVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupversion-post.html",
                "description": "Grants permission to create a version of a group that has already been defined.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinition-post.html",
                "description": "Grants permission to create a logger definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoggerDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing logger definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "CreateResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinition-post.html",
                "description": "Grants permission to create a resource definition that contains a list of resources to be used in a group.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateResourceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing resource definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "CreateSoftwareUpdateJob": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createsoftwareupdatejob-post.html",
                "description": "Grants permission to create an AWS IoT job that will trigger your Greengrass cores to update the software they are running.",
                "accessLevel": "Write"
            },
            "CreateSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinition-post.html",
                "description": "Grants permission to create a subscription definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSubscriptionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing subscription definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deleteconnectordefinition-delete.html",
                "description": "Grants permission to delete a connector definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletecoredefinition-delete.html",
                "description": "Grants permission to delete a core definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletedevicedefinition-delete.html",
                "description": "Grants permission to delete a device definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletefunctiondefinition-delete.html",
                "description": "Grants permission to delete a Lambda function definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletegroup-delete.html",
                "description": "Grants permission to delete a group that is not currently in use in a deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deleteloggerdefinition-delete.html",
                "description": "Grants permission to delete a logger definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deleteresourcedefinition-delete.html",
                "description": "Grants permission to delete a resource definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletesubscriptiondefinition-delete.html",
                "description": "Grants permission to delete a subscription definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "DisassociateRoleFromGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/disassociaterolefromgroup-delete.html",
                "description": "Grants permission to disassociate the role from a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DisassociateServiceRoleFromAccount": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/disassociateservicerolefromaccount-delete.html",
                "description": "Grants permission to disassociate the service role from an account. Without a service role, deployments will not work.",
                "accessLevel": "Write"
            },
            "Discover": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-discover-api.html",
                "description": "Grants permission to retrieve information required to connect to a Greengrass core.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetAssociatedRole": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getassociatedrole-get.html",
                "description": "Grants permission to retrieve the role associated with a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetBulkDeploymentStatus": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getbulkdeploymentstatus-get.html",
                "description": "Grants permission to return the status of a bulk deployment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": true
                    }
                }
            },
            "GetConnectivityInfo": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectivityinfo-get.html",
                "description": "Grants permission to retrieve the connectivity information for a core.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectivityInfo": {
                        "required": true
                    }
                }
            },
            "GetConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinition-get.html",
                "description": "Grants permission to retrieve information about a connector definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "GetConnectorDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a connector definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    },
                    "connectorDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinition-get.html",
                "description": "Grants permission to retrieve information about a core definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "GetCoreDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a core definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    },
                    "coreDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetDeploymentStatus": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getdeploymentstatus-get.html",
                "description": "Grants permission to return the status of a deployment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deployment": {
                        "required": true
                    },
                    "group": {
                        "required": true
                    }
                }
            },
            "GetDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinition-get.html",
                "description": "Grants permission to retrieve information about a device definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "GetDeviceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a device definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    },
                    "deviceDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinition-get.html",
                "description": "Grants permission to retrieve information about a Lambda function definition, such as its creation time and latest version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "GetFunctionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a Lambda function definition version, such as which Lambda functions are included in the version and their configurations.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    },
                    "functionDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroup-get.html",
                "description": "Grants permission to retrieve information about a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateauthority-get.html",
                "description": "Grants permission to return the public key of the CA associated with a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificateAuthority": {
                        "required": true
                    },
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupCertificateConfiguration": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateconfiguration-get.html",
                "description": "Grants permission to retrieve the current configuration for the CA used by a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupversion-get.html",
                "description": "Grants permission to retrieve information about a group version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    },
                    "groupVersion": {
                        "required": true
                    }
                }
            },
            "GetLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinition-get.html",
                "description": "Grants permission to retrieve information about a logger definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "GetLoggerDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a logger definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    },
                    "loggerDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinition-get.html",
                "description": "Grants permission to retrieve information about a resource definition, such as its creation time and latest version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "GetResourceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a resource definition version, such as which resources are included in the version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    },
                    "resourceDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetServiceRoleForAccount": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getserviceroleforaccount-get.html",
                "description": "Grants permission to retrieve the service role that is attached to an account.",
                "accessLevel": "Read"
            },
            "GetSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinition-get.html",
                "description": "Grants permission to retrieve information about a subscription definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "GetSubscriptionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a subscription definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    },
                    "subscriptionDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "ListBulkDeploymentDetailedReports": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeploymentdetailedreports-get.html",
                "description": "Grants permission to retrieve a paginated list of the deployments that have been started in a bulk deployment operation and their current deployment status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": true
                    }
                }
            },
            "ListBulkDeployments": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeployments-get.html",
                "description": "Grants permission to retrieve a list of bulk deployments.",
                "accessLevel": "List"
            },
            "ListConnectorDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitionversions-get.html",
                "description": "Grants permission to list the versions of a connector definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "ListConnectorDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitions-get.html",
                "description": "Grants permission to retrieve a list of connector definitions.",
                "accessLevel": "List"
            },
            "ListCoreDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitionversions-get.html",
                "description": "Grants permission to list the versions of a core definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "ListCoreDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitions-get.html",
                "description": "Grants permission to retrieve a list of core definitions.",
                "accessLevel": "List"
            },
            "ListDeployments": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listdeployments-get.html",
                "description": "Grants permission to retrieve a list of all deployments for a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListDeviceDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitionversions-get.html",
                "description": "Grants permission to list the versions of a device definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "ListDeviceDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitions-get.html",
                "description": "Grants permission to retrieve a list of device definitions.",
                "accessLevel": "List"
            },
            "ListFunctionDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitionversions-get.html",
                "description": "Grants permission to list the versions of a Lambda function definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "ListFunctionDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitions-get.html",
                "description": "Grants permission to retrieve a list of Lambda function definitions.",
                "accessLevel": "List"
            },
            "ListGroupCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupcertificateauthorities-get.html",
                "description": "Grants permission to retrieve a list of current CAs for a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroupVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupversions-get.html",
                "description": "Grants permission to list the versions of a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listgroups-get.html",
                "description": "Grants permission to retrieve a list of groups.",
                "accessLevel": "List"
            },
            "ListLoggerDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitionversions-get.html",
                "description": "Grants permission to list the versions of a logger definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "ListLoggerDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitions-get.html",
                "description": "Grants permission to retrieve a list of logger definitions.",
                "accessLevel": "List"
            },
            "ListResourceDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitionversions-get.html",
                "description": "Grants permission to list the versions of a resource definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "ListResourceDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitions-get.html",
                "description": "Grants permission to retrieve a list of resource definitions.",
                "accessLevel": "List"
            },
            "ListSubscriptionDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitionversions-get.html",
                "description": "Grants permission to list the versions of a subscription definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "ListSubscriptionDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitions-get.html",
                "description": "Grants permission to retrieve a list of subscription definitions.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listtagsforresource-get.html",
                "description": "Grants permission to list the tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": false
                    },
                    "connectorDefinition": {
                        "required": false
                    },
                    "coreDefinition": {
                        "required": false
                    },
                    "deviceDefinition": {
                        "required": false
                    },
                    "functionDefinition": {
                        "required": false
                    },
                    "group": {
                        "required": false
                    },
                    "loggerDefinition": {
                        "required": false
                    },
                    "resourceDefinition": {
                        "required": false
                    },
                    "subscriptionDefinition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResetDeployments": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/resetdeployments-post.html",
                "description": "Grants permission to reset a group's deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "StartBulkDeployment": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/startbulkdeployment-post.html",
                "description": "Grants permission to deploy multiple groups in one operation.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StopBulkDeployment": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/stopbulkdeployment-put.html",
                "description": "Grants permission to stop the execution of a bulk deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/tagresource-post.html",
                "description": "Grants permission to add tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": false
                    },
                    "connectorDefinition": {
                        "required": false
                    },
                    "coreDefinition": {
                        "required": false
                    },
                    "deviceDefinition": {
                        "required": false
                    },
                    "functionDefinition": {
                        "required": false
                    },
                    "group": {
                        "required": false
                    },
                    "loggerDefinition": {
                        "required": false
                    },
                    "resourceDefinition": {
                        "required": false
                    },
                    "subscriptionDefinition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/untagresource-delete.html",
                "description": "Grants permission to remove tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": false
                    },
                    "connectorDefinition": {
                        "required": false
                    },
                    "coreDefinition": {
                        "required": false
                    },
                    "deviceDefinition": {
                        "required": false
                    },
                    "functionDefinition": {
                        "required": false
                    },
                    "group": {
                        "required": false
                    },
                    "loggerDefinition": {
                        "required": false
                    },
                    "resourceDefinition": {
                        "required": false
                    },
                    "subscriptionDefinition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateConnectivityInfo": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectivityinfo-put.html",
                "description": "Grants permission to update the connectivity information for a Greengrass core. Any devices that belong to the group that has this core will receive this information in order to find the location of the core and connect to it.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectivityInfo": {
                        "required": true
                    }
                }
            },
            "UpdateConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectordefinition-put.html",
                "description": "Grants permission to update a connector definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatecoredefinition-put.html",
                "description": "Grants permission to update a core definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatedevicedefinition-put.html",
                "description": "Grants permission to update a device definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatefunctiondefinition-put.html",
                "description": "Grants permission to update a Lambda function definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updategroup-put.html",
                "description": "Grants permission to update a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateGroupCertificateConfiguration": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updategroupcertificateconfiguration-put.html",
                "description": "Grants permission to update the certificate expiry time for a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateloggerdefinition-put.html",
                "description": "Grants permission to update a logger definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateresourcedefinition-put.html",
                "description": "Grants permission to update a resource definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatesubscriptiondefinition-put.html",
                "description": "Grants permission to update a subscription definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "connectivityInfo": {
                "name": "connectivityInfo",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectivityinfo.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/things/${ThingName}/connectivityInfo",
                "conditionKeys": []
            },
            "artifact": {
                "name": "artifact",
                "url": "",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}/artifacts/lambda/${ArtifactId}",
                "conditionKeys": []
            },
            "certificateAuthority": {
                "name": "certificateAuthority",
                "url": "https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-sec.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/certificateauthorities/${CertificateAuthorityId}",
                "conditionKeys": []
            },
            "deployment": {
                "name": "deployment",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-createdeploymentrequest.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}",
                "conditionKeys": []
            },
            "bulkDeployment": {
                "name": "bulkDeployment",
                "url": "https://docs.aws.amazon.com/greengrass/latest/developerguide/bulk-deploy-cli.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/bulk/deployments/${BulkDeploymentId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupinformation.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "groupVersion": {
                "name": "groupVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "coreDefinition": {
                "name": "coreDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-core.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "coreDefinitionVersion": {
                "name": "coreDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-coredefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "deviceDefinition": {
                "name": "deviceDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-device.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deviceDefinitionVersion": {
                "name": "deviceDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-devicedefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "functionDefinition": {
                "name": "functionDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-function.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "functionDefinitionVersion": {
                "name": "functionDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-functiondefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "subscriptionDefinition": {
                "name": "subscriptionDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscription.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "subscriptionDefinitionVersion": {
                "name": "subscriptionDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscriptiondefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "loggerDefinition": {
                "name": "loggerDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-logger.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "loggerDefinitionVersion": {
                "name": "loggerDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-loggerdefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "resourceDefinition": {
                "name": "resourceDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resource.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "resourceDefinitionVersion": {
                "name": "resourceDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resourcedefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "connectorDefinition": {
                "name": "connectorDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connector.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "connectorDefinitionVersion": {
                "name": "connectorDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectordefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "thing": {
                "name": "thing",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate a role with a group. The role's permissions must allow Greengrass core Lambda functions and connectors to perform actions in other AWS services.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/associateroletogroup-put.html
     */
    toAssociateRoleToGroup() {
        this.add('greengrass:AssociateRoleToGroup');
        return this;
    }
    /**
     * Grants permission to associate a role with your account. AWS IoT Greengrass uses this role to access your Lambda functions and AWS IoT resources.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/associateserviceroletoaccount-put.html
     */
    toAssociateServiceRoleToAccount() {
        this.add('greengrass:AssociateServiceRoleToAccount');
        return this;
    }
    /**
     * Grants permission to create a connector definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinition-post.html
     */
    toCreateConnectorDefinition() {
        this.add('greengrass:CreateConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinitionversion-post.html
     */
    toCreateConnectorDefinitionVersion() {
        this.add('greengrass:CreateConnectorDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a core definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinition-post.html
     */
    toCreateCoreDefinition() {
        this.add('greengrass:CreateCoreDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing core definition. Greengrass groups must each contain exactly one Greengrass core.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinitionversion-post.html
     */
    toCreateCoreDefinitionVersion() {
        this.add('greengrass:CreateCoreDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdeployment-post.html
     */
    toCreateDeployment() {
        this.add('greengrass:CreateDeployment');
        return this;
    }
    /**
     * Grants permission to create a device definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinition-post.html
     */
    toCreateDeviceDefinition() {
        this.add('greengrass:CreateDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing device definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinitionversion-post.html
     */
    toCreateDeviceDefinitionVersion() {
        this.add('greengrass:CreateDeviceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a Lambda function definition to be used in a group that contains a list of Lambda functions and their configurations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinition-post.html
     */
    toCreateFunctionDefinition() {
        this.add('greengrass:CreateFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing Lambda function definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinitionversion-post.html
     */
    toCreateFunctionDefinitionVersion() {
        this.add('greengrass:CreateFunctionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroup-post.html
     */
    toCreateGroup() {
        this.add('greengrass:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a CA for the group, or rotate the existing CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupcertificateauthority-post.html
     */
    toCreateGroupCertificateAuthority() {
        this.add('greengrass:CreateGroupCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to create a version of a group that has already been defined.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupversion-post.html
     */
    toCreateGroupVersion() {
        this.add('greengrass:CreateGroupVersion');
        return this;
    }
    /**
     * Grants permission to create a logger definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinition-post.html
     */
    toCreateLoggerDefinition() {
        this.add('greengrass:CreateLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing logger definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinitionversion-post.html
     */
    toCreateLoggerDefinitionVersion() {
        this.add('greengrass:CreateLoggerDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a resource definition that contains a list of resources to be used in a group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinition-post.html
     */
    toCreateResourceDefinition() {
        this.add('greengrass:CreateResourceDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinitionversion-post.html
     */
    toCreateResourceDefinitionVersion() {
        this.add('greengrass:CreateResourceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create an AWS IoT job that will trigger your Greengrass cores to update the software they are running.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsoftwareupdatejob-post.html
     */
    toCreateSoftwareUpdateJob() {
        this.add('greengrass:CreateSoftwareUpdateJob');
        return this;
    }
    /**
     * Grants permission to create a subscription definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinition-post.html
     */
    toCreateSubscriptionDefinition() {
        this.add('greengrass:CreateSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing subscription definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinitionversion-post.html
     */
    toCreateSubscriptionDefinitionVersion() {
        this.add('greengrass:CreateSubscriptionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to delete a connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteconnectordefinition-delete.html
     */
    toDeleteConnectorDefinition() {
        this.add('greengrass:DeleteConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to delete a core definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletecoredefinition-delete.html
     */
    toDeleteCoreDefinition() {
        this.add('greengrass:DeleteCoreDefinition');
        return this;
    }
    /**
     * Grants permission to delete a device definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletedevicedefinition-delete.html
     */
    toDeleteDeviceDefinition() {
        this.add('greengrass:DeleteDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to delete a Lambda function definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletefunctiondefinition-delete.html
     */
    toDeleteFunctionDefinition() {
        this.add('greengrass:DeleteFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to delete a group that is not currently in use in a deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletegroup-delete.html
     */
    toDeleteGroup() {
        this.add('greengrass:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete a logger definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteloggerdefinition-delete.html
     */
    toDeleteLoggerDefinition() {
        this.add('greengrass:DeleteLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to delete a resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteresourcedefinition-delete.html
     */
    toDeleteResourceDefinition() {
        this.add('greengrass:DeleteResourceDefinition');
        return this;
    }
    /**
     * Grants permission to delete a subscription definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletesubscriptiondefinition-delete.html
     */
    toDeleteSubscriptionDefinition() {
        this.add('greengrass:DeleteSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to disassociate the role from a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/disassociaterolefromgroup-delete.html
     */
    toDisassociateRoleFromGroup() {
        this.add('greengrass:DisassociateRoleFromGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the service role from an account. Without a service role, deployments will not work.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/disassociateservicerolefromaccount-delete.html
     */
    toDisassociateServiceRoleFromAccount() {
        this.add('greengrass:DisassociateServiceRoleFromAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information required to connect to a Greengrass core.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-discover-api.html
     */
    toDiscover() {
        this.add('greengrass:Discover');
        return this;
    }
    /**
     * Grants permission to retrieve the role associated with a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getassociatedrole-get.html
     */
    toGetAssociatedRole() {
        this.add('greengrass:GetAssociatedRole');
        return this;
    }
    /**
     * Grants permission to return the status of a bulk deployment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getbulkdeploymentstatus-get.html
     */
    toGetBulkDeploymentStatus() {
        this.add('greengrass:GetBulkDeploymentStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the connectivity information for a core.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectivityinfo-get.html
     */
    toGetConnectivityInfo() {
        this.add('greengrass:GetConnectivityInfo');
        return this;
    }
    /**
     * Grants permission to retrieve information about a connector definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinition-get.html
     */
    toGetConnectorDefinition() {
        this.add('greengrass:GetConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a connector definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinitionversion-get.html
     */
    toGetConnectorDefinitionVersion() {
        this.add('greengrass:GetConnectorDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a core definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinition-get.html
     */
    toGetCoreDefinition() {
        this.add('greengrass:GetCoreDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a core definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinitionversion-get.html
     */
    toGetCoreDefinitionVersion() {
        this.add('greengrass:GetCoreDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to return the status of a deployment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdeploymentstatus-get.html
     */
    toGetDeploymentStatus() {
        this.add('greengrass:GetDeploymentStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about a device definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinition-get.html
     */
    toGetDeviceDefinition() {
        this.add('greengrass:GetDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a device definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinitionversion-get.html
     */
    toGetDeviceDefinitionVersion() {
        this.add('greengrass:GetDeviceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a Lambda function definition, such as its creation time and latest version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinition-get.html
     */
    toGetFunctionDefinition() {
        this.add('greengrass:GetFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a Lambda function definition version, such as which Lambda functions are included in the version and their configurations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinitionversion-get.html
     */
    toGetFunctionDefinitionVersion() {
        this.add('greengrass:GetFunctionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroup-get.html
     */
    toGetGroup() {
        this.add('greengrass:GetGroup');
        return this;
    }
    /**
     * Grants permission to return the public key of the CA associated with a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateauthority-get.html
     */
    toGetGroupCertificateAuthority() {
        this.add('greengrass:GetGroupCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to retrieve the current configuration for the CA used by a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateconfiguration-get.html
     */
    toGetGroupCertificateConfiguration() {
        this.add('greengrass:GetGroupCertificateConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupversion-get.html
     */
    toGetGroupVersion() {
        this.add('greengrass:GetGroupVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a logger definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinition-get.html
     */
    toGetLoggerDefinition() {
        this.add('greengrass:GetLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a logger definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinitionversion-get.html
     */
    toGetLoggerDefinitionVersion() {
        this.add('greengrass:GetLoggerDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a resource definition, such as its creation time and latest version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinition-get.html
     */
    toGetResourceDefinition() {
        this.add('greengrass:GetResourceDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a resource definition version, such as which resources are included in the version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinitionversion-get.html
     */
    toGetResourceDefinitionVersion() {
        this.add('greengrass:GetResourceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve the service role that is attached to an account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getserviceroleforaccount-get.html
     */
    toGetServiceRoleForAccount() {
        this.add('greengrass:GetServiceRoleForAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information about a subscription definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinition-get.html
     */
    toGetSubscriptionDefinition() {
        this.add('greengrass:GetSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a subscription definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinitionversion-get.html
     */
    toGetSubscriptionDefinitionVersion() {
        this.add('greengrass:GetSubscriptionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a paginated list of the deployments that have been started in a bulk deployment operation and their current deployment status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeploymentdetailedreports-get.html
     */
    toListBulkDeploymentDetailedReports() {
        this.add('greengrass:ListBulkDeploymentDetailedReports');
        return this;
    }
    /**
     * Grants permission to retrieve a list of bulk deployments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeployments-get.html
     */
    toListBulkDeployments() {
        this.add('greengrass:ListBulkDeployments');
        return this;
    }
    /**
     * Grants permission to list the versions of a connector definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitionversions-get.html
     */
    toListConnectorDefinitionVersions() {
        this.add('greengrass:ListConnectorDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connector definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitions-get.html
     */
    toListConnectorDefinitions() {
        this.add('greengrass:ListConnectorDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a core definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitionversions-get.html
     */
    toListCoreDefinitionVersions() {
        this.add('greengrass:ListCoreDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of core definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitions-get.html
     */
    toListCoreDefinitions() {
        this.add('greengrass:ListCoreDefinitions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all deployments for a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdeployments-get.html
     */
    toListDeployments() {
        this.add('greengrass:ListDeployments');
        return this;
    }
    /**
     * Grants permission to list the versions of a device definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitionversions-get.html
     */
    toListDeviceDefinitionVersions() {
        this.add('greengrass:ListDeviceDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of device definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitions-get.html
     */
    toListDeviceDefinitions() {
        this.add('greengrass:ListDeviceDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a Lambda function definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitionversions-get.html
     */
    toListFunctionDefinitionVersions() {
        this.add('greengrass:ListFunctionDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of Lambda function definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitions-get.html
     */
    toListFunctionDefinitions() {
        this.add('greengrass:ListFunctionDefinitions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of current CAs for a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupcertificateauthorities-get.html
     */
    toListGroupCertificateAuthorities() {
        this.add('greengrass:ListGroupCertificateAuthorities');
        return this;
    }
    /**
     * Grants permission to list the versions of a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupversions-get.html
     */
    toListGroupVersions() {
        this.add('greengrass:ListGroupVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroups-get.html
     */
    toListGroups() {
        this.add('greengrass:ListGroups');
        return this;
    }
    /**
     * Grants permission to list the versions of a logger definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitionversions-get.html
     */
    toListLoggerDefinitionVersions() {
        this.add('greengrass:ListLoggerDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of logger definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitions-get.html
     */
    toListLoggerDefinitions() {
        this.add('greengrass:ListLoggerDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a resource definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitionversions-get.html
     */
    toListResourceDefinitionVersions() {
        this.add('greengrass:ListResourceDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of resource definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitions-get.html
     */
    toListResourceDefinitions() {
        this.add('greengrass:ListResourceDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a subscription definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitionversions-get.html
     */
    toListSubscriptionDefinitionVersions() {
        this.add('greengrass:ListSubscriptionDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of subscription definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitions-get.html
     */
    toListSubscriptionDefinitions() {
        this.add('greengrass:ListSubscriptionDefinitions');
        return this;
    }
    /**
     * Grants permission to list the tags for a resource.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listtagsforresource-get.html
     */
    toListTagsForResource() {
        this.add('greengrass:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to reset a group's deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/resetdeployments-post.html
     */
    toResetDeployments() {
        this.add('greengrass:ResetDeployments');
        return this;
    }
    /**
     * Grants permission to deploy multiple groups in one operation.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/startbulkdeployment-post.html
     */
    toStartBulkDeployment() {
        this.add('greengrass:StartBulkDeployment');
        return this;
    }
    /**
     * Grants permission to stop the execution of a bulk deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/stopbulkdeployment-put.html
     */
    toStopBulkDeployment() {
        this.add('greengrass:StopBulkDeployment');
        return this;
    }
    /**
     * Grants permission to add tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/tagresource-post.html
     */
    toTagResource() {
        this.add('greengrass:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/untagresource-delete.html
     */
    toUntagResource() {
        this.add('greengrass:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the connectivity information for a Greengrass core. Any devices that belong to the group that has this core will receive this information in order to find the location of the core and connect to it.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectivityinfo-put.html
     */
    toUpdateConnectivityInfo() {
        this.add('greengrass:UpdateConnectivityInfo');
        return this;
    }
    /**
     * Grants permission to update a connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectordefinition-put.html
     */
    toUpdateConnectorDefinition() {
        this.add('greengrass:UpdateConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to update a core definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatecoredefinition-put.html
     */
    toUpdateCoreDefinition() {
        this.add('greengrass:UpdateCoreDefinition');
        return this;
    }
    /**
     * Grants permission to update a device definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatedevicedefinition-put.html
     */
    toUpdateDeviceDefinition() {
        this.add('greengrass:UpdateDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to update a Lambda function definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatefunctiondefinition-put.html
     */
    toUpdateFunctionDefinition() {
        this.add('greengrass:UpdateFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to update a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updategroup-put.html
     */
    toUpdateGroup() {
        this.add('greengrass:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to update the certificate expiry time for a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updategroupcertificateconfiguration-put.html
     */
    toUpdateGroupCertificateConfiguration() {
        this.add('greengrass:UpdateGroupCertificateConfiguration');
        return this;
    }
    /**
     * Grants permission to update a logger definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateloggerdefinition-put.html
     */
    toUpdateLoggerDefinition() {
        this.add('greengrass:UpdateLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to update a resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateresourcedefinition-put.html
     */
    toUpdateResourceDefinition() {
        this.add('greengrass:UpdateResourceDefinition');
        return this;
    }
    /**
     * Grants permission to update a subscription definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatesubscriptiondefinition-put.html
     */
    toUpdateSubscriptionDefinition() {
        this.add('greengrass:UpdateSubscriptionDefinition');
        return this;
    }
    /**
     * Adds a resource of type connectivityInfo to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectivityinfo.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectivityInfo(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/things/${ThingName}/connectivityInfo';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type artifact to the statement
     *
     * @param groupId - Identifier for the groupId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param artifactId - Identifier for the artifactId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArtifact(groupId, deploymentId, artifactId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}/artifacts/lambda/${ArtifactId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${DeploymentId}', deploymentId);
        arn = arn.replace('${ArtifactId}', artifactId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type certificateAuthority to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-sec.html
     *
     * @param groupId - Identifier for the groupId.
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAuthority(groupId, certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/certificateauthorities/${CertificateAuthorityId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-createdeploymentrequest.html
     *
     * @param groupId - Identifier for the groupId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeployment(groupId, deploymentId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${DeploymentId}', deploymentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bulkDeployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/bulk-deploy-cli.html
     *
     * @param bulkDeploymentId - Identifier for the bulkDeploymentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBulkDeployment(bulkDeploymentId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/bulk/deployments/${BulkDeploymentId}';
        arn = arn.replace('${BulkDeploymentId}', bulkDeploymentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupinformation.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type groupVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupversion.html
     *
     * @param groupId - Identifier for the groupId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroupVersion(groupId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/versions/${VersionId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type coreDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-core.html
     *
     * @param coreDefinitionId - Identifier for the coreDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCoreDefinition(coreDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}';
        arn = arn.replace('${CoreDefinitionId}', coreDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type coreDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-coredefinitionversion.html
     *
     * @param coreDefinitionId - Identifier for the coreDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCoreDefinitionVersion(coreDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${CoreDefinitionId}', coreDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-device.html
     *
     * @param deviceDefinitionId - Identifier for the deviceDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeviceDefinition(deviceDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}';
        arn = arn.replace('${DeviceDefinitionId}', deviceDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-devicedefinitionversion.html
     *
     * @param deviceDefinitionId - Identifier for the deviceDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeviceDefinitionVersion(deviceDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${DeviceDefinitionId}', deviceDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type functionDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-function.html
     *
     * @param functionDefinitionId - Identifier for the functionDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionDefinition(functionDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}';
        arn = arn.replace('${FunctionDefinitionId}', functionDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type functionDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-functiondefinitionversion.html
     *
     * @param functionDefinitionId - Identifier for the functionDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionDefinitionVersion(functionDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${FunctionDefinitionId}', functionDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subscriptionDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscription.html
     *
     * @param subscriptionDefinitionId - Identifier for the subscriptionDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubscriptionDefinition(subscriptionDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}';
        arn = arn.replace('${SubscriptionDefinitionId}', subscriptionDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subscriptionDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscriptiondefinitionversion.html
     *
     * @param subscriptionDefinitionId - Identifier for the subscriptionDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSubscriptionDefinitionVersion(subscriptionDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${SubscriptionDefinitionId}', subscriptionDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loggerDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-logger.html
     *
     * @param loggerDefinitionId - Identifier for the loggerDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLoggerDefinition(loggerDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}';
        arn = arn.replace('${LoggerDefinitionId}', loggerDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loggerDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-loggerdefinitionversion.html
     *
     * @param loggerDefinitionId - Identifier for the loggerDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoggerDefinitionVersion(loggerDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${LoggerDefinitionId}', loggerDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourceDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resource.html
     *
     * @param resourceDefinitionId - Identifier for the resourceDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourceDefinition(resourceDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}';
        arn = arn.replace('${ResourceDefinitionId}', resourceDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourceDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resourcedefinitionversion.html
     *
     * @param resourceDefinitionId - Identifier for the resourceDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceDefinitionVersion(resourceDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${ResourceDefinitionId}', resourceDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connectorDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connector.html
     *
     * @param connectorDefinitionId - Identifier for the connectorDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectorDefinition(connectorDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}';
        arn = arn.replace('${ConnectorDefinitionId}', connectorDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connectorDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectordefinitionversion.html
     *
     * @param connectorDefinitionId - Identifier for the connectorDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectorDefinitionVersion(connectorDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${ConnectorDefinitionId}', connectorDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thing to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onThing(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Greengrass = Greengrass;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzaW90Z3JlZW5ncmFzcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImF3c2lvdGdyZWVuZ3Jhc3MudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsc0NBQW9FO0FBRXBFOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQWtsQzdDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBdmxDTixrQkFBYSxHQUFHLFlBQVksQ0FBQztRQUMxQixlQUFVLEdBQVk7WUFDOUIsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSxpTEFBaUw7Z0JBQ2hNLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsK0JBQStCLEVBQUU7Z0JBQy9CLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSxtSkFBbUo7Z0JBQ2xLLGFBQWEsRUFBRSx3QkFBd0I7YUFDeEM7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGtDQUFrQyxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsdUdBQXVHO2dCQUM5RyxhQUFhLEVBQUUsNEVBQTRFO2dCQUMzRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHFCQUFxQixFQUFFO3dCQUNyQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHNCQUFzQixFQUFFO2dCQUN0QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsZ0RBQWdEO2dCQUMvRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSx3SUFBd0k7Z0JBQ3ZKLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSwyQ0FBMkM7Z0JBQzFELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLG9HQUFvRztnQkFDM0csYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLCtGQUErRjtnQkFDdEcsYUFBYSxFQUFFLG1KQUFtSjtnQkFDbEssYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGlDQUFpQyxFQUFFO2dCQUNqQyxLQUFLLEVBQUUsc0dBQXNHO2dCQUM3RyxhQUFhLEVBQUUsa0ZBQWtGO2dCQUNqRyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsc0NBQXNDO2dCQUNyRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsaUNBQWlDLEVBQUU7Z0JBQ2pDLEtBQUssRUFBRSxzR0FBc0c7Z0JBQzdHLGFBQWEsRUFBRSw0RUFBNEU7Z0JBQzNGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSxpRkFBaUY7Z0JBQ2hHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLG9HQUFvRztnQkFDM0csYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLCtGQUErRjtnQkFDdEcsYUFBYSxFQUFFLDRHQUE0RztnQkFDM0gsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGlDQUFpQyxFQUFFO2dCQUNqQyxLQUFLLEVBQUUsc0dBQXNHO2dCQUM3RyxhQUFhLEVBQUUsMkVBQTJFO2dCQUMxRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsOEZBQThGO2dCQUNyRyxhQUFhLEVBQUUsNkhBQTZIO2dCQUM1SSxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELDhCQUE4QixFQUFFO2dCQUM5QixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxhQUFhLEVBQUUsd0RBQXdEO2dCQUN2RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QscUNBQXFDLEVBQUU7Z0JBQ3JDLEtBQUssRUFBRSwwR0FBMEc7Z0JBQ2pILGFBQWEsRUFBRSwrRUFBK0U7Z0JBQzlGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysd0JBQXdCLEVBQUU7d0JBQ3hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxxREFBcUQ7Z0JBQ3BFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YscUJBQXFCLEVBQUU7d0JBQ3JCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSwySUFBMkk7Z0JBQzFKLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw2SUFBNkk7Z0JBQzVKLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSxzSkFBc0o7Z0JBQ3JLLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxtRkFBbUY7Z0JBQ2xHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw2SUFBNkk7Z0JBQzVKLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSxvREFBb0Q7Z0JBQ25FLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsOEJBQThCLEVBQUU7Z0JBQzlCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSxtSkFBbUo7Z0JBQ2xLLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysd0JBQXdCLEVBQUU7d0JBQ3hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSwwREFBMEQ7Z0JBQ3pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0NBQW9DLEVBQUU7Z0JBQ3BDLEtBQUssRUFBRSwyR0FBMkc7Z0JBQ2xILGFBQWEsRUFBRSx3SEFBd0g7Z0JBQ3ZJLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSxxRkFBcUY7Z0JBQ3BHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QseUJBQXlCLEVBQUU7Z0JBQ3pCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSw4REFBOEQ7Z0JBQzdFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSx3RUFBd0U7Z0JBQ3ZGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YscUJBQXFCLEVBQUU7d0JBQ3JCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsK0JBQStCLEVBQUU7Z0JBQy9CLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSxpRkFBaUY7Z0JBQ2hHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YscUJBQXFCLEVBQUU7d0JBQ3JCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCw0QkFBNEIsRUFBRTt3QkFDNUIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLG9FQUFvRTtnQkFDbkYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLDhGQUE4RjtnQkFDckcsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELHVCQUF1QixFQUFFO3dCQUN2QixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUseUZBQXlGO2dCQUNoRyxhQUFhLEVBQUUseURBQXlEO2dCQUN4RSxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFlBQVksRUFBRTt3QkFDWixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSxzRUFBc0U7Z0JBQ3JGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw4RUFBOEU7Z0JBQzdGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCx5QkFBeUIsRUFBRTt3QkFDekIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLDZIQUE2SDtnQkFDNUksYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLDRLQUE0SztnQkFDM0wsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELDJCQUEyQixFQUFFO3dCQUMzQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELFVBQVUsRUFBRTtnQkFDVixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsMERBQTBEO2dCQUN6RSxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDhCQUE4QixFQUFFO2dCQUM5QixLQUFLLEVBQUUsa0dBQWtHO2dCQUN6RyxhQUFhLEVBQUUsK0VBQStFO2dCQUM5RixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLHNCQUFzQixFQUFFO3dCQUN0QixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0NBQWtDLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSxzR0FBc0c7Z0JBQzdHLGFBQWEsRUFBRSxxRkFBcUY7Z0JBQ3BHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSxrRUFBa0U7Z0JBQ2pGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLDhFQUE4RTtnQkFDN0YsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELHlCQUF5QixFQUFFO3dCQUN6QixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHVCQUF1QixFQUFFO2dCQUN2QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsc0hBQXNIO2dCQUNySSxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDhCQUE4QixFQUFFO2dCQUM5QixLQUFLLEVBQUUsa0dBQWtHO2dCQUN6RyxhQUFhLEVBQUUscUlBQXFJO2dCQUNwSixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsMkJBQTJCLEVBQUU7d0JBQzNCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxnRkFBZ0Y7Z0JBQy9GLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw0RUFBNEU7Z0JBQzNGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysd0JBQXdCLEVBQUU7d0JBQ3hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0NBQWtDLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSxzR0FBc0c7Z0JBQzdHLGFBQWEsRUFBRSxvRkFBb0Y7Z0JBQ25HLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysd0JBQXdCLEVBQUU7d0JBQ3hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCwrQkFBK0IsRUFBRTt3QkFDL0IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQ0FBbUMsRUFBRTtnQkFDbkMsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLDhKQUE4SjtnQkFDN0ssYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsYUFBYSxFQUFFLG1FQUFtRTtnQkFDbEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixxQkFBcUIsRUFBRTt3QkFDckIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLDhGQUE4RjtnQkFDckcsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxnQ0FBZ0MsRUFBRTtnQkFDaEMsS0FBSyxFQUFFLG9HQUFvRztnQkFDM0csYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsYUFBYSxFQUFFLGtFQUFrRTtnQkFDakYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLG9EQUFvRDtnQkFDbkUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxnQ0FBZ0MsRUFBRTtnQkFDaEMsS0FBSyxFQUFFLG9HQUFvRztnQkFDM0csYUFBYSxFQUFFLGtFQUFrRTtnQkFDakYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLCtEQUErRDtnQkFDOUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxvQ0FBb0MsRUFBRTtnQkFDcEMsS0FBSyxFQUFFLHdHQUF3RztnQkFDL0csYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZix3QkFBd0IsRUFBRTt3QkFDeEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw2QkFBNkIsRUFBRTtnQkFDN0IsS0FBSyxFQUFFLGlHQUFpRztnQkFDeEcsYUFBYSxFQUFFLG1FQUFtRTtnQkFDbEYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLG9EQUFvRDtnQkFDbkUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELHFCQUFxQixFQUFFO3dCQUNyQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsd0JBQXdCLEVBQUU7d0JBQ3hCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLCtEQUErRDtnQkFDOUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUsK0RBQStEO2dCQUM5RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGdCQUFnQixFQUFFO3dCQUNoQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsOENBQThDO2dCQUM3RCxhQUFhLEVBQUUsU0FBUztnQkFDeEIsZUFBZSxFQUFFO29CQUNmLGdCQUFnQixFQUFFO3dCQUNoQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QscUJBQXFCLEVBQUU7d0JBQ3JCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGtCQUFrQixFQUFFO3dCQUNsQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGtCQUFrQixFQUFFO3dCQUNsQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCx3QkFBd0IsRUFBRTt3QkFDeEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixhQUFhLEVBQUUsbURBQW1EO2dCQUNsRSxhQUFhLEVBQUUsU0FBUztnQkFDeEIsZUFBZSxFQUFFO29CQUNmLGdCQUFnQixFQUFFO3dCQUNoQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QscUJBQXFCLEVBQUU7d0JBQ3JCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGtCQUFrQixFQUFFO3dCQUNsQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGtCQUFrQixFQUFFO3dCQUNsQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCx3QkFBd0IsRUFBRTt3QkFDeEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLG9PQUFvTztnQkFDblAsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLCtGQUErRjtnQkFDdEcsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixxQkFBcUIsRUFBRTt3QkFDckIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLGdEQUFnRDtnQkFDL0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLGtEQUFrRDtnQkFDakUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLDhGQUE4RjtnQkFDckcsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQ0FBcUMsRUFBRTtnQkFDckMsS0FBSyxFQUFFLHlHQUF5RztnQkFDaEgsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLGtEQUFrRDtnQkFDakUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLDhGQUE4RjtnQkFDckcsYUFBYSxFQUFFLG9EQUFvRDtnQkFDbkUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLHdEQUF3RDtnQkFDdkUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZix3QkFBd0IsRUFBRTt3QkFDeEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO1FBQ1Esa0JBQWEsR0FBa0I7WUFDdkMsa0JBQWtCLEVBQUU7Z0JBQ2xCLE1BQU0sRUFBRSxrQkFBa0I7Z0JBQzFCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxVQUFVO2dCQUNsQixLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsMklBQTJJO2dCQUNsSixlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELHNCQUFzQixFQUFFO2dCQUN0QixNQUFNLEVBQUUsc0JBQXNCO2dCQUM5QixLQUFLLEVBQUUsMEVBQTBFO2dCQUNqRixLQUFLLEVBQUUsaUlBQWlJO2dCQUN4SSxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELFlBQVksRUFBRTtnQkFDWixNQUFNLEVBQUUsWUFBWTtnQkFDcEIsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsS0FBSyxFQUFFLDRHQUE0RztnQkFDbkgsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsTUFBTSxFQUFFLGdCQUFnQjtnQkFDeEIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsS0FBSyxFQUFFLG1HQUFtRztnQkFDMUcsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsT0FBTztnQkFDZixLQUFLLEVBQUUsOEZBQThGO2dCQUNyRyxLQUFLLEVBQUUsZ0ZBQWdGO2dCQUN2RixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLE1BQU0sRUFBRSxjQUFjO2dCQUN0QixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxLQUFLLEVBQUUsc0dBQXNHO2dCQUM3RyxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixNQUFNLEVBQUUsZ0JBQWdCO2dCQUN4QixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLE1BQU0sRUFBRSx1QkFBdUI7Z0JBQy9CLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLEtBQUssRUFBRSx5SEFBeUg7Z0JBQ2hJLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLE1BQU0sRUFBRSxrQkFBa0I7Z0JBQzFCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLEtBQUssRUFBRSx1R0FBdUc7Z0JBQzlHLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsTUFBTSxFQUFFLHlCQUF5QjtnQkFDakMsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsS0FBSyxFQUFFLDZIQUE2SDtnQkFDcEksZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsTUFBTSxFQUFFLG9CQUFvQjtnQkFDNUIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsS0FBSyxFQUFFLDJHQUEyRztnQkFDbEgsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixNQUFNLEVBQUUsMkJBQTJCO2dCQUNuQyxLQUFLLEVBQUUsdUdBQXVHO2dCQUM5RyxLQUFLLEVBQUUsaUlBQWlJO2dCQUN4SSxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELHdCQUF3QixFQUFFO2dCQUN4QixNQUFNLEVBQUUsd0JBQXdCO2dCQUNoQyxLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxLQUFLLEVBQUUsbUhBQW1IO2dCQUMxSCxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsK0JBQStCLEVBQUU7Z0JBQy9CLE1BQU0sRUFBRSwrQkFBK0I7Z0JBQ3ZDLEtBQUssRUFBRSwyR0FBMkc7Z0JBQ2xILEtBQUssRUFBRSx5SUFBeUk7Z0JBQ2hKLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLE1BQU0sRUFBRSxrQkFBa0I7Z0JBQzFCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLEtBQUssRUFBRSx1R0FBdUc7Z0JBQzlHLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsTUFBTSxFQUFFLHlCQUF5QjtnQkFDakMsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsS0FBSyxFQUFFLDZIQUE2SDtnQkFDcEksZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsTUFBTSxFQUFFLG9CQUFvQjtnQkFDNUIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsS0FBSyxFQUFFLDJHQUEyRztnQkFDbEgsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixNQUFNLEVBQUUsMkJBQTJCO2dCQUNuQyxLQUFLLEVBQUUsdUdBQXVHO2dCQUM5RyxLQUFLLEVBQUUsaUlBQWlJO2dCQUN4SSxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixNQUFNLEVBQUUscUJBQXFCO2dCQUM3QixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixLQUFLLEVBQUUsNkdBQTZHO2dCQUNwSCxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLE1BQU0sRUFBRSw0QkFBNEI7Z0JBQ3BDLEtBQUssRUFBRSx3R0FBd0c7Z0JBQy9HLEtBQUssRUFBRSxtSUFBbUk7Z0JBQzFJLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRSxPQUFPO2dCQUNmLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLEtBQUssRUFBRSw4REFBOEQ7Z0JBQ3JFLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxJQUFJLENBQUMsR0FBRyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7UUFDckQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDakQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLElBQUksQ0FBQyxHQUFHLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxHQUFHLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxHQUFHLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDhCQUE4QjtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLElBQUksQ0FBQyxHQUFHLENBQUMsZ0RBQWdELENBQUMsQ0FBQztRQUMzRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDakQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0NBQStDLENBQUMsQ0FBQztRQUMxRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxHQUFHLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFDbEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxHQUFHLENBQUMseUNBQXlDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ3hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxHQUFHLENBQUMseUNBQXlDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDakQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLElBQUksQ0FBQyxHQUFHLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQ0FBbUM7UUFDeEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1FBQ3pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLElBQUksQ0FBQyxHQUFHLENBQUMsNENBQTRDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFDbEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLElBQUksQ0FBQyxHQUFHLENBQUMsNENBQTRDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0NBQStDLENBQUMsQ0FBQztRQUMxRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsR0FBRyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7UUFDM0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksa0JBQWtCLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoRyxJQUFJLEdBQUcsR0FBRyxtR0FBbUcsQ0FBQztRQUM5RyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxVQUFVLENBQUMsT0FBZSxFQUFFLFlBQW9CLEVBQUUsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoSSxJQUFJLEdBQUcsR0FBRywySUFBMkksQ0FBQztRQUN0SixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDekMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDbkQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHNCQUFzQixDQUFDLE9BQWUsRUFBRSxzQkFBOEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNsSSxJQUFJLEdBQUcsR0FBRyxpSUFBaUksQ0FBQztRQUM1SSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDekMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsMkJBQTJCLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztRQUN2RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxZQUFZLENBQUMsT0FBZSxFQUFFLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUcsSUFBSSxHQUFHLEdBQUcsNEdBQTRHLENBQUM7UUFDdkgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsSUFBSSxHQUFHLEdBQUcsbUdBQW1HLENBQUM7UUFDOUcsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMscUJBQXFCLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUMzRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE9BQU8sQ0FBQyxPQUFlLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkYsSUFBSSxHQUFHLEdBQUcsZ0ZBQWdGLENBQUM7UUFDM0YsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWMsQ0FBQyxPQUFlLEVBQUUsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RyxJQUFJLEdBQUcsR0FBRyxzR0FBc0csQ0FBQztRQUNqSCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDekMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsSUFBSSxHQUFHLEdBQUcsbUdBQW1HLENBQUM7UUFDOUcsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMscUJBQXFCLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUMzRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx1QkFBdUIsQ0FBQyxnQkFBd0IsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9ILElBQUksR0FBRyxHQUFHLHlIQUF5SCxDQUFDO1FBQ3BJLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLHFCQUFxQixFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFDM0QsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsa0JBQTBCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekcsSUFBSSxHQUFHLEdBQUcsdUdBQXVHLENBQUM7UUFDbEgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsdUJBQXVCLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUMvRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx5QkFBeUIsQ0FBQyxrQkFBMEIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25JLElBQUksR0FBRyxHQUFHLDZIQUE2SCxDQUFDO1FBQ3hJLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLHVCQUF1QixFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDL0QsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CLENBQUMsb0JBQTRCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0csSUFBSSxHQUFHLEdBQUcsMkdBQTJHLENBQUM7UUFDdEgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMseUJBQXlCLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztRQUNuRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwyQkFBMkIsQ0FBQyxvQkFBNEIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZJLElBQUksR0FBRyxHQUFHLGlJQUFpSSxDQUFDO1FBQzVJLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLHlCQUF5QixFQUFFLG9CQUFvQixDQUFDLENBQUM7UUFDbkUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksd0JBQXdCLENBQUMsd0JBQWdDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckgsSUFBSSxHQUFHLEdBQUcsbUhBQW1ILENBQUM7UUFDOUgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsNkJBQTZCLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztRQUMzRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwrQkFBK0IsQ0FBQyx3QkFBZ0MsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9JLElBQUksR0FBRyxHQUFHLHlJQUF5SSxDQUFDO1FBQ3BKLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLDZCQUE2QixFQUFFLHdCQUF3QixDQUFDLENBQUM7UUFDM0UsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsa0JBQTBCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekcsSUFBSSxHQUFHLEdBQUcsdUdBQXVHLENBQUM7UUFDbEgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsdUJBQXVCLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUMvRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx5QkFBeUIsQ0FBQyxrQkFBMEIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25JLElBQUksR0FBRyxHQUFHLDZIQUE2SCxDQUFDO1FBQ3hJLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLHVCQUF1QixFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDL0QsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CLENBQUMsb0JBQTRCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0csSUFBSSxHQUFHLEdBQUcsMkdBQTJHLENBQUM7UUFDdEgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMseUJBQXlCLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztRQUNuRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwyQkFBMkIsQ0FBQyxvQkFBNEIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZJLElBQUksR0FBRyxHQUFHLGlJQUFpSSxDQUFDO1FBQzVJLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLHlCQUF5QixFQUFFLG9CQUFvQixDQUFDLENBQUM7UUFDbkUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kscUJBQXFCLENBQUMscUJBQTZCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0csSUFBSSxHQUFHLEdBQUcsNkdBQTZHLENBQUM7UUFDeEgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsMEJBQTBCLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUNyRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSw0QkFBNEIsQ0FBQyxxQkFBNkIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pJLElBQUksR0FBRyxHQUFHLG1JQUFtSSxDQUFDO1FBQzlJLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLDBCQUEwQixFQUFFLHFCQUFxQixDQUFDLENBQUM7UUFDckUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksT0FBTyxDQUFDLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsSUFBSSxHQUFHLEdBQUcsOERBQThELENBQUM7UUFDekUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUFocUZELGdDQWdxRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtncmVlbmdyYXNzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3Npb3RncmVlbmdyYXNzLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgR3JlZW5ncmFzcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2dyZWVuZ3Jhc3MnO1xuICBwcm90ZWN0ZWQgYWN0aW9uTGlzdDogQWN0aW9ucyA9IHtcbiAgICBcIkFzc29jaWF0ZVJvbGVUb0dyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9hc3NvY2lhdGVyb2xldG9ncm91cC1wdXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhIHJvbGUgd2l0aCBhIGdyb3VwLiBUaGUgcm9sZSdzIHBlcm1pc3Npb25zIG11c3QgYWxsb3cgR3JlZW5ncmFzcyBjb3JlIExhbWJkYSBmdW5jdGlvbnMgYW5kIGNvbm5lY3RvcnMgdG8gcGVyZm9ybSBhY3Rpb25zIGluIG90aGVyIEFXUyBzZXJ2aWNlcy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXNzb2NpYXRlU2VydmljZVJvbGVUb0FjY291bnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2Fzc29jaWF0ZXNlcnZpY2Vyb2xldG9hY2NvdW50LXB1dC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGEgcm9sZSB3aXRoIHlvdXIgYWNjb3VudC4gQVdTIElvVCBHcmVlbmdyYXNzIHVzZXMgdGhpcyByb2xlIHRvIGFjY2VzcyB5b3VyIExhbWJkYSBmdW5jdGlvbnMgYW5kIEFXUyBJb1QgcmVzb3VyY2VzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIlxuICAgIH0sXG4gICAgXCJDcmVhdGVDb25uZWN0b3JEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVjb25uZWN0b3JkZWZpbml0aW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGNvbm5lY3RvciBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlQ29ubmVjdG9yRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWNvbm5lY3RvcmRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHZlcnNpb24gb2YgYW4gZXhpc3RpbmcgY29ubmVjdG9yIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY29ubmVjdG9yRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlQ29yZURlZmluaXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWNvcmVkZWZpbml0aW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGNvcmUgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZUNvcmVEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlY29yZWRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHZlcnNpb24gb2YgYW4gZXhpc3RpbmcgY29yZSBkZWZpbml0aW9uLiBHcmVlbmdyYXNzIGdyb3VwcyBtdXN0IGVhY2ggY29udGFpbiBleGFjdGx5IG9uZSBHcmVlbmdyYXNzIGNvcmUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY29yZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZURlcGxveW1lbnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWRlcGxveW1lbnQtcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgZGVwbG95bWVudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlRGV2aWNlRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlZGV2aWNlZGVmaW5pdGlvbi1wb3N0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBkZXZpY2UgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZURldmljZURlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVkZXZpY2VkZWZpbml0aW9udmVyc2lvbi1wb3N0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB2ZXJzaW9uIG9mIGFuIGV4aXN0aW5nIGRldmljZSBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldmljZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUZ1bmN0aW9uRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlZnVuY3Rpb25kZWZpbml0aW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIExhbWJkYSBmdW5jdGlvbiBkZWZpbml0aW9uIHRvIGJlIHVzZWQgaW4gYSBncm91cCB0aGF0IGNvbnRhaW5zIGEgbGlzdCBvZiBMYW1iZGEgZnVuY3Rpb25zIGFuZCB0aGVpciBjb25maWd1cmF0aW9ucy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZUZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWZ1bmN0aW9uZGVmaW5pdGlvbnZlcnNpb24tcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdmVyc2lvbiBvZiBhbiBleGlzdGluZyBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmdW5jdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUdyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVncm91cC1wb3N0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBncm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZUdyb3VwQ2VydGlmaWNhdGVBdXRob3JpdHlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWdyb3VwY2VydGlmaWNhdGVhdXRob3JpdHktcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgQ0EgZm9yIHRoZSBncm91cCwgb3Igcm90YXRlIHRoZSBleGlzdGluZyBDQS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlR3JvdXBWZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVncm91cHZlcnNpb24tcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdmVyc2lvbiBvZiBhIGdyb3VwIHRoYXQgaGFzIGFscmVhZHkgYmVlbiBkZWZpbmVkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVMb2dnZXJEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVsb2dnZXJkZWZpbml0aW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGxvZ2dlciBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlTG9nZ2VyRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWxvZ2dlcmRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHZlcnNpb24gb2YgYW4gZXhpc3RpbmcgbG9nZ2VyIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nZ2VyRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVyZXNvdXJjZWRlZmluaXRpb24tcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgcmVzb3VyY2UgZGVmaW5pdGlvbiB0aGF0IGNvbnRhaW5zIGEgbGlzdCBvZiByZXNvdXJjZXMgdG8gYmUgdXNlZCBpbiBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlcmVzb3VyY2VkZWZpbml0aW9udmVyc2lvbi1wb3N0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB2ZXJzaW9uIG9mIGFuIGV4aXN0aW5nIHJlc291cmNlIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVTb2Z0d2FyZVVwZGF0ZUpvYlwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlc29mdHdhcmV1cGRhdGVqb2ItcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFXUyBJb1Qgam9iIHRoYXQgd2lsbCB0cmlnZ2VyIHlvdXIgR3JlZW5ncmFzcyBjb3JlcyB0byB1cGRhdGUgdGhlIHNvZnR3YXJlIHRoZXkgYXJlIHJ1bm5pbmcuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJDcmVhdGVTdWJzY3JpcHRpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVzdWJzY3JpcHRpb25kZWZpbml0aW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHN1YnNjcmlwdGlvbiBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlU3Vic2NyaXB0aW9uRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZXN1YnNjcmlwdGlvbmRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHZlcnNpb24gb2YgYW4gZXhpc3Rpbmcgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3Vic2NyaXB0aW9uRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlQ29ubmVjdG9yRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVsZXRlY29ubmVjdG9yZGVmaW5pdGlvbi1kZWxldGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNvbm5lY3RvciBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbm5lY3RvckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUNvcmVEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVjb3JlZGVmaW5pdGlvbi1kZWxldGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNvcmUgZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY29yZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZURldmljZURlZmluaXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlbGV0ZWRldmljZWRlZmluaXRpb24tZGVsZXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBkZXZpY2UgZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV2aWNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlRnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVmdW5jdGlvbmRlZmluaXRpb24tZGVsZXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVsZXRlZ3JvdXAtZGVsZXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBncm91cCB0aGF0IGlzIG5vdCBjdXJyZW50bHkgaW4gdXNlIGluIGEgZGVwbG95bWVudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlTG9nZ2VyRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVsZXRlbG9nZ2VyZGVmaW5pdGlvbi1kZWxldGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGxvZ2dlciBkZWZpbml0aW9uLiBEZWxldGluZyBhIGRlZmluaXRpb24gdGhhdCBpcyBjdXJyZW50bHkgaW4gdXNlIGluIGEgZGVwbG95bWVudCBhZmZlY3RzIGZ1dHVyZSBkZXBsb3ltZW50cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2dnZXJEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSZXNvdXJjZURlZmluaXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlbGV0ZXJlc291cmNlZGVmaW5pdGlvbi1kZWxldGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHJlc291cmNlIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVTdWJzY3JpcHRpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVzdWJzY3JpcHRpb25kZWZpbml0aW9uLWRlbGV0ZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uIERlbGV0aW5nIGEgZGVmaW5pdGlvbiB0aGF0IGlzIGN1cnJlbnRseSBpbiB1c2UgaW4gYSBkZXBsb3ltZW50IGFmZmVjdHMgZnV0dXJlIGRlcGxveW1lbnRzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN1YnNjcmlwdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZVJvbGVGcm9tR3JvdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2Rpc2Fzc29jaWF0ZXJvbGVmcm9tZ3JvdXAtZGVsZXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgdGhlIHJvbGUgZnJvbSBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEaXNhc3NvY2lhdGVTZXJ2aWNlUm9sZUZyb21BY2NvdW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kaXNhc3NvY2lhdGVzZXJ2aWNlcm9sZWZyb21hY2NvdW50LWRlbGV0ZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIHRoZSBzZXJ2aWNlIHJvbGUgZnJvbSBhbiBhY2NvdW50LiBXaXRob3V0IGEgc2VydmljZSByb2xlLCBkZXBsb3ltZW50cyB3aWxsIG5vdCB3b3JrLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGlzY292ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZ2ctZGlzY292ZXItYXBpLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiByZXF1aXJlZCB0byBjb25uZWN0IHRvIGEgR3JlZW5ncmFzcyBjb3JlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGhpbmdcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEFzc29jaWF0ZWRSb2xlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRhc3NvY2lhdGVkcm9sZS1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSByb2xlIGFzc29jaWF0ZWQgd2l0aCBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEJ1bGtEZXBsb3ltZW50U3RhdHVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRidWxrZGVwbG95bWVudHN0YXR1cy1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgc3RhdHVzIG9mIGEgYnVsayBkZXBsb3ltZW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVsa0RlcGxveW1lbnRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldENvbm5lY3Rpdml0eUluZm9cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGNvbm5lY3Rpdml0eWluZm8tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgY29ubmVjdGl2aXR5IGluZm9ybWF0aW9uIGZvciBhIGNvcmUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjb25uZWN0aXZpdHlJbmZvXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRDb25uZWN0b3JEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRjb25uZWN0b3JkZWZpbml0aW9uLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBjb25uZWN0b3IgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbm5lY3RvckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldENvbm5lY3RvckRlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRjb25uZWN0b3JkZWZpbml0aW9udmVyc2lvbi1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgY29ubmVjdG9yIGRlZmluaXRpb24gdmVyc2lvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbm5lY3RvckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcImNvbm5lY3RvckRlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRDb3JlRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Y29yZWRlZmluaXRpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGNvcmUgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvcmVEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRDb3JlRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGNvcmVkZWZpbml0aW9udmVyc2lvbi1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgY29yZSBkZWZpbml0aW9uIHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjb3JlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwiY29yZURlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXREZXBsb3ltZW50U3RhdHVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRkZXBsb3ltZW50c3RhdHVzLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBzdGF0dXMgb2YgYSBkZXBsb3ltZW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGVwbG95bWVudFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldERldmljZURlZmluaXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGRldmljZWRlZmluaXRpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGRldmljZSBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV2aWNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0RGV2aWNlRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGRldmljZWRlZmluaXRpb252ZXJzaW9uLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBkZXZpY2UgZGVmaW5pdGlvbiB2ZXJzaW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV2aWNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwiZGV2aWNlRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEZ1bmN0aW9uRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0ZnVuY3Rpb25kZWZpbml0aW9uLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbiwgc3VjaCBhcyBpdHMgY3JlYXRpb24gdGltZSBhbmQgbGF0ZXN0IHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmdW5jdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGZ1bmN0aW9uZGVmaW5pdGlvbnZlcnNpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIExhbWJkYSBmdW5jdGlvbiBkZWZpbml0aW9uIHZlcnNpb24sIHN1Y2ggYXMgd2hpY2ggTGFtYmRhIGZ1bmN0aW9ucyBhcmUgaW5jbHVkZWQgaW4gdGhlIHZlcnNpb24gYW5kIHRoZWlyIGNvbmZpZ3VyYXRpb25zLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJmdW5jdGlvbkRlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Z3JvdXAtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEdyb3VwQ2VydGlmaWNhdGVBdXRob3JpdHlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGdyb3VwY2VydGlmaWNhdGVhdXRob3JpdHktZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHB1YmxpYyBrZXkgb2YgdGhlIENBIGFzc29jaWF0ZWQgd2l0aCBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2VydGlmaWNhdGVBdXRob3JpdHlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRHcm91cENlcnRpZmljYXRlQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Z3JvdXBjZXJ0aWZpY2F0ZWNvbmZpZ3VyYXRpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgY3VycmVudCBjb25maWd1cmF0aW9uIGZvciB0aGUgQ0EgdXNlZCBieSBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEdyb3VwVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Z3JvdXB2ZXJzaW9uLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBncm91cCB2ZXJzaW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcImdyb3VwVmVyc2lvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0TG9nZ2VyRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0bG9nZ2VyZGVmaW5pdGlvbi1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgbG9nZ2VyIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2dnZXJEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRMb2dnZXJEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0bG9nZ2VyZGVmaW5pdGlvbnZlcnNpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGxvZ2dlciBkZWZpbml0aW9uIHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2dnZXJEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJsb2dnZXJEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRyZXNvdXJjZWRlZmluaXRpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIHJlc291cmNlIGRlZmluaXRpb24sIHN1Y2ggYXMgaXRzIGNyZWF0aW9uIHRpbWUgYW5kIGxhdGVzdCB2ZXJzaW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRSZXNvdXJjZURlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRyZXNvdXJjZWRlZmluaXRpb252ZXJzaW9uLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSByZXNvdXJjZSBkZWZpbml0aW9uIHZlcnNpb24sIHN1Y2ggYXMgd2hpY2ggcmVzb3VyY2VzIGFyZSBpbmNsdWRlZCBpbiB0aGUgdmVyc2lvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlc291cmNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwicmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0U2VydmljZVJvbGVGb3JBY2NvdW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRzZXJ2aWNlcm9sZWZvcmFjY291bnQtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgc2VydmljZSByb2xlIHRoYXQgaXMgYXR0YWNoZWQgdG8gYW4gYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0U3Vic2NyaXB0aW9uRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0c3Vic2NyaXB0aW9uZGVmaW5pdGlvbi1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRTdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0c3Vic2NyaXB0aW9uZGVmaW5pdGlvbnZlcnNpb24tZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIHN1YnNjcmlwdGlvbiBkZWZpbml0aW9uIHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdEJ1bGtEZXBsb3ltZW50RGV0YWlsZWRSZXBvcnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0YnVsa2RlcGxveW1lbnRkZXRhaWxlZHJlcG9ydHMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIHBhZ2luYXRlZCBsaXN0IG9mIHRoZSBkZXBsb3ltZW50cyB0aGF0IGhhdmUgYmVlbiBzdGFydGVkIGluIGEgYnVsayBkZXBsb3ltZW50IG9wZXJhdGlvbiBhbmQgdGhlaXIgY3VycmVudCBkZXBsb3ltZW50IHN0YXR1cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1bGtEZXBsb3ltZW50XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0QnVsa0RlcGxveW1lbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0YnVsa2RlcGxveW1lbnRzLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGJ1bGsgZGVwbG95bWVudHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RDb25uZWN0b3JEZWZpbml0aW9uVmVyc2lvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3Rjb25uZWN0b3JkZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSB2ZXJzaW9ucyBvZiBhIGNvbm5lY3RvciBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY29ubmVjdG9yRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdENvbm5lY3RvckRlZmluaXRpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0Y29ubmVjdG9yZGVmaW5pdGlvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgY29ubmVjdG9yIGRlZmluaXRpb25zLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0Q29yZURlZmluaXRpb25WZXJzaW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGNvcmVkZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSB2ZXJzaW9ucyBvZiBhIGNvcmUgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvcmVEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0Q29yZURlZmluaXRpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0Y29yZWRlZmluaXRpb25zLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGNvcmUgZGVmaW5pdGlvbnMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3REZXBsb3ltZW50c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGRlcGxveW1lbnRzLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGFsbCBkZXBsb3ltZW50cyBmb3IgYSBncm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0RGV2aWNlRGVmaW5pdGlvblZlcnNpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0ZGV2aWNlZGVmaW5pdGlvbnZlcnNpb25zLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdmVyc2lvbnMgb2YgYSBkZXZpY2UgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldmljZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3REZXZpY2VEZWZpbml0aW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGRldmljZWRlZmluaXRpb25zLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGRldmljZSBkZWZpbml0aW9ucy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdEZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0ZnVuY3Rpb25kZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSB2ZXJzaW9ucyBvZiBhIExhbWJkYSBmdW5jdGlvbiBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0RnVuY3Rpb25EZWZpbml0aW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGZ1bmN0aW9uZGVmaW5pdGlvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgTGFtYmRhIGZ1bmN0aW9uIGRlZmluaXRpb25zLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0R3JvdXBDZXJ0aWZpY2F0ZUF1dGhvcml0aWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0Z3JvdXBjZXJ0aWZpY2F0ZWF1dGhvcml0aWVzLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGN1cnJlbnQgQ0FzIGZvciBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RHcm91cFZlcnNpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0Z3JvdXB2ZXJzaW9ucy1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHZlcnNpb25zIG9mIGEgZ3JvdXAuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdEdyb3Vwc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGdyb3Vwcy1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBncm91cHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RMb2dnZXJEZWZpbml0aW9uVmVyc2lvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3Rsb2dnZXJkZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSB2ZXJzaW9ucyBvZiBhIGxvZ2dlciBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nZ2VyRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdExvZ2dlckRlZmluaXRpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0bG9nZ2VyZGVmaW5pdGlvbnMtZ2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgbG9nZ2VyIGRlZmluaXRpb25zLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0UmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RyZXNvdXJjZWRlZmluaXRpb252ZXJzaW9ucy1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHZlcnNpb25zIG9mIGEgcmVzb3VyY2UgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlc291cmNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdFJlc291cmNlRGVmaW5pdGlvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RyZXNvdXJjZWRlZmluaXRpb25zLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIHJlc291cmNlIGRlZmluaXRpb25zLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0U3Vic2NyaXB0aW9uRGVmaW5pdGlvblZlcnNpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0c3Vic2NyaXB0aW9uZGVmaW5pdGlvbnZlcnNpb25zLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdmVyc2lvbnMgb2YgYSBzdWJzY3JpcHRpb24gZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN1YnNjcmlwdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RTdWJzY3JpcHRpb25EZWZpbml0aW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdHN1YnNjcmlwdGlvbmRlZmluaXRpb25zLWdldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIHN1YnNjcmlwdGlvbiBkZWZpbml0aW9ucy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdHRhZ3Nmb3JyZXNvdXJjZS1nZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHRhZ3MgZm9yIGEgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWxrRGVwbG95bWVudFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImNvbm5lY3RvckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJjb3JlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImRldmljZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJmdW5jdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImxvZ2dlckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJyZXNvdXJjZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUmVzZXREZXBsb3ltZW50c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvcmVzZXRkZXBsb3ltZW50cy1wb3N0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXNldCBhIGdyb3VwJ3MgZGVwbG95bWVudHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlN0YXJ0QnVsa0RlcGxveW1lbnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL3N0YXJ0YnVsa2RlcGxveW1lbnQtcG9zdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVwbG95IG11bHRpcGxlIGdyb3VwcyBpbiBvbmUgb3BlcmF0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiU3RvcEJ1bGtEZXBsb3ltZW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9zdG9wYnVsa2RlcGxveW1lbnQtcHV0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzdG9wIHRoZSBleGVjdXRpb24gb2YgYSBidWxrIGRlcGxveW1lbnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVsa0RlcGxveW1lbnRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS90YWdyZXNvdXJjZS1wb3N0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgdGFncyB0byBhIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVsa0RlcGxveW1lbnRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJjb25uZWN0b3JEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiY29yZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJkZXZpY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiZnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJsb2dnZXJEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwicmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwic3Vic2NyaXB0aW9uRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVudGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL3VudGFncmVzb3VyY2UtZGVsZXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFncyBmcm9tIGEgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWxrRGVwbG95bWVudFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImNvbm5lY3RvckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJjb3JlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImRldmljZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJmdW5jdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImxvZ2dlckRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJyZXNvdXJjZURlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVXBkYXRlQ29ubmVjdGl2aXR5SW5mb1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlY29ubmVjdGl2aXR5aW5mby1wdXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgY29ubmVjdGl2aXR5IGluZm9ybWF0aW9uIGZvciBhIEdyZWVuZ3Jhc3MgY29yZS4gQW55IGRldmljZXMgdGhhdCBiZWxvbmcgdG8gdGhlIGdyb3VwIHRoYXQgaGFzIHRoaXMgY29yZSB3aWxsIHJlY2VpdmUgdGhpcyBpbmZvcm1hdGlvbiBpbiBvcmRlciB0byBmaW5kIHRoZSBsb2NhdGlvbiBvZiB0aGUgY29yZSBhbmQgY29ubmVjdCB0byBpdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjb25uZWN0aXZpdHlJbmZvXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVDb25uZWN0b3JEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVjb25uZWN0b3JkZWZpbml0aW9uLXB1dC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY29ubmVjdG9yIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY29ubmVjdG9yRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlQ29yZURlZmluaXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL3VwZGF0ZWNvcmVkZWZpbml0aW9uLXB1dC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY29yZSBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvcmVEZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVEZXZpY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVkZXZpY2VkZWZpbml0aW9uLXB1dC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgZGV2aWNlIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV2aWNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlRnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVmdW5jdGlvbmRlZmluaXRpb24tcHV0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmdW5jdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUdyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVncm91cC1wdXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVHcm91cENlcnRpZmljYXRlQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlZ3JvdXBjZXJ0aWZpY2F0ZWNvbmZpZ3VyYXRpb24tcHV0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGNlcnRpZmljYXRlIGV4cGlyeSB0aW1lIGZvciBhIGdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVMb2dnZXJEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVsb2dnZXJkZWZpbml0aW9uLXB1dC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgbG9nZ2VyIGRlZmluaXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nZ2VyRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlUmVzb3VyY2VEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVyZXNvdXJjZWRlZmluaXRpb24tcHV0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSByZXNvdXJjZSBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlc291cmNlRGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlU3Vic2NyaXB0aW9uRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlc3Vic2NyaXB0aW9uZGVmaW5pdGlvbi1wdXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIHN1YnNjcmlwdGlvbiBkZWZpbml0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN1YnNjcmlwdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9O1xuICBwcm90ZWN0ZWQgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcImNvbm5lY3Rpdml0eUluZm9cIjoge1xuICAgICAgXCJuYW1lXCI6IFwiY29ubmVjdGl2aXR5SW5mb1wiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWNvbm5lY3Rpdml0eWluZm8uaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvdGhpbmdzLyR7VGhpbmdOYW1lfS9jb25uZWN0aXZpdHlJbmZvXCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwiYXJ0aWZhY3RcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiYXJ0aWZhY3RcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9ncm91cHMvJHtHcm91cElkfS9kZXBsb3ltZW50cy8ke0RlcGxveW1lbnRJZH0vYXJ0aWZhY3RzL2xhbWJkYS8ke0FydGlmYWN0SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwiY2VydGlmaWNhdGVBdXRob3JpdHlcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiY2VydGlmaWNhdGVBdXRob3JpdHlcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2dnLXNlYy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9ncm91cHMvJHtHcm91cElkfS9jZXJ0aWZpY2F0ZWF1dGhvcml0aWVzLyR7Q2VydGlmaWNhdGVBdXRob3JpdHlJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJkZXBsb3ltZW50XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImRlcGxveW1lbnRcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1jcmVhdGVkZXBsb3ltZW50cmVxdWVzdC5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9ncm91cHMvJHtHcm91cElkfS9kZXBsb3ltZW50cy8ke0RlcGxveW1lbnRJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJidWxrRGVwbG95bWVudFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJidWxrRGVwbG95bWVudFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYnVsay1kZXBsb3ktY2xpLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2J1bGsvZGVwbG95bWVudHMvJHtCdWxrRGVwbG95bWVudElkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiZ3JvdXBcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZ3JvdXBcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1ncm91cGluZm9ybWF0aW9uLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2dyb3Vwcy8ke0dyb3VwSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJncm91cFZlcnNpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZ3JvdXBWZXJzaW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtZ3JvdXB2ZXJzaW9uLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2dyb3Vwcy8ke0dyb3VwSWR9L3ZlcnNpb25zLyR7VmVyc2lvbklkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcImNvcmVEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImNvcmVEZWZpbml0aW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtY29yZS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9kZWZpbml0aW9uL2NvcmVzLyR7Q29yZURlZmluaXRpb25JZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcImNvcmVEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJjb3JlRGVmaW5pdGlvblZlcnNpb25cIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1jb3JlZGVmaW5pdGlvbnZlcnNpb24uaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9jb3Jlcy8ke0NvcmVEZWZpbml0aW9uSWR9L3ZlcnNpb25zLyR7VmVyc2lvbklkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcImRldmljZURlZmluaXRpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZGV2aWNlRGVmaW5pdGlvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWRldmljZS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9kZWZpbml0aW9uL2RldmljZXMvJHtEZXZpY2VEZWZpbml0aW9uSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJkZXZpY2VEZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJkZXZpY2VEZWZpbml0aW9uVmVyc2lvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWRldmljZWRlZmluaXRpb252ZXJzaW9uLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vZGV2aWNlcy8ke0RldmljZURlZmluaXRpb25JZH0vdmVyc2lvbnMvJHtWZXJzaW9uSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwiZnVuY3Rpb25EZWZpbml0aW9uXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImZ1bmN0aW9uRGVmaW5pdGlvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWZ1bmN0aW9uLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vZnVuY3Rpb25zLyR7RnVuY3Rpb25EZWZpbml0aW9uSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJmdW5jdGlvbkRlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb25cIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1mdW5jdGlvbmRlZmluaXRpb252ZXJzaW9uLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vZnVuY3Rpb25zLyR7RnVuY3Rpb25EZWZpbml0aW9uSWR9L3ZlcnNpb25zLyR7VmVyc2lvbklkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcInN1YnNjcmlwdGlvbkRlZmluaXRpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwic3Vic2NyaXB0aW9uRGVmaW5pdGlvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLXN1YnNjcmlwdGlvbi5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9kZWZpbml0aW9uL3N1YnNjcmlwdGlvbnMvJHtTdWJzY3JpcHRpb25EZWZpbml0aW9uSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvblwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJzdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLXN1YnNjcmlwdGlvbmRlZmluaXRpb252ZXJzaW9uLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vc3Vic2NyaXB0aW9ucy8ke1N1YnNjcmlwdGlvbkRlZmluaXRpb25JZH0vdmVyc2lvbnMvJHtWZXJzaW9uSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwibG9nZ2VyRGVmaW5pdGlvblwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJsb2dnZXJEZWZpbml0aW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtbG9nZ2VyLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vbG9nZ2Vycy8ke0xvZ2dlckRlZmluaXRpb25JZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcImxvZ2dlckRlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImxvZ2dlckRlZmluaXRpb25WZXJzaW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtbG9nZ2VyZGVmaW5pdGlvbnZlcnNpb24uaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9sb2dnZXJzLyR7TG9nZ2VyRGVmaW5pdGlvbklkfS92ZXJzaW9ucy8ke1ZlcnNpb25JZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJyZXNvdXJjZURlZmluaXRpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwicmVzb3VyY2VEZWZpbml0aW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtcmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9yZXNvdXJjZXMvJHtSZXNvdXJjZURlZmluaXRpb25JZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcInJlc291cmNlRGVmaW5pdGlvblZlcnNpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwicmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLXJlc291cmNlZGVmaW5pdGlvbnZlcnNpb24uaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9yZXNvdXJjZXMvJHtSZXNvdXJjZURlZmluaXRpb25JZH0vdmVyc2lvbnMvJHtWZXJzaW9uSWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwiY29ubmVjdG9yRGVmaW5pdGlvblwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJjb25uZWN0b3JEZWZpbml0aW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtY29ubmVjdG9yLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vY29ubmVjdG9ycy8ke0Nvbm5lY3RvckRlZmluaXRpb25JZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcImNvbm5lY3RvckRlZmluaXRpb25WZXJzaW9uXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImNvbm5lY3RvckRlZmluaXRpb25WZXJzaW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtY29ubmVjdG9yZGVmaW5pdGlvbnZlcnNpb24uaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9jb25uZWN0b3JzLyR7Q29ubmVjdG9yRGVmaW5pdGlvbklkfS92ZXJzaW9ucy8ke1ZlcnNpb25JZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJ0aGluZ1wiOiB7XG4gICAgICBcIm5hbWVcIjogXCJ0aGluZ1wiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90L2xhdGVzdC9kZXZlbG9wZXJndWlkZS90aGluZy1yZWdpc3RyeS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06aW90OiR7UmVnaW9ufToke0FjY291bnR9OnRoaW5nLyR7VGhpbmdOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2dyZWVuZ3Jhc3NdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2lvdGdyZWVuZ3Jhc3MuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGEgcm9sZSB3aXRoIGEgZ3JvdXAuIFRoZSByb2xlJ3MgcGVybWlzc2lvbnMgbXVzdCBhbGxvdyBHcmVlbmdyYXNzIGNvcmUgTGFtYmRhIGZ1bmN0aW9ucyBhbmQgY29ubmVjdG9ycyB0byBwZXJmb3JtIGFjdGlvbnMgaW4gb3RoZXIgQVdTIHNlcnZpY2VzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvYXNzb2NpYXRlcm9sZXRvZ3JvdXAtcHV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZVJvbGVUb0dyb3VwKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkFzc29jaWF0ZVJvbGVUb0dyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGEgcm9sZSB3aXRoIHlvdXIgYWNjb3VudC4gQVdTIElvVCBHcmVlbmdyYXNzIHVzZXMgdGhpcyByb2xlIHRvIGFjY2VzcyB5b3VyIExhbWJkYSBmdW5jdGlvbnMgYW5kIEFXUyBJb1QgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9hc3NvY2lhdGVzZXJ2aWNlcm9sZXRvYWNjb3VudC1wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlU2VydmljZVJvbGVUb0FjY291bnQoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6QXNzb2NpYXRlU2VydmljZVJvbGVUb0FjY291bnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBjb25uZWN0b3IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWNvbm5lY3RvcmRlZmluaXRpb24tcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDb25uZWN0b3JEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkNyZWF0ZUNvbm5lY3RvckRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB2ZXJzaW9uIG9mIGFuIGV4aXN0aW5nIGNvbm5lY3RvciBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlY29ubmVjdG9yZGVmaW5pdGlvbnZlcnNpb24tcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDb25uZWN0b3JEZWZpbml0aW9uVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVDb25uZWN0b3JEZWZpbml0aW9uVmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGNvcmUgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWNvcmVkZWZpbml0aW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ29yZURlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6Q3JlYXRlQ29yZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB2ZXJzaW9uIG9mIGFuIGV4aXN0aW5nIGNvcmUgZGVmaW5pdGlvbi4gR3JlZW5ncmFzcyBncm91cHMgbXVzdCBlYWNoIGNvbnRhaW4gZXhhY3RseSBvbmUgR3JlZW5ncmFzcyBjb3JlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlY29yZWRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ29yZURlZmluaXRpb25WZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkNyZWF0ZUNvcmVEZWZpbml0aW9uVmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGRlcGxveW1lbnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVkZXBsb3ltZW50LXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGVwbG95bWVudCgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVEZXBsb3ltZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgZGV2aWNlIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVkZXZpY2VkZWZpbml0aW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGV2aWNlRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVEZXZpY2VEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdmVyc2lvbiBvZiBhbiBleGlzdGluZyBkZXZpY2UgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWRldmljZWRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGV2aWNlRGVmaW5pdGlvblZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6Q3JlYXRlRGV2aWNlRGVmaW5pdGlvblZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbiB0byBiZSB1c2VkIGluIGEgZ3JvdXAgdGhhdCBjb250YWlucyBhIGxpc3Qgb2YgTGFtYmRhIGZ1bmN0aW9ucyBhbmQgdGhlaXIgY29uZmlndXJhdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVmdW5jdGlvbmRlZmluaXRpb24tcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVGdW5jdGlvbkRlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6Q3JlYXRlRnVuY3Rpb25EZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdmVyc2lvbiBvZiBhbiBleGlzdGluZyBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWZ1bmN0aW9uZGVmaW5pdGlvbnZlcnNpb24tcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVGdW5jdGlvbkRlZmluaXRpb25WZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkNyZWF0ZUZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWdyb3VwLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6Q3JlYXRlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBDQSBmb3IgdGhlIGdyb3VwLCBvciByb3RhdGUgdGhlIGV4aXN0aW5nIENBLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlZ3JvdXBjZXJ0aWZpY2F0ZWF1dGhvcml0eS1wb3N0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUdyb3VwQ2VydGlmaWNhdGVBdXRob3JpdHkoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6Q3JlYXRlR3JvdXBDZXJ0aWZpY2F0ZUF1dGhvcml0eScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHZlcnNpb24gb2YgYSBncm91cCB0aGF0IGhhcyBhbHJlYWR5IGJlZW4gZGVmaW5lZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWdyb3VwdmVyc2lvbi1wb3N0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUdyb3VwVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVHcm91cFZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBsb2dnZXIgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZWxvZ2dlcmRlZmluaXRpb24tcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMb2dnZXJEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkNyZWF0ZUxvZ2dlckRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB2ZXJzaW9uIG9mIGFuIGV4aXN0aW5nIGxvZ2dlciBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvY3JlYXRlbG9nZ2VyZGVmaW5pdGlvbnZlcnNpb24tcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMb2dnZXJEZWZpbml0aW9uVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVMb2dnZXJEZWZpbml0aW9uVmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHJlc291cmNlIGRlZmluaXRpb24gdGhhdCBjb250YWlucyBhIGxpc3Qgb2YgcmVzb3VyY2VzIHRvIGJlIHVzZWQgaW4gYSBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZXJlc291cmNlZGVmaW5pdGlvbi1wb3N0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJlc291cmNlRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVSZXNvdXJjZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB2ZXJzaW9uIG9mIGFuIGV4aXN0aW5nIHJlc291cmNlIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVyZXNvdXJjZWRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVSZXNvdXJjZURlZmluaXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFXUyBJb1Qgam9iIHRoYXQgd2lsbCB0cmlnZ2VyIHlvdXIgR3JlZW5ncmFzcyBjb3JlcyB0byB1cGRhdGUgdGhlIHNvZnR3YXJlIHRoZXkgYXJlIHJ1bm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVzb2Z0d2FyZXVwZGF0ZWpvYi1wb3N0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVNvZnR3YXJlVXBkYXRlSm9iKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkNyZWF0ZVNvZnR3YXJlVXBkYXRlSm9iJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jcmVhdGVzdWJzY3JpcHRpb25kZWZpbml0aW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU3Vic2NyaXB0aW9uRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpDcmVhdGVTdWJzY3JpcHRpb25EZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdmVyc2lvbiBvZiBhbiBleGlzdGluZyBzdWJzY3JpcHRpb24gZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2NyZWF0ZXN1YnNjcmlwdGlvbmRlZmluaXRpb252ZXJzaW9uLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU3Vic2NyaXB0aW9uRGVmaW5pdGlvblZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6Q3JlYXRlU3Vic2NyaXB0aW9uRGVmaW5pdGlvblZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBjb25uZWN0b3IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlbGV0ZWNvbm5lY3RvcmRlZmluaXRpb24tZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNvbm5lY3RvckRlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6RGVsZXRlQ29ubmVjdG9yRGVmaW5pdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNvcmUgZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVjb3JlZGVmaW5pdGlvbi1kZWxldGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ29yZURlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6RGVsZXRlQ29yZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBkZXZpY2UgZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVkZXZpY2VkZWZpbml0aW9uLWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEZXZpY2VEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkRlbGV0ZURldmljZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVmdW5jdGlvbmRlZmluaXRpb24tZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUZ1bmN0aW9uRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpEZWxldGVGdW5jdGlvbkRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBncm91cCB0aGF0IGlzIG5vdCBjdXJyZW50bHkgaW4gdXNlIGluIGEgZGVwbG95bWVudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlbGV0ZWdyb3VwLWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpEZWxldGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGxvZ2dlciBkZWZpbml0aW9uLiBEZWxldGluZyBhIGRlZmluaXRpb24gdGhhdCBpcyBjdXJyZW50bHkgaW4gdXNlIGluIGEgZGVwbG95bWVudCBhZmZlY3RzIGZ1dHVyZSBkZXBsb3ltZW50cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlbGV0ZWxvZ2dlcmRlZmluaXRpb24tZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUxvZ2dlckRlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6RGVsZXRlTG9nZ2VyRGVmaW5pdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHJlc291cmNlIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVyZXNvdXJjZWRlZmluaXRpb24tZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJlc291cmNlRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpEZWxldGVSZXNvdXJjZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBzdWJzY3JpcHRpb24gZGVmaW5pdGlvbi4gRGVsZXRpbmcgYSBkZWZpbml0aW9uIHRoYXQgaXMgY3VycmVudGx5IGluIHVzZSBpbiBhIGRlcGxveW1lbnQgYWZmZWN0cyBmdXR1cmUgZGVwbG95bWVudHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWxldGVzdWJzY3JpcHRpb25kZWZpbml0aW9uLWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTdWJzY3JpcHRpb25EZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkRlbGV0ZVN1YnNjcmlwdGlvbkRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgdGhlIHJvbGUgZnJvbSBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGlzYXNzb2NpYXRlcm9sZWZyb21ncm91cC1kZWxldGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlUm9sZUZyb21Hcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpEaXNhc3NvY2lhdGVSb2xlRnJvbUdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIHRoZSBzZXJ2aWNlIHJvbGUgZnJvbSBhbiBhY2NvdW50LiBXaXRob3V0IGEgc2VydmljZSByb2xlLCBkZXBsb3ltZW50cyB3aWxsIG5vdCB3b3JrLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGlzYXNzb2NpYXRlc2VydmljZXJvbGVmcm9tYWNjb3VudC1kZWxldGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlU2VydmljZVJvbGVGcm9tQWNjb3VudCgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpEaXNhc3NvY2lhdGVTZXJ2aWNlUm9sZUZyb21BY2NvdW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gcmVxdWlyZWQgdG8gY29ubmVjdCB0byBhIEdyZWVuZ3Jhc3MgY29yZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9nZy1kaXNjb3Zlci1hcGkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzY292ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6RGlzY292ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgcm9sZSBhc3NvY2lhdGVkIHdpdGggYSBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0YXNzb2NpYXRlZHJvbGUtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFzc29jaWF0ZWRSb2xlKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldEFzc29jaWF0ZWRSb2xlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBzdGF0dXMgb2YgYSBidWxrIGRlcGxveW1lbnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGJ1bGtkZXBsb3ltZW50c3RhdHVzLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCdWxrRGVwbG95bWVudFN0YXR1cygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpHZXRCdWxrRGVwbG95bWVudFN0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBjb25uZWN0aXZpdHkgaW5mb3JtYXRpb24gZm9yIGEgY29yZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Y29ubmVjdGl2aXR5aW5mby1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q29ubmVjdGl2aXR5SW5mbygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpHZXRDb25uZWN0aXZpdHlJbmZvJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBjb25uZWN0b3IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Y29ubmVjdG9yZGVmaW5pdGlvbi1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q29ubmVjdG9yRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpHZXRDb25uZWN0b3JEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBjb25uZWN0b3IgZGVmaW5pdGlvbiB2ZXJzaW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRjb25uZWN0b3JkZWZpbml0aW9udmVyc2lvbi1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q29ubmVjdG9yRGVmaW5pdGlvblZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0Q29ubmVjdG9yRGVmaW5pdGlvblZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGNvcmUgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Y29yZWRlZmluaXRpb24tZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldENvcmVEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldENvcmVEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSBjb3JlIGRlZmluaXRpb24gdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0Y29yZWRlZmluaXRpb252ZXJzaW9uLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRDb3JlRGVmaW5pdGlvblZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0Q29yZURlZmluaXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBzdGF0dXMgb2YgYSBkZXBsb3ltZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRkZXBsb3ltZW50c3RhdHVzLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREZXBsb3ltZW50U3RhdHVzKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldERlcGxveW1lbnRTdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGRldmljZSBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRkZXZpY2VkZWZpbml0aW9uLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREZXZpY2VEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldERldmljZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGRldmljZSBkZWZpbml0aW9uIHZlcnNpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGRldmljZWRlZmluaXRpb252ZXJzaW9uLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREZXZpY2VEZWZpbml0aW9uVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpHZXREZXZpY2VEZWZpbml0aW9uVmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgTGFtYmRhIGZ1bmN0aW9uIGRlZmluaXRpb24sIHN1Y2ggYXMgaXRzIGNyZWF0aW9uIHRpbWUgYW5kIGxhdGVzdCB2ZXJzaW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRmdW5jdGlvbmRlZmluaXRpb24tZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEZ1bmN0aW9uRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpHZXRGdW5jdGlvbkRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIExhbWJkYSBmdW5jdGlvbiBkZWZpbml0aW9uIHZlcnNpb24sIHN1Y2ggYXMgd2hpY2ggTGFtYmRhIGZ1bmN0aW9ucyBhcmUgaW5jbHVkZWQgaW4gdGhlIHZlcnNpb24gYW5kIHRoZWlyIGNvbmZpZ3VyYXRpb25zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRmdW5jdGlvbmRlZmluaXRpb252ZXJzaW9uLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRGdW5jdGlvbkRlZmluaXRpb25WZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldEZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRncm91cC1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0R3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0R3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHB1YmxpYyBrZXkgb2YgdGhlIENBIGFzc29jaWF0ZWQgd2l0aCBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRncm91cGNlcnRpZmljYXRlYXV0aG9yaXR5LWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRHcm91cENlcnRpZmljYXRlQXV0aG9yaXR5KCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldEdyb3VwQ2VydGlmaWNhdGVBdXRob3JpdHknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgY3VycmVudCBjb25maWd1cmF0aW9uIGZvciB0aGUgQ0EgdXNlZCBieSBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9nZXRncm91cGNlcnRpZmljYXRlY29uZmlndXJhdGlvbi1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0R3JvdXBDZXJ0aWZpY2F0ZUNvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0R3JvdXBDZXJ0aWZpY2F0ZUNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGdyb3VwIHZlcnNpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGdyb3VwdmVyc2lvbi1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0R3JvdXBWZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldEdyb3VwVmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgbG9nZ2VyIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldGxvZ2dlcmRlZmluaXRpb24tZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExvZ2dlckRlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0TG9nZ2VyRGVmaW5pdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgbG9nZ2VyIGRlZmluaXRpb24gdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0bG9nZ2VyZGVmaW5pdGlvbnZlcnNpb24tZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExvZ2dlckRlZmluaXRpb25WZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldExvZ2dlckRlZmluaXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSByZXNvdXJjZSBkZWZpbml0aW9uLCBzdWNoIGFzIGl0cyBjcmVhdGlvbiB0aW1lIGFuZCBsYXRlc3QgdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0cmVzb3VyY2VkZWZpbml0aW9uLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZXNvdXJjZURlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0UmVzb3VyY2VEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYSByZXNvdXJjZSBkZWZpbml0aW9uIHZlcnNpb24sIHN1Y2ggYXMgd2hpY2ggcmVzb3VyY2VzIGFyZSBpbmNsdWRlZCBpbiB0aGUgdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0cmVzb3VyY2VkZWZpbml0aW9udmVyc2lvbi1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpHZXRSZXNvdXJjZURlZmluaXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIHNlcnZpY2Ugcm9sZSB0aGF0IGlzIGF0dGFjaGVkIHRvIGFuIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldHNlcnZpY2Vyb2xlZm9yYWNjb3VudC1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2VydmljZVJvbGVGb3JBY2NvdW50KCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldFNlcnZpY2VSb2xlRm9yQWNjb3VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2dldHN1YnNjcmlwdGlvbmRlZmluaXRpb24tZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFN1YnNjcmlwdGlvbkRlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6R2V0U3Vic2NyaXB0aW9uRGVmaW5pdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24gdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZ2V0c3Vic2NyaXB0aW9uZGVmaW5pdGlvbnZlcnNpb24tZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFN1YnNjcmlwdGlvbkRlZmluaXRpb25WZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkdldFN1YnNjcmlwdGlvbkRlZmluaXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBwYWdpbmF0ZWQgbGlzdCBvZiB0aGUgZGVwbG95bWVudHMgdGhhdCBoYXZlIGJlZW4gc3RhcnRlZCBpbiBhIGJ1bGsgZGVwbG95bWVudCBvcGVyYXRpb24gYW5kIHRoZWlyIGN1cnJlbnQgZGVwbG95bWVudCBzdGF0dXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RidWxrZGVwbG95bWVudGRldGFpbGVkcmVwb3J0cy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEJ1bGtEZXBsb3ltZW50RGV0YWlsZWRSZXBvcnRzKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RCdWxrRGVwbG95bWVudERldGFpbGVkUmVwb3J0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBidWxrIGRlcGxveW1lbnRzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0YnVsa2RlcGxveW1lbnRzLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QnVsa0RlcGxveW1lbnRzKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RCdWxrRGVwbG95bWVudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSB2ZXJzaW9ucyBvZiBhIGNvbm5lY3RvciBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0Y29ubmVjdG9yZGVmaW5pdGlvbnZlcnNpb25zLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Q29ubmVjdG9yRGVmaW5pdGlvblZlcnNpb25zKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RDb25uZWN0b3JEZWZpbml0aW9uVmVyc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgY29ubmVjdG9yIGRlZmluaXRpb25zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0Y29ubmVjdG9yZGVmaW5pdGlvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDb25uZWN0b3JEZWZpbml0aW9ucygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0Q29ubmVjdG9yRGVmaW5pdGlvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSB2ZXJzaW9ucyBvZiBhIGNvcmUgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGNvcmVkZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDb3JlRGVmaW5pdGlvblZlcnNpb25zKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RDb3JlRGVmaW5pdGlvblZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGNvcmUgZGVmaW5pdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3Rjb3JlZGVmaW5pdGlvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDb3JlRGVmaW5pdGlvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6TGlzdENvcmVEZWZpbml0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBhbGwgZGVwbG95bWVudHMgZm9yIGEgZ3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RkZXBsb3ltZW50cy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERlcGxveW1lbnRzKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3REZXBsb3ltZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHZlcnNpb25zIG9mIGEgZGV2aWNlIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RkZXZpY2VkZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REZXZpY2VEZWZpbml0aW9uVmVyc2lvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6TGlzdERldmljZURlZmluaXRpb25WZXJzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBkZXZpY2UgZGVmaW5pdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RkZXZpY2VkZWZpbml0aW9ucy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERldmljZURlZmluaXRpb25zKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3REZXZpY2VEZWZpbml0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHZlcnNpb25zIG9mIGEgTGFtYmRhIGZ1bmN0aW9uIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RmdW5jdGlvbmRlZmluaXRpb252ZXJzaW9ucy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb25zKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RGdW5jdGlvbkRlZmluaXRpb25WZXJzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBMYW1iZGEgZnVuY3Rpb24gZGVmaW5pdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RmdW5jdGlvbmRlZmluaXRpb25zLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RnVuY3Rpb25EZWZpbml0aW9ucygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0RnVuY3Rpb25EZWZpbml0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBjdXJyZW50IENBcyBmb3IgYSBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGdyb3VwY2VydGlmaWNhdGVhdXRob3JpdGllcy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEdyb3VwQ2VydGlmaWNhdGVBdXRob3JpdGllcygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0R3JvdXBDZXJ0aWZpY2F0ZUF1dGhvcml0aWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdmVyc2lvbnMgb2YgYSBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGdyb3VwdmVyc2lvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RHcm91cFZlcnNpb25zKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RHcm91cFZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGdyb3Vwcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGdyb3Vwcy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEdyb3VwcygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0R3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdmVyc2lvbnMgb2YgYSBsb2dnZXIgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGxvZ2dlcmRlZmluaXRpb252ZXJzaW9ucy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdExvZ2dlckRlZmluaXRpb25WZXJzaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0TG9nZ2VyRGVmaW5pdGlvblZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGxvZ2dlciBkZWZpbml0aW9ucy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvbGlzdGxvZ2dlcmRlZmluaXRpb25zLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TG9nZ2VyRGVmaW5pdGlvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6TGlzdExvZ2dlckRlZmluaXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdmVyc2lvbnMgb2YgYSByZXNvdXJjZSBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9saXN0cmVzb3VyY2VkZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZXNvdXJjZURlZmluaXRpb25WZXJzaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0UmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgcmVzb3VyY2UgZGVmaW5pdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RyZXNvdXJjZWRlZmluaXRpb25zLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UmVzb3VyY2VEZWZpbml0aW9ucygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpMaXN0UmVzb3VyY2VEZWZpbml0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHZlcnNpb25zIG9mIGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RzdWJzY3JpcHRpb25kZWZpbml0aW9udmVyc2lvbnMtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6TGlzdFN1YnNjcmlwdGlvbkRlZmluaXRpb25WZXJzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBzdWJzY3JpcHRpb24gZGVmaW5pdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3RzdWJzY3JpcHRpb25kZWZpbml0aW9ucy1nZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFN1YnNjcmlwdGlvbkRlZmluaXRpb25zKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOkxpc3RTdWJzY3JpcHRpb25EZWZpbml0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHRhZ3MgZm9yIGEgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2xpc3R0YWdzZm9ycmVzb3VyY2UtZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6TGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlc2V0IGEgZ3JvdXAncyBkZXBsb3ltZW50cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL3Jlc2V0ZGVwbG95bWVudHMtcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXNldERlcGxveW1lbnRzKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOlJlc2V0RGVwbG95bWVudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXBsb3kgbXVsdGlwbGUgZ3JvdXBzIGluIG9uZSBvcGVyYXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9zdGFydGJ1bGtkZXBsb3ltZW50LXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRCdWxrRGVwbG95bWVudCgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpTdGFydEJ1bGtEZXBsb3ltZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCB0aGUgZXhlY3V0aW9uIG9mIGEgYnVsayBkZXBsb3ltZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2Uvc3RvcGJ1bGtkZXBsb3ltZW50LXB1dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wQnVsa0RlcGxveW1lbnQoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6U3RvcEJ1bGtEZXBsb3ltZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIHRhZ3MgdG8gYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdGFncmVzb3VyY2UtcG9zdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdW50YWdyZXNvdXJjZS1kZWxldGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBjb25uZWN0aXZpdHkgaW5mb3JtYXRpb24gZm9yIGEgR3JlZW5ncmFzcyBjb3JlLiBBbnkgZGV2aWNlcyB0aGF0IGJlbG9uZyB0byB0aGUgZ3JvdXAgdGhhdCBoYXMgdGhpcyBjb3JlIHdpbGwgcmVjZWl2ZSB0aGlzIGluZm9ybWF0aW9uIGluIG9yZGVyIHRvIGZpbmQgdGhlIGxvY2F0aW9uIG9mIHRoZSBjb3JlIGFuZCBjb25uZWN0IHRvIGl0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlY29ubmVjdGl2aXR5aW5mby1wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ29ubmVjdGl2aXR5SW5mbygpIHtcbiAgICB0aGlzLmFkZCgnZ3JlZW5ncmFzczpVcGRhdGVDb25uZWN0aXZpdHlJbmZvJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY29ubmVjdG9yIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVjb25uZWN0b3JkZWZpbml0aW9uLXB1dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDb25uZWN0b3JEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOlVwZGF0ZUNvbm5lY3RvckRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBjb3JlIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVjb3JlZGVmaW5pdGlvbi1wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ29yZURlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6VXBkYXRlQ29yZURlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBkZXZpY2UgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL3VwZGF0ZWRldmljZWRlZmluaXRpb24tcHV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURldmljZURlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6VXBkYXRlRGV2aWNlRGVmaW5pdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIExhbWJkYSBmdW5jdGlvbiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlZnVuY3Rpb25kZWZpbml0aW9uLXB1dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVGdW5jdGlvbkRlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6VXBkYXRlRnVuY3Rpb25EZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgZ3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVncm91cC1wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6VXBkYXRlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGNlcnRpZmljYXRlIGV4cGlyeSB0aW1lIGZvciBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlZ3JvdXBjZXJ0aWZpY2F0ZWNvbmZpZ3VyYXRpb24tcHV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUdyb3VwQ2VydGlmaWNhdGVDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOlVwZGF0ZUdyb3VwQ2VydGlmaWNhdGVDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgbG9nZ2VyIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVsb2dnZXJkZWZpbml0aW9uLXB1dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVMb2dnZXJEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOlVwZGF0ZUxvZ2dlckRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSByZXNvdXJjZSBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvdXBkYXRlcmVzb3VyY2VkZWZpbml0aW9uLXB1dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSZXNvdXJjZURlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2dyZWVuZ3Jhc3M6VXBkYXRlUmVzb3VyY2VEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgc3Vic2NyaXB0aW9uIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS91cGRhdGVzdWJzY3JpcHRpb25kZWZpbml0aW9uLXB1dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTdWJzY3JpcHRpb25EZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdncmVlbmdyYXNzOlVwZGF0ZVN1YnNjcmlwdGlvbkRlZmluaXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjb25uZWN0aXZpdHlJbmZvIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1jb25uZWN0aXZpdHlpbmZvLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHRoaW5nTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB0aGluZ05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Db25uZWN0aXZpdHlJbmZvKHRoaW5nTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL3RoaW5ncy8ke1RoaW5nTmFtZX0vY29ubmVjdGl2aXR5SW5mbyc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VGhpbmdOYW1lfScsIHRoaW5nTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXJ0aWZhY3QgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gZ3JvdXBJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBncm91cElkLlxuICAgKiBAcGFyYW0gZGVwbG95bWVudElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRlcGxveW1lbnRJZC5cbiAgICogQHBhcmFtIGFydGlmYWN0SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXJ0aWZhY3RJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkFydGlmYWN0KGdyb3VwSWQ6IHN0cmluZywgZGVwbG95bWVudElkOiBzdHJpbmcsIGFydGlmYWN0SWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9ncm91cHMvJHtHcm91cElkfS9kZXBsb3ltZW50cy8ke0RlcGxveW1lbnRJZH0vYXJ0aWZhY3RzL2xhbWJkYS8ke0FydGlmYWN0SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtHcm91cElkfScsIGdyb3VwSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0RlcGxveW1lbnRJZH0nLCBkZXBsb3ltZW50SWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FydGlmYWN0SWR9JywgYXJ0aWZhY3RJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2VydGlmaWNhdGVBdXRob3JpdHkgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZ2ctc2VjLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGdyb3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JvdXBJZC5cbiAgICogQHBhcmFtIGNlcnRpZmljYXRlQXV0aG9yaXR5SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY2VydGlmaWNhdGVBdXRob3JpdHlJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkNlcnRpZmljYXRlQXV0aG9yaXR5KGdyb3VwSWQ6IHN0cmluZywgY2VydGlmaWNhdGVBdXRob3JpdHlJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2dyb3Vwcy8ke0dyb3VwSWR9L2NlcnRpZmljYXRlYXV0aG9yaXRpZXMvJHtDZXJ0aWZpY2F0ZUF1dGhvcml0eUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7R3JvdXBJZH0nLCBncm91cElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtDZXJ0aWZpY2F0ZUF1dGhvcml0eUlkfScsIGNlcnRpZmljYXRlQXV0aG9yaXR5SWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRlcGxveW1lbnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWNyZWF0ZWRlcGxveW1lbnRyZXF1ZXN0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGdyb3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JvdXBJZC5cbiAgICogQHBhcmFtIGRlcGxveW1lbnRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkZXBsb3ltZW50SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25EZXBsb3ltZW50KGdyb3VwSWQ6IHN0cmluZywgZGVwbG95bWVudElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZ3JvdXBzLyR7R3JvdXBJZH0vZGVwbG95bWVudHMvJHtEZXBsb3ltZW50SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtHcm91cElkfScsIGdyb3VwSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0RlcGxveW1lbnRJZH0nLCBkZXBsb3ltZW50SWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGJ1bGtEZXBsb3ltZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2J1bGstZGVwbG95LWNsaS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBidWxrRGVwbG95bWVudElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGJ1bGtEZXBsb3ltZW50SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQnVsa0RlcGxveW1lbnQoYnVsa0RlcGxveW1lbnRJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2J1bGsvZGVwbG95bWVudHMvJHtCdWxrRGVwbG95bWVudElkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QnVsa0RlcGxveW1lbnRJZH0nLCBidWxrRGVwbG95bWVudElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBncm91cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtZ3JvdXBpbmZvcm1hdGlvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBncm91cElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGdyb3VwSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uR3JvdXAoZ3JvdXBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2dyb3Vwcy8ke0dyb3VwSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtHcm91cElkfScsIGdyb3VwSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGdyb3VwVmVyc2lvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtZ3JvdXB2ZXJzaW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGdyb3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JvdXBJZC5cbiAgICogQHBhcmFtIHZlcnNpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSB2ZXJzaW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Hcm91cFZlcnNpb24oZ3JvdXBJZDogc3RyaW5nLCB2ZXJzaW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9ncm91cHMvJHtHcm91cElkfS92ZXJzaW9ucy8ke1ZlcnNpb25JZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0dyb3VwSWR9JywgZ3JvdXBJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VmVyc2lvbklkfScsIHZlcnNpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29yZURlZmluaXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWNvcmUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29yZURlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjb3JlRGVmaW5pdGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNvcmVEZWZpbml0aW9uKGNvcmVEZWZpbml0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9kZWZpbml0aW9uL2NvcmVzLyR7Q29yZURlZmluaXRpb25JZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0NvcmVEZWZpbml0aW9uSWR9JywgY29yZURlZmluaXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29yZURlZmluaXRpb25WZXJzaW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1jb3JlZGVmaW5pdGlvbnZlcnNpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29yZURlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjb3JlRGVmaW5pdGlvbklkLlxuICAgKiBAcGFyYW0gdmVyc2lvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHZlcnNpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkNvcmVEZWZpbml0aW9uVmVyc2lvbihjb3JlRGVmaW5pdGlvbklkOiBzdHJpbmcsIHZlcnNpb25JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vY29yZXMvJHtDb3JlRGVmaW5pdGlvbklkfS92ZXJzaW9ucy8ke1ZlcnNpb25JZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0NvcmVEZWZpbml0aW9uSWR9JywgY29yZURlZmluaXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VmVyc2lvbklkfScsIHZlcnNpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZGV2aWNlRGVmaW5pdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtZGV2aWNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRldmljZURlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkZXZpY2VEZWZpbml0aW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGV2aWNlRGVmaW5pdGlvbihkZXZpY2VEZWZpbml0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9kZWZpbml0aW9uL2RldmljZXMvJHtEZXZpY2VEZWZpbml0aW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtEZXZpY2VEZWZpbml0aW9uSWR9JywgZGV2aWNlRGVmaW5pdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkZXZpY2VEZWZpbml0aW9uVmVyc2lvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtZGV2aWNlZGVmaW5pdGlvbnZlcnNpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGV2aWNlRGVmaW5pdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRldmljZURlZmluaXRpb25JZC5cbiAgICogQHBhcmFtIHZlcnNpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSB2ZXJzaW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25EZXZpY2VEZWZpbml0aW9uVmVyc2lvbihkZXZpY2VEZWZpbml0aW9uSWQ6IHN0cmluZywgdmVyc2lvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9kZXZpY2VzLyR7RGV2aWNlRGVmaW5pdGlvbklkfS92ZXJzaW9ucy8ke1ZlcnNpb25JZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0RldmljZURlZmluaXRpb25JZH0nLCBkZXZpY2VEZWZpbml0aW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1ZlcnNpb25JZH0nLCB2ZXJzaW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGZ1bmN0aW9uRGVmaW5pdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtZnVuY3Rpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZnVuY3Rpb25EZWZpbml0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZnVuY3Rpb25EZWZpbml0aW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRnVuY3Rpb25EZWZpbml0aW9uKGZ1bmN0aW9uRGVmaW5pdGlvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9mdW5jdGlvbnMvJHtGdW5jdGlvbkRlZmluaXRpb25JZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0Z1bmN0aW9uRGVmaW5pdGlvbklkfScsIGZ1bmN0aW9uRGVmaW5pdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBmdW5jdGlvbkRlZmluaXRpb25WZXJzaW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dyZWVuZ3Jhc3MvbGF0ZXN0L2FwaXJlZmVyZW5jZS9kZWZpbml0aW9ucy1mdW5jdGlvbmRlZmluaXRpb252ZXJzaW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGZ1bmN0aW9uRGVmaW5pdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGZ1bmN0aW9uRGVmaW5pdGlvbklkLlxuICAgKiBAcGFyYW0gdmVyc2lvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHZlcnNpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkZ1bmN0aW9uRGVmaW5pdGlvblZlcnNpb24oZnVuY3Rpb25EZWZpbml0aW9uSWQ6IHN0cmluZywgdmVyc2lvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9mdW5jdGlvbnMvJHtGdW5jdGlvbkRlZmluaXRpb25JZH0vdmVyc2lvbnMvJHtWZXJzaW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtGdW5jdGlvbkRlZmluaXRpb25JZH0nLCBmdW5jdGlvbkRlZmluaXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VmVyc2lvbklkfScsIHZlcnNpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc3Vic2NyaXB0aW9uRGVmaW5pdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtc3Vic2NyaXB0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHN1YnNjcmlwdGlvbkRlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzdWJzY3JpcHRpb25EZWZpbml0aW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU3Vic2NyaXB0aW9uRGVmaW5pdGlvbihzdWJzY3JpcHRpb25EZWZpbml0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Z3JlZW5ncmFzczoke1JlZ2lvbn06JHtBY2NvdW50fTovZ3JlZW5ncmFzcy9kZWZpbml0aW9uL3N1YnNjcmlwdGlvbnMvJHtTdWJzY3JpcHRpb25EZWZpbml0aW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtTdWJzY3JpcHRpb25EZWZpbml0aW9uSWR9Jywgc3Vic2NyaXB0aW9uRGVmaW5pdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtc3Vic2NyaXB0aW9uZGVmaW5pdGlvbnZlcnNpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc3Vic2NyaXB0aW9uRGVmaW5pdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHN1YnNjcmlwdGlvbkRlZmluaXRpb25JZC5cbiAgICogQHBhcmFtIHZlcnNpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSB2ZXJzaW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25TdWJzY3JpcHRpb25EZWZpbml0aW9uVmVyc2lvbihzdWJzY3JpcHRpb25EZWZpbml0aW9uSWQ6IHN0cmluZywgdmVyc2lvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9zdWJzY3JpcHRpb25zLyR7U3Vic2NyaXB0aW9uRGVmaW5pdGlvbklkfS92ZXJzaW9ucy8ke1ZlcnNpb25JZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1N1YnNjcmlwdGlvbkRlZmluaXRpb25JZH0nLCBzdWJzY3JpcHRpb25EZWZpbml0aW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1ZlcnNpb25JZH0nLCB2ZXJzaW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGxvZ2dlckRlZmluaXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWxvZ2dlci5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBsb2dnZXJEZWZpbml0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgbG9nZ2VyRGVmaW5pdGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkxvZ2dlckRlZmluaXRpb24obG9nZ2VyRGVmaW5pdGlvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9sb2dnZXJzLyR7TG9nZ2VyRGVmaW5pdGlvbklkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7TG9nZ2VyRGVmaW5pdGlvbklkfScsIGxvZ2dlckRlZmluaXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbG9nZ2VyRGVmaW5pdGlvblZlcnNpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWxvZ2dlcmRlZmluaXRpb252ZXJzaW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGxvZ2dlckRlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBsb2dnZXJEZWZpbml0aW9uSWQuXG4gICAqIEBwYXJhbSB2ZXJzaW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdmVyc2lvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uTG9nZ2VyRGVmaW5pdGlvblZlcnNpb24obG9nZ2VyRGVmaW5pdGlvbklkOiBzdHJpbmcsIHZlcnNpb25JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vbG9nZ2Vycy8ke0xvZ2dlckRlZmluaXRpb25JZH0vdmVyc2lvbnMvJHtWZXJzaW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtMb2dnZXJEZWZpbml0aW9uSWR9JywgbG9nZ2VyRGVmaW5pdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtWZXJzaW9uSWR9JywgdmVyc2lvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSByZXNvdXJjZURlZmluaXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLXJlc291cmNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlRGVmaW5pdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlRGVmaW5pdGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJlc291cmNlRGVmaW5pdGlvbihyZXNvdXJjZURlZmluaXRpb25JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vcmVzb3VyY2VzLyR7UmVzb3VyY2VEZWZpbml0aW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZURlZmluaXRpb25JZH0nLCByZXNvdXJjZURlZmluaXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcmVzb3VyY2VEZWZpbml0aW9uVmVyc2lvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ncmVlbmdyYXNzL2xhdGVzdC9hcGlyZWZlcmVuY2UvZGVmaW5pdGlvbnMtcmVzb3VyY2VkZWZpbml0aW9udmVyc2lvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZURlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZURlZmluaXRpb25JZC5cbiAgICogQHBhcmFtIHZlcnNpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSB2ZXJzaW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25SZXNvdXJjZURlZmluaXRpb25WZXJzaW9uKHJlc291cmNlRGVmaW5pdGlvbklkOiBzdHJpbmcsIHZlcnNpb25JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vcmVzb3VyY2VzLyR7UmVzb3VyY2VEZWZpbml0aW9uSWR9L3ZlcnNpb25zLyR7VmVyc2lvbklkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VEZWZpbml0aW9uSWR9JywgcmVzb3VyY2VEZWZpbml0aW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1ZlcnNpb25JZH0nLCB2ZXJzaW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNvbm5lY3RvckRlZmluaXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWNvbm5lY3Rvci5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjb25uZWN0b3JEZWZpbml0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY29ubmVjdG9yRGVmaW5pdGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNvbm5lY3RvckRlZmluaXRpb24oY29ubmVjdG9yRGVmaW5pdGlvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmdyZWVuZ3Jhc3M6JHtSZWdpb259OiR7QWNjb3VudH06L2dyZWVuZ3Jhc3MvZGVmaW5pdGlvbi9jb25uZWN0b3JzLyR7Q29ubmVjdG9yRGVmaW5pdGlvbklkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Q29ubmVjdG9yRGVmaW5pdGlvbklkfScsIGNvbm5lY3RvckRlZmluaXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29ubmVjdG9yRGVmaW5pdGlvblZlcnNpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ3JlZW5ncmFzcy9sYXRlc3QvYXBpcmVmZXJlbmNlL2RlZmluaXRpb25zLWNvbm5lY3RvcmRlZmluaXRpb252ZXJzaW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNvbm5lY3RvckRlZmluaXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjb25uZWN0b3JEZWZpbml0aW9uSWQuXG4gICAqIEBwYXJhbSB2ZXJzaW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdmVyc2lvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uQ29ubmVjdG9yRGVmaW5pdGlvblZlcnNpb24oY29ubmVjdG9yRGVmaW5pdGlvbklkOiBzdHJpbmcsIHZlcnNpb25JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpncmVlbmdyYXNzOiR7UmVnaW9ufToke0FjY291bnR9Oi9ncmVlbmdyYXNzL2RlZmluaXRpb24vY29ubmVjdG9ycy8ke0Nvbm5lY3RvckRlZmluaXRpb25JZH0vdmVyc2lvbnMvJHtWZXJzaW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtDb25uZWN0b3JEZWZpbml0aW9uSWR9JywgY29ubmVjdG9yRGVmaW5pdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtWZXJzaW9uSWR9JywgdmVyc2lvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0aGluZyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3QvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3RoaW5nLXJlZ2lzdHJ5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHRoaW5nTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB0aGluZ05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25UaGluZyh0aGluZ05hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06aW90OiR7UmVnaW9ufToke0FjY291bnR9OnRoaW5nLyR7VGhpbmdOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VGhpbmdOYW1lfScsIHRoaW5nTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==