"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotsitewise = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotsitewise](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotsitewise.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotsitewise extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotsitewise](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotsitewise.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotsitewise';
        this.actionList = {
            "AssociateAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_AssociateAssets.html",
                "description": "Grants permission to associate a child asset to a parent asset by a hierarchy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "BatchAssociateProjectAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchAssociateProjectAssets.html",
                "description": "Grants permission to associate assets to a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchDisassociateProjectAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchDisassociateProjectAssets.html",
                "description": "Grants permission to disassociate assets from a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchPutAssetPropertyValue": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchPutAssetPropertyValue.html",
                "description": "Grants permission to put property values for asset properties",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "CreateAccessPolicy": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAccessPolicy.html",
                "description": "Grants permission to create an access policy for a portal or a project",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "portal": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateAsset": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAsset.html",
                "description": "Grants permission to create an asset from an asset model",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset-model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateAssetModel": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModel.html",
                "description": "Grants permission to create an asset model",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDashboard": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDashboard.html",
                "description": "Grants permission to create a dashboard in a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGateway": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateGateway.html",
                "description": "Grants permission to create a gateway",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePortal": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreatePortal.html",
                "description": "Grants permission to create a portal",
                "accessLevel": "Write",
                "dependentActions": [
                    "sso:CreateManagedApplicationInstance",
                    "sso:DescribeRegisteredRegions"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateProject.html",
                "description": "Grants permission to create a project in a portal",
                "accessLevel": "Write",
                "resourceTypes": {
                    "portal": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccessPolicy": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAccessPolicy.html",
                "description": "Grants permission to delete an access policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "access-policy": {
                        "required": true
                    }
                }
            },
            "DeleteAsset": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAsset.html",
                "description": "Grants permission to delete an asset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "DeleteAssetModel": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAssetModel.html",
                "description": "Grants permission to delete an asset model",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset-model": {
                        "required": true
                    }
                }
            },
            "DeleteDashboard": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteDashboard.html",
                "description": "Grants permission to delete a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DeleteGateway": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteGateway.html",
                "description": "Grants permission to delete a gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gateway": {
                        "required": true
                    }
                }
            },
            "DeletePortal": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeletePortal.html",
                "description": "Grants permission to delete a portal",
                "accessLevel": "Write",
                "dependentActions": [
                    "sso:DeleteManagedApplicationInstance"
                ],
                "resourceTypes": {
                    "portal": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteProject.html",
                "description": "Grants permission to delete a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeAccessPolicy": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAccessPolicy.html",
                "description": "Grants permission to describe an access policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "access-policy": {
                        "required": true
                    }
                }
            },
            "DescribeAsset": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAsset.html",
                "description": "Grants permission to describe an asset",
                "accessLevel": "Read",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "DescribeAssetModel": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetModel.html",
                "description": "Grants permission to describe an asset model",
                "accessLevel": "Read",
                "resourceTypes": {
                    "asset-model": {
                        "required": true
                    }
                }
            },
            "DescribeAssetProperty": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetProperty.html",
                "description": "Grants permission to describe an asset property",
                "accessLevel": "Read",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "DescribeDashboard": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeDashboard.html",
                "description": "Grants permission to describe a dashboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DescribeGateway": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeGateway.html",
                "description": "Grants permission to describe a gateway",
                "accessLevel": "Read",
                "resourceTypes": {
                    "gateway": {
                        "required": true
                    }
                }
            },
            "DescribeGatewayCapabilityConfiguration": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeGatewayCapabilityConfiguration.html",
                "description": "Grants permission to describe a capability configuration for a gateway",
                "accessLevel": "Read",
                "resourceTypes": {
                    "gateway": {
                        "required": true
                    }
                }
            },
            "DescribeLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeLoggingOptions.html",
                "description": "Grants permission to describe logging options for the AWS account",
                "accessLevel": "Read"
            },
            "DescribePortal": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribePortal.html",
                "description": "Grants permission to describe a portal",
                "accessLevel": "Read",
                "resourceTypes": {
                    "portal": {
                        "required": true
                    }
                }
            },
            "DescribeProject": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeProject.html",
                "description": "Grants permission to describe a project",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DisassociateAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DisassociateAssets.html",
                "description": "Grants permission to disassociate a child asset from a parent asset by a hierarchy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "GetAssetPropertyAggregates": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyAggregates.html",
                "description": "Grants permission to retrieve computed aggregates for an asset property",
                "accessLevel": "Read",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "GetAssetPropertyValue": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyValue.html",
                "description": "Grants permission to retrieve the latest value for an asset property",
                "accessLevel": "Read",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "GetAssetPropertyValueHistory": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyValueHistory.html",
                "description": "Grants permission to retrieve the value history for an asset property",
                "accessLevel": "Read",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "ListAccessPolicies": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAccessPolicies.html",
                "description": "Grants permission to list all access policies for an identity or a resource",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "portal": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                }
            },
            "ListAssetModels": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetModels.html",
                "description": "Grants permission to list all asset models",
                "accessLevel": "List"
            },
            "ListAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssets.html",
                "description": "Grants permission to list all assets",
                "accessLevel": "List",
                "resourceTypes": {
                    "asset-model": {
                        "required": false
                    }
                }
            },
            "ListAssociatedAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssociatedAssets.html",
                "description": "Grants permission to list all assets associated to an asset by a hierarchy",
                "accessLevel": "List",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "ListDashboards": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListDashboards.html",
                "description": "Grants permission to list all dashboards in a project",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListGateways": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListGateways.html",
                "description": "Grants permission to list all gateways",
                "accessLevel": "List"
            },
            "ListPortals": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListPortals.html",
                "description": "Grants permission to list all portals",
                "accessLevel": "List"
            },
            "ListProjectAssets": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListProjectAssets.html",
                "description": "Grants permission to list all assets associated with a project",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListProjects.html",
                "description": "Grants permission to list all projects in a portal",
                "accessLevel": "List",
                "resourceTypes": {
                    "portal": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list all tags for a resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "access-policy": {
                        "required": false
                    },
                    "asset": {
                        "required": false
                    },
                    "asset-model": {
                        "required": false
                    },
                    "dashboard": {
                        "required": false
                    },
                    "gateway": {
                        "required": false
                    },
                    "portal": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "PutLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_PutLoggingOptions.html",
                "description": "Grants permission to set logging options for the AWS account",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to tag a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "access-policy": {
                        "required": false
                    },
                    "asset": {
                        "required": false
                    },
                    "asset-model": {
                        "required": false
                    },
                    "dashboard": {
                        "required": false
                    },
                    "gateway": {
                        "required": false
                    },
                    "portal": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "access-policy": {
                        "required": false
                    },
                    "asset": {
                        "required": false
                    },
                    "asset-model": {
                        "required": false
                    },
                    "dashboard": {
                        "required": false
                    },
                    "gateway": {
                        "required": false
                    },
                    "portal": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAccessPolicy": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAccessPolicy.html",
                "description": "Grants permission to update an access policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "access-policy": {
                        "required": true
                    }
                }
            },
            "UpdateAsset": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAsset.html",
                "description": "Grants permission to update an asset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "UpdateAssetModel": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetModel.html",
                "description": "Grants permission to update an asset model",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset-model": {
                        "required": true
                    }
                }
            },
            "UpdateAssetProperty": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetProperty.html",
                "description": "Grants permission to update an asset property",
                "accessLevel": "Write",
                "resourceTypes": {
                    "asset": {
                        "required": true
                    }
                }
            },
            "UpdateDashboard": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateDashboard.html",
                "description": "Grants permission to update a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateGateway": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateGateway.html",
                "description": "Grants permission to update a gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gateway": {
                        "required": true
                    }
                }
            },
            "UpdateGatewayCapabilityConfiguration": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateGatewayCapabilityConfiguration.html",
                "description": "Grants permission to update a capability configuration for a gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gateway": {
                        "required": true
                    }
                }
            },
            "UpdatePortal": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdatePortal.html",
                "description": "Grants permission to update a portal",
                "accessLevel": "Write",
                "resourceTypes": {
                    "portal": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateProject.html",
                "description": "Grants permission to update a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "asset": {
                "name": "asset",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAsset.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:asset/${AssetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "asset-model": {
                "name": "asset-model",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModel.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:asset-model/${AssetModelId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "gateway": {
                "name": "gateway",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateGateway.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:gateway/${GatewayId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "portal": {
                "name": "portal",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreatePortal.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:portal/${PortalId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateProject.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:project/${ProjectId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dashboard": {
                "name": "dashboard",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDashboard.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:dashboard/${DashboardId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "access-policy": {
                "name": "access-policy",
                "url": "https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAccessPolicy.html",
                "arn": "arn:${Partition}:iotsitewise:${Region}:${Account}:access-policy/${AccessPolicyId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate a child asset to a parent asset by a hierarchy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_AssociateAssets.html
     */
    toAssociateAssets() {
        this.add('iotsitewise:AssociateAssets');
        return this;
    }
    /**
     * Grants permission to associate assets to a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchAssociateProjectAssets.html
     */
    toBatchAssociateProjectAssets() {
        this.add('iotsitewise:BatchAssociateProjectAssets');
        return this;
    }
    /**
     * Grants permission to disassociate assets from a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchDisassociateProjectAssets.html
     */
    toBatchDisassociateProjectAssets() {
        this.add('iotsitewise:BatchDisassociateProjectAssets');
        return this;
    }
    /**
     * Grants permission to put property values for asset properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchPutAssetPropertyValue.html
     */
    toBatchPutAssetPropertyValue() {
        this.add('iotsitewise:BatchPutAssetPropertyValue');
        return this;
    }
    /**
     * Grants permission to create an access policy for a portal or a project
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAccessPolicy.html
     */
    toCreateAccessPolicy() {
        this.add('iotsitewise:CreateAccessPolicy');
        return this;
    }
    /**
     * Grants permission to create an asset from an asset model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAsset.html
     */
    toCreateAsset() {
        this.add('iotsitewise:CreateAsset');
        return this;
    }
    /**
     * Grants permission to create an asset model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModel.html
     */
    toCreateAssetModel() {
        this.add('iotsitewise:CreateAssetModel');
        return this;
    }
    /**
     * Grants permission to create a dashboard in a project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard() {
        this.add('iotsitewise:CreateDashboard');
        return this;
    }
    /**
     * Grants permission to create a gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateGateway.html
     */
    toCreateGateway() {
        this.add('iotsitewise:CreateGateway');
        return this;
    }
    /**
     * Grants permission to create a portal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sso:CreateManagedApplicationInstance
     * - sso:DescribeRegisteredRegions
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreatePortal.html
     */
    toCreatePortal() {
        this.add('iotsitewise:CreatePortal');
        return this;
    }
    /**
     * Grants permission to create a project in a portal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        this.add('iotsitewise:CreateProject');
        return this;
    }
    /**
     * Grants permission to delete an access policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAccessPolicy.html
     */
    toDeleteAccessPolicy() {
        this.add('iotsitewise:DeleteAccessPolicy');
        return this;
    }
    /**
     * Grants permission to delete an asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAsset.html
     */
    toDeleteAsset() {
        this.add('iotsitewise:DeleteAsset');
        return this;
    }
    /**
     * Grants permission to delete an asset model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAssetModel.html
     */
    toDeleteAssetModel() {
        this.add('iotsitewise:DeleteAssetModel');
        return this;
    }
    /**
     * Grants permission to delete a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard() {
        this.add('iotsitewise:DeleteDashboard');
        return this;
    }
    /**
     * Grants permission to delete a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteGateway.html
     */
    toDeleteGateway() {
        this.add('iotsitewise:DeleteGateway');
        return this;
    }
    /**
     * Grants permission to delete a portal
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteManagedApplicationInstance
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeletePortal.html
     */
    toDeletePortal() {
        this.add('iotsitewise:DeletePortal');
        return this;
    }
    /**
     * Grants permission to delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        this.add('iotsitewise:DeleteProject');
        return this;
    }
    /**
     * Grants permission to describe an access policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAccessPolicy.html
     */
    toDescribeAccessPolicy() {
        this.add('iotsitewise:DescribeAccessPolicy');
        return this;
    }
    /**
     * Grants permission to describe an asset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAsset.html
     */
    toDescribeAsset() {
        this.add('iotsitewise:DescribeAsset');
        return this;
    }
    /**
     * Grants permission to describe an asset model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetModel.html
     */
    toDescribeAssetModel() {
        this.add('iotsitewise:DescribeAssetModel');
        return this;
    }
    /**
     * Grants permission to describe an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetProperty.html
     */
    toDescribeAssetProperty() {
        this.add('iotsitewise:DescribeAssetProperty');
        return this;
    }
    /**
     * Grants permission to describe a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeDashboard.html
     */
    toDescribeDashboard() {
        this.add('iotsitewise:DescribeDashboard');
        return this;
    }
    /**
     * Grants permission to describe a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeGateway.html
     */
    toDescribeGateway() {
        this.add('iotsitewise:DescribeGateway');
        return this;
    }
    /**
     * Grants permission to describe a capability configuration for a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeGatewayCapabilityConfiguration.html
     */
    toDescribeGatewayCapabilityConfiguration() {
        this.add('iotsitewise:DescribeGatewayCapabilityConfiguration');
        return this;
    }
    /**
     * Grants permission to describe logging options for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeLoggingOptions.html
     */
    toDescribeLoggingOptions() {
        this.add('iotsitewise:DescribeLoggingOptions');
        return this;
    }
    /**
     * Grants permission to describe a portal
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribePortal.html
     */
    toDescribePortal() {
        this.add('iotsitewise:DescribePortal');
        return this;
    }
    /**
     * Grants permission to describe a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeProject.html
     */
    toDescribeProject() {
        this.add('iotsitewise:DescribeProject');
        return this;
    }
    /**
     * Grants permission to disassociate a child asset from a parent asset by a hierarchy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DisassociateAssets.html
     */
    toDisassociateAssets() {
        this.add('iotsitewise:DisassociateAssets');
        return this;
    }
    /**
     * Grants permission to retrieve computed aggregates for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyAggregates.html
     */
    toGetAssetPropertyAggregates() {
        this.add('iotsitewise:GetAssetPropertyAggregates');
        return this;
    }
    /**
     * Grants permission to retrieve the latest value for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyValue.html
     */
    toGetAssetPropertyValue() {
        this.add('iotsitewise:GetAssetPropertyValue');
        return this;
    }
    /**
     * Grants permission to retrieve the value history for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyValueHistory.html
     */
    toGetAssetPropertyValueHistory() {
        this.add('iotsitewise:GetAssetPropertyValueHistory');
        return this;
    }
    /**
     * Grants permission to list all access policies for an identity or a resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAccessPolicies.html
     */
    toListAccessPolicies() {
        this.add('iotsitewise:ListAccessPolicies');
        return this;
    }
    /**
     * Grants permission to list all asset models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetModels.html
     */
    toListAssetModels() {
        this.add('iotsitewise:ListAssetModels');
        return this;
    }
    /**
     * Grants permission to list all assets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssets.html
     */
    toListAssets() {
        this.add('iotsitewise:ListAssets');
        return this;
    }
    /**
     * Grants permission to list all assets associated to an asset by a hierarchy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssociatedAssets.html
     */
    toListAssociatedAssets() {
        this.add('iotsitewise:ListAssociatedAssets');
        return this;
    }
    /**
     * Grants permission to list all dashboards in a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        this.add('iotsitewise:ListDashboards');
        return this;
    }
    /**
     * Grants permission to list all gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListGateways.html
     */
    toListGateways() {
        this.add('iotsitewise:ListGateways');
        return this;
    }
    /**
     * Grants permission to list all portals
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListPortals.html
     */
    toListPortals() {
        this.add('iotsitewise:ListPortals');
        return this;
    }
    /**
     * Grants permission to list all assets associated with a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListProjectAssets.html
     */
    toListProjectAssets() {
        this.add('iotsitewise:ListProjectAssets');
        return this;
    }
    /**
     * Grants permission to list all projects in a portal
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        this.add('iotsitewise:ListProjects');
        return this;
    }
    /**
     * Grants permission to list all tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('iotsitewise:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to set logging options for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        this.add('iotsitewise:PutLoggingOptions');
        return this;
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('iotsitewise:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('iotsitewise:UntagResource');
        return this;
    }
    /**
     * Grants permission to update an access policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAccessPolicy.html
     */
    toUpdateAccessPolicy() {
        this.add('iotsitewise:UpdateAccessPolicy');
        return this;
    }
    /**
     * Grants permission to update an asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAsset.html
     */
    toUpdateAsset() {
        this.add('iotsitewise:UpdateAsset');
        return this;
    }
    /**
     * Grants permission to update an asset model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetModel.html
     */
    toUpdateAssetModel() {
        this.add('iotsitewise:UpdateAssetModel');
        return this;
    }
    /**
     * Grants permission to update an asset property
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetProperty.html
     */
    toUpdateAssetProperty() {
        this.add('iotsitewise:UpdateAssetProperty');
        return this;
    }
    /**
     * Grants permission to update a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard() {
        this.add('iotsitewise:UpdateDashboard');
        return this;
    }
    /**
     * Grants permission to update a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateGateway.html
     */
    toUpdateGateway() {
        this.add('iotsitewise:UpdateGateway');
        return this;
    }
    /**
     * Grants permission to update a capability configuration for a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateGatewayCapabilityConfiguration.html
     */
    toUpdateGatewayCapabilityConfiguration() {
        this.add('iotsitewise:UpdateGatewayCapabilityConfiguration');
        return this;
    }
    /**
     * Grants permission to update a portal
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdatePortal.html
     */
    toUpdatePortal() {
        this.add('iotsitewise:UpdatePortal');
        return this;
    }
    /**
     * Grants permission to update a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        this.add('iotsitewise:UpdateProject');
        return this;
    }
    /**
     * Adds a resource of type asset to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAsset.html
     *
     * @param assetId - Identifier for the assetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAsset(assetId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:asset/${AssetId}';
        arn = arn.replace('${AssetId}', assetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type asset-model to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModel.html
     *
     * @param assetModelId - Identifier for the assetModelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssetModel(assetModelId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:asset-model/${AssetModelId}';
        arn = arn.replace('${AssetModelId}', assetModelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gateway to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateGateway.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGateway(gatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:gateway/${GatewayId}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type portal to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreatePortal.html
     *
     * @param portalId - Identifier for the portalId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortal(portalId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:portal/${PortalId}';
        arn = arn.replace('${PortalId}', portalId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateProject.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDashboard.html
     *
     * @param dashboardId - Identifier for the dashboardId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(dashboardId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:dashboard/${DashboardId}';
        arn = arn.replace('${DashboardId}', dashboardId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type access-policy to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAccessPolicy.html
     *
     * @param accessPolicyId - Identifier for the accessPolicyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccessPolicy(accessPolicyId, account, region, partition) {
        var arn = 'arn:${Partition}:iotsitewise:${Region}:${Account}:access-policy/${AccessPolicyId}';
        arn = arn.replace('${AccessPolicyId}', accessPolicyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by an asset hierarchy path, which is the string of asset IDs in the asset's hierarchy, each separated by a forward slash
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssetHierarchyPath(value, operator) {
        return this.if(`iotsitewise:assetHierarchyPath`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of a child asset being associated to a parent asset
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChildAssetId(value, operator) {
        return this.if(`iotsitewise:childAssetId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of an AWS Single Sign-On group
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGroup(value, operator) {
        return this.if(`iotsitewise:group`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of a portal
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPortal(value, operator) {
        return this.if(`iotsitewise:portal`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of a project
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProject(value, operator) {
        return this.if(`iotsitewise:project`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of an asset property
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPropertyId(value, operator) {
        return this.if(`iotsitewise:propertyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of an AWS Single Sign-On user
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUser(value, operator) {
        return this.if(`iotsitewise:user`, value, operator || 'StringLike');
    }
}
exports.Iotsitewise = Iotsitewise;
//# sourceMappingURL=data:application/json;base64,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