"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotthingsgraph = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotthingsgraph](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotthingsgraph.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotthingsgraph extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotthingsgraph](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotthingsgraph.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotthingsgraph';
        this.actionList = {
            "AssociateEntityToThing": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_AssociateEntityToThing.html",
                "description": "Associates a device with a concrete thing that is in the user's registry. A thing can be associated with only one device at a time. If you associate a thing with a new device id, its previous association will be removed.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iot:DescribeThing",
                    "iot:DescribeThingGroup"
                ]
            },
            "CreateFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateFlowTemplate.html",
                "description": "Creates a workflow template. Workflows can be created only in the user's namespace. (The public namespace contains only entities.) The workflow can contain only entities in the specified namespace. The workflow is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.",
                "accessLevel": "Write"
            },
            "CreateSystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemInstance.html",
                "description": "Creates an instance of a system with specified configurations and Things.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemTemplate.html",
                "description": "Creates a system. The system is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.",
                "accessLevel": "Write"
            },
            "DeleteFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteFlowTemplate.html",
                "description": "Deletes a workflow. Any new system or system instance that contains this workflow will fail to update or deploy. Existing system instances that contain the workflow will continue to run (since they use a snapshot of the workflow taken at the time of deploying the system instance).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "DeleteNamespace": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteNamespace.html",
                "description": "Deletes the specified namespace. This action deletes all of the entities in the namespace. Delete the systems and flows in the namespace before performing this action.",
                "accessLevel": "Write"
            },
            "DeleteSystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemInstance.html",
                "description": "Deletes a system instance. Only instances that have never been deployed, or that have been undeployed from the target can be deleted. Users can create a new system instance that has the same ID as a deleted system instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "DeleteSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemTemplate.html",
                "description": "Deletes a system. New system instances can't contain the system after its deletion. Existing system instances that contain the system will continue to work because they use a snapshot of the system that is taken when it is deployed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "DeploySystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeploySystemInstance.html",
                "description": "Deploys the system instance to the target specified in CreateSystemInstance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "DeprecateFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateFlowTemplate.html",
                "description": "Deprecates the specified workflow. This action marks the workflow for deletion. Deprecated flows can't be deployed, but existing system instances that use the flow will continue to run.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "DeprecateSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateSystemTemplate.html",
                "description": "Deprecates the specified system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "DescribeNamespace": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DescribeNamespace.html",
                "description": "Gets the latest version of the user's namespace and the public version that it is tracking.",
                "accessLevel": "Read"
            },
            "DissociateEntityFromThing": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DissociateEntityFromThing.html",
                "description": "Dissociates a device entity from a concrete thing. The action takes only the type of the entity that you need to dissociate because only one entity of a particular type can be associated with a thing.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iot:DescribeThing",
                    "iot:DescribeThingGroup"
                ]
            },
            "GetEntities": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetEntities.html",
                "description": "Gets descriptions of the specified entities. Uses the latest version of the user's namespace by default.",
                "accessLevel": "Read"
            },
            "GetFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplate.html",
                "description": "Gets the latest version of the DefinitionDocument and FlowTemplateSummary for the specified workflow.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "GetFlowTemplateRevisions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplateRevisions.html",
                "description": "Gets revisions of the specified workflow. Only the last 100 revisions are stored. If the workflow has been deprecated, this action will return revisions that occurred before the deprecation. This action won't work for workflows that have been deleted.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "GetNamespaceDeletionStatus": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetNamespaceDeletionStatus.html",
                "description": "Gets the status of a namespace deletion task.",
                "accessLevel": "Read"
            },
            "GetSystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemInstance.html",
                "description": "Gets a system instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "GetSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplate.html",
                "description": "Gets a system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "GetSystemTemplateRevisions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplateRevisions.html",
                "description": "Gets revisions made to the specified system template. Only the previous 100 revisions are stored. If the system has been deprecated, this action will return the revisions that occurred before its deprecation. This action won't work with systems that have been deleted.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "GetUploadStatus": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetUploadStatus.html",
                "description": "Gets the status of the specified upload.",
                "accessLevel": "Read"
            },
            "ListFlowExecutionMessages": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListFlowExecutionMessages.html",
                "description": "Lists details of a single workflow execution",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListTagsForResource.html",
                "description": "Lists all tags for a given resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": false
                    }
                }
            },
            "SearchEntities": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchEntities.html",
                "description": "Searches for entities of the specified type. You can search for entities in your namespace and the public namespace that you're tracking.",
                "accessLevel": "Read"
            },
            "SearchFlowExecutions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowExecutions.html",
                "description": "Searches for workflow executions of a system instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "SearchFlowTemplates": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowTemplates.html",
                "description": "Searches for summary information about workflows.",
                "accessLevel": "Read"
            },
            "SearchSystemInstances": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemInstances.html",
                "description": "Searches for system instances in the user's account.",
                "accessLevel": "Read"
            },
            "SearchSystemTemplates": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemTemplates.html",
                "description": "Searches for summary information about systems in the user's account. You can filter by the ID of a workflow to return only systems that use the specified workflow.",
                "accessLevel": "Read"
            },
            "SearchThings": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchThings.html",
                "description": "Searches for things associated with the specified entity. You can search by both device and device model.",
                "accessLevel": "Read"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_TagResource.html",
                "description": "Tag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UndeploySystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UndeploySystemInstance.html",
                "description": "Removes the system instance and associated triggers from the target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UntagResource.html",
                "description": "Untag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateFlowTemplate.html",
                "description": "Updates the specified workflow. All deployed systems and system instances that use the workflow will see the changes in the flow when it is redeployed. The workflow can contain only entities in the specified namespace.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "UpdateSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateSystemTemplate.html",
                "description": "Updates the specified system. You don't need to run this action after updating a workflow. Any system instance that uses the system will see the changes in the system when it is redeployed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "UploadEntityDefinitions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UploadEntityDefinitions.html",
                "description": "Asynchronously uploads one or more entity definitions to the user's namespace.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Workflow": {
                "name": "Workflow",
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-workflow.html",
                "arn": "arn:${Partition}:iotthingsgraph:${Region}:${Account}:Workflow/${NamespacePath}",
                "conditionKeys": []
            },
            "System": {
                "name": "System",
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-system.html",
                "arn": "arn:${Partition}:iotthingsgraph:${Region}:${Account}:System/${NamespacePath}",
                "conditionKeys": []
            },
            "SystemInstance": {
                "name": "SystemInstance",
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-sdc-deployconfig.html",
                "arn": "arn:${Partition}:iotthingsgraph:${Region}:${Account}:Deployment/${NamespacePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Associates a device with a concrete thing that is in the user's registry. A thing can be associated with only one device at a time. If you associate a thing with a new device id, its previous association will be removed.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_AssociateEntityToThing.html
     */
    toAssociateEntityToThing() {
        this.add('iotthingsgraph:AssociateEntityToThing');
        return this;
    }
    /**
     * Creates a workflow template. Workflows can be created only in the user's namespace. (The public namespace contains only entities.) The workflow can contain only entities in the specified namespace. The workflow is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateFlowTemplate.html
     */
    toCreateFlowTemplate() {
        this.add('iotthingsgraph:CreateFlowTemplate');
        return this;
    }
    /**
     * Creates an instance of a system with specified configurations and Things.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemInstance.html
     */
    toCreateSystemInstance() {
        this.add('iotthingsgraph:CreateSystemInstance');
        return this;
    }
    /**
     * Creates a system. The system is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemTemplate.html
     */
    toCreateSystemTemplate() {
        this.add('iotthingsgraph:CreateSystemTemplate');
        return this;
    }
    /**
     * Deletes a workflow. Any new system or system instance that contains this workflow will fail to update or deploy. Existing system instances that contain the workflow will continue to run (since they use a snapshot of the workflow taken at the time of deploying the system instance).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteFlowTemplate.html
     */
    toDeleteFlowTemplate() {
        this.add('iotthingsgraph:DeleteFlowTemplate');
        return this;
    }
    /**
     * Deletes the specified namespace. This action deletes all of the entities in the namespace. Delete the systems and flows in the namespace before performing this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteNamespace.html
     */
    toDeleteNamespace() {
        this.add('iotthingsgraph:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a system instance. Only instances that have never been deployed, or that have been undeployed from the target can be deleted. Users can create a new system instance that has the same ID as a deleted system instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemInstance.html
     */
    toDeleteSystemInstance() {
        this.add('iotthingsgraph:DeleteSystemInstance');
        return this;
    }
    /**
     * Deletes a system. New system instances can't contain the system after its deletion. Existing system instances that contain the system will continue to work because they use a snapshot of the system that is taken when it is deployed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemTemplate.html
     */
    toDeleteSystemTemplate() {
        this.add('iotthingsgraph:DeleteSystemTemplate');
        return this;
    }
    /**
     * Deploys the system instance to the target specified in CreateSystemInstance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeploySystemInstance.html
     */
    toDeploySystemInstance() {
        this.add('iotthingsgraph:DeploySystemInstance');
        return this;
    }
    /**
     * Deprecates the specified workflow. This action marks the workflow for deletion. Deprecated flows can't be deployed, but existing system instances that use the flow will continue to run.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateFlowTemplate.html
     */
    toDeprecateFlowTemplate() {
        this.add('iotthingsgraph:DeprecateFlowTemplate');
        return this;
    }
    /**
     * Deprecates the specified system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateSystemTemplate.html
     */
    toDeprecateSystemTemplate() {
        this.add('iotthingsgraph:DeprecateSystemTemplate');
        return this;
    }
    /**
     * Gets the latest version of the user's namespace and the public version that it is tracking.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DescribeNamespace.html
     */
    toDescribeNamespace() {
        this.add('iotthingsgraph:DescribeNamespace');
        return this;
    }
    /**
     * Dissociates a device entity from a concrete thing. The action takes only the type of the entity that you need to dissociate because only one entity of a particular type can be associated with a thing.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DissociateEntityFromThing.html
     */
    toDissociateEntityFromThing() {
        this.add('iotthingsgraph:DissociateEntityFromThing');
        return this;
    }
    /**
     * Gets descriptions of the specified entities. Uses the latest version of the user's namespace by default.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetEntities.html
     */
    toGetEntities() {
        this.add('iotthingsgraph:GetEntities');
        return this;
    }
    /**
     * Gets the latest version of the DefinitionDocument and FlowTemplateSummary for the specified workflow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplate.html
     */
    toGetFlowTemplate() {
        this.add('iotthingsgraph:GetFlowTemplate');
        return this;
    }
    /**
     * Gets revisions of the specified workflow. Only the last 100 revisions are stored. If the workflow has been deprecated, this action will return revisions that occurred before the deprecation. This action won't work for workflows that have been deleted.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplateRevisions.html
     */
    toGetFlowTemplateRevisions() {
        this.add('iotthingsgraph:GetFlowTemplateRevisions');
        return this;
    }
    /**
     * Gets the status of a namespace deletion task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetNamespaceDeletionStatus.html
     */
    toGetNamespaceDeletionStatus() {
        this.add('iotthingsgraph:GetNamespaceDeletionStatus');
        return this;
    }
    /**
     * Gets a system instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemInstance.html
     */
    toGetSystemInstance() {
        this.add('iotthingsgraph:GetSystemInstance');
        return this;
    }
    /**
     * Gets a system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplate.html
     */
    toGetSystemTemplate() {
        this.add('iotthingsgraph:GetSystemTemplate');
        return this;
    }
    /**
     * Gets revisions made to the specified system template. Only the previous 100 revisions are stored. If the system has been deprecated, this action will return the revisions that occurred before its deprecation. This action won't work with systems that have been deleted.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplateRevisions.html
     */
    toGetSystemTemplateRevisions() {
        this.add('iotthingsgraph:GetSystemTemplateRevisions');
        return this;
    }
    /**
     * Gets the status of the specified upload.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetUploadStatus.html
     */
    toGetUploadStatus() {
        this.add('iotthingsgraph:GetUploadStatus');
        return this;
    }
    /**
     * Lists details of a single workflow execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListFlowExecutionMessages.html
     */
    toListFlowExecutionMessages() {
        this.add('iotthingsgraph:ListFlowExecutionMessages');
        return this;
    }
    /**
     * Lists all tags for a given resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('iotthingsgraph:ListTagsForResource');
        return this;
    }
    /**
     * Searches for entities of the specified type. You can search for entities in your namespace and the public namespace that you're tracking.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchEntities.html
     */
    toSearchEntities() {
        this.add('iotthingsgraph:SearchEntities');
        return this;
    }
    /**
     * Searches for workflow executions of a system instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowExecutions.html
     */
    toSearchFlowExecutions() {
        this.add('iotthingsgraph:SearchFlowExecutions');
        return this;
    }
    /**
     * Searches for summary information about workflows.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowTemplates.html
     */
    toSearchFlowTemplates() {
        this.add('iotthingsgraph:SearchFlowTemplates');
        return this;
    }
    /**
     * Searches for system instances in the user's account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemInstances.html
     */
    toSearchSystemInstances() {
        this.add('iotthingsgraph:SearchSystemInstances');
        return this;
    }
    /**
     * Searches for summary information about systems in the user's account. You can filter by the ID of a workflow to return only systems that use the specified workflow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemTemplates.html
     */
    toSearchSystemTemplates() {
        this.add('iotthingsgraph:SearchSystemTemplates');
        return this;
    }
    /**
     * Searches for things associated with the specified entity. You can search by both device and device model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchThings.html
     */
    toSearchThings() {
        this.add('iotthingsgraph:SearchThings');
        return this;
    }
    /**
     * Tag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_TagResource.html
     */
    toTagResource() {
        this.add('iotthingsgraph:TagResource');
        return this;
    }
    /**
     * Removes the system instance and associated triggers from the target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UndeploySystemInstance.html
     */
    toUndeploySystemInstance() {
        this.add('iotthingsgraph:UndeploySystemInstance');
        return this;
    }
    /**
     * Untag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UntagResource.html
     */
    toUntagResource() {
        this.add('iotthingsgraph:UntagResource');
        return this;
    }
    /**
     * Updates the specified workflow. All deployed systems and system instances that use the workflow will see the changes in the flow when it is redeployed. The workflow can contain only entities in the specified namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateFlowTemplate.html
     */
    toUpdateFlowTemplate() {
        this.add('iotthingsgraph:UpdateFlowTemplate');
        return this;
    }
    /**
     * Updates the specified system. You don't need to run this action after updating a workflow. Any system instance that uses the system will see the changes in the system when it is redeployed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateSystemTemplate.html
     */
    toUpdateSystemTemplate() {
        this.add('iotthingsgraph:UpdateSystemTemplate');
        return this;
    }
    /**
     * Asynchronously uploads one or more entity definitions to the user's namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UploadEntityDefinitions.html
     */
    toUploadEntityDefinitions() {
        this.add('iotthingsgraph:UploadEntityDefinitions');
        return this;
    }
    /**
     * Adds a resource of type Workflow to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-workflow.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkflow(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:Workflow/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type System to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-system.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSystem(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:System/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type SystemInstance to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-sdc-deployconfig.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSystemInstance(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:Deployment/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Iotthingsgraph = Iotthingsgraph;
//# sourceMappingURL=data:application/json;base64,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