"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Organizations = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [organizations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsorganizations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Organizations extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [organizations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsorganizations.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'organizations';
        this.actionList = {
            "AcceptHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html",
                "description": "Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html",
                "description": "Grants permission to attach a policy to a root, an organizational unit, or an individual account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    },
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "CancelHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html",
                "description": "Grants permission to cancel a handshake.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "CreateAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html",
                "description": "Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGovCloudAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html",
                "description": "Grants permission to create an AWS GovCloud (US) account.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html",
                "description": "Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the master account of the new organization.",
                "accessLevel": "Write"
            },
            "CreateOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html",
                "description": "Grants permission to create an organizational unit (OU) within a root or parent OU.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html",
                "description": "Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:PolicyType",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeclineHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html",
                "description": "Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "DeleteOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html",
                "description": "Grants permission to delete the organization.",
                "accessLevel": "Write"
            },
            "DeleteOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html",
                "description": "Grants permission to delete an organizational unit from a root or another OU.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html",
                "description": "Grants permission to delete a policy from your organization.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DeregisterDelegatedAdministrator": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html",
                "description": "Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "DescribeAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html",
                "description": "Grants permission to retrieve Organizations-related details about the specified account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                }
            },
            "DescribeCreateAccountStatus": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html",
                "description": "Grants permission to retrieve the current status of an asynchronous request to create an account.",
                "accessLevel": "Read"
            },
            "DescribeEffectivePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html",
                "description": "Grants permission to retrieve the effective policy for an account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DescribeHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html",
                "description": "Grants permission to retrieve details about a previously requested handshake.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "DescribeOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html",
                "description": "Grants permission to retrieves details about the organization that the calling credentials belong to.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html",
                "description": "Grants permission to retrieve details about an organizational unit (OU).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": true
                    }
                }
            },
            "DescribePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html",
                "description": "Grants permission to retrieves details about a policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html",
                "description": "Grants permission to detach a policy from a target root, organizational unit, or account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    },
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DisableAWSServiceAccess": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html",
                "description": "Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "DisablePolicyType": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html",
                "description": "Grants permission to disable an organization policy type in a root.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "root": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "EnableAWSServiceAccess": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html",
                "description": "Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "EnableAllFeatures": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html",
                "description": "Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features.",
                "accessLevel": "Write"
            },
            "EnablePolicyType": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html",
                "description": "Grants permission to enable a policy type in a root.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "root": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "InviteAccountToOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html",
                "description": "Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "LeaveOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html",
                "description": "Grants permission to remove a member account from its parent organization.",
                "accessLevel": "Write"
            },
            "ListAWSServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html",
                "description": "Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization.",
                "accessLevel": "List"
            },
            "ListAccounts": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html",
                "description": "Grants permission to list all of the the accounts in the organization.",
                "accessLevel": "List"
            },
            "ListAccountsForParent": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html",
                "description": "Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU).",
                "accessLevel": "List",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListChildren": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html",
                "description": "Grants permission to list all of the OUs or accounts that are contained in a parent OU or root.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListCreateAccountStatus": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html",
                "description": "Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization.",
                "accessLevel": "List"
            },
            "ListDelegatedAdministrators": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html",
                "description": "Grants permission to list the AWS accounts that are designated as delegated administrators in this organization.",
                "accessLevel": "List",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "ListDelegatedServicesForAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html",
                "description": "Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization.",
                "accessLevel": "List",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                }
            },
            "ListHandshakesForAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html",
                "description": "Grants permission to list all of the handshakes that are associated with an account.",
                "accessLevel": "List"
            },
            "ListHandshakesForOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html",
                "description": "Grants permission to list the handshakes that are associated with the organization.",
                "accessLevel": "List"
            },
            "ListOrganizationalUnitsForParent": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html",
                "description": "Grants permission to lists all of the organizational units (OUs) in a parent organizational unit or root.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListParents": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html",
                "description": "Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    }
                }
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html",
                "description": "Grants permission to list all of the policies in an organization.",
                "accessLevel": "List",
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "ListPoliciesForTarget": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html",
                "description": "Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "ListRoots": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html",
                "description": "Grants permission to list all of the roots that are defined in the organization.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list all tags for the specified resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "policy": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListTargetsForPolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html",
                "description": "Grants permission to list all the roots, OUs, and accounts to which a policy is attached.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "MoveAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html",
                "description": "Grants permission to move an account from its current root or OU to another parent root or OU.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": true
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "RegisterDelegatedAdministrator": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html",
                "description": "Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "RemoveAccountFromOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html",
                "description": "Grants permission to removes the specified account from the organization.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to add one or more tags to the specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "policy": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove one or more tags from the specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "policy": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html",
                "description": "Grants permission to rename an organizational unit (OU).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": true
                    }
                }
            },
            "UpdatePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html",
                "description": "Grants permission to update an existing policy with a new name, description, or content.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            }
        };
        this.resourceTypes = {
            "account": {
                "name": "account",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:account/o-${OrganizationId}/${AccountId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "handshake": {
                "name": "handshake",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:handshake/o-${OrganizationId}/${HandshakeType}/h-${HandshakeId}",
                "conditionKeys": []
            },
            "organization": {
                "name": "organization",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:organization/o-${OrganizationId}",
                "conditionKeys": []
            },
            "organizationalunit": {
                "name": "organizationalunit",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:ou/o-${OrganizationId}/ou-${OrganizationalUnitId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:policy/o-${OrganizationId}/${PolicyType}/p-${PolicyId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "awspolicy": {
                "name": "awspolicy",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::aws:policy/${PolicyType}/p-${PolicyId}",
                "conditionKeys": []
            },
            "root": {
                "name": "root",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:root/o-${OrganizationId}/r-${RootId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html
     */
    toAcceptHandshake() {
        this.add('organizations:AcceptHandshake');
        return this;
    }
    /**
     * Grants permission to attach a policy to a root, an organizational unit, or an individual account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        this.add('organizations:AttachPolicy');
        return this;
    }
    /**
     * Grants permission to cancel a handshake.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html
     */
    toCancelHandshake() {
        this.add('organizations:CancelHandshake');
        return this;
    }
    /**
     * Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        this.add('organizations:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create an AWS GovCloud (US) account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html
     */
    toCreateGovCloudAccount() {
        this.add('organizations:CreateGovCloudAccount');
        return this;
    }
    /**
     * Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the master account of the new organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html
     */
    toCreateOrganization() {
        this.add('organizations:CreateOrganization');
        return this;
    }
    /**
     * Grants permission to create an organizational unit (OU) within a root or parent OU.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html
     */
    toCreateOrganizationalUnit() {
        this.add('organizations:CreateOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        this.add('organizations:CreatePolicy');
        return this;
    }
    /**
     * Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html
     */
    toDeclineHandshake() {
        this.add('organizations:DeclineHandshake');
        return this;
    }
    /**
     * Grants permission to delete the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html
     */
    toDeleteOrganization() {
        this.add('organizations:DeleteOrganization');
        return this;
    }
    /**
     * Grants permission to delete an organizational unit from a root or another OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html
     */
    toDeleteOrganizationalUnit() {
        this.add('organizations:DeleteOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to delete a policy from your organization.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.add('organizations:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html
     */
    toDeregisterDelegatedAdministrator() {
        this.add('organizations:DeregisterDelegatedAdministrator');
        return this;
    }
    /**
     * Grants permission to retrieve Organizations-related details about the specified account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html
     */
    toDescribeAccount() {
        this.add('organizations:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the current status of an asynchronous request to create an account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html
     */
    toDescribeCreateAccountStatus() {
        this.add('organizations:DescribeCreateAccountStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the effective policy for an account.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html
     */
    toDescribeEffectivePolicy() {
        this.add('organizations:DescribeEffectivePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve details about a previously requested handshake.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html
     */
    toDescribeHandshake() {
        this.add('organizations:DescribeHandshake');
        return this;
    }
    /**
     * Grants permission to retrieves details about the organization that the calling credentials belong to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        this.add('organizations:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to retrieve details about an organizational unit (OU).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html
     */
    toDescribeOrganizationalUnit() {
        this.add('organizations:DescribeOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to retrieves details about a policy.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html
     */
    toDescribePolicy() {
        this.add('organizations:DescribePolicy');
        return this;
    }
    /**
     * Grants permission to detach a policy from a target root, organizational unit, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        this.add('organizations:DetachPolicy');
        return this;
    }
    /**
     * Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html
     */
    toDisableAWSServiceAccess() {
        this.add('organizations:DisableAWSServiceAccess');
        return this;
    }
    /**
     * Grants permission to disable an organization policy type in a root.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html
     */
    toDisablePolicyType() {
        this.add('organizations:DisablePolicyType');
        return this;
    }
    /**
     * Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html
     */
    toEnableAWSServiceAccess() {
        this.add('organizations:EnableAWSServiceAccess');
        return this;
    }
    /**
     * Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html
     */
    toEnableAllFeatures() {
        this.add('organizations:EnableAllFeatures');
        return this;
    }
    /**
     * Grants permission to enable a policy type in a root.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html
     */
    toEnablePolicyType() {
        this.add('organizations:EnablePolicyType');
        return this;
    }
    /**
     * Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html
     */
    toInviteAccountToOrganization() {
        this.add('organizations:InviteAccountToOrganization');
        return this;
    }
    /**
     * Grants permission to remove a member account from its parent organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html
     */
    toLeaveOrganization() {
        this.add('organizations:LeaveOrganization');
        return this;
    }
    /**
     * Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html
     */
    toListAWSServiceAccessForOrganization() {
        this.add('organizations:ListAWSServiceAccessForOrganization');
        return this;
    }
    /**
     * Grants permission to list all of the the accounts in the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        this.add('organizations:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html
     */
    toListAccountsForParent() {
        this.add('organizations:ListAccountsForParent');
        return this;
    }
    /**
     * Grants permission to list all of the OUs or accounts that are contained in a parent OU or root.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html
     */
    toListChildren() {
        this.add('organizations:ListChildren');
        return this;
    }
    /**
     * Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html
     */
    toListCreateAccountStatus() {
        this.add('organizations:ListCreateAccountStatus');
        return this;
    }
    /**
     * Grants permission to list the AWS accounts that are designated as delegated administrators in this organization.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html
     */
    toListDelegatedAdministrators() {
        this.add('organizations:ListDelegatedAdministrators');
        return this;
    }
    /**
     * Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html
     */
    toListDelegatedServicesForAccount() {
        this.add('organizations:ListDelegatedServicesForAccount');
        return this;
    }
    /**
     * Grants permission to list all of the handshakes that are associated with an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html
     */
    toListHandshakesForAccount() {
        this.add('organizations:ListHandshakesForAccount');
        return this;
    }
    /**
     * Grants permission to list the handshakes that are associated with the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html
     */
    toListHandshakesForOrganization() {
        this.add('organizations:ListHandshakesForOrganization');
        return this;
    }
    /**
     * Grants permission to lists all of the organizational units (OUs) in a parent organizational unit or root.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html
     */
    toListOrganizationalUnitsForParent() {
        this.add('organizations:ListOrganizationalUnitsForParent');
        return this;
    }
    /**
     * Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html
     */
    toListParents() {
        this.add('organizations:ListParents');
        return this;
    }
    /**
     * Grants permission to list all of the policies in an organization.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        this.add('organizations:ListPolicies');
        return this;
    }
    /**
     * Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html
     */
    toListPoliciesForTarget() {
        this.add('organizations:ListPoliciesForTarget');
        return this;
    }
    /**
     * Grants permission to list all of the roots that are defined in the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html
     */
    toListRoots() {
        this.add('organizations:ListRoots');
        return this;
    }
    /**
     * Grants permission to list all tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('organizations:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list all the roots, OUs, and accounts to which a policy is attached.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html
     */
    toListTargetsForPolicy() {
        this.add('organizations:ListTargetsForPolicy');
        return this;
    }
    /**
     * Grants permission to move an account from its current root or OU to another parent root or OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html
     */
    toMoveAccount() {
        this.add('organizations:MoveAccount');
        return this;
    }
    /**
     * Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html
     */
    toRegisterDelegatedAdministrator() {
        this.add('organizations:RegisterDelegatedAdministrator');
        return this;
    }
    /**
     * Grants permission to removes the specified account from the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html
     */
    toRemoveAccountFromOrganization() {
        this.add('organizations:RemoveAccountFromOrganization');
        return this;
    }
    /**
     * Grants permission to add one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('organizations:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('organizations:UntagResource');
        return this;
    }
    /**
     * Grants permission to rename an organizational unit (OU).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html
     */
    toUpdateOrganizationalUnit() {
        this.add('organizations:UpdateOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to update an existing policy with a new name, description, or content.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html
     */
    toUpdatePolicy() {
        this.add('organizations:UpdatePolicy');
        return this;
    }
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccount(masterAccountId, organizationId, accountId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:account/o-${OrganizationId}/${AccountId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type handshake to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param handshakeType - Identifier for the handshakeType.
     * @param handshakeId - Identifier for the handshakeId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHandshake(masterAccountId, organizationId, handshakeType, handshakeId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:handshake/o-${OrganizationId}/${HandshakeType}/h-${HandshakeId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${HandshakeType}', handshakeType);
        arn = arn.replace('${HandshakeId}', handshakeId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganization(masterAccountId, organizationId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:organization/o-${OrganizationId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type organizationalunit to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param organizationalUnitId - Identifier for the organizationalUnitId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganizationalunit(masterAccountId, organizationId, organizationalUnitId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:ou/o-${OrganizationId}/ou-${OrganizationalUnitId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${OrganizationalUnitId}', organizationalUnitId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(masterAccountId, organizationId, policyType, policyId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:policy/o-${OrganizationId}/${PolicyType}/p-${PolicyId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${PolicyType}', policyType);
        arn = arn.replace('${PolicyId}', policyId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type awspolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAwspolicy(policyType, policyId, partition) {
        var arn = 'arn:${Partition}:organizations::aws:policy/${PolicyType}/p-${PolicyId}';
        arn = arn.replace('${PolicyType}', policyType);
        arn = arn.replace('${PolicyId}', policyId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type root to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param rootId - Identifier for the rootId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoot(masterAccountId, organizationId, rootId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:root/o-${OrganizationId}/r-${RootId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${RootId}', rootId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to filter the request to only the specified policy type names.
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toAttachPolicy()
     * - .toCreatePolicy()
     * - .toDeletePolicy()
     * - .toDescribeEffectivePolicy()
     * - .toDescribePolicy()
     * - .toDetachPolicy()
     * - .toDisablePolicyType()
     * - .toEnablePolicyType()
     * - .toListPolicies()
     * - .toListPoliciesForTarget()
     * - .toListTargetsForPolicy()
     * - .toUpdatePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPolicyType(value, operator) {
        return this.if(`organizations:PolicyType`, value, operator || 'StringLike');
    }
    /**
     * Enables you to filter the request to only the specified service principal names.
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toDeregisterDelegatedAdministrator()
     * - .toDisableAWSServiceAccess()
     * - .toEnableAWSServiceAccess()
     * - .toListDelegatedAdministrators()
     * - .toRegisterDelegatedAdministrator()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServicePrincipal(value, operator) {
        return this.if(`organizations:ServicePrincipal`, value, operator || 'StringLike');
    }
}
exports.Organizations = Organizations;
//# sourceMappingURL=data:application/json;base64,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