"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Secretsmanager = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecretsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Secretsmanager extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecretsmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'secretsmanager';
        this.actionList = {
            "CancelRotateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to cancel an in-progress secret rotation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "CreateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to create a secret that stores encrypted data that can be queried and rotated.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:Name",
                    "secretsmanager:Description",
                    "secretsmanager:KmsKeyId",
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to delete the resource policy attached to a secret.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DeleteSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to delete a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:RecoveryWindowInDays",
                    "secretsmanager:ForceDeleteWithoutRecovery",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DescribeSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to retrieve the metadata about a secret, but not the encrypted data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "GetRandomPassword": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to generate a random string for use in password creation.",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to get the resource policy attached to a secret.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "GetSecretValue": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to retrieve and decrypt the encrypted data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:VersionId",
                    "secretsmanager:VersionStage",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ListSecretVersionIds": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to list the available versions of a secret.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ListSecrets": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to list the available secrets.",
                "accessLevel": "List"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to attach a resource policy to a secret.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key",
                    "secretsmanager:BlockPublicPolicy"
                ]
            },
            "PutSecretValue": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to create a new version of the secret with new encrypted data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "RestoreSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to cancel deletion of a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "RotateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to start rotation of a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:RotationLambdaARN",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to add tags to a secret.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to remove tags from a secret.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "aws:TagKeys",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UpdateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to update a secret with new metadata or with a new version of the encrypted data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:Description",
                    "secretsmanager:KmsKeyId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UpdateSecretVersionStage": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to move a stage from one secret to another.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:VersionStage",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ValidateResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to validate a resource policy before attaching policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            }
        };
        this.resourceTypes = {
            "Secret": {
                "name": "Secret",
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources",
                "arn": "arn:${Partition}:secretsmanager:${Region}:${Account}:secret:${SecretId}",
                "conditionKeys": [
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:ResourceTag/tag-key",
                    "secretsmanager:resource/AllowRotationLambdaArn"
                ]
            }
        };
    }
    /**
     * Enables the user to cancel an in-progress secret rotation.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toCancelRotateSecret() {
        this.add('secretsmanager:CancelRotateSecret');
        return this;
    }
    /**
     * Enables the user to create a secret that stores encrypted data that can be queried and rotated.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifName()
     * - .ifDescription()
     * - .ifKmsKeyId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toCreateSecret() {
        this.add('secretsmanager:CreateSecret');
        return this;
    }
    /**
     * Enables the user to delete the resource policy attached to a secret.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toDeleteResourcePolicy() {
        this.add('secretsmanager:DeleteResourcePolicy');
        return this;
    }
    /**
     * Enables the user to delete a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifRecoveryWindowInDays()
     * - .ifForceDeleteWithoutRecovery()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toDeleteSecret() {
        this.add('secretsmanager:DeleteSecret');
        return this;
    }
    /**
     * Enables the user to retrieve the metadata about a secret, but not the encrypted data.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toDescribeSecret() {
        this.add('secretsmanager:DescribeSecret');
        return this;
    }
    /**
     * Enables the user to generate a random string for use in password creation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toGetRandomPassword() {
        this.add('secretsmanager:GetRandomPassword');
        return this;
    }
    /**
     * Enables the user to get the resource policy attached to a secret.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toGetResourcePolicy() {
        this.add('secretsmanager:GetResourcePolicy');
        return this;
    }
    /**
     * Enables the user to retrieve and decrypt the encrypted data.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toGetSecretValue() {
        this.add('secretsmanager:GetSecretValue');
        return this;
    }
    /**
     * Enables the user to list the available versions of a secret.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toListSecretVersionIds() {
        this.add('secretsmanager:ListSecretVersionIds');
        return this;
    }
    /**
     * Enables the user to list the available secrets.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toListSecrets() {
        this.add('secretsmanager:ListSecrets');
        return this;
    }
    /**
     * Enables the user to attach a resource policy to a secret.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifBlockPublicPolicy()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toPutResourcePolicy() {
        this.add('secretsmanager:PutResourcePolicy');
        return this;
    }
    /**
     * Enables the user to create a new version of the secret with new encrypted data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toPutSecretValue() {
        this.add('secretsmanager:PutSecretValue');
        return this;
    }
    /**
     * Enables the user to cancel deletion of a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toRestoreSecret() {
        this.add('secretsmanager:RestoreSecret');
        return this;
    }
    /**
     * Enables the user to start rotation of a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifRotationLambdaARN()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toRotateSecret() {
        this.add('secretsmanager:RotateSecret');
        return this;
    }
    /**
     * Enables the user to add tags to a secret.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toTagResource() {
        this.add('secretsmanager:TagResource');
        return this;
    }
    /**
     * Enables the user to remove tags from a secret.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toUntagResource() {
        this.add('secretsmanager:UntagResource');
        return this;
    }
    /**
     * Enables the user to update a secret with new metadata or with a new version of the encrypted data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifDescription()
     * - .ifKmsKeyId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toUpdateSecret() {
        this.add('secretsmanager:UpdateSecret');
        return this;
    }
    /**
     * Enables the user to move a stage from one secret to another.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toUpdateSecretVersionStage() {
        this.add('secretsmanager:UpdateSecretVersionStage');
        return this;
    }
    /**
     * Enables the user to validate a resource policy before attaching policy.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    toValidateResourcePolicy() {
        this.add('secretsmanager:ValidateResourcePolicy');
        return this;
    }
    /**
     * Adds a resource of type Secret to the statement
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources
     *
     * @param secretId - Identifier for the secretId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     * - .ifResource()
     */
    onSecret(secretId, account, region, partition) {
        var arn = 'arn:${Partition}:secretsmanager:${Region}:${Account}:secret:${SecretId}';
        arn = arn.replace('${SecretId}', secretId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by whether the resource policy blocks broad AWS account access.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toPutResourcePolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBlockPublicPolicy(value) {
        return this.if(`secretsmanager:BlockPublicPolicy`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the description text in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toUpdateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDescription(value, operator) {
        return this.if(`secretsmanager:Description`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the secret is to be deleted immediately without any recovery window.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toDeleteSecret()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifForceDeleteWithoutRecovery(value) {
        return this.if(`secretsmanager:ForceDeleteWithoutRecovery`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of the KMS key in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toUpdateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKmsKeyId(value, operator) {
        return this.if(`secretsmanager:KmsKeyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the friendly name of the secret in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`secretsmanager:Name`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the number of days that Secrets Manager waits before it can delete the secret.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toDeleteSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRecoveryWindowInDays(value, operator) {
        return this.if(`secretsmanager:RecoveryWindowInDays`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by a tag key and value pair.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toCreateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagkey, value, operator) {
        return this.if(`secretsmanager:ResourceTag/${tagkey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toRotateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRotationLambdaARN(value, operator) {
        return this.if(`secretsmanager:RotationLambdaARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the SecretID value in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSecretId(value, operator) {
        return this.if(`secretsmanager:SecretId`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the unique identifier of the version of the secret in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toGetSecretValue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`secretsmanager:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of version stages in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toGetSecretValue()
     * - .toUpdateSecretVersionStage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionStage(value, operator) {
        return this.if(`secretsmanager:VersionStage`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function associated with the secret.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param allowRotationLambdaArn The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifResource(allowRotationLambdaArn, value, operator) {
        return this.if(`secretsmanager:resource/${allowRotationLambdaArn}`, value, operator || 'ArnEquals');
    }
}
exports.Secretsmanager = Secretsmanager;
//# sourceMappingURL=data:application/json;base64,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