"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securityhub = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [securityhub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecurityhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securityhub extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [securityhub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecurityhub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'securityhub';
        this.actionList = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html",
                "description": "Grants permission to accept Security Hub invitations to become a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "BatchDisableStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html",
                "description": "Grants permission to disable standards in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "BatchEnableStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html",
                "description": "Grants permission to enable standards in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "BatchImportFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html",
                "description": "Grants permission to import findings into Security Hub from an integrated product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "product": {
                        "required": true
                    }
                },
                "conditions": [
                    "securityhub:TargetAccount"
                ]
            },
            "BatchUpdateFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html",
                "description": "Grants permission to update customer-controlled fields for a selected set of Security Hub findings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                },
                "conditions": [
                    "securityhub:ASFFSyntaxPath/${ASFFSyntaxPath}"
                ]
            },
            "CreateActionTarget": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html",
                "description": "Grants permission to create custom actions in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "CreateInsight": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html",
                "description": "Grants permission to create insights in Security Hub. Insights are collections of related findings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "CreateMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html",
                "description": "Grants permission to create member accounts in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeclineInvitations": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html",
                "description": "Grants permission to decline Security Hub invitations to become a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteActionTarget": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html",
                "description": "Grants permission to delete custom actions in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteInsight": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html",
                "description": "Grants permission to delete insights from Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteInvitations": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html",
                "description": "Grants permission to delete Security Hub invitations to become a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html",
                "description": "Grants permission to delete Security Hub member accounts",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeActionTargets": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html",
                "description": "Grants permission to retrieve a list of custom actions using the API",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeHub": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html",
                "description": "Grants permission to retrieve information about the hub resource in your account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeProducts": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html",
                "description": "Grants permission to retrieve information about the available Security Hub product integrations",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html",
                "description": "Grants permission to retrieve information about Security Hub standards",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeStandardsControls": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html",
                "description": "Grants permission to retrieve information about Security Hub standards controls",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisableImportFindingsForProduct": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html",
                "description": "Grants permission to disable the findings importing for a Security Hub integrated product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisableSecurityHub": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html",
                "description": "Grants permission to disable Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisassociateFromMasterAccount": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html",
                "description": "Grants permission to a Security Hub member account to disassociate from the associated master account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisassociateMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html",
                "description": "Grants permission to disassociate Security Hub member accounts from the associated master account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "EnableImportFindingsForProduct": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html",
                "description": "Grants permission to enable the findings importing for a Security Hub integrated product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "EnableSecurityHub": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html",
                "description": "Grants permission to enable Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "GetEnabledStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html",
                "description": "Grants permission to retrieve a list of the standards that are enabled in Security Hub",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindings.html",
                "description": "Grants permission to retrieve a list of findings from Security Hub",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetInsightResults": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html",
                "description": "Grants permission to retrieve insight results from Security Hub",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetInsights": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html",
                "description": "Grants permission to retrieve Security Hub insights",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetInvitationsCount": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html",
                "description": "Grants permission to retrieve the count of Security Hub membership invitations sent to the account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetMasterAccount": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html",
                "description": "Grants permission to retrieve details about the Security Hub master account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html",
                "description": "Grants permission to retrieve the details of Security Hub member accounts",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "InviteMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html",
                "description": "Grants permission to invite other AWS accounts to become Security Hub member accounts",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListEnabledProductsForImport": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html",
                "description": "Grants permission to retrieve the Security Hub integrated products that are currently enabled",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html",
                "description": "Grants permission to retrieve the Security Hub invitations sent to the account",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html",
                "description": "Grants permission to retrieve details about Security Hub member accounts associated with the master account",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list of tags associated with a resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html",
                "description": "Grants permission to add tags to a Security Hub resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove tags from a Security Hub resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": true
                    }
                }
            },
            "UpdateActionTarget": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html",
                "description": "Grants permission to update custom actions in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html",
                "description": "Grants permission to update Security Hub findings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateInsight": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html",
                "description": "Grants permission to update insights in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateSecurityHubConfiguration": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityHubConfiguration.html",
                "description": "Grants permission to update Security Hub configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateStandardsControl": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html",
                "description": "Grants permission to update Security Hub standards controls",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "hub": {
                "name": "hub",
                "url": "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources",
                "arn": "arn:${Partition}:securityhub:${Region}:${Account}:hub/default",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "product": {
                "name": "product",
                "url": "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources",
                "arn": "arn:${Partition}:securityhub:${Region}:${Account}:product/${Company}/${ProductId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        this.add('securityhub:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to disable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html
     */
    toBatchDisableStandards() {
        this.add('securityhub:BatchDisableStandards');
        return this;
    }
    /**
     * Grants permission to enable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html
     */
    toBatchEnableStandards() {
        this.add('securityhub:BatchEnableStandards');
        return this;
    }
    /**
     * Grants permission to import findings into Security Hub from an integrated product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetAccount()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html
     */
    toBatchImportFindings() {
        this.add('securityhub:BatchImportFindings');
        return this;
    }
    /**
     * Grants permission to update customer-controlled fields for a selected set of Security Hub findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifASFFSyntaxPath()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html
     */
    toBatchUpdateFindings() {
        this.add('securityhub:BatchUpdateFindings');
        return this;
    }
    /**
     * Grants permission to create custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html
     */
    toCreateActionTarget() {
        this.add('securityhub:CreateActionTarget');
        return this;
    }
    /**
     * Grants permission to create insights in Security Hub. Insights are collections of related findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html
     */
    toCreateInsight() {
        this.add('securityhub:CreateInsight');
        return this;
    }
    /**
     * Grants permission to create member accounts in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        this.add('securityhub:CreateMembers');
        return this;
    }
    /**
     * Grants permission to decline Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        this.add('securityhub:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html
     */
    toDeleteActionTarget() {
        this.add('securityhub:DeleteActionTarget');
        return this;
    }
    /**
     * Grants permission to delete insights from Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html
     */
    toDeleteInsight() {
        this.add('securityhub:DeleteInsight');
        return this;
    }
    /**
     * Grants permission to delete Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        this.add('securityhub:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        this.add('securityhub:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of custom actions using the API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html
     */
    toDescribeActionTargets() {
        this.add('securityhub:DescribeActionTargets');
        return this;
    }
    /**
     * Grants permission to retrieve information about the hub resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html
     */
    toDescribeHub() {
        this.add('securityhub:DescribeHub');
        return this;
    }
    /**
     * Grants permission to retrieve information about the available Security Hub product integrations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html
     */
    toDescribeProducts() {
        this.add('securityhub:DescribeProducts');
        return this;
    }
    /**
     * Grants permission to retrieve information about Security Hub standards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html
     */
    toDescribeStandards() {
        this.add('securityhub:DescribeStandards');
        return this;
    }
    /**
     * Grants permission to retrieve information about Security Hub standards controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html
     */
    toDescribeStandardsControls() {
        this.add('securityhub:DescribeStandardsControls');
        return this;
    }
    /**
     * Grants permission to disable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html
     */
    toDisableImportFindingsForProduct() {
        this.add('securityhub:DisableImportFindingsForProduct');
        return this;
    }
    /**
     * Grants permission to disable Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html
     */
    toDisableSecurityHub() {
        this.add('securityhub:DisableSecurityHub');
        return this;
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        this.add('securityhub:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants permission to disassociate Security Hub member accounts from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        this.add('securityhub:DisassociateMembers');
        return this;
    }
    /**
     * Grants permission to enable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html
     */
    toEnableImportFindingsForProduct() {
        this.add('securityhub:EnableImportFindingsForProduct');
        return this;
    }
    /**
     * Grants permission to enable Security Hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html
     */
    toEnableSecurityHub() {
        this.add('securityhub:EnableSecurityHub');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the standards that are enabled in Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html
     */
    toGetEnabledStandards() {
        this.add('securityhub:GetEnabledStandards');
        return this;
    }
    /**
     * Grants permission to retrieve a list of findings from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        this.add('securityhub:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve insight results from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html
     */
    toGetInsightResults() {
        this.add('securityhub:GetInsightResults');
        return this;
    }
    /**
     * Grants permission to retrieve Security Hub insights
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html
     */
    toGetInsights() {
        this.add('securityhub:GetInsights');
        return this;
    }
    /**
     * Grants permission to retrieve the count of Security Hub membership invitations sent to the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        this.add('securityhub:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve details about the Security Hub master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        this.add('securityhub:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the details of Security Hub member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        this.add('securityhub:GetMembers');
        return this;
    }
    /**
     * Grants permission to invite other AWS accounts to become Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        this.add('securityhub:InviteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve the Security Hub integrated products that are currently enabled
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html
     */
    toListEnabledProductsForImport() {
        this.add('securityhub:ListEnabledProductsForImport');
        return this;
    }
    /**
     * Grants permission to retrieve the Security Hub invitations sent to the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.add('securityhub:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve details about Security Hub member accounts associated with the master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.add('securityhub:ListMembers');
        return this;
    }
    /**
     * Grants permission to list of tags associated with a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('securityhub:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to add tags to a Security Hub resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('securityhub:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a Security Hub resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('securityhub:UntagResource');
        return this;
    }
    /**
     * Grants permission to update custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html
     */
    toUpdateActionTarget() {
        this.add('securityhub:UpdateActionTarget');
        return this;
    }
    /**
     * Grants permission to update Security Hub findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        this.add('securityhub:UpdateFindings');
        return this;
    }
    /**
     * Grants permission to update insights in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html
     */
    toUpdateInsight() {
        this.add('securityhub:UpdateInsight');
        return this;
    }
    /**
     * Grants permission to update Security Hub configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityHubConfiguration.html
     */
    toUpdateSecurityHubConfiguration() {
        this.add('securityhub:UpdateSecurityHubConfiguration');
        return this;
    }
    /**
     * Grants permission to update Security Hub standards controls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html
     */
    toUpdateStandardsControl() {
        this.add('securityhub:UpdateStandardsControl');
        return this;
    }
    /**
     * Adds a resource of type hub to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHub(account, region, partition) {
        var arn = 'arn:${Partition}:securityhub:${Region}:${Account}:hub/default';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type product to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param company - Identifier for the company.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProduct(company, productId, account, region, partition) {
        var arn = 'arn:${Partition}:securityhub:${Region}:${Account}:product/${Company}/${ProductId}';
        arn = arn.replace('${Company}', company);
        arn = arn.replace('${ProductId}', productId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access based on the presence of specific fields and values in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-asffsyntaxpath
     *
     * Applies to actions:
     * - .toBatchUpdateFindings()
     *
     * @param aSFFSyntaxPath The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifASFFSyntaxPath(aSFFSyntaxPath, value, operator) {
        return this.if(`securityhub:ASFFSyntaxPath/${aSFFSyntaxPath}`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on the presence of AwsAccountId field in the requests
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#conditions
     *
     * Applies to actions:
     * - .toBatchImportFindings()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetAccount(value, operator) {
        return this.if(`securityhub:TargetAccount`, value, operator || 'StringLike');
    }
}
exports.Securityhub = Securityhub;
//# sourceMappingURL=data:application/json;base64,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