"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicecatalog = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsservicecatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicecatalog extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsservicecatalog.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicecatalog';
        this.actionList = {
            "AcceptPortfolioShare": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AcceptPortfolioShare.html",
                "description": "Grants permission to accept a portfolio that has been shared with you",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "AssociateBudgetWithResource": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateBudgetWithResource.html",
                "description": "Grants permission to associate a budget with a resource",
                "accessLevel": "Write"
            },
            "AssociatePrincipalWithPortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociatePrincipalWithPortfolio.html",
                "description": "Grants permission to associate an IAM principal with a portfolio, giving the specified principal access to any products associated with the specified portfolio",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "AssociateProductWithPortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateProductWithPortfolio.html",
                "description": "Grants permission to associate a product with a portfolio",
                "accessLevel": "Write"
            },
            "AssociateServiceActionWithProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateServiceActionWithProvisioningArtifact.html",
                "description": "Grants permission to associate an action with a provisioning artifact",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "AssociateTagOptionWithResource": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateTagOptionWithResource.html",
                "description": "Grants permission to associate the specified TagOption with the specified portfolio or product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": false
                    },
                    "Product": {
                        "required": false
                    }
                }
            },
            "BatchAssociateServiceActionWithProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchAssociateServiceActionWithProvisioningArtifact.html",
                "description": "Grants permission to associate multiple self-service actions with provisioning artifacts",
                "accessLevel": "Write"
            },
            "BatchDisassociateServiceActionFromProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchDisassociateServiceActionFromProvisioningArtifact.html",
                "description": "Grants permission to disassociate a batch of self-service actions from the specified provisioning artifact",
                "accessLevel": "Write"
            },
            "CopyProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CopyProduct.html",
                "description": "Grants permission to copy the specified source product to the specified target product or a new product",
                "accessLevel": "Write"
            },
            "CreateConstraint": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateConstraint.html",
                "description": "Grants permission to create a constraint on an associated product and portfolio",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "CreatePortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolio.html",
                "description": "Grants permission to create a portfolio",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePortfolioShare": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolioShare.html",
                "description": "Grants permission to share a portfolio you own with another AWS account",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "CreateProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProduct.html",
                "description": "Grants permission to create a product and that product's first provisioning artifact",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateProvisionedProductPlan": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisionedProductPlan.html",
                "description": "Grants permission to add a new provisioned product plan",
                "accessLevel": "Write"
            },
            "CreateProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisioningArtifact.html",
                "description": "Grants permission to add a new provisioning artifact to an existing product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "CreateServiceAction": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateServiceAction.html",
                "description": "Grants permission to create a self-service action",
                "accessLevel": "Write"
            },
            "CreateTagOption": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateTagOption.html",
                "description": "Grants permission to create a TagOption",
                "accessLevel": "Write"
            },
            "DeleteConstraint": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteConstraint.html",
                "description": "Grants permission to remove and delete an existing constraint from an associated product and portfolio",
                "accessLevel": "Write"
            },
            "DeletePortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolio.html",
                "description": "Grants permission to delete a portfolio if all associations and shares have been removed from the portfolio",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "DeletePortfolioShare": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolioShare.html",
                "description": "Grants permission to unshare a portfolio you own from an AWS account you previously shared the portfolio with",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "DeleteProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProduct.html",
                "description": "Grants permission to delete a product if all associations have been removed from the product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DeleteProvisionedProductPlan": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisionedProductPlan.html",
                "description": "Grants permission to delete a provisioned product plan",
                "accessLevel": "Write"
            },
            "DeleteProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisioningArtifact.html",
                "description": "Grants permission to delete a provisioning artifact from a product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DeleteServiceAction": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteServiceAction.html",
                "description": "Grants permission to delete a self-service action",
                "accessLevel": "Write"
            },
            "DeleteTagOption": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteTagOption.html",
                "description": "Grants permission to delete the specified TagOption",
                "accessLevel": "Write"
            },
            "DescribeConstraint": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeConstraint.html",
                "description": "Grants permission to describe a constraint",
                "accessLevel": "Read"
            },
            "DescribeCopyProductStatus": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeCopyProductStatus.html",
                "description": "Grants permission to get the status of the specified copy product operation",
                "accessLevel": "Read"
            },
            "DescribePortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolio.html",
                "description": "Grants permission to describe a portfolio",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "DescribePortfolioShareStatus": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolioShareStatus.html",
                "description": "Grants permission to get the status of the specified portfolio share operation",
                "accessLevel": "Read"
            },
            "DescribeProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProduct.html",
                "description": "Grants permission to describe a product as an end-user",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DescribeProductAsAdmin": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductAsAdmin.html",
                "description": "Grants permission to describe a product as an admin",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DescribeProductView": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductView.html",
                "description": "Grants permission to describe a product as an end-user",
                "accessLevel": "Read"
            },
            "DescribeProvisionedProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProduct.html",
                "description": "Grants permission to describe a provisioned product",
                "accessLevel": "Read"
            },
            "DescribeProvisionedProductPlan": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProductPlan.html",
                "description": "Grants permission to describe a provisioned product plan",
                "accessLevel": "Read"
            },
            "DescribeProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningArtifact.html",
                "description": "Grants permission to describe a provisioning artifact",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningParameters": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningParameters.html",
                "description": "Grants permission to describe the parameters that you need to specify to successfully provision a specified provisioning artifact",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DescribeRecord": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeRecord.html",
                "description": "Grants permission to describe a record and lists any outputs",
                "accessLevel": "Read",
                "conditions": [
                    "servicecatalog:accountLevel",
                    "servicecatalog:roleLevel",
                    "servicecatalog:userLevel"
                ]
            },
            "DescribeServiceAction": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceAction.html",
                "description": "Grants permission to describe a self-service action",
                "accessLevel": "Read"
            },
            "DescribeServiceActionExecutionParameters": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceActionExecutionParameters.html",
                "description": "Grants permission to get the default parameters if you executed the specified Service Action on the specified Provisioned Product",
                "accessLevel": "Read"
            },
            "DescribeTagOption": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeTagOption.html",
                "description": "Grants permission to get information about the specified TagOption",
                "accessLevel": "Read"
            },
            "DisableAWSOrganizationsAccess": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisableAWSOrganizationsAccess.html",
                "description": "Grants permission to disable portfolio sharing through AWS Organizations feature",
                "accessLevel": "Write"
            },
            "DisassociateBudgetFromResource": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateBudgetFromResource.html",
                "description": "Grants permission to disassociate a budget from a resource",
                "accessLevel": "Write"
            },
            "DisassociatePrincipalFromPortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociatePrincipalFromPortfolio.html",
                "description": "Grants permission to disassociate an IAM principal from a portfolio",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "DisassociateProductFromPortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateProductFromPortfolio.html",
                "description": "Grants permission to disassociate a product from a portfolio",
                "accessLevel": "Write"
            },
            "DisassociateServiceActionFromProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateServiceActionFromProvisioningArtifact.html",
                "description": "Grants permission to disassociate the specified self-service action association from the specified provisioning artifact",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "DisassociateTagOptionFromResource": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateTagOptionFromResource.html",
                "description": "Grants permission to disassociate the specified TagOption from the specified resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": false
                    },
                    "Product": {
                        "required": false
                    }
                }
            },
            "EnableAWSOrganizationsAccess": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_EnableAWSOrganizationsAccess.html",
                "description": "Grants permission to enable portfolio sharing feature through AWS Organizations",
                "accessLevel": "Write"
            },
            "ExecuteProvisionedProductPlan": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductPlan.html",
                "description": "Grants permission to execute a provisioned product plan",
                "accessLevel": "Write"
            },
            "ExecuteProvisionedProductServiceAction": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductServiceAction.html",
                "description": "Grants permission to executes a provisioned product plan",
                "accessLevel": "Write"
            },
            "GetAWSOrganizationsAccessStatus": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetAWSOrganizationsAccessStatus.html",
                "description": "Grants permission to get the access status of AWS Organization portfolio share feature",
                "accessLevel": "Read"
            },
            "ListAcceptedPortfolioShares": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAcceptedPortfolioShares.html",
                "description": "Grants permission to list the portfolios that have been shared with you and you have accepted",
                "accessLevel": "List"
            },
            "ListBudgetsForResource": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListBudgetsForResource.html",
                "description": "Grants permission to list all the budgets associated to a resource",
                "accessLevel": "List"
            },
            "ListConstraintsForPortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListConstraintsForPortfolio.html",
                "description": "Grants permission to list constraints associated with a given portfolio",
                "accessLevel": "List"
            },
            "ListLaunchPaths": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListLaunchPaths.html",
                "description": "Grants permission to list the different ways to launch a given product as an end-user",
                "accessLevel": "List",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "ListOrganizationPortfolioAccess": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListOrganizationPortfolioAccess.html",
                "description": "Grants permission to list the organization nodes that have access to the specified portfolio",
                "accessLevel": "List"
            },
            "ListPortfolioAccess": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolioAccess.html",
                "description": "Grants permission to list the AWS accounts you have shared a given portfolio with",
                "accessLevel": "List",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "ListPortfolios": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolios.html",
                "description": "Grants permission to list the portfolios in your account",
                "accessLevel": "List"
            },
            "ListPortfoliosForProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfoliosForProduct.html",
                "description": "Grants permission to list the portfolios associated with a given product",
                "accessLevel": "List",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "ListPrincipalsForPortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPrincipalsForPortfolio.html",
                "description": "Grants permission to list the IAM principals associated with a given portfolio",
                "accessLevel": "List",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "ListProvisionedProductPlans": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisionedProductPlans.html",
                "description": "Grants permission to list the provisioned product plans",
                "accessLevel": "List"
            },
            "ListProvisioningArtifacts": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifacts.html",
                "description": "Grants permission to list the provisioning artifacts associated with a given product",
                "accessLevel": "List",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "ListProvisioningArtifactsForServiceAction": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifactsForServiceAction.html",
                "description": "Grants permission to list all provisioning artifacts for the specified self-service action",
                "accessLevel": "List"
            },
            "ListRecordHistory": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListRecordHistory.html",
                "description": "Grants permission to list all the records in your account or all the records related to a given provisioned product",
                "accessLevel": "List",
                "conditions": [
                    "servicecatalog:accountLevel",
                    "servicecatalog:roleLevel",
                    "servicecatalog:userLevel"
                ]
            },
            "ListResourcesForTagOption": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListResourcesForTagOption.html",
                "description": "Grants permission to list the resources associated with the specified TagOption",
                "accessLevel": "List"
            },
            "ListServiceActions": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActions.html",
                "description": "Grants permission to list all self-service actions",
                "accessLevel": "List"
            },
            "ListServiceActionsForProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActionsForProvisioningArtifact.html",
                "description": "Grants permission to list all the service actions associated with the specified provisioning artifact in your account",
                "accessLevel": "List",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "ListStackInstancesForProvisionedProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListStackInstancesForProvisionedProduct.html",
                "description": "Grants permission to list account, region and status of each stack instances that are associated with a CFN_STACKSET type provisioned product",
                "accessLevel": "List"
            },
            "ListTagOptions": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListTagOptions.html",
                "description": "Grants permission to list the specified TagOptions or all TagOptions",
                "accessLevel": "List"
            },
            "ProvisionProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProvisionProduct.html",
                "description": "Grants permission to provision a product with a specified provisioning artifact and launch parameters",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "RejectPortfolioShare": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_RejectPortfolioShare.html",
                "description": "Grants permission to reject a portfolio that has been shared with you that you previously accepted",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                }
            },
            "ScanProvisionedProducts": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ScanProvisionedProducts.html",
                "description": "Grants permission to list all the provisioned products in your account",
                "accessLevel": "List",
                "conditions": [
                    "servicecatalog:accountLevel",
                    "servicecatalog:roleLevel",
                    "servicecatalog:userLevel"
                ]
            },
            "SearchProducts": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProducts.html",
                "description": "Grants permission to list the products available to you as an end-user",
                "accessLevel": "List"
            },
            "SearchProductsAsAdmin": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProductsAsAdmin.html",
                "description": "Grants permission to list all the products in your account or all the products associated with a given portfolio",
                "accessLevel": "List"
            },
            "SearchProvisionedProducts": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProvisionedProducts.html",
                "description": "Grants permission to list all the provisioned products in your account",
                "accessLevel": "List",
                "conditions": [
                    "servicecatalog:accountLevel",
                    "servicecatalog:roleLevel",
                    "servicecatalog:userLevel"
                ]
            },
            "TerminateProvisionedProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_TerminateProvisionedProduct.html",
                "description": "Grants permission to terminate an existing provisioned product",
                "accessLevel": "Write",
                "conditions": [
                    "servicecatalog:accountLevel",
                    "servicecatalog:roleLevel",
                    "servicecatalog:userLevel"
                ]
            },
            "UpdateConstraint": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateConstraint.html",
                "description": "Grants permission to update the metadata fields of an existing constraint",
                "accessLevel": "Write"
            },
            "UpdatePortfolio": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdatePortfolio.html",
                "description": "Grants permission to update the metadata fields and/or tags of an existing portfolio",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Portfolio": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProduct.html",
                "description": "Grants permission to update the metadata fields and/or tags of an existing product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateProvisionedProduct": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProduct.html",
                "description": "Grants permission to update an existing provisioned product",
                "accessLevel": "Write",
                "conditions": [
                    "servicecatalog:accountLevel",
                    "servicecatalog:roleLevel",
                    "servicecatalog:userLevel"
                ]
            },
            "UpdateProvisionedProductProperties": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProductProperties.html",
                "description": "Grants permission to update the properties of an existing provisioned product",
                "accessLevel": "Write"
            },
            "UpdateProvisioningArtifact": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisioningArtifact.html",
                "description": "Grants permission to update the metadata fields of an existing provisioning artifact",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Product": {
                        "required": true
                    }
                }
            },
            "UpdateServiceAction": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateServiceAction.html",
                "description": "Grants permission to update a self-service action",
                "accessLevel": "Write"
            },
            "UpdateTagOption": {
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateTagOption.html",
                "description": "Grants permission to update the specified TagOption",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Portfolio": {
                "name": "Portfolio",
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_PortfolioDetail.html",
                "arn": "arn:${Partition}:catalog:${Region}:${Account}:portfolio/${PortfolioId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Product": {
                "name": "Product",
                "url": "https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProductViewDetail.html",
                "arn": "arn:${Partition}:catalog:${Region}:${Account}:product/${ProductId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept a portfolio that has been shared with you
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AcceptPortfolioShare.html
     */
    toAcceptPortfolioShare() {
        this.add('servicecatalog:AcceptPortfolioShare');
        return this;
    }
    /**
     * Grants permission to associate a budget with a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateBudgetWithResource.html
     */
    toAssociateBudgetWithResource() {
        this.add('servicecatalog:AssociateBudgetWithResource');
        return this;
    }
    /**
     * Grants permission to associate an IAM principal with a portfolio, giving the specified principal access to any products associated with the specified portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociatePrincipalWithPortfolio.html
     */
    toAssociatePrincipalWithPortfolio() {
        this.add('servicecatalog:AssociatePrincipalWithPortfolio');
        return this;
    }
    /**
     * Grants permission to associate a product with a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateProductWithPortfolio.html
     */
    toAssociateProductWithPortfolio() {
        this.add('servicecatalog:AssociateProductWithPortfolio');
        return this;
    }
    /**
     * Grants permission to associate an action with a provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateServiceActionWithProvisioningArtifact.html
     */
    toAssociateServiceActionWithProvisioningArtifact() {
        this.add('servicecatalog:AssociateServiceActionWithProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to associate the specified TagOption with the specified portfolio or product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateTagOptionWithResource.html
     */
    toAssociateTagOptionWithResource() {
        this.add('servicecatalog:AssociateTagOptionWithResource');
        return this;
    }
    /**
     * Grants permission to associate multiple self-service actions with provisioning artifacts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchAssociateServiceActionWithProvisioningArtifact.html
     */
    toBatchAssociateServiceActionWithProvisioningArtifact() {
        this.add('servicecatalog:BatchAssociateServiceActionWithProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to disassociate a batch of self-service actions from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchDisassociateServiceActionFromProvisioningArtifact.html
     */
    toBatchDisassociateServiceActionFromProvisioningArtifact() {
        this.add('servicecatalog:BatchDisassociateServiceActionFromProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to copy the specified source product to the specified target product or a new product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CopyProduct.html
     */
    toCopyProduct() {
        this.add('servicecatalog:CopyProduct');
        return this;
    }
    /**
     * Grants permission to create a constraint on an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateConstraint.html
     */
    toCreateConstraint() {
        this.add('servicecatalog:CreateConstraint');
        return this;
    }
    /**
     * Grants permission to create a portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolio.html
     */
    toCreatePortfolio() {
        this.add('servicecatalog:CreatePortfolio');
        return this;
    }
    /**
     * Grants permission to share a portfolio you own with another AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolioShare.html
     */
    toCreatePortfolioShare() {
        this.add('servicecatalog:CreatePortfolioShare');
        return this;
    }
    /**
     * Grants permission to create a product and that product's first provisioning artifact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProduct.html
     */
    toCreateProduct() {
        this.add('servicecatalog:CreateProduct');
        return this;
    }
    /**
     * Grants permission to add a new provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisionedProductPlan.html
     */
    toCreateProvisionedProductPlan() {
        this.add('servicecatalog:CreateProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to add a new provisioning artifact to an existing product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisioningArtifact.html
     */
    toCreateProvisioningArtifact() {
        this.add('servicecatalog:CreateProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to create a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateServiceAction.html
     */
    toCreateServiceAction() {
        this.add('servicecatalog:CreateServiceAction');
        return this;
    }
    /**
     * Grants permission to create a TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateTagOption.html
     */
    toCreateTagOption() {
        this.add('servicecatalog:CreateTagOption');
        return this;
    }
    /**
     * Grants permission to remove and delete an existing constraint from an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteConstraint.html
     */
    toDeleteConstraint() {
        this.add('servicecatalog:DeleteConstraint');
        return this;
    }
    /**
     * Grants permission to delete a portfolio if all associations and shares have been removed from the portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolio.html
     */
    toDeletePortfolio() {
        this.add('servicecatalog:DeletePortfolio');
        return this;
    }
    /**
     * Grants permission to unshare a portfolio you own from an AWS account you previously shared the portfolio with
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolioShare.html
     */
    toDeletePortfolioShare() {
        this.add('servicecatalog:DeletePortfolioShare');
        return this;
    }
    /**
     * Grants permission to delete a product if all associations have been removed from the product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProduct.html
     */
    toDeleteProduct() {
        this.add('servicecatalog:DeleteProduct');
        return this;
    }
    /**
     * Grants permission to delete a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisionedProductPlan.html
     */
    toDeleteProvisionedProductPlan() {
        this.add('servicecatalog:DeleteProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to delete a provisioning artifact from a product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisioningArtifact.html
     */
    toDeleteProvisioningArtifact() {
        this.add('servicecatalog:DeleteProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to delete a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteServiceAction.html
     */
    toDeleteServiceAction() {
        this.add('servicecatalog:DeleteServiceAction');
        return this;
    }
    /**
     * Grants permission to delete the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteTagOption.html
     */
    toDeleteTagOption() {
        this.add('servicecatalog:DeleteTagOption');
        return this;
    }
    /**
     * Grants permission to describe a constraint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeConstraint.html
     */
    toDescribeConstraint() {
        this.add('servicecatalog:DescribeConstraint');
        return this;
    }
    /**
     * Grants permission to get the status of the specified copy product operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeCopyProductStatus.html
     */
    toDescribeCopyProductStatus() {
        this.add('servicecatalog:DescribeCopyProductStatus');
        return this;
    }
    /**
     * Grants permission to describe a portfolio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolio.html
     */
    toDescribePortfolio() {
        this.add('servicecatalog:DescribePortfolio');
        return this;
    }
    /**
     * Grants permission to get the status of the specified portfolio share operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolioShareStatus.html
     */
    toDescribePortfolioShareStatus() {
        this.add('servicecatalog:DescribePortfolioShareStatus');
        return this;
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProduct.html
     */
    toDescribeProduct() {
        this.add('servicecatalog:DescribeProduct');
        return this;
    }
    /**
     * Grants permission to describe a product as an admin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductAsAdmin.html
     */
    toDescribeProductAsAdmin() {
        this.add('servicecatalog:DescribeProductAsAdmin');
        return this;
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductView.html
     */
    toDescribeProductView() {
        this.add('servicecatalog:DescribeProductView');
        return this;
    }
    /**
     * Grants permission to describe a provisioned product
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProduct.html
     */
    toDescribeProvisionedProduct() {
        this.add('servicecatalog:DescribeProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to describe a provisioned product plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProductPlan.html
     */
    toDescribeProvisionedProductPlan() {
        this.add('servicecatalog:DescribeProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to describe a provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningArtifact.html
     */
    toDescribeProvisioningArtifact() {
        this.add('servicecatalog:DescribeProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to describe the parameters that you need to specify to successfully provision a specified provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningParameters.html
     */
    toDescribeProvisioningParameters() {
        this.add('servicecatalog:DescribeProvisioningParameters');
        return this;
    }
    /**
     * Grants permission to describe a record and lists any outputs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeRecord.html
     */
    toDescribeRecord() {
        this.add('servicecatalog:DescribeRecord');
        return this;
    }
    /**
     * Grants permission to describe a self-service action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceAction.html
     */
    toDescribeServiceAction() {
        this.add('servicecatalog:DescribeServiceAction');
        return this;
    }
    /**
     * Grants permission to get the default parameters if you executed the specified Service Action on the specified Provisioned Product
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceActionExecutionParameters.html
     */
    toDescribeServiceActionExecutionParameters() {
        this.add('servicecatalog:DescribeServiceActionExecutionParameters');
        return this;
    }
    /**
     * Grants permission to get information about the specified TagOption
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeTagOption.html
     */
    toDescribeTagOption() {
        this.add('servicecatalog:DescribeTagOption');
        return this;
    }
    /**
     * Grants permission to disable portfolio sharing through AWS Organizations feature
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisableAWSOrganizationsAccess.html
     */
    toDisableAWSOrganizationsAccess() {
        this.add('servicecatalog:DisableAWSOrganizationsAccess');
        return this;
    }
    /**
     * Grants permission to disassociate a budget from a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateBudgetFromResource.html
     */
    toDisassociateBudgetFromResource() {
        this.add('servicecatalog:DisassociateBudgetFromResource');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM principal from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociatePrincipalFromPortfolio.html
     */
    toDisassociatePrincipalFromPortfolio() {
        this.add('servicecatalog:DisassociatePrincipalFromPortfolio');
        return this;
    }
    /**
     * Grants permission to disassociate a product from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateProductFromPortfolio.html
     */
    toDisassociateProductFromPortfolio() {
        this.add('servicecatalog:DisassociateProductFromPortfolio');
        return this;
    }
    /**
     * Grants permission to disassociate the specified self-service action association from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateServiceActionFromProvisioningArtifact.html
     */
    toDisassociateServiceActionFromProvisioningArtifact() {
        this.add('servicecatalog:DisassociateServiceActionFromProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to disassociate the specified TagOption from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateTagOptionFromResource.html
     */
    toDisassociateTagOptionFromResource() {
        this.add('servicecatalog:DisassociateTagOptionFromResource');
        return this;
    }
    /**
     * Grants permission to enable portfolio sharing feature through AWS Organizations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_EnableAWSOrganizationsAccess.html
     */
    toEnableAWSOrganizationsAccess() {
        this.add('servicecatalog:EnableAWSOrganizationsAccess');
        return this;
    }
    /**
     * Grants permission to execute a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductPlan.html
     */
    toExecuteProvisionedProductPlan() {
        this.add('servicecatalog:ExecuteProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to executes a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductServiceAction.html
     */
    toExecuteProvisionedProductServiceAction() {
        this.add('servicecatalog:ExecuteProvisionedProductServiceAction');
        return this;
    }
    /**
     * Grants permission to get the access status of AWS Organization portfolio share feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetAWSOrganizationsAccessStatus.html
     */
    toGetAWSOrganizationsAccessStatus() {
        this.add('servicecatalog:GetAWSOrganizationsAccessStatus');
        return this;
    }
    /**
     * Grants permission to list the portfolios that have been shared with you and you have accepted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAcceptedPortfolioShares.html
     */
    toListAcceptedPortfolioShares() {
        this.add('servicecatalog:ListAcceptedPortfolioShares');
        return this;
    }
    /**
     * Grants permission to list all the budgets associated to a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListBudgetsForResource.html
     */
    toListBudgetsForResource() {
        this.add('servicecatalog:ListBudgetsForResource');
        return this;
    }
    /**
     * Grants permission to list constraints associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListConstraintsForPortfolio.html
     */
    toListConstraintsForPortfolio() {
        this.add('servicecatalog:ListConstraintsForPortfolio');
        return this;
    }
    /**
     * Grants permission to list the different ways to launch a given product as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListLaunchPaths.html
     */
    toListLaunchPaths() {
        this.add('servicecatalog:ListLaunchPaths');
        return this;
    }
    /**
     * Grants permission to list the organization nodes that have access to the specified portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListOrganizationPortfolioAccess.html
     */
    toListOrganizationPortfolioAccess() {
        this.add('servicecatalog:ListOrganizationPortfolioAccess');
        return this;
    }
    /**
     * Grants permission to list the AWS accounts you have shared a given portfolio with
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolioAccess.html
     */
    toListPortfolioAccess() {
        this.add('servicecatalog:ListPortfolioAccess');
        return this;
    }
    /**
     * Grants permission to list the portfolios in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolios.html
     */
    toListPortfolios() {
        this.add('servicecatalog:ListPortfolios');
        return this;
    }
    /**
     * Grants permission to list the portfolios associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfoliosForProduct.html
     */
    toListPortfoliosForProduct() {
        this.add('servicecatalog:ListPortfoliosForProduct');
        return this;
    }
    /**
     * Grants permission to list the IAM principals associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPrincipalsForPortfolio.html
     */
    toListPrincipalsForPortfolio() {
        this.add('servicecatalog:ListPrincipalsForPortfolio');
        return this;
    }
    /**
     * Grants permission to list the provisioned product plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisionedProductPlans.html
     */
    toListProvisionedProductPlans() {
        this.add('servicecatalog:ListProvisionedProductPlans');
        return this;
    }
    /**
     * Grants permission to list the provisioning artifacts associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifacts.html
     */
    toListProvisioningArtifacts() {
        this.add('servicecatalog:ListProvisioningArtifacts');
        return this;
    }
    /**
     * Grants permission to list all provisioning artifacts for the specified self-service action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifactsForServiceAction.html
     */
    toListProvisioningArtifactsForServiceAction() {
        this.add('servicecatalog:ListProvisioningArtifactsForServiceAction');
        return this;
    }
    /**
     * Grants permission to list all the records in your account or all the records related to a given provisioned product
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListRecordHistory.html
     */
    toListRecordHistory() {
        this.add('servicecatalog:ListRecordHistory');
        return this;
    }
    /**
     * Grants permission to list the resources associated with the specified TagOption
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListResourcesForTagOption.html
     */
    toListResourcesForTagOption() {
        this.add('servicecatalog:ListResourcesForTagOption');
        return this;
    }
    /**
     * Grants permission to list all self-service actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActions.html
     */
    toListServiceActions() {
        this.add('servicecatalog:ListServiceActions');
        return this;
    }
    /**
     * Grants permission to list all the service actions associated with the specified provisioning artifact in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActionsForProvisioningArtifact.html
     */
    toListServiceActionsForProvisioningArtifact() {
        this.add('servicecatalog:ListServiceActionsForProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to list account, region and status of each stack instances that are associated with a CFN_STACKSET type provisioned product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListStackInstancesForProvisionedProduct.html
     */
    toListStackInstancesForProvisionedProduct() {
        this.add('servicecatalog:ListStackInstancesForProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to list the specified TagOptions or all TagOptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListTagOptions.html
     */
    toListTagOptions() {
        this.add('servicecatalog:ListTagOptions');
        return this;
    }
    /**
     * Grants permission to provision a product with a specified provisioning artifact and launch parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProvisionProduct.html
     */
    toProvisionProduct() {
        this.add('servicecatalog:ProvisionProduct');
        return this;
    }
    /**
     * Grants permission to reject a portfolio that has been shared with you that you previously accepted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_RejectPortfolioShare.html
     */
    toRejectPortfolioShare() {
        this.add('servicecatalog:RejectPortfolioShare');
        return this;
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ScanProvisionedProducts.html
     */
    toScanProvisionedProducts() {
        this.add('servicecatalog:ScanProvisionedProducts');
        return this;
    }
    /**
     * Grants permission to list the products available to you as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProducts.html
     */
    toSearchProducts() {
        this.add('servicecatalog:SearchProducts');
        return this;
    }
    /**
     * Grants permission to list all the products in your account or all the products associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProductsAsAdmin.html
     */
    toSearchProductsAsAdmin() {
        this.add('servicecatalog:SearchProductsAsAdmin');
        return this;
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProvisionedProducts.html
     */
    toSearchProvisionedProducts() {
        this.add('servicecatalog:SearchProvisionedProducts');
        return this;
    }
    /**
     * Grants permission to terminate an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_TerminateProvisionedProduct.html
     */
    toTerminateProvisionedProduct() {
        this.add('servicecatalog:TerminateProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to update the metadata fields of an existing constraint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateConstraint.html
     */
    toUpdateConstraint() {
        this.add('servicecatalog:UpdateConstraint');
        return this;
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdatePortfolio.html
     */
    toUpdatePortfolio() {
        this.add('servicecatalog:UpdatePortfolio');
        return this;
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProduct.html
     */
    toUpdateProduct() {
        this.add('servicecatalog:UpdateProduct');
        return this;
    }
    /**
     * Grants permission to update an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProduct.html
     */
    toUpdateProvisionedProduct() {
        this.add('servicecatalog:UpdateProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to update the properties of an existing provisioned product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProductProperties.html
     */
    toUpdateProvisionedProductProperties() {
        this.add('servicecatalog:UpdateProvisionedProductProperties');
        return this;
    }
    /**
     * Grants permission to update the metadata fields of an existing provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisioningArtifact.html
     */
    toUpdateProvisioningArtifact() {
        this.add('servicecatalog:UpdateProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to update a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateServiceAction.html
     */
    toUpdateServiceAction() {
        this.add('servicecatalog:UpdateServiceAction');
        return this;
    }
    /**
     * Grants permission to update the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateTagOption.html
     */
    toUpdateTagOption() {
        this.add('servicecatalog:UpdateTagOption');
        return this;
    }
    /**
     * Adds a resource of type Portfolio to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_PortfolioDetail.html
     *
     * @param portfolioId - Identifier for the portfolioId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortfolio(portfolioId, account, region, partition) {
        var arn = 'arn:${Partition}:catalog:${Region}:${Account}:portfolio/${PortfolioId}';
        arn = arn.replace('${PortfolioId}', portfolioId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Product to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProductViewDetail.html
     *
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProduct(productId, account, region, partition) {
        var arn = 'arn:${Partition}:catalog:${Region}:${Account}:product/${ProductId}';
        arn = arn.replace('${ProductId}', productId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters users to see and perform actions on resources created by anyone in the account
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountLevel(value, operator) {
        return this.if(`servicecatalog:accountLevel`, value, operator || 'StringLike');
    }
    /**
     * Filters users to see and perform actions on resources created either by them or by anyone federating into the same role as them
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoleLevel(value, operator) {
        return this.if(`servicecatalog:roleLevel`, value, operator || 'StringLike');
    }
    /**
     * Filters users to see and perform actions on only resources that they created
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserLevel(value, operator) {
        return this.if(`servicecatalog:userLevel`, value, operator || 'StringLike');
    }
}
exports.Servicecatalog = Servicecatalog;
//# sourceMappingURL=data:application/json;base64,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