"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sso = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sso](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssso.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sso extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sso](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssso.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sso';
        this.actionList = {
            "AssociateDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Connect a directory to be used by AWS Single Sign-On",
                "accessLevel": "Write"
            },
            "AssociateProfile": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Create an association between a directory user or group and a profile",
                "accessLevel": "Write"
            },
            "AttachManagedPolicyToPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Attaches an AWS managed policy to a permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "CreateAccountAssignment": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Assigns access to a Principal for a specified AWS account using a specified permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Account": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "CreateApplicationInstance": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Add an application instance to AWS Single Sign-On",
                "accessLevel": "Write"
            },
            "CreateApplicationInstanceCertificate": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Add a new certificate for an application instance",
                "accessLevel": "Write"
            },
            "CreateManagedApplicationInstance": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Add a managed application instance to AWS Single Sign-On",
                "accessLevel": "Write"
            },
            "CreatePermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Create a permission set",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "CreateProfile": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Create a profile for an application instance",
                "accessLevel": "Write"
            },
            "CreateTrust": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Create a federation trust in a target account",
                "accessLevel": "Write"
            },
            "DeleteAccountAssignment": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes a Principal's access from a specified AWS account using a specified permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Account": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationInstance": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete the application instance",
                "accessLevel": "Write"
            },
            "DeleteApplicationInstanceCertificate": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete an inactive or expired certificate from the application instance",
                "accessLevel": "Write"
            },
            "DeleteInlinePolicyFromPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes the inline policy from a specified permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "DeleteManagedApplicationInstance": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete the managed application instance",
                "accessLevel": "Write"
            },
            "DeletePermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete a permission set",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "DeletePermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete the permission policy associated with a permission set",
                "accessLevel": "Write"
            },
            "DeleteProfile": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete the profile for an application instance",
                "accessLevel": "Write"
            },
            "DescribeAccountAssignmentCreationStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Describes the status of the assignment creation request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAssignmentDeletionStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Describes the status of an assignment deletion request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DescribePermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Describes a permission set",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "DescribePermissionSetProvisioningStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Describes the status for the given Permission Set Provisioning request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DescribePermissionsPolicies": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all the permissions policies associated with a permission set",
                "accessLevel": "Read"
            },
            "DescribeRegisteredRegions": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Obtains the regions where your organization has enabled AWS Single Sign-on",
                "accessLevel": "Read"
            },
            "DetachManagedPolicyFromPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Detaches the attached AWS managed policy from the specified permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "DisassociateDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Disassociate a directory to be used by AWS Single Sign-On",
                "accessLevel": "Write"
            },
            "DisassociateProfile": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Disassociate a directory user or group from a profile",
                "accessLevel": "Write"
            },
            "GetApplicationInstance": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve details for an application instance",
                "accessLevel": "Read"
            },
            "GetApplicationTemplate": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve application template details",
                "accessLevel": "Read"
            },
            "GetInlinePolicyForPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Obtains the inline policy assigned to the permission set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "GetManagedApplicationInstance": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve details for an application instance",
                "accessLevel": "Read"
            },
            "GetMfaDeviceManagementForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve Mfa Device Management settings for the directory",
                "accessLevel": "Read"
            },
            "GetPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve details of a permission set",
                "accessLevel": "Read"
            },
            "GetPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all permission policies associated with a permission set",
                "accessLevel": "Read",
                "dependentActions": [
                    "sso:DescribePermissionsPolicies"
                ]
            },
            "GetProfile": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve a profile for an application instance",
                "accessLevel": "Read"
            },
            "GetSSOStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Check if AWS Single Sign-On is enabled",
                "accessLevel": "Read"
            },
            "GetSharedSsoConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve shared configuration for the current SSO instance",
                "accessLevel": "Read"
            },
            "GetSsoConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve configuration for the current SSO instance",
                "accessLevel": "Read"
            },
            "GetTrust": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve the federation trust in a target account",
                "accessLevel": "Read"
            },
            "ImportApplicationInstanceServiceProviderMetadata": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the application instance by uploading an application SAML metadata file provided by the service provider",
                "accessLevel": "Write"
            },
            "ListAccountAssignmentCreationStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the status of the AWS account assignment creation requests for a specified SSO instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "ListAccountAssignmentDeletionStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the status of the AWS account assignment deletion requests for a specified SSO instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "ListAccountAssignments": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the assignee of the specified AWS account with the specified permission set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Account": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "ListAccountsForProvisionedPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists all the AWS accounts where the specified permission set is provisioned.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "ListApplicationInstanceCertificates": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all of the certificates for a given application instance",
                "accessLevel": "Read"
            },
            "ListApplicationInstances": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all application instances",
                "accessLevel": "List",
                "dependentActions": [
                    "sso:GetApplicationInstance"
                ]
            },
            "ListApplicationTemplates": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all supported application templates",
                "accessLevel": "Read",
                "dependentActions": [
                    "sso:GetApplicationTemplate"
                ]
            },
            "ListApplications": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all supported applications",
                "accessLevel": "Read"
            },
            "ListDirectoryAssociations": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve details about the directory connected to AWS Single Sign-On",
                "accessLevel": "Read"
            },
            "ListInstances": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the SSO Instances that the caller has access to.",
                "accessLevel": "Read"
            },
            "ListManagedPoliciesInPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the AWS managed policies that are attached to a specified permission set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "ListPermissionSetProvisioningStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the status of the Permission Set Provisioning requests for a specified SSO instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "ListPermissionSets": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all permission sets",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "ListPermissionSetsProvisionedToAccount": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists all the permission sets that are provisioned to a specified AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Account": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    }
                }
            },
            "ListProfileAssociations": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve the directory user or group associated with the profile",
                "accessLevel": "Read"
            },
            "ListProfiles": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve all profiles for an application instance",
                "accessLevel": "Read",
                "dependentActions": [
                    "sso:GetProfile"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists the tags that are attached to a specified resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "ProvisionPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "The process by which a specified permission set is provisioned to the specified target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Account": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "PutInlinePolicyToPermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Attaches an IAM inline policy to a permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "PutMfaDeviceManagementForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Put Mfa Device Management settings for the directory",
                "accessLevel": "Write"
            },
            "PutPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Add a policy to a permission set",
                "accessLevel": "Write"
            },
            "StartSSO": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Initialize AWS Single Sign-On",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Associates a set of tags with a specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Disassociates a set of tags from a specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateApplicationInstanceActiveCertificate": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Set a certificate as the active one for this application instance",
                "accessLevel": "Write"
            },
            "UpdateApplicationInstanceDisplayData": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update display data of an application instance",
                "accessLevel": "Write"
            },
            "UpdateApplicationInstanceResponseConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update federation response configuration for the application instance",
                "accessLevel": "Write"
            },
            "UpdateApplicationInstanceResponseSchemaConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update federation response schema configuration for the application instance",
                "accessLevel": "Write"
            },
            "UpdateApplicationInstanceSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update security details for the application instance",
                "accessLevel": "Write"
            },
            "UpdateApplicationInstanceServiceProviderConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update service provider related configuration for the application instance",
                "accessLevel": "Write"
            },
            "UpdateApplicationInstanceStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the status of an application instance",
                "accessLevel": "Write"
            },
            "UpdateDirectoryAssociation": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the user attribute mappings for your connected directory",
                "accessLevel": "Write"
            },
            "UpdateManagedApplicationInstanceStatus": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the status of a managed application instance",
                "accessLevel": "Write"
            },
            "UpdatePermissionSet": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the permission set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "PermissionSet": {
                        "required": true
                    }
                }
            },
            "UpdateProfile": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the profile for an application instance",
                "accessLevel": "Write"
            },
            "UpdateSSOConfiguration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the configuration for the current SSO instance",
                "accessLevel": "Write"
            },
            "UpdateTrust": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update the federation trust in a target account",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "PermissionSet": {
                "name": "PermissionSet",
                "url": "",
                "arn": "arn:${Partition}:sso:::permissionSet/${InstanceId}/${PermissionSetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Account": {
                "name": "Account",
                "url": "",
                "arn": "arn:${Partition}:sso:::account/${AccountId}",
                "conditionKeys": []
            },
            "Instance": {
                "name": "Instance",
                "url": "",
                "arn": "arn:${Partition}:sso:::instance/${InstanceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Connect a directory to be used by AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateDirectory() {
        this.add('sso:AssociateDirectory');
        return this;
    }
    /**
     * Create an association between a directory user or group and a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateProfile() {
        this.add('sso:AssociateProfile');
        return this;
    }
    /**
     * Attaches an AWS managed policy to a permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAttachManagedPolicyToPermissionSet() {
        this.add('sso:AttachManagedPolicyToPermissionSet');
        return this;
    }
    /**
     * Assigns access to a Principal for a specified AWS account using a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateAccountAssignment() {
        this.add('sso:CreateAccountAssignment');
        return this;
    }
    /**
     * Add an application instance to AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstance() {
        this.add('sso:CreateApplicationInstance');
        return this;
    }
    /**
     * Add a new certificate for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstanceCertificate() {
        this.add('sso:CreateApplicationInstanceCertificate');
        return this;
    }
    /**
     * Add a managed application instance to AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateManagedApplicationInstance() {
        this.add('sso:CreateManagedApplicationInstance');
        return this;
    }
    /**
     * Create a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreatePermissionSet() {
        this.add('sso:CreatePermissionSet');
        return this;
    }
    /**
     * Create a profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProfile() {
        this.add('sso:CreateProfile');
        return this;
    }
    /**
     * Create a federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateTrust() {
        this.add('sso:CreateTrust');
        return this;
    }
    /**
     * Deletes a Principal's access from a specified AWS account using a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteAccountAssignment() {
        this.add('sso:DeleteAccountAssignment');
        return this;
    }
    /**
     * Delete the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstance() {
        this.add('sso:DeleteApplicationInstance');
        return this;
    }
    /**
     * Delete an inactive or expired certificate from the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstanceCertificate() {
        this.add('sso:DeleteApplicationInstanceCertificate');
        return this;
    }
    /**
     * Deletes the inline policy from a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteInlinePolicyFromPermissionSet() {
        this.add('sso:DeleteInlinePolicyFromPermissionSet');
        return this;
    }
    /**
     * Delete the managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteManagedApplicationInstance() {
        this.add('sso:DeleteManagedApplicationInstance');
        return this;
    }
    /**
     * Delete a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionSet() {
        this.add('sso:DeletePermissionSet');
        return this;
    }
    /**
     * Delete the permission policy associated with a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionsPolicy() {
        this.add('sso:DeletePermissionsPolicy');
        return this;
    }
    /**
     * Delete the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProfile() {
        this.add('sso:DeleteProfile');
        return this;
    }
    /**
     * Describes the status of the assignment creation request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeAccountAssignmentCreationStatus() {
        this.add('sso:DescribeAccountAssignmentCreationStatus');
        return this;
    }
    /**
     * Describes the status of an assignment deletion request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeAccountAssignmentDeletionStatus() {
        this.add('sso:DescribeAccountAssignmentDeletionStatus');
        return this;
    }
    /**
     * Describes a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionSet() {
        this.add('sso:DescribePermissionSet');
        return this;
    }
    /**
     * Describes the status for the given Permission Set Provisioning request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionSetProvisioningStatus() {
        this.add('sso:DescribePermissionSetProvisioningStatus');
        return this;
    }
    /**
     * Retrieve all the permissions policies associated with a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionsPolicies() {
        this.add('sso:DescribePermissionsPolicies');
        return this;
    }
    /**
     * Obtains the regions where your organization has enabled AWS Single Sign-on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeRegisteredRegions() {
        this.add('sso:DescribeRegisteredRegions');
        return this;
    }
    /**
     * Detaches the attached AWS managed policy from the specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDetachManagedPolicyFromPermissionSet() {
        this.add('sso:DetachManagedPolicyFromPermissionSet');
        return this;
    }
    /**
     * Disassociate a directory to be used by AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateDirectory() {
        this.add('sso:DisassociateDirectory');
        return this;
    }
    /**
     * Disassociate a directory user or group from a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateProfile() {
        this.add('sso:DisassociateProfile');
        return this;
    }
    /**
     * Retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationInstance() {
        this.add('sso:GetApplicationInstance');
        return this;
    }
    /**
     * Retrieve application template details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationTemplate() {
        this.add('sso:GetApplicationTemplate');
        return this;
    }
    /**
     * Obtains the inline policy assigned to the permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetInlinePolicyForPermissionSet() {
        this.add('sso:GetInlinePolicyForPermissionSet');
        return this;
    }
    /**
     * Retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetManagedApplicationInstance() {
        this.add('sso:GetManagedApplicationInstance');
        return this;
    }
    /**
     * Retrieve Mfa Device Management settings for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetMfaDeviceManagementForDirectory() {
        this.add('sso:GetMfaDeviceManagementForDirectory');
        return this;
    }
    /**
     * Retrieve details of a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionSet() {
        this.add('sso:GetPermissionSet');
        return this;
    }
    /**
     * Retrieve all permission policies associated with a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:DescribePermissionsPolicies
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionsPolicy() {
        this.add('sso:GetPermissionsPolicy');
        return this;
    }
    /**
     * Retrieve a profile for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetProfile() {
        this.add('sso:GetProfile');
        return this;
    }
    /**
     * Check if AWS Single Sign-On is enabled
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSSOStatus() {
        this.add('sso:GetSSOStatus');
        return this;
    }
    /**
     * Retrieve shared configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSharedSsoConfiguration() {
        this.add('sso:GetSharedSsoConfiguration');
        return this;
    }
    /**
     * Retrieve configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSsoConfiguration() {
        this.add('sso:GetSsoConfiguration');
        return this;
    }
    /**
     * Retrieve the federation trust in a target account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetTrust() {
        this.add('sso:GetTrust');
        return this;
    }
    /**
     * Update the application instance by uploading an application SAML metadata file provided by the service provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toImportApplicationInstanceServiceProviderMetadata() {
        this.add('sso:ImportApplicationInstanceServiceProviderMetadata');
        return this;
    }
    /**
     * Lists the status of the AWS account assignment creation requests for a specified SSO instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignmentCreationStatus() {
        this.add('sso:ListAccountAssignmentCreationStatus');
        return this;
    }
    /**
     * Lists the status of the AWS account assignment deletion requests for a specified SSO instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignmentDeletionStatus() {
        this.add('sso:ListAccountAssignmentDeletionStatus');
        return this;
    }
    /**
     * Lists the assignee of the specified AWS account with the specified permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignments() {
        this.add('sso:ListAccountAssignments');
        return this;
    }
    /**
     * Lists all the AWS accounts where the specified permission set is provisioned.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountsForProvisionedPermissionSet() {
        this.add('sso:ListAccountsForProvisionedPermissionSet');
        return this;
    }
    /**
     * Retrieve all of the certificates for a given application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstanceCertificates() {
        this.add('sso:ListApplicationInstanceCertificates');
        return this;
    }
    /**
     * Retrieve all application instances
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationInstance
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstances() {
        this.add('sso:ListApplicationInstances');
        return this;
    }
    /**
     * Retrieve all supported application templates
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:GetApplicationTemplate
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationTemplates() {
        this.add('sso:ListApplicationTemplates');
        return this;
    }
    /**
     * Retrieve all supported applications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplications() {
        this.add('sso:ListApplications');
        return this;
    }
    /**
     * Retrieve details about the directory connected to AWS Single Sign-On
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListDirectoryAssociations() {
        this.add('sso:ListDirectoryAssociations');
        return this;
    }
    /**
     * Lists the SSO Instances that the caller has access to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListInstances() {
        this.add('sso:ListInstances');
        return this;
    }
    /**
     * Lists the AWS managed policies that are attached to a specified permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListManagedPoliciesInPermissionSet() {
        this.add('sso:ListManagedPoliciesInPermissionSet');
        return this;
    }
    /**
     * Lists the status of the Permission Set Provisioning requests for a specified SSO instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSetProvisioningStatus() {
        this.add('sso:ListPermissionSetProvisioningStatus');
        return this;
    }
    /**
     * Retrieve all permission sets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSets() {
        this.add('sso:ListPermissionSets');
        return this;
    }
    /**
     * Lists all the permission sets that are provisioned to a specified AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSetsProvisionedToAccount() {
        this.add('sso:ListPermissionSetsProvisionedToAccount');
        return this;
    }
    /**
     * Retrieve the directory user or group associated with the profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfileAssociations() {
        this.add('sso:ListProfileAssociations');
        return this;
    }
    /**
     * Retrieve all profiles for an application instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:GetProfile
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfiles() {
        this.add('sso:ListProfiles');
        return this;
    }
    /**
     * Lists the tags that are attached to a specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListTagsForResource() {
        this.add('sso:ListTagsForResource');
        return this;
    }
    /**
     * The process by which a specified permission set is provisioned to the specified target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toProvisionPermissionSet() {
        this.add('sso:ProvisionPermissionSet');
        return this;
    }
    /**
     * Attaches an IAM inline policy to a permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutInlinePolicyToPermissionSet() {
        this.add('sso:PutInlinePolicyToPermissionSet');
        return this;
    }
    /**
     * Put Mfa Device Management settings for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutMfaDeviceManagementForDirectory() {
        this.add('sso:PutMfaDeviceManagementForDirectory');
        return this;
    }
    /**
     * Add a policy to a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutPermissionsPolicy() {
        this.add('sso:PutPermissionsPolicy');
        return this;
    }
    /**
     * Initialize AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartSSO() {
        this.add('sso:StartSSO');
        return this;
    }
    /**
     * Associates a set of tags with a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toTagResource() {
        this.add('sso:TagResource');
        return this;
    }
    /**
     * Disassociates a set of tags from a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUntagResource() {
        this.add('sso:UntagResource');
        return this;
    }
    /**
     * Set a certificate as the active one for this application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceActiveCertificate() {
        this.add('sso:UpdateApplicationInstanceActiveCertificate');
        return this;
    }
    /**
     * Update display data of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceDisplayData() {
        this.add('sso:UpdateApplicationInstanceDisplayData');
        return this;
    }
    /**
     * Update federation response configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseConfiguration() {
        this.add('sso:UpdateApplicationInstanceResponseConfiguration');
        return this;
    }
    /**
     * Update federation response schema configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseSchemaConfiguration() {
        this.add('sso:UpdateApplicationInstanceResponseSchemaConfiguration');
        return this;
    }
    /**
     * Update security details for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceSecurityConfiguration() {
        this.add('sso:UpdateApplicationInstanceSecurityConfiguration');
        return this;
    }
    /**
     * Update service provider related configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceServiceProviderConfiguration() {
        this.add('sso:UpdateApplicationInstanceServiceProviderConfiguration');
        return this;
    }
    /**
     * Update the status of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceStatus() {
        this.add('sso:UpdateApplicationInstanceStatus');
        return this;
    }
    /**
     * Update the user attribute mappings for your connected directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateDirectoryAssociation() {
        this.add('sso:UpdateDirectoryAssociation');
        return this;
    }
    /**
     * Update the status of a managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateManagedApplicationInstanceStatus() {
        this.add('sso:UpdateManagedApplicationInstanceStatus');
        return this;
    }
    /**
     * Update the permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdatePermissionSet() {
        this.add('sso:UpdatePermissionSet');
        return this;
    }
    /**
     * Update the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateProfile() {
        this.add('sso:UpdateProfile');
        return this;
    }
    /**
     * Update the configuration for the current SSO instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateSSOConfiguration() {
        this.add('sso:UpdateSSOConfiguration');
        return this;
    }
    /**
     * Update the federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateTrust() {
        this.add('sso:UpdateTrust');
        return this;
    }
    /**
     * Adds a resource of type PermissionSet to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param permissionSetId - Identifier for the permissionSetId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPermissionSet(instanceId, permissionSetId, partition) {
        var arn = 'arn:${Partition}:sso:::permissionSet/${InstanceId}/${PermissionSetId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${PermissionSetId}', permissionSetId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccount(accountId, partition) {
        var arn = 'arn:${Partition}:sso:::account/${AccountId}';
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstance(instanceId, partition) {
        var arn = 'arn:${Partition}:sso:::instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sso = Sso;
//# sourceMappingURL=data:application/json;base64,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