"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ssm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ssm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssystemsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ssm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ssm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssystemsmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ssm';
        this.actionList = {
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AddTagsToResource.html",
                "description": "Grants permission to add or overwrite one or more tags for a specified AWS resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "document": {
                        "required": false
                    },
                    "maintenancewindow": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    },
                    "parameter": {
                        "required": false
                    },
                    "patchbaseline": {
                        "required": false
                    }
                }
            },
            "CancelCommand": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelCommand.html",
                "description": "Grants permission to cancel a specified Run Command command",
                "accessLevel": "Write"
            },
            "CancelMaintenanceWindowExecution": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelMaintenanceWindowExecution.html",
                "description": "Grants permission to cancel an in-progress maintenance window execution",
                "accessLevel": "Write"
            },
            "CreateActivation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateActivation.html",
                "description": "Grants permission to create an activation that is used to register on-premises servers and virtual machines (VMs) with Systems Manager",
                "accessLevel": "Write"
            },
            "CreateAssociation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociation.html",
                "description": "Grants permission to associate a specified Systems Manager document with specified instances or other targets",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "CreateAssociationBatch": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociationBatch.html",
                "description": "Grants permission to combine entries for multiple CreateAssociation operations in a single command",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "CreateDocument": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateDocument.html",
                "description": "Grants permission to create a Systems Manager SSM document",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateMaintenanceWindow.html",
                "description": "Grants permission to create a maintenance window",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateOpsItem": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateOpsItem.html",
                "description": "Grants permission to create an OpsItem in OpsCenter",
                "accessLevel": "Write"
            },
            "CreatePatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreatePatchBaseline.html",
                "description": "Grants permission to create a patch baseline",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateResourceDataSync": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateResourceDataSync.html",
                "description": "Grants permission to create a resource data sync configuration, which regularly collects inventory data from managed instances and updates the data in an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourcedatasync": {
                        "required": true
                    }
                },
                "conditions": [
                    "ssm:SyncType"
                ]
            },
            "DeleteActivation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteActivation.html",
                "description": "Grants permission to delete a specified activation for managed instances",
                "accessLevel": "Write"
            },
            "DeleteAssociation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteAssociation.html",
                "description": "Grants permission to disassociate a specified SSM document from a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "association": {
                        "required": false
                    },
                    "document": {
                        "required": false
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "DeleteDocument": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteDocument.html",
                "description": "Grants permission to delete a specified SSM document and its instance associations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "DeleteInventory": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteInventory.html",
                "description": "Grants permission to delete a specified custom inventory type, or the data associated with a custom inventory type",
                "accessLevel": "Write"
            },
            "DeleteMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteMaintenanceWindow.html",
                "description": "Grants permission to delete a specified maintenance window",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "DeleteParameter": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameter.html",
                "description": "Grants permission to delete a specified SSM parameter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteParameters": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameters.html",
                "description": "Grants permission to delete multiple specified SSM parameters",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeletePatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeletePatchBaseline.html",
                "description": "Grants permission to delete a specified patch baseline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "DeleteResourceDataSync": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteResourceDataSync.html",
                "description": "Grants permission to delete a specified resource data sync",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourcedatasync": {
                        "required": true
                    }
                },
                "conditions": [
                    "ssm:SyncType"
                ]
            },
            "DeregisterManagedInstance": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterManagedInstance.html",
                "description": "Grants permission to deregister a specified on-premises server or virtual machine (VM) from Systems Manager",
                "accessLevel": "Write",
                "resourceTypes": {
                    "managed-instance": {
                        "required": true
                    }
                }
            },
            "DeregisterPatchBaselineForPatchGroup": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterPatchBaselineForPatchGroup.html",
                "description": "Grants permission to deregister a specified patch baseline from being the default patch baseline for a specified patch group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "DeregisterTargetFromMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTargetFromMaintenanceWindow.html",
                "description": "Grants permission to deregister a specified target from a maintenance window",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "DeregisterTaskFromMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTaskFromMaintenanceWindow.html",
                "description": "Grants permission to deregister a specified task from a maintenance window",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "DescribeActivations": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeActivations.html",
                "description": "Grants permission to view details about a specified managed instance activation, such as when it was created and the number of instances registered using the activation",
                "accessLevel": "Read"
            },
            "DescribeAssociation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociation.html",
                "description": "Grants permission to view details about the specified association for a specified instance or target",
                "accessLevel": "Read",
                "resourceTypes": {
                    "association": {
                        "required": false
                    },
                    "document": {
                        "required": false
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "DescribeAssociationExecutionTargets": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutionTargets.html",
                "description": "Grants permission to view information about a specified association execution",
                "accessLevel": "Read"
            },
            "DescribeAssociationExecutions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutions.html",
                "description": "Grants permission to view all executions for a specified association",
                "accessLevel": "Read"
            },
            "DescribeAutomationExecutions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationExecutions.html",
                "description": "Grants permission to view details about all active and terminated Automation executions",
                "accessLevel": "Read"
            },
            "DescribeAutomationStepExecutions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationStepExecutions.html",
                "description": "Grants permission to view information about all active and terminated step executions in an Automation workflow",
                "accessLevel": "Read"
            },
            "DescribeAvailablePatches": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAvailablePatches.html",
                "description": "Grants permission to view all patches eligible to include in a patch baseline",
                "accessLevel": "Read"
            },
            "DescribeDocument": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocument.html",
                "description": "Grants permission to view details about a specified SSM document",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "DescribeDocumentParameters": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html",
                "description": "Grants permission to display information about SSM document parameters in the Systems Manager console (internal Systems Manager action)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "DescribeDocumentPermission": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocumentPermission.html",
                "description": "Grants permission to view the permissions for a specified SSM document",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "DescribeEffectiveInstanceAssociations": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectiveInstanceAssociations.html",
                "description": "Grants permission to view all current associations for a specified instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "DescribeEffectivePatchesForPatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectivePatchesForPatchBaseline.html",
                "description": "Grants permission to view details about the patches currently associated with the specified patch baseline (Windows only)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "DescribeInstanceAssociationsStatus": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceAssociationsStatus.html",
                "description": "Grants permission to view the status of the associations for a specified instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "DescribeInstanceInformation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceInformation.html",
                "description": "Grants permission to view details about a specified instance",
                "accessLevel": "Read"
            },
            "DescribeInstancePatchStates": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStates.html",
                "description": "Grants permission to view status details about patches on a specified instance",
                "accessLevel": "Read"
            },
            "DescribeInstancePatchStatesForPatchGroup": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStatesForPatchGroup.html",
                "description": "Grants permission to describe the high-level patch state for the instances in the specified patch group",
                "accessLevel": "Read"
            },
            "DescribeInstancePatches": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatches.html",
                "description": "Grants permission to view general details about the patches on a specified instance",
                "accessLevel": "Read"
            },
            "DescribeInstanceProperties": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html",
                "description": "Grants permission to user's Amazon EC2 console to render managed instances' nodes",
                "accessLevel": "Read"
            },
            "DescribeInventoryDeletions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInventoryDeletions.html",
                "description": "Grants permission to view details about a specified inventory deletion",
                "accessLevel": "Read"
            },
            "DescribeMaintenanceWindowExecutionTaskInvocations": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTaskInvocations.html",
                "description": "Grants permission to view details of a specified task execution for a maintenance window",
                "accessLevel": "List"
            },
            "DescribeMaintenanceWindowExecutionTasks": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTasks.html",
                "description": "Grants permission to view details about the tasks that ran during a specified maintenance window execution",
                "accessLevel": "List"
            },
            "DescribeMaintenanceWindowExecutions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutions.html",
                "description": "Grants permission to view the executions of a specified maintenance window",
                "accessLevel": "List",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "DescribeMaintenanceWindowSchedule": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowSchedule.html",
                "description": "Grants permission to view details about upcoming executions of a specified maintenance window",
                "accessLevel": "List"
            },
            "DescribeMaintenanceWindowTargets": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTargets.html",
                "description": "Grants permission to view a list of the targets associated with a specified maintenance window",
                "accessLevel": "List",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "DescribeMaintenanceWindowTasks": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTasks.html",
                "description": "Grants permission to view a list of the tasks associated with a specified maintenance window",
                "accessLevel": "List",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "DescribeMaintenanceWindows": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindows.html",
                "description": "Grants permission to view information about all or specified maintenance windows",
                "accessLevel": "List"
            },
            "DescribeMaintenanceWindowsForTarget": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowsForTarget.html",
                "description": "Grants permission to view information about the maintenance window targets and tasks associated with a specified instance",
                "accessLevel": "List"
            },
            "DescribeOpsItems": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeOpsItems.html",
                "description": "Grants permission to view details about specified OpsItems",
                "accessLevel": "Read"
            },
            "DescribeParameters": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeParameters.html",
                "description": "Grants permission to view details about a specified SSM parameter",
                "accessLevel": "List"
            },
            "DescribePatchBaselines": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchBaselines.html",
                "description": "Grants permission to view information about patch baselines that meet the specified criteria",
                "accessLevel": "List"
            },
            "DescribePatchGroupState": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroupState.html",
                "description": "Grants permission to view aggregated status details for patches for a specified patch group",
                "accessLevel": "Read"
            },
            "DescribePatchGroups": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroups.html",
                "description": "Grants permission to view information about the patch baseline for a specified patch group",
                "accessLevel": "List"
            },
            "DescribePatchProperties": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchProperties.html",
                "description": "Grants permission to view details of available patches for a specified operating system and patch property",
                "accessLevel": "List"
            },
            "DescribeSessions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeSessions.html",
                "description": "Grants permission to view a list of recent Session Manager sessions that meet the specified search criteria",
                "accessLevel": "List"
            },
            "GetAutomationExecution": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AutomationExecution.html",
                "description": "Grants permission to view details of a specified Automation execution",
                "accessLevel": "Read"
            },
            "GetCommandInvocation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetCommandInvocation.html",
                "description": "Grants permission to view details about the command execution of a specified invocation or plugin",
                "accessLevel": "Read"
            },
            "GetConnectionStatus": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetConnectionStatus.html",
                "description": "Grants permission to view the Session Manager connection status for a specified managed instance",
                "accessLevel": "Read"
            },
            "GetDefaultPatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDefaultPatchBaseline.html",
                "description": "Grants permission to view the current default patch baseline for a specified operating system type",
                "accessLevel": "Read",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "GetDeployablePatchSnapshotForInstance": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDeployablePatchSnapshotForInstance.html",
                "description": "Grants permission to retrieve the current patch baseline snapshot for a specified instance",
                "accessLevel": "Read"
            },
            "GetDocument": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDocument.html",
                "description": "Grants permission to view the contents of a specified SSM document",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "GetInventory": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventory.html",
                "description": "Grants permission to view instance inventory details per the specified criteria",
                "accessLevel": "Read"
            },
            "GetInventorySchema": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventorySchema.html",
                "description": "Grants permission to view a list of inventory types or attribute names for a specified inventory item type",
                "accessLevel": "Read"
            },
            "GetMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindow.html",
                "description": "Grants permission to view details about a specified maintenance window",
                "accessLevel": "Read",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "GetMaintenanceWindowExecution": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecution.html",
                "description": "Grants permission to view details about a specified maintenance window execution",
                "accessLevel": "Read"
            },
            "GetMaintenanceWindowExecutionTask": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTask.html",
                "description": "Grants permission to view details about a specified maintenance window execution task",
                "accessLevel": "Read"
            },
            "GetMaintenanceWindowExecutionTaskInvocation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTaskInvocation.html",
                "description": "Grants permission to view details about a specific maintenance window task running on a specific target",
                "accessLevel": "Read"
            },
            "GetMaintenanceWindowTask": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowTask.html",
                "description": "Grants permission to view details about tasks registered with a specified maintenance window",
                "accessLevel": "Read",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "GetManifest": {
                "url": "",
                "description": "Used by Systems Manager and SSM Agent to determine package installation requirements for an instance (internal Systems Manager call)",
                "accessLevel": "Read"
            },
            "GetOpsItem": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsItem.html",
                "description": "Grants permission to view information about a specified OpsItem",
                "accessLevel": "Read"
            },
            "GetOpsSummary": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsSummary.html",
                "description": "Grants permission to view summary information about OpsItems based on specified filters and aggregators",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resourcedatasync": {
                        "required": true
                    }
                }
            },
            "GetParameter": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameter.html",
                "description": "Grants permission to view information about a specified parameter",
                "accessLevel": "Read",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "GetParameterHistory": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameterHistory.html",
                "description": "Grants permission to view details and changes for a specified parameter",
                "accessLevel": "Read",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "GetParameters": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameters.html",
                "description": "Grants permission to view information about multiple specified parameters",
                "accessLevel": "Read",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "GetParametersByPath": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParametersByPath.html",
                "description": "Grants permission to view information about parameters in a specified hierarchy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                }
            },
            "GetPatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaseline.html",
                "description": "Grants permission to view information about a specified patch baseline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "GetPatchBaselineForPatchGroup": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaselineForPatchGroup.html",
                "description": "Grants permission to view the ID of the current patch baseline for a specified patch group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "GetServiceSetting": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetServiceSetting.html",
                "description": "Grants permission to view the account-level setting for an AWS service",
                "accessLevel": "Read",
                "resourceTypes": {
                    "servicesetting": {
                        "required": true
                    }
                }
            },
            "LabelParameterVersion": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_LabelParameterVersion.html",
                "description": "Grants permission to apply an identifying label to a specified version of a parameter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                }
            },
            "ListAssociationVersions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociationVersions.html",
                "description": "Grants permission to list versions of the specified association",
                "accessLevel": "List"
            },
            "ListAssociations": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociations.html",
                "description": "Grants permission to list the associations for a specified SSM document or managed instance",
                "accessLevel": "List"
            },
            "ListCommandInvocations": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommandInvocations.html",
                "description": "Grants permission to list information about command invocations sent to a specified instance",
                "accessLevel": "Read"
            },
            "ListCommands": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommands.html",
                "description": "Grants permission to list the commands sent to a specified instance",
                "accessLevel": "Read"
            },
            "ListComplianceItems": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceItems.html",
                "description": "Grants permission to list compliance status for specified resource types on a specified resource",
                "accessLevel": "List"
            },
            "ListComplianceSummaries": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceSummaries.html",
                "description": "Grants permission to list a summary count of compliant and noncompliant resources for a specified compliance type",
                "accessLevel": "List"
            },
            "ListDocumentVersions": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocumentVersions.html",
                "description": "Grants permission to list all versions of a specified document",
                "accessLevel": "List",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "ListDocuments": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocuments.html",
                "description": "Grants permission to view information about a specified SSM document",
                "accessLevel": "List"
            },
            "ListInstanceAssociations": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html",
                "description": "Used by SSM Agent to check for new State Manager associations (internal Systems Manager call)",
                "accessLevel": "List",
                "resourceTypes": {
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "ListInventoryEntries": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListInventoryEntries.html",
                "description": "Grants permission to view a list of specified inventory types for a specified instance",
                "accessLevel": "List"
            },
            "ListResourceComplianceSummaries": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceComplianceSummaries.html",
                "description": "Grants permission to list resource-level summary count",
                "accessLevel": "List"
            },
            "ListResourceDataSync": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceDataSync.html",
                "description": "Grants permission to list information about resource data sync configurations in an account",
                "accessLevel": "List",
                "conditions": [
                    "ssm:SyncType"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to view a list of resource tags for a specified resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document": {
                        "required": false
                    },
                    "maintenancewindow": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    },
                    "parameter": {
                        "required": false
                    },
                    "patchbaseline": {
                        "required": false
                    }
                }
            },
            "ModifyDocumentPermission": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ModifyDocumentPermission.html",
                "description": "Grants permission to share a custom SSM document publicly or privately with specified AWS accounts",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "PutComplianceItems": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutComplianceItems.html",
                "description": "Grants permission to register a compliance type and other compliance details on a specified resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "PutConfigurePackageResult": {
                "url": "",
                "description": "Used by SSM Agent to generate a report of the results of specific agent requests (internal Systems Manager call)",
                "accessLevel": "Read"
            },
            "PutInventory": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutInventory.html",
                "description": "Grants permission to add or update inventory items on multiple specified managed instances",
                "accessLevel": "Write"
            },
            "PutParameter": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutParameter.html",
                "description": "Grants permission to create an SSM parameter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parameter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RegisterDefaultPatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterDefaultPatchBaseline.html",
                "description": "Grants permission to specify the default patch baseline for an operating system type",
                "accessLevel": "Write",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "RegisterPatchBaselineForPatchGroup": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterPatchBaselineForPatchGroup.html",
                "description": "Grants permission to specify the default patch baseline for a specified patch group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "RegisterTargetWithMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTargetWithMaintenanceWindow.html",
                "description": "Grants permission to register a target with a specified maintenance window",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "RegisterTaskWithMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTaskWithMaintenanceWindow.html",
                "description": "Grants permission to register a task with a specified maintenance window",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RemoveTagsFromResource.html",
                "description": "Grants permission to remove a specified tag key from a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "document": {
                        "required": false
                    },
                    "maintenancewindow": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    },
                    "parameter": {
                        "required": false
                    },
                    "patchbaseline": {
                        "required": false
                    }
                }
            },
            "ResetServiceSetting": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResetServiceSetting.html",
                "description": "Grants permission to reset the service setting for an AWS account to the default value",
                "accessLevel": "Write",
                "resourceTypes": {
                    "servicesetting": {
                        "required": true
                    }
                }
            },
            "ResumeSession": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResumeSession.html",
                "description": "Grants permission to reconnect a Session Manager session to a managed instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "session": {
                        "required": true
                    }
                }
            },
            "SendAutomationSignal": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendAutomationSignal.html",
                "description": "Grants permission to send a signal to change the current behavior or status of a specified Automation execution",
                "accessLevel": "Write"
            },
            "SendCommand": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendCommand.html",
                "description": "Grants permission to run commands on one or more specified managed instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "StartAssociationsOnce": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAssociationsOnce.html",
                "description": "Grants permission to run a specified association manually",
                "accessLevel": "Write",
                "resourceTypes": {
                    "association": {
                        "required": true
                    }
                }
            },
            "StartAutomationExecution": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAutomationExecution.html",
                "description": "Grants permission to initiate the execution of an Automation document",
                "accessLevel": "Write",
                "resourceTypes": {
                    "automation-definition": {
                        "required": true
                    }
                }
            },
            "StartSession": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartSession.html",
                "description": "Grants permission to initiate a connection to a specified target for a Session Manager session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    },
                    "document": {
                        "required": false
                    }
                },
                "conditions": [
                    "ssm:SessionDocumentAccessCheck"
                ]
            },
            "StopAutomationExecution": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StopAutomationExecution.html",
                "description": "Grants permission to stop a specified Automation execution that is already in progress",
                "accessLevel": "Write"
            },
            "TerminateSession": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_TerminateSession.html",
                "description": "Grants permission to permanently end a Session Manager connection to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "session": {
                        "required": true
                    }
                }
            },
            "UpdateAssociation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociation.html",
                "description": "Grants permission to update an association and immediately run the association on the specified targets",
                "accessLevel": "Write",
                "resourceTypes": {
                    "association": {
                        "required": true
                    },
                    "document": {
                        "required": false
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "UpdateAssociationStatus": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociationStatus.html",
                "description": "Grants permission to update the status of the SSM document associated with a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "UpdateDocument": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocument.html",
                "description": "Grants permission to update one or more values for an SSM document",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "UpdateDocumentDefaultVersion": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocumentDefaultVersion.html",
                "description": "Grants permission to change the default version of an SSM document",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document": {
                        "required": true
                    }
                }
            },
            "UpdateInstanceAssociationStatus": {
                "url": "",
                "description": "Used by SSM Agent to update the status of the association that it is currently running (internal Systems Manager call)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "association": {
                        "required": true
                    },
                    "instance": {
                        "required": false
                    },
                    "managed-instance": {
                        "required": false
                    }
                }
            },
            "UpdateInstanceInformation": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html",
                "description": "Used by SSM Agent to send a heartbeat signal to the Systems Manager service in the cloud",
                "accessLevel": "Write"
            },
            "UpdateMaintenanceWindow": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindow.html",
                "description": "Grants permission to update a specified maintenance window",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "UpdateMaintenanceWindowTarget": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTarget.html",
                "description": "Grants permission to update a specified maintenance window target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "UpdateMaintenanceWindowTask": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTask.html",
                "description": "Grants permission to update a specified maintenance window task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "maintenancewindow": {
                        "required": true
                    }
                }
            },
            "UpdateManagedInstanceRole": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateManagedInstanceRole.html",
                "description": "Grants permission to assign or change the IAM role assigned to a specified managed instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "managed-instance": {
                        "required": true
                    }
                }
            },
            "UpdateOpsItem": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateOpsItem.html",
                "description": "Grants permission to edit or change an OpsItem",
                "accessLevel": "Write"
            },
            "UpdatePatchBaseline": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdatePatchBaseline.html",
                "description": "Grants permission to update a specified patch baseline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "patchbaseline": {
                        "required": true
                    }
                }
            },
            "UpdateResourceDataSync": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateResourceDataSync.html",
                "description": "Grants permission to update a resource data sync",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourcedatasync": {
                        "required": true
                    }
                },
                "conditions": [
                    "ssm:SyncType"
                ]
            },
            "UpdateServiceSetting": {
                "url": "https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateServiceSetting.html",
                "description": "Grants permission to update the service setting for an AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "servicesetting": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "association": {
                "name": "association",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-associations.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:association/${AssociationId}",
                "conditionKeys": []
            },
            "automation-execution": {
                "name": "automation-execution",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-working.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:automation-execution/${AutomationExecutionId}",
                "conditionKeys": []
            },
            "automation-definition": {
                "name": "automation-definition",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-documents.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:automation-definition/${AutomationDefinitionName}:${VersionId}",
                "conditionKeys": []
            },
            "document": {
                "name": "document",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-ssm-docs.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:document/${DocumentName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "instance": {
                "name": "instance",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "maintenancewindow": {
                "name": "maintenancewindow",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-maintenance.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:maintenancewindow/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "managed-instance": {
                "name": "managed-instance",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/what-is-systems-manager.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:managed-instance/${ManagedInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "managed-instance-inventory": {
                "name": "managed-instance-inventory",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-configuring.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:managed-instance-inventory/${InstanceId}",
                "conditionKeys": []
            },
            "opsitem": {
                "name": "opsitem",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:opsitem/${ResourceId}",
                "conditionKeys": []
            },
            "parameter": {
                "name": "parameter",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-parameter-store.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:parameter/${FullyQualifiedParameterName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "patchbaseline": {
                "name": "patchbaseline",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-patch.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:patchbaseline/${PatchBaselineIdResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ssm:resourceTag/tag-key"
                ]
            },
            "session": {
                "name": "session",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:session/${SessionId}",
                "conditionKeys": []
            },
            "resourcedatasync": {
                "name": "resourcedatasync",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-datasync.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:resource-data-sync/${SyncName}",
                "conditionKeys": []
            },
            "servicesetting": {
                "name": "servicesetting",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/API_ServiceSetting.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:servicesetting/${ResourceId}",
                "conditionKeys": []
            },
            "windowtarget": {
                "name": "windowtarget",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-targets.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:windowtarget/${WindowTargetId}",
                "conditionKeys": []
            },
            "windowtask": {
                "name": "windowtask",
                "url": "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-tasks.html",
                "arn": "arn:${Partition}:ssm:${Region}:${Account}:windowtask/${WindowTaskId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add or overwrite one or more tags for a specified AWS resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.add('ssm:AddTagsToResource');
        return this;
    }
    /**
     * Grants permission to cancel a specified Run Command command
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelCommand.html
     */
    toCancelCommand() {
        this.add('ssm:CancelCommand');
        return this;
    }
    /**
     * Grants permission to cancel an in-progress maintenance window execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelMaintenanceWindowExecution.html
     */
    toCancelMaintenanceWindowExecution() {
        this.add('ssm:CancelMaintenanceWindowExecution');
        return this;
    }
    /**
     * Grants permission to create an activation that is used to register on-premises servers and virtual machines (VMs) with Systems Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateActivation.html
     */
    toCreateActivation() {
        this.add('ssm:CreateActivation');
        return this;
    }
    /**
     * Grants permission to associate a specified Systems Manager document with specified instances or other targets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociation.html
     */
    toCreateAssociation() {
        this.add('ssm:CreateAssociation');
        return this;
    }
    /**
     * Grants permission to combine entries for multiple CreateAssociation operations in a single command
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociationBatch.html
     */
    toCreateAssociationBatch() {
        this.add('ssm:CreateAssociationBatch');
        return this;
    }
    /**
     * Grants permission to create a Systems Manager SSM document
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateDocument.html
     */
    toCreateDocument() {
        this.add('ssm:CreateDocument');
        return this;
    }
    /**
     * Grants permission to create a maintenance window
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateMaintenanceWindow.html
     */
    toCreateMaintenanceWindow() {
        this.add('ssm:CreateMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to create an OpsItem in OpsCenter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateOpsItem.html
     */
    toCreateOpsItem() {
        this.add('ssm:CreateOpsItem');
        return this;
    }
    /**
     * Grants permission to create a patch baseline
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreatePatchBaseline.html
     */
    toCreatePatchBaseline() {
        this.add('ssm:CreatePatchBaseline');
        return this;
    }
    /**
     * Grants permission to create a resource data sync configuration, which regularly collects inventory data from managed instances and updates the data in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateResourceDataSync.html
     */
    toCreateResourceDataSync() {
        this.add('ssm:CreateResourceDataSync');
        return this;
    }
    /**
     * Grants permission to delete a specified activation for managed instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteActivation.html
     */
    toDeleteActivation() {
        this.add('ssm:DeleteActivation');
        return this;
    }
    /**
     * Grants permission to disassociate a specified SSM document from a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteAssociation.html
     */
    toDeleteAssociation() {
        this.add('ssm:DeleteAssociation');
        return this;
    }
    /**
     * Grants permission to delete a specified SSM document and its instance associations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteDocument.html
     */
    toDeleteDocument() {
        this.add('ssm:DeleteDocument');
        return this;
    }
    /**
     * Grants permission to delete a specified custom inventory type, or the data associated with a custom inventory type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteInventory.html
     */
    toDeleteInventory() {
        this.add('ssm:DeleteInventory');
        return this;
    }
    /**
     * Grants permission to delete a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteMaintenanceWindow.html
     */
    toDeleteMaintenanceWindow() {
        this.add('ssm:DeleteMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to delete a specified SSM parameter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameter.html
     */
    toDeleteParameter() {
        this.add('ssm:DeleteParameter');
        return this;
    }
    /**
     * Grants permission to delete multiple specified SSM parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameters.html
     */
    toDeleteParameters() {
        this.add('ssm:DeleteParameters');
        return this;
    }
    /**
     * Grants permission to delete a specified patch baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeletePatchBaseline.html
     */
    toDeletePatchBaseline() {
        this.add('ssm:DeletePatchBaseline');
        return this;
    }
    /**
     * Grants permission to delete a specified resource data sync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteResourceDataSync.html
     */
    toDeleteResourceDataSync() {
        this.add('ssm:DeleteResourceDataSync');
        return this;
    }
    /**
     * Grants permission to deregister a specified on-premises server or virtual machine (VM) from Systems Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterManagedInstance.html
     */
    toDeregisterManagedInstance() {
        this.add('ssm:DeregisterManagedInstance');
        return this;
    }
    /**
     * Grants permission to deregister a specified patch baseline from being the default patch baseline for a specified patch group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterPatchBaselineForPatchGroup.html
     */
    toDeregisterPatchBaselineForPatchGroup() {
        this.add('ssm:DeregisterPatchBaselineForPatchGroup');
        return this;
    }
    /**
     * Grants permission to deregister a specified target from a maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTargetFromMaintenanceWindow.html
     */
    toDeregisterTargetFromMaintenanceWindow() {
        this.add('ssm:DeregisterTargetFromMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to deregister a specified task from a maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTaskFromMaintenanceWindow.html
     */
    toDeregisterTaskFromMaintenanceWindow() {
        this.add('ssm:DeregisterTaskFromMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to view details about a specified managed instance activation, such as when it was created and the number of instances registered using the activation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeActivations.html
     */
    toDescribeActivations() {
        this.add('ssm:DescribeActivations');
        return this;
    }
    /**
     * Grants permission to view details about the specified association for a specified instance or target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociation.html
     */
    toDescribeAssociation() {
        this.add('ssm:DescribeAssociation');
        return this;
    }
    /**
     * Grants permission to view information about a specified association execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutionTargets.html
     */
    toDescribeAssociationExecutionTargets() {
        this.add('ssm:DescribeAssociationExecutionTargets');
        return this;
    }
    /**
     * Grants permission to view all executions for a specified association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutions.html
     */
    toDescribeAssociationExecutions() {
        this.add('ssm:DescribeAssociationExecutions');
        return this;
    }
    /**
     * Grants permission to view details about all active and terminated Automation executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationExecutions.html
     */
    toDescribeAutomationExecutions() {
        this.add('ssm:DescribeAutomationExecutions');
        return this;
    }
    /**
     * Grants permission to view information about all active and terminated step executions in an Automation workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationStepExecutions.html
     */
    toDescribeAutomationStepExecutions() {
        this.add('ssm:DescribeAutomationStepExecutions');
        return this;
    }
    /**
     * Grants permission to view all patches eligible to include in a patch baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAvailablePatches.html
     */
    toDescribeAvailablePatches() {
        this.add('ssm:DescribeAvailablePatches');
        return this;
    }
    /**
     * Grants permission to view details about a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocument.html
     */
    toDescribeDocument() {
        this.add('ssm:DescribeDocument');
        return this;
    }
    /**
     * Grants permission to display information about SSM document parameters in the Systems Manager console (internal Systems Manager action)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDescribeDocumentParameters() {
        this.add('ssm:DescribeDocumentParameters');
        return this;
    }
    /**
     * Grants permission to view the permissions for a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocumentPermission.html
     */
    toDescribeDocumentPermission() {
        this.add('ssm:DescribeDocumentPermission');
        return this;
    }
    /**
     * Grants permission to view all current associations for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectiveInstanceAssociations.html
     */
    toDescribeEffectiveInstanceAssociations() {
        this.add('ssm:DescribeEffectiveInstanceAssociations');
        return this;
    }
    /**
     * Grants permission to view details about the patches currently associated with the specified patch baseline (Windows only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectivePatchesForPatchBaseline.html
     */
    toDescribeEffectivePatchesForPatchBaseline() {
        this.add('ssm:DescribeEffectivePatchesForPatchBaseline');
        return this;
    }
    /**
     * Grants permission to view the status of the associations for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceAssociationsStatus.html
     */
    toDescribeInstanceAssociationsStatus() {
        this.add('ssm:DescribeInstanceAssociationsStatus');
        return this;
    }
    /**
     * Grants permission to view details about a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceInformation.html
     */
    toDescribeInstanceInformation() {
        this.add('ssm:DescribeInstanceInformation');
        return this;
    }
    /**
     * Grants permission to view status details about patches on a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStates.html
     */
    toDescribeInstancePatchStates() {
        this.add('ssm:DescribeInstancePatchStates');
        return this;
    }
    /**
     * Grants permission to describe the high-level patch state for the instances in the specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStatesForPatchGroup.html
     */
    toDescribeInstancePatchStatesForPatchGroup() {
        this.add('ssm:DescribeInstancePatchStatesForPatchGroup');
        return this;
    }
    /**
     * Grants permission to view general details about the patches on a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatches.html
     */
    toDescribeInstancePatches() {
        this.add('ssm:DescribeInstancePatches');
        return this;
    }
    /**
     * Grants permission to user's Amazon EC2 console to render managed instances' nodes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDescribeInstanceProperties() {
        this.add('ssm:DescribeInstanceProperties');
        return this;
    }
    /**
     * Grants permission to view details about a specified inventory deletion
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInventoryDeletions.html
     */
    toDescribeInventoryDeletions() {
        this.add('ssm:DescribeInventoryDeletions');
        return this;
    }
    /**
     * Grants permission to view details of a specified task execution for a maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTaskInvocations.html
     */
    toDescribeMaintenanceWindowExecutionTaskInvocations() {
        this.add('ssm:DescribeMaintenanceWindowExecutionTaskInvocations');
        return this;
    }
    /**
     * Grants permission to view details about the tasks that ran during a specified maintenance window execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTasks.html
     */
    toDescribeMaintenanceWindowExecutionTasks() {
        this.add('ssm:DescribeMaintenanceWindowExecutionTasks');
        return this;
    }
    /**
     * Grants permission to view the executions of a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutions.html
     */
    toDescribeMaintenanceWindowExecutions() {
        this.add('ssm:DescribeMaintenanceWindowExecutions');
        return this;
    }
    /**
     * Grants permission to view details about upcoming executions of a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowSchedule.html
     */
    toDescribeMaintenanceWindowSchedule() {
        this.add('ssm:DescribeMaintenanceWindowSchedule');
        return this;
    }
    /**
     * Grants permission to view a list of the targets associated with a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTargets.html
     */
    toDescribeMaintenanceWindowTargets() {
        this.add('ssm:DescribeMaintenanceWindowTargets');
        return this;
    }
    /**
     * Grants permission to view a list of the tasks associated with a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTasks.html
     */
    toDescribeMaintenanceWindowTasks() {
        this.add('ssm:DescribeMaintenanceWindowTasks');
        return this;
    }
    /**
     * Grants permission to view information about all or specified maintenance windows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindows.html
     */
    toDescribeMaintenanceWindows() {
        this.add('ssm:DescribeMaintenanceWindows');
        return this;
    }
    /**
     * Grants permission to view information about the maintenance window targets and tasks associated with a specified instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowsForTarget.html
     */
    toDescribeMaintenanceWindowsForTarget() {
        this.add('ssm:DescribeMaintenanceWindowsForTarget');
        return this;
    }
    /**
     * Grants permission to view details about specified OpsItems
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeOpsItems.html
     */
    toDescribeOpsItems() {
        this.add('ssm:DescribeOpsItems');
        return this;
    }
    /**
     * Grants permission to view details about a specified SSM parameter
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeParameters.html
     */
    toDescribeParameters() {
        this.add('ssm:DescribeParameters');
        return this;
    }
    /**
     * Grants permission to view information about patch baselines that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchBaselines.html
     */
    toDescribePatchBaselines() {
        this.add('ssm:DescribePatchBaselines');
        return this;
    }
    /**
     * Grants permission to view aggregated status details for patches for a specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroupState.html
     */
    toDescribePatchGroupState() {
        this.add('ssm:DescribePatchGroupState');
        return this;
    }
    /**
     * Grants permission to view information about the patch baseline for a specified patch group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroups.html
     */
    toDescribePatchGroups() {
        this.add('ssm:DescribePatchGroups');
        return this;
    }
    /**
     * Grants permission to view details of available patches for a specified operating system and patch property
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchProperties.html
     */
    toDescribePatchProperties() {
        this.add('ssm:DescribePatchProperties');
        return this;
    }
    /**
     * Grants permission to view a list of recent Session Manager sessions that meet the specified search criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeSessions.html
     */
    toDescribeSessions() {
        this.add('ssm:DescribeSessions');
        return this;
    }
    /**
     * Grants permission to view details of a specified Automation execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AutomationExecution.html
     */
    toGetAutomationExecution() {
        this.add('ssm:GetAutomationExecution');
        return this;
    }
    /**
     * Grants permission to view details about the command execution of a specified invocation or plugin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetCommandInvocation.html
     */
    toGetCommandInvocation() {
        this.add('ssm:GetCommandInvocation');
        return this;
    }
    /**
     * Grants permission to view the Session Manager connection status for a specified managed instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetConnectionStatus.html
     */
    toGetConnectionStatus() {
        this.add('ssm:GetConnectionStatus');
        return this;
    }
    /**
     * Grants permission to view the current default patch baseline for a specified operating system type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDefaultPatchBaseline.html
     */
    toGetDefaultPatchBaseline() {
        this.add('ssm:GetDefaultPatchBaseline');
        return this;
    }
    /**
     * Grants permission to retrieve the current patch baseline snapshot for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDeployablePatchSnapshotForInstance.html
     */
    toGetDeployablePatchSnapshotForInstance() {
        this.add('ssm:GetDeployablePatchSnapshotForInstance');
        return this;
    }
    /**
     * Grants permission to view the contents of a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDocument.html
     */
    toGetDocument() {
        this.add('ssm:GetDocument');
        return this;
    }
    /**
     * Grants permission to view instance inventory details per the specified criteria
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventory.html
     */
    toGetInventory() {
        this.add('ssm:GetInventory');
        return this;
    }
    /**
     * Grants permission to view a list of inventory types or attribute names for a specified inventory item type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventorySchema.html
     */
    toGetInventorySchema() {
        this.add('ssm:GetInventorySchema');
        return this;
    }
    /**
     * Grants permission to view details about a specified maintenance window
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindow.html
     */
    toGetMaintenanceWindow() {
        this.add('ssm:GetMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to view details about a specified maintenance window execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecution.html
     */
    toGetMaintenanceWindowExecution() {
        this.add('ssm:GetMaintenanceWindowExecution');
        return this;
    }
    /**
     * Grants permission to view details about a specified maintenance window execution task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTask.html
     */
    toGetMaintenanceWindowExecutionTask() {
        this.add('ssm:GetMaintenanceWindowExecutionTask');
        return this;
    }
    /**
     * Grants permission to view details about a specific maintenance window task running on a specific target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTaskInvocation.html
     */
    toGetMaintenanceWindowExecutionTaskInvocation() {
        this.add('ssm:GetMaintenanceWindowExecutionTaskInvocation');
        return this;
    }
    /**
     * Grants permission to view details about tasks registered with a specified maintenance window
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowTask.html
     */
    toGetMaintenanceWindowTask() {
        this.add('ssm:GetMaintenanceWindowTask');
        return this;
    }
    /**
     * Used by Systems Manager and SSM Agent to determine package installation requirements for an instance (internal Systems Manager call)
     *
     * Access Level: Read
     */
    toGetManifest() {
        this.add('ssm:GetManifest');
        return this;
    }
    /**
     * Grants permission to view information about a specified OpsItem
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsItem.html
     */
    toGetOpsItem() {
        this.add('ssm:GetOpsItem');
        return this;
    }
    /**
     * Grants permission to view summary information about OpsItems based on specified filters and aggregators
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsSummary.html
     */
    toGetOpsSummary() {
        this.add('ssm:GetOpsSummary');
        return this;
    }
    /**
     * Grants permission to view information about a specified parameter
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameter.html
     */
    toGetParameter() {
        this.add('ssm:GetParameter');
        return this;
    }
    /**
     * Grants permission to view details and changes for a specified parameter
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameterHistory.html
     */
    toGetParameterHistory() {
        this.add('ssm:GetParameterHistory');
        return this;
    }
    /**
     * Grants permission to view information about multiple specified parameters
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameters.html
     */
    toGetParameters() {
        this.add('ssm:GetParameters');
        return this;
    }
    /**
     * Grants permission to view information about parameters in a specified hierarchy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParametersByPath.html
     */
    toGetParametersByPath() {
        this.add('ssm:GetParametersByPath');
        return this;
    }
    /**
     * Grants permission to view information about a specified patch baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaseline.html
     */
    toGetPatchBaseline() {
        this.add('ssm:GetPatchBaseline');
        return this;
    }
    /**
     * Grants permission to view the ID of the current patch baseline for a specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaselineForPatchGroup.html
     */
    toGetPatchBaselineForPatchGroup() {
        this.add('ssm:GetPatchBaselineForPatchGroup');
        return this;
    }
    /**
     * Grants permission to view the account-level setting for an AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetServiceSetting.html
     */
    toGetServiceSetting() {
        this.add('ssm:GetServiceSetting');
        return this;
    }
    /**
     * Grants permission to apply an identifying label to a specified version of a parameter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_LabelParameterVersion.html
     */
    toLabelParameterVersion() {
        this.add('ssm:LabelParameterVersion');
        return this;
    }
    /**
     * Grants permission to list versions of the specified association
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociationVersions.html
     */
    toListAssociationVersions() {
        this.add('ssm:ListAssociationVersions');
        return this;
    }
    /**
     * Grants permission to list the associations for a specified SSM document or managed instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociations.html
     */
    toListAssociations() {
        this.add('ssm:ListAssociations');
        return this;
    }
    /**
     * Grants permission to list information about command invocations sent to a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommandInvocations.html
     */
    toListCommandInvocations() {
        this.add('ssm:ListCommandInvocations');
        return this;
    }
    /**
     * Grants permission to list the commands sent to a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommands.html
     */
    toListCommands() {
        this.add('ssm:ListCommands');
        return this;
    }
    /**
     * Grants permission to list compliance status for specified resource types on a specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceItems.html
     */
    toListComplianceItems() {
        this.add('ssm:ListComplianceItems');
        return this;
    }
    /**
     * Grants permission to list a summary count of compliant and noncompliant resources for a specified compliance type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceSummaries.html
     */
    toListComplianceSummaries() {
        this.add('ssm:ListComplianceSummaries');
        return this;
    }
    /**
     * Grants permission to list all versions of a specified document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocumentVersions.html
     */
    toListDocumentVersions() {
        this.add('ssm:ListDocumentVersions');
        return this;
    }
    /**
     * Grants permission to view information about a specified SSM document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocuments.html
     */
    toListDocuments() {
        this.add('ssm:ListDocuments');
        return this;
    }
    /**
     * Used by SSM Agent to check for new State Manager associations (internal Systems Manager call)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toListInstanceAssociations() {
        this.add('ssm:ListInstanceAssociations');
        return this;
    }
    /**
     * Grants permission to view a list of specified inventory types for a specified instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListInventoryEntries.html
     */
    toListInventoryEntries() {
        this.add('ssm:ListInventoryEntries');
        return this;
    }
    /**
     * Grants permission to list resource-level summary count
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceComplianceSummaries.html
     */
    toListResourceComplianceSummaries() {
        this.add('ssm:ListResourceComplianceSummaries');
        return this;
    }
    /**
     * Grants permission to list information about resource data sync configurations in an account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceDataSync.html
     */
    toListResourceDataSync() {
        this.add('ssm:ListResourceDataSync');
        return this;
    }
    /**
     * Grants permission to view a list of resource tags for a specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('ssm:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to share a custom SSM document publicly or privately with specified AWS accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ModifyDocumentPermission.html
     */
    toModifyDocumentPermission() {
        this.add('ssm:ModifyDocumentPermission');
        return this;
    }
    /**
     * Grants permission to register a compliance type and other compliance details on a specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutComplianceItems.html
     */
    toPutComplianceItems() {
        this.add('ssm:PutComplianceItems');
        return this;
    }
    /**
     * Used by SSM Agent to generate a report of the results of specific agent requests (internal Systems Manager call)
     *
     * Access Level: Read
     */
    toPutConfigurePackageResult() {
        this.add('ssm:PutConfigurePackageResult');
        return this;
    }
    /**
     * Grants permission to add or update inventory items on multiple specified managed instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutInventory.html
     */
    toPutInventory() {
        this.add('ssm:PutInventory');
        return this;
    }
    /**
     * Grants permission to create an SSM parameter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutParameter.html
     */
    toPutParameter() {
        this.add('ssm:PutParameter');
        return this;
    }
    /**
     * Grants permission to specify the default patch baseline for an operating system type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterDefaultPatchBaseline.html
     */
    toRegisterDefaultPatchBaseline() {
        this.add('ssm:RegisterDefaultPatchBaseline');
        return this;
    }
    /**
     * Grants permission to specify the default patch baseline for a specified patch group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterPatchBaselineForPatchGroup.html
     */
    toRegisterPatchBaselineForPatchGroup() {
        this.add('ssm:RegisterPatchBaselineForPatchGroup');
        return this;
    }
    /**
     * Grants permission to register a target with a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTargetWithMaintenanceWindow.html
     */
    toRegisterTargetWithMaintenanceWindow() {
        this.add('ssm:RegisterTargetWithMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to register a task with a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTaskWithMaintenanceWindow.html
     */
    toRegisterTaskWithMaintenanceWindow() {
        this.add('ssm:RegisterTaskWithMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to remove a specified tag key from a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.add('ssm:RemoveTagsFromResource');
        return this;
    }
    /**
     * Grants permission to reset the service setting for an AWS account to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResetServiceSetting.html
     */
    toResetServiceSetting() {
        this.add('ssm:ResetServiceSetting');
        return this;
    }
    /**
     * Grants permission to reconnect a Session Manager session to a managed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResumeSession.html
     */
    toResumeSession() {
        this.add('ssm:ResumeSession');
        return this;
    }
    /**
     * Grants permission to send a signal to change the current behavior or status of a specified Automation execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendAutomationSignal.html
     */
    toSendAutomationSignal() {
        this.add('ssm:SendAutomationSignal');
        return this;
    }
    /**
     * Grants permission to run commands on one or more specified managed instances
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendCommand.html
     */
    toSendCommand() {
        this.add('ssm:SendCommand');
        return this;
    }
    /**
     * Grants permission to run a specified association manually
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAssociationsOnce.html
     */
    toStartAssociationsOnce() {
        this.add('ssm:StartAssociationsOnce');
        return this;
    }
    /**
     * Grants permission to initiate the execution of an Automation document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAutomationExecution.html
     */
    toStartAutomationExecution() {
        this.add('ssm:StartAutomationExecution');
        return this;
    }
    /**
     * Grants permission to initiate a connection to a specified target for a Session Manager session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionDocumentAccessCheck()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartSession.html
     */
    toStartSession() {
        this.add('ssm:StartSession');
        return this;
    }
    /**
     * Grants permission to stop a specified Automation execution that is already in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StopAutomationExecution.html
     */
    toStopAutomationExecution() {
        this.add('ssm:StopAutomationExecution');
        return this;
    }
    /**
     * Grants permission to permanently end a Session Manager connection to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_TerminateSession.html
     */
    toTerminateSession() {
        this.add('ssm:TerminateSession');
        return this;
    }
    /**
     * Grants permission to update an association and immediately run the association on the specified targets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociation.html
     */
    toUpdateAssociation() {
        this.add('ssm:UpdateAssociation');
        return this;
    }
    /**
     * Grants permission to update the status of the SSM document associated with a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociationStatus.html
     */
    toUpdateAssociationStatus() {
        this.add('ssm:UpdateAssociationStatus');
        return this;
    }
    /**
     * Grants permission to update one or more values for an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocument.html
     */
    toUpdateDocument() {
        this.add('ssm:UpdateDocument');
        return this;
    }
    /**
     * Grants permission to change the default version of an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocumentDefaultVersion.html
     */
    toUpdateDocumentDefaultVersion() {
        this.add('ssm:UpdateDocumentDefaultVersion');
        return this;
    }
    /**
     * Used by SSM Agent to update the status of the association that it is currently running (internal Systems Manager call)
     *
     * Access Level: Write
     */
    toUpdateInstanceAssociationStatus() {
        this.add('ssm:UpdateInstanceAssociationStatus');
        return this;
    }
    /**
     * Used by SSM Agent to send a heartbeat signal to the Systems Manager service in the cloud
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toUpdateInstanceInformation() {
        this.add('ssm:UpdateInstanceInformation');
        return this;
    }
    /**
     * Grants permission to update a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindow.html
     */
    toUpdateMaintenanceWindow() {
        this.add('ssm:UpdateMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to update a specified maintenance window target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTarget.html
     */
    toUpdateMaintenanceWindowTarget() {
        this.add('ssm:UpdateMaintenanceWindowTarget');
        return this;
    }
    /**
     * Grants permission to update a specified maintenance window task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTask.html
     */
    toUpdateMaintenanceWindowTask() {
        this.add('ssm:UpdateMaintenanceWindowTask');
        return this;
    }
    /**
     * Grants permission to assign or change the IAM role assigned to a specified managed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateManagedInstanceRole.html
     */
    toUpdateManagedInstanceRole() {
        this.add('ssm:UpdateManagedInstanceRole');
        return this;
    }
    /**
     * Grants permission to edit or change an OpsItem
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateOpsItem.html
     */
    toUpdateOpsItem() {
        this.add('ssm:UpdateOpsItem');
        return this;
    }
    /**
     * Grants permission to update a specified patch baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdatePatchBaseline.html
     */
    toUpdatePatchBaseline() {
        this.add('ssm:UpdatePatchBaseline');
        return this;
    }
    /**
     * Grants permission to update a resource data sync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateResourceDataSync.html
     */
    toUpdateResourceDataSync() {
        this.add('ssm:UpdateResourceDataSync');
        return this;
    }
    /**
     * Grants permission to update the service setting for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateServiceSetting.html
     */
    toUpdateServiceSetting() {
        this.add('ssm:UpdateServiceSetting');
        return this;
    }
    /**
     * Adds a resource of type association to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-associations.html
     *
     * @param associationId - Identifier for the associationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssociation(associationId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:association/${AssociationId}';
        arn = arn.replace('${AssociationId}', associationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type automation-execution to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-working.html
     *
     * @param automationExecutionId - Identifier for the automationExecutionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAutomationExecution(automationExecutionId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:automation-execution/${AutomationExecutionId}';
        arn = arn.replace('${AutomationExecutionId}', automationExecutionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type automation-definition to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-documents.html
     *
     * @param automationDefinitionName - Identifier for the automationDefinitionName.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAutomationDefinition(automationDefinitionName, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:automation-definition/${AutomationDefinitionName}:${VersionId}';
        arn = arn.replace('${AutomationDefinitionName}', automationDefinitionName);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type document to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-ssm-docs.html
     *
     * @param documentName - Identifier for the documentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDocument(documentName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:document/${DocumentName}';
        arn = arn.replace('${DocumentName}', documentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type maintenancewindow to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-maintenance.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMaintenancewindow(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:maintenancewindow/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type managed-instance to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/what-is-systems-manager.html
     *
     * @param managedInstanceName - Identifier for the managedInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onManagedInstance(managedInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:managed-instance/${ManagedInstanceName}';
        arn = arn.replace('${ManagedInstanceName}', managedInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type managed-instance-inventory to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-configuring.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onManagedInstanceInventory(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:managed-instance-inventory/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type opsitem to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOpsitem(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:opsitem/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type parameter to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-parameter-store.html
     *
     * @param fullyQualifiedParameterName - Identifier for the fullyQualifiedParameterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onParameter(fullyQualifiedParameterName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:parameter/${FullyQualifiedParameterName}';
        arn = arn.replace('${FullyQualifiedParameterName}', fullyQualifiedParameterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type patchbaseline to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-patch.html
     *
     * @param patchBaselineIdResourceId - Identifier for the patchBaselineIdResourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onPatchbaseline(patchBaselineIdResourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:patchbaseline/${PatchBaselineIdResourceId}';
        arn = arn.replace('${PatchBaselineIdResourceId}', patchBaselineIdResourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type session to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager.html
     *
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSession(sessionId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:session/${SessionId}';
        arn = arn.replace('${SessionId}', sessionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourcedatasync to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-datasync.html
     *
     * @param syncName - Identifier for the syncName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourcedatasync(syncName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:resource-data-sync/${SyncName}';
        arn = arn.replace('${SyncName}', syncName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type servicesetting to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/API_ServiceSetting.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onServicesetting(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:servicesetting/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type windowtarget to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-targets.html
     *
     * @param windowTargetId - Identifier for the windowTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWindowtarget(windowTargetId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:windowtarget/${WindowTargetId}';
        arn = arn.replace('${WindowTargetId}', windowTargetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type windowtask to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-tasks.html
     *
     * @param windowTaskId - Identifier for the windowTaskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWindowtask(windowTaskId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:windowtask/${WindowTaskId}';
        arn = arn.replace('${WindowTaskId}', windowTaskId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by controlling whether the values for specified resources can be overwritten.
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOverwrite(value, operator) {
        return this.if(`ssm:Overwrite`, value, operator || 'StringLike');
    }
    /**
     * Filters access for resources created in a hierarchical structure.
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecursive(value, operator) {
        return this.if(`ssm:Recursive`, value, operator || 'StringLike');
    }
    /**
     * Filters access by verifying that a user has permission to access either the default Session Manager configuration document or the custom configuration document specified in a request.
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-sessiondocumentaccesscheck.html
     *
     * Applies to actions:
     * - .toStartSession()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSessionDocumentAccessCheck(value) {
        return this.if(`ssm:SessionDocumentAccessCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by verifying that a user also has access to the ResourceDataSync SyncType specified in the request
     *
     * Applies to actions:
     * - .toCreateResourceDataSync()
     * - .toDeleteResourceDataSync()
     * - .toListResourceDataSync()
     * - .toUpdateResourceDataSync()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSyncType(value, operator) {
        return this.if(`ssm:SyncType`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on a tag key-value pair assigned to the Systems Manager resource
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toSendCommand()
     *
     * Applies to resource types:
     * - document
     * - instance
     * - maintenancewindow
     * - managed-instance
     * - parameter
     * - patchbaseline
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagkey, value, operator) {
        return this.if(`ssm:resourceTag/${tagkey}`, value, operator || 'StringLike');
    }
}
exports.Ssm = Ssm;
//# sourceMappingURL=data:application/json;base64,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