"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trustedadvisor = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Trustedadvisor extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'trustedadvisor';
        this.actionList = {
            "DescribeAccount": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the AWS Support plan and various AWS Trusted Advisor preferences",
                "accessLevel": "Read"
            },
            "DescribeAccountAccess": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view if the AWS account has enabled or disabled AWS Trusted Advisor",
                "accessLevel": "Read"
            },
            "DescribeCheckItems": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view details for the check items",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckRefreshStatuses": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the refresh statuses for AWS Trusted Advisor checks",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckSummaries": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view AWS Trusted Advisor check summaries",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeChecks": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view details for AWS Trusted Advisor checks",
                "accessLevel": "Read"
            },
            "DescribeNotificationPreferences": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the notification preferences for the AWS account",
                "accessLevel": "Read"
            },
            "DescribeOrganization": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view if the AWS account meets the requirements to enable the organizational view feature",
                "accessLevel": "Read"
            },
            "DescribeOrganizationAccounts": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the linked AWS accounts that are in the organization",
                "accessLevel": "Read"
            },
            "DescribeReports": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view details for organizational view reports, such as the report name, runtime, date created, status, and format",
                "accessLevel": "Read"
            },
            "DescribeServiceMetadata": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view information about organizational view reports, such as the AWS Regions, check categories, check names, and resource statuses",
                "accessLevel": "Read"
            },
            "ExcludeCheckItems": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to exclude recommendations for AWS Trusted Advisor checks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "GenerateReport": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to create a report for AWS Trusted Advisor checks in your organization",
                "accessLevel": "Write"
            },
            "IncludeCheckItems": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to include recommendations for AWS Trusted Advisor checks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "RefreshCheck": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to refresh an AWS Trusted Advisor check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "SetAccountAccess": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to enable or disable AWS Trusted Advisor for the account",
                "accessLevel": "Write"
            },
            "SetOrganizationAccess": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to enable the organizational view feature for AWS Trusted Advisor",
                "accessLevel": "Write"
            },
            "UpdateNotificationPreferences": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to update notification preferences for AWS Trusted Advisor",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "checks": {
                "name": "checks",
                "url": "https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html",
                "arn": "arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to view the AWS Support plan and various AWS Trusted Advisor preferences
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeAccount() {
        this.add('trustedadvisor:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to view if the AWS account has enabled or disabled AWS Trusted Advisor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeAccountAccess() {
        this.add('trustedadvisor:DescribeAccountAccess');
        return this;
    }
    /**
     * Grants permission to view details for the check items
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckItems() {
        this.add('trustedadvisor:DescribeCheckItems');
        return this;
    }
    /**
     * Grants permission to view the refresh statuses for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckRefreshStatuses() {
        this.add('trustedadvisor:DescribeCheckRefreshStatuses');
        return this;
    }
    /**
     * Grants permission to view AWS Trusted Advisor check summaries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckSummaries() {
        this.add('trustedadvisor:DescribeCheckSummaries');
        return this;
    }
    /**
     * Grants permission to view details for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeChecks() {
        this.add('trustedadvisor:DescribeChecks');
        return this;
    }
    /**
     * Grants permission to view the notification preferences for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeNotificationPreferences() {
        this.add('trustedadvisor:DescribeNotificationPreferences');
        return this;
    }
    /**
     * Grants permission to view if the AWS account meets the requirements to enable the organizational view feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeOrganization() {
        this.add('trustedadvisor:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to view the linked AWS accounts that are in the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeOrganizationAccounts() {
        this.add('trustedadvisor:DescribeOrganizationAccounts');
        return this;
    }
    /**
     * Grants permission to view details for organizational view reports, such as the report name, runtime, date created, status, and format
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeReports() {
        this.add('trustedadvisor:DescribeReports');
        return this;
    }
    /**
     * Grants permission to view information about organizational view reports, such as the AWS Regions, check categories, check names, and resource statuses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeServiceMetadata() {
        this.add('trustedadvisor:DescribeServiceMetadata');
        return this;
    }
    /**
     * Grants permission to exclude recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toExcludeCheckItems() {
        this.add('trustedadvisor:ExcludeCheckItems');
        return this;
    }
    /**
     * Grants permission to create a report for AWS Trusted Advisor checks in your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toGenerateReport() {
        this.add('trustedadvisor:GenerateReport');
        return this;
    }
    /**
     * Grants permission to include recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toIncludeCheckItems() {
        this.add('trustedadvisor:IncludeCheckItems');
        return this;
    }
    /**
     * Grants permission to refresh an AWS Trusted Advisor check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toRefreshCheck() {
        this.add('trustedadvisor:RefreshCheck');
        return this;
    }
    /**
     * Grants permission to enable or disable AWS Trusted Advisor for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toSetAccountAccess() {
        this.add('trustedadvisor:SetAccountAccess');
        return this;
    }
    /**
     * Grants permission to enable the organizational view feature for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toSetOrganizationAccess() {
        this.add('trustedadvisor:SetOrganizationAccess');
        return this;
    }
    /**
     * Grants permission to update notification preferences for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toUpdateNotificationPreferences() {
        this.add('trustedadvisor:UpdateNotificationPreferences');
        return this;
    }
    /**
     * Adds a resource of type checks to the statement
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html
     *
     * @param categoryCode - Identifier for the categoryCode.
     * @param checkId - Identifier for the checkId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChecks(categoryCode, checkId, account, region, partition) {
        var arn = 'arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}';
        arn = arn.replace('${CategoryCode}', categoryCode);
        arn = arn.replace('${CheckId}', checkId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Trustedadvisor = Trustedadvisor;
//# sourceMappingURL=data:application/json;base64,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