"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Xray = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [xray](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsx-ray.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Xray extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [xray](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsx-ray.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'xray';
        this.actionList = {
            "BatchGetTraces": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html",
                "description": "Grants permission to retrieve a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs",
                "accessLevel": "List"
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html",
                "description": "Grants permission to create a group resource with a name and a filter expression",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSamplingRule": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html",
                "description": "Grants permission to create a rule to control sampling behavior for instrumented applications",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sampling-rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html",
                "description": "Grants permission to delete a group resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteSamplingRule": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html",
                "description": "Grants permission to delete a sampling rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sampling-rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html",
                "description": "Grants permission to retrieve the current encryption configuration for X-Ray data",
                "accessLevel": "Permissions management"
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html",
                "description": "Grants permission to retrieve group resource details",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetGroups": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html",
                "description": "Grants permission to retrieve all active group details",
                "accessLevel": "Read"
            },
            "GetInsight": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetInsight.html",
                "description": "Grants permission to retrieve the details of a specific insight",
                "accessLevel": "Read"
            },
            "GetInsightEvents": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetInsightEvents.html",
                "description": "Grants permission to retrieve the events of a specific insight",
                "accessLevel": "Read"
            },
            "GetInsightImpactGraph": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetInsightImpactGraph.html",
                "description": "Grants permission to retrieve the part of the service graph which is impacted for a specific insight",
                "accessLevel": "Read"
            },
            "GetInsightSummaries": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetInsightSummaries.html",
                "description": "Grants permission to retrieve the summary of all insights for a group and time range with optional filters",
                "accessLevel": "Read"
            },
            "GetSamplingRules": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html",
                "description": "Grants permission to retrieve all sampling rules",
                "accessLevel": "Read"
            },
            "GetSamplingStatisticSummaries": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html",
                "description": "Grants permission to retrieve information about recent sampling results for all sampling rules",
                "accessLevel": "Read"
            },
            "GetSamplingTargets": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html",
                "description": "Grants permission to request a sampling quota for rules that the service is using to sample requests",
                "accessLevel": "Read"
            },
            "GetServiceGraph": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html",
                "description": "Grants permission to retrieve a document that describes services that process incoming requests, and downstream services that they call as a result",
                "accessLevel": "Read"
            },
            "GetTimeSeriesServiceStatistics": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html",
                "description": "Grants permission to retrieve an aggregation of service statistics defined by a specific time range bucketed into time intervals",
                "accessLevel": "Read"
            },
            "GetTraceGraph": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html",
                "description": "Grants permission to retrieve a service graph for one or more specific trace IDs",
                "accessLevel": "Read"
            },
            "GetTraceSummaries": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html",
                "description": "Grants permission to retrieve IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for an X-Ray resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "sampling-rule": {
                        "required": false
                    }
                }
            },
            "PutEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html",
                "description": "Grants permission to update the encryption configuration for X-Ray data",
                "accessLevel": "Permissions management"
            },
            "PutTelemetryRecords": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html",
                "description": "Grants permission to send AWS X-Ray daemon telemetry to the service",
                "accessLevel": "Write"
            },
            "PutTraceSegments": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html",
                "description": "Grants permission to upload segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_TagResource.html",
                "description": "Grants permission to add tags to an X-Ray resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "sampling-rule": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_UntagResource.html",
                "description": "Grants permission to remove tags from an X-Ray resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "sampling-rule": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html",
                "description": "Grants permission to update a group resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateSamplingRule": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html",
                "description": "Grants permission to modify a sampling rule's configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sampling-rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups",
                "arn": "arn:${Partition}:xray:${Region}:${Account}:group/${GroupName}/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "sampling-rule": {
                "name": "sampling-rule",
                "url": "https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling",
                "arn": "arn:${Partition}:xray:${Region}:${Account}:sampling-rule/${SamplingRuleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to retrieve a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html
     */
    toBatchGetTraces() {
        this.add('xray:BatchGetTraces');
        return this;
    }
    /**
     * Grants permission to create a group resource with a name and a filter expression
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html
     */
    toCreateGroup() {
        this.add('xray:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a rule to control sampling behavior for instrumented applications
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html
     */
    toCreateSamplingRule() {
        this.add('xray:CreateSamplingRule');
        return this;
    }
    /**
     * Grants permission to delete a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.add('xray:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete a sampling rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html
     */
    toDeleteSamplingRule() {
        this.add('xray:DeleteSamplingRule');
        return this;
    }
    /**
     * Grants permission to retrieve the current encryption configuration for X-Ray data
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html
     */
    toGetEncryptionConfig() {
        this.add('xray:GetEncryptionConfig');
        return this;
    }
    /**
     * Grants permission to retrieve group resource details
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html
     */
    toGetGroup() {
        this.add('xray:GetGroup');
        return this;
    }
    /**
     * Grants permission to retrieve all active group details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html
     */
    toGetGroups() {
        this.add('xray:GetGroups');
        return this;
    }
    /**
     * Grants permission to retrieve the details of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsight.html
     */
    toGetInsight() {
        this.add('xray:GetInsight');
        return this;
    }
    /**
     * Grants permission to retrieve the events of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightEvents.html
     */
    toGetInsightEvents() {
        this.add('xray:GetInsightEvents');
        return this;
    }
    /**
     * Grants permission to retrieve the part of the service graph which is impacted for a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightImpactGraph.html
     */
    toGetInsightImpactGraph() {
        this.add('xray:GetInsightImpactGraph');
        return this;
    }
    /**
     * Grants permission to retrieve the summary of all insights for a group and time range with optional filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightSummaries.html
     */
    toGetInsightSummaries() {
        this.add('xray:GetInsightSummaries');
        return this;
    }
    /**
     * Grants permission to retrieve all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html
     */
    toGetSamplingRules() {
        this.add('xray:GetSamplingRules');
        return this;
    }
    /**
     * Grants permission to retrieve information about recent sampling results for all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html
     */
    toGetSamplingStatisticSummaries() {
        this.add('xray:GetSamplingStatisticSummaries');
        return this;
    }
    /**
     * Grants permission to request a sampling quota for rules that the service is using to sample requests
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html
     */
    toGetSamplingTargets() {
        this.add('xray:GetSamplingTargets');
        return this;
    }
    /**
     * Grants permission to retrieve a document that describes services that process incoming requests, and downstream services that they call as a result
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html
     */
    toGetServiceGraph() {
        this.add('xray:GetServiceGraph');
        return this;
    }
    /**
     * Grants permission to retrieve an aggregation of service statistics defined by a specific time range bucketed into time intervals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html
     */
    toGetTimeSeriesServiceStatistics() {
        this.add('xray:GetTimeSeriesServiceStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve a service graph for one or more specific trace IDs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html
     */
    toGetTraceGraph() {
        this.add('xray:GetTraceGraph');
        return this;
    }
    /**
     * Grants permission to retrieve IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html
     */
    toGetTraceSummaries() {
        this.add('xray:GetTraceSummaries');
        return this;
    }
    /**
     * Grants permission to list tags for an X-Ray resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('xray:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to update the encryption configuration for X-Ray data
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html
     */
    toPutEncryptionConfig() {
        this.add('xray:PutEncryptionConfig');
        return this;
    }
    /**
     * Grants permission to send AWS X-Ray daemon telemetry to the service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html
     */
    toPutTelemetryRecords() {
        this.add('xray:PutTelemetryRecords');
        return this;
    }
    /**
     * Grants permission to upload segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html
     */
    toPutTraceSegments() {
        this.add('xray:PutTraceSegments');
        return this;
    }
    /**
     * Grants permission to add tags to an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.add('xray:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.add('xray:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.add('xray:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to modify a sampling rule's configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html
     */
    toUpdateSamplingRule() {
        this.add('xray:UpdateSamplingRule');
        return this;
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups
     *
     * @param groupName - Identifier for the groupName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupName, id, account, region, partition) {
        var arn = 'arn:${Partition}:xray:${Region}:${Account}:group/${GroupName}/${Id}';
        arn = arn.replace('${GroupName}', groupName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sampling-rule to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling
     *
     * @param samplingRuleName - Identifier for the samplingRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSamplingRule(samplingRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:xray:${Region}:${Account}:sampling-rule/${SamplingRuleName}';
        arn = arn.replace('${SamplingRuleName}', samplingRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Xray = Xray;
//# sourceMappingURL=data:application/json;base64,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