"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iam = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iam](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_identityandaccessmanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iam extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iam](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_identityandaccessmanagement.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iam';
        this.actionList = {
            "AddClientIDToOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddClientIDToOpenIDConnectProvider.html",
                "description": "Grants permission to add a new client ID (audience) to the list of registered IDs for the specified IAM OpenID Connect (OIDC) provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "AddRoleToInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddRoleToInstanceProfile.html",
                "description": "Grants permission to add an IAM role to the specified instance profile",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "AddUserToGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddUserToGroup.html",
                "description": "Grants permission to add an IAM user to the specified IAM group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "AttachGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachGroupPolicy.html",
                "description": "Grants permission to attach a managed policy to the specified IAM group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN"
                ]
            },
            "AttachRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachRolePolicy.html",
                "description": "Grants permission to attach a managed policy to the specified IAM role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "AttachUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachUserPolicy.html",
                "description": "Grants permission to attach a managed policy to the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "ChangePassword": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ChangePassword.html",
                "description": "Grants permission for an IAM user to to change their own password",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateAccessKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccessKey.html",
                "description": "Grants permission to create access key and secret access key for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateAccountAlias": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccountAlias.html",
                "description": "Grants permission to create an alias for your AWS account",
                "accessLevel": "Write"
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateGroup.html",
                "description": "Grants permission to create a new group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateInstanceProfile.html",
                "description": "Grants permission to create a new instance profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "CreateLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateLoginProfile.html",
                "description": "Grants permission to create a password for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateOpenIDConnectProvider.html",
                "description": "Grants permission to create an IAM resource that describes an identity provider (IdP) that supports OpenID Connect (OIDC)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicy.html",
                "description": "Grants permission to create a new managed policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreatePolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicyVersion.html",
                "description": "Grants permission to create a new version of the specified managed policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreateRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html",
                "description": "Grants permission to create a new role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "CreateSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateSAMLProvider.html",
                "description": "Grants permission to create an IAM resource that describes an identity provider (IdP) that supports SAML 2.0",
                "accessLevel": "Write",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "CreateServiceLinkedRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceLinkedRole.html",
                "description": "Grants permission to create an IAM role that allows an AWS service to perform actions on your behalf",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:AWSServiceName"
                ]
            },
            "CreateServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceSpecificCredential.html",
                "description": "Grants permission to create a new service-specific credential for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a new IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "CreateVirtualMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateVirtualMFADevice.html",
                "description": "Grants permission to create a new virtual MFA device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mfa": {
                        "required": true
                    }
                }
            },
            "DeactivateMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeactivateMFADevice.html",
                "description": "Grants permission to deactivate the specified MFA device and remove its association with the IAM user for which it was originally enabled",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteAccessKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccessKey.html",
                "description": "Grants permission to delete the access key pair that is associated with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteAccountAlias": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountAlias.html",
                "description": "Grants permission to delete the specified AWS account alias",
                "accessLevel": "Write"
            },
            "DeleteAccountPasswordPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountPasswordPolicy.html",
                "description": "Grants permission to delete the password policy for the AWS account",
                "accessLevel": "Permissions management"
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroup.html",
                "description": "Grants permission to delete the specified IAM group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroupPolicy.html",
                "description": "Grants permission to delete the specified inline policy from its group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteInstanceProfile.html",
                "description": "Grants permission to delete the specified instance profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "DeleteLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteLoginProfile.html",
                "description": "Grants permission to delete the password for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteOpenIDConnectProvider.html",
                "description": "Grants permission to delete an OpenID Connect identity provider (IdP) resource object in IAM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicy.html",
                "description": "Grants permission to delete the specified managed policy and remove it from any IAM entities (users, groups, or roles) to which it is attached",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeletePolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicyVersion.html",
                "description": "Grants permission to delete a version from the specified managed policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeleteRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRole.html",
                "description": "Grants permission to delete the specified role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "DeleteRolePermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePermissionsBoundary.html",
                "description": "Grants permission to remove the permissions boundary from a role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePolicy.html",
                "description": "Grants permission to delete the specified inline policy from the specified role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSAMLProvider.html",
                "description": "Grants permission to delete a SAML provider resource in IAM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "DeleteSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSSHPublicKey.html",
                "description": "Grants permission to delete the specified SSH public key",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServerCertificate.html",
                "description": "Grants permission to delete the specified server certificate",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "DeleteServiceLinkedRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceLinkedRole.html",
                "description": "Grants permission to delete an IAM role that is linked to a specific AWS service, if the service is no longer using it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "DeleteServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceSpecificCredential.html",
                "description": "Grants permission to delete the specified service-specific credential for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteSigningCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSigningCertificate.html",
                "description": "Grants permission to delete a signing certificate that is associated with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUser.html",
                "description": "Grants permission to delete the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUserPermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPermissionsBoundary.html",
                "description": "Grants permission to remove the permissions boundary from the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPolicy.html",
                "description": "Grants permission to delete the specified inline policy from an IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteVirtualMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteVirtualMFADevice.html",
                "description": "Grants permission to delete a virtual MFA device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mfa": {
                        "required": false
                    },
                    "sms-mfa": {
                        "required": false
                    }
                }
            },
            "DetachGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachGroupPolicy.html",
                "description": "Grants permission to detach a managed policy from the specified IAM group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN"
                ]
            },
            "DetachRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachRolePolicy.html",
                "description": "Grants permission to detach a managed policy from the specified role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "DetachUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachUserPolicy.html",
                "description": "Grants permission to detach a managed policy from the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "EnableMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_EnableMFADevice.html",
                "description": "Grants permission to enable an MFA device and associate it with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GenerateCredentialReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateCredentialReport.html",
                "description": "Grants permission to generate a credential report for the AWS account",
                "accessLevel": "Read"
            },
            "GenerateOrganizationsAccessReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateOrganizationsAccessReport.html",
                "description": "Grants permission to generate an access report for an AWS Organizations entity",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:DescribePolicy",
                    "organizations:ListChildren",
                    "organizations:ListParents",
                    "organizations:ListPoliciesForTarget",
                    "organizations:ListRoots",
                    "organizations:ListTargetsForPolicy"
                ],
                "resourceTypes": {
                    "access-report": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:OrganizationsPolicyId"
                ]
            },
            "GenerateServiceLastAccessedDetails": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateServiceLastAccessedDetails.html",
                "description": "Grants permission to generate a service last accessed data report for an IAM resource",
                "accessLevel": "Read"
            },
            "GetAccessKeyLastUsed": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccessKeyLastUsed.html",
                "description": "Grants permission to retrieve information about when the specified access key was last used",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetAccountAuthorizationDetails": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountAuthorizationDetails.html",
                "description": "Grants permission to retrieve information about all IAM users, groups, roles, and policies in your AWS account, including their relationships to one another",
                "accessLevel": "Read"
            },
            "GetAccountPasswordPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountPasswordPolicy.html",
                "description": "Grants permission to retrieve the password policy for the AWS account",
                "accessLevel": "Read"
            },
            "GetAccountSummary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountSummary.html",
                "description": "Grants permission to retrieve information about IAM entity usage and IAM quotas in the AWS account",
                "accessLevel": "List"
            },
            "GetContextKeysForCustomPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForCustomPolicy.html",
                "description": "Grants permission to retrieve a list of all of the context keys that are referenced in the specified policy",
                "accessLevel": "Read"
            },
            "GetContextKeysForPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForPrincipalPolicy.html",
                "description": "Grants permission to retrieve a list of all context keys that are referenced in all IAM policies that are attached to the specified IAM identity (user, group, or role)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "role": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                }
            },
            "GetCredentialReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetCredentialReport.html",
                "description": "Grants permission to retrieve a credential report for the AWS account",
                "accessLevel": "Read"
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroup.html",
                "description": "Grants permission to retrieve a list of IAM users in the specified IAM group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroupPolicy.html",
                "description": "Grants permission to retrieve an inline policy document that is embedded in the specified IAM group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetInstanceProfile.html",
                "description": "Grants permission to retrieve information about the specified instance profile, including the instance profile's path, GUID, ARN, and role",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "GetLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetLoginProfile.html",
                "description": "Grants permission to retrieve the user name and password creation date for the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOpenIDConnectProvider.html",
                "description": "Grants permission to retrieve information about the specified OpenID Connect (OIDC) provider resource in IAM",
                "accessLevel": "Read",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "GetOrganizationsAccessReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOrganizationsAccessReport.html",
                "description": "Grants permission to retrieve an AWS Organizations access report",
                "accessLevel": "Read"
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicy.html",
                "description": "Grants permission to retrieve information about the specified managed policy, including the policy's default version and the total number of identities to which the policy is attached",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetPolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicyVersion.html",
                "description": "Grants permission to retrieve information about a version of the specified managed policy, including the policy document",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRole.html",
                "description": "Grants permission to retrieve information about the specified role, including the role's path, GUID, ARN, and the role's trust policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "GetRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRolePolicy.html",
                "description": "Grants permission to retrieve an inline policy document that is embedded with the specified IAM role",
                "accessLevel": "Read",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "GetSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSAMLProvider.html",
                "description": "Grants permission to retrieve the SAML provider metadocument that was uploaded when the IAM SAML provider resource was created or updated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "GetSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSSHPublicKey.html",
                "description": "Grants permission to retrieve the specified SSH public key, including metadata about the key",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServerCertificate.html",
                "description": "Grants permission to retrieve information about the specified server certificate stored in IAM",
                "accessLevel": "Read",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "GetServiceLastAccessedDetails": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetails.html",
                "description": "Grants permission to retrieve information about the service last accessed data report",
                "accessLevel": "Read"
            },
            "GetServiceLastAccessedDetailsWithEntities": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetailsWithEntities.html",
                "description": "Grants permission to retrieve information about the entities from the service last accessed data report",
                "accessLevel": "Read"
            },
            "GetServiceLinkedRoleDeletionStatus": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLinkedRoleDeletionStatus.html",
                "description": "Grants permission to retrieve an IAM service-linked role deletion status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "GetUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUser.html",
                "description": "Grants permission to retrieve information about the specified IAM user, including the user's creation date, path, unique ID, and ARN",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUserPolicy.html",
                "description": "Grants permission to retrieve an inline policy document that is embedded in the specified IAM user",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListAccessKeys": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccessKeys.html",
                "description": "Grants permission to list information about the access key IDs that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListAccountAliases": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccountAliases.html",
                "description": "Grants permission to list the account alias that is associated with the AWS account",
                "accessLevel": "List"
            },
            "ListAttachedGroupPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedGroupPolicies.html",
                "description": "Grants permission to list all managed policies that are attached to the specified IAM group",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListAttachedRolePolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedRolePolicies.html",
                "description": "Grants permission to list all managed policies that are attached to the specified IAM role",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListAttachedUserPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedUserPolicies.html",
                "description": "Grants permission to list all managed policies that are attached to the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListEntitiesForPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListEntitiesForPolicy.html",
                "description": "Grants permission to list all IAM identities to which the specified managed policy is attached",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListGroupPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupPolicies.html",
                "description": "Grants permission to list the names of the inline policies that are embedded in the specified IAM group",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroups.html",
                "description": "Grants permission to list the IAM groups that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListGroupsForUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupsForUser.html",
                "description": "Grants permission to list the IAM groups that the specified IAM user belongs to",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListInstanceProfiles": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfiles.html",
                "description": "Grants permission to list the instance profiles that have the specified path prefix",
                "accessLevel": "List",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "ListInstanceProfilesForRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfilesForRole.html",
                "description": "Grants permission to list the instance profiles that have the specified associated IAM role",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListMFADevices": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListMFADevices.html",
                "description": "Grants permission to list the MFA devices for an IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": false
                    }
                }
            },
            "ListOpenIDConnectProviders": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListOpenIDConnectProviders.html",
                "description": "Grants permission to list information about the IAM OpenID Connect (OIDC) provider resource objects that are defined in the AWS account",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicies.html",
                "description": "Grants permission to list all managed policies",
                "accessLevel": "List"
            },
            "ListPoliciesGrantingServiceAccess": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPoliciesGrantingServiceAccess.html",
                "description": "Grants permission to list information about the policies that grant an entity access to a specific service",
                "accessLevel": "List"
            },
            "ListPolicyVersions": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicyVersions.html",
                "description": "Grants permission to list information about the versions of the specified managed policy, including the version that is currently set as the policy's default version",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListRolePolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRolePolicies.html",
                "description": "Grants permission to list the names of the inline policies that are embedded in the specified IAM role",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListRoleTags": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoleTags.html",
                "description": "Grants permission to list the tags that are attached to the specified IAM role.",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListRoles": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoles.html",
                "description": "Grants permission to list the IAM roles that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListSAMLProviders": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSAMLProviders.html",
                "description": "Grants permission to list the SAML provider resources in IAM",
                "accessLevel": "List"
            },
            "ListSSHPublicKeys": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSSHPublicKeys.html",
                "description": "Grants permission to list information about the SSH public keys that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListServerCertificates": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServerCertificates.html",
                "description": "Grants permission to list the server certificates that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListServiceSpecificCredentials": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServiceSpecificCredentials.html",
                "description": "Grants permission to list the service-specific credentials that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListSigningCertificates": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSigningCertificates.html",
                "description": "Grants permission to list information about the signing certificates that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUserPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserPolicies.html",
                "description": "Grants permission to list the names of the inline policies that are embedded in the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUserTags": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserTags.html",
                "description": "Grants permission to list the tags that are attached to the specified IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUsers.html",
                "description": "Grants permission to list the IAM users that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListVirtualMFADevices": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListVirtualMFADevices.html",
                "description": "Grants permission to list virtual MFA devices by assignment status",
                "accessLevel": "List"
            },
            "PassRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html",
                "description": "Grants permission to pass a role to a service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:AssociatedResourceArn",
                    "iam:PassedToService"
                ]
            },
            "PutGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutGroupPolicy.html",
                "description": "Grants permission to create or update an inline policy document that is embedded in the specified IAM group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "PutRolePermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePermissionsBoundary.html",
                "description": "Grants permission to set a managed policy as a permissions boundary for a role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "PutRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePolicy.html",
                "description": "Grants permission to create or update an inline policy document that is embedded in the specified IAM role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "PutUserPermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPermissionsBoundary.html",
                "description": "Grants permission to set a managed policy as a permissions boundary for an IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "PutUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPolicy.html",
                "description": "Grants permission to create or update an inline policy document that is embedded in the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "RemoveClientIDFromOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveClientIDFromOpenIDConnectProvider.html",
                "description": "Grants permission to remove the client ID (audience) from the list of client IDs in the specified IAM OpenID Connect (OIDC) provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "RemoveRoleFromInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveRoleFromInstanceProfile.html",
                "description": "Grants permission to remove an IAM role from the specified EC2 instance profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "RemoveUserFromGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveUserFromGroup.html",
                "description": "Grants permission to remove an IAM user from the specified group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ResetServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResetServiceSpecificCredential.html",
                "description": "Grants permission to reset the password for an existing service-specific credential for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ResyncMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResyncMFADevice.html",
                "description": "Grants permission to synchronize the specified MFA device with its IAM entity (user or role)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "SetDefaultPolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetDefaultPolicyVersion.html",
                "description": "Grants permission to set the version of the specified policy as the policy's default version",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "SetSecurityTokenServicePreferences": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetSecurityTokenServicePreferences.html",
                "description": "Grants permission to set the STS global endpoint token version",
                "accessLevel": "Write"
            },
            "SimulateCustomPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulateCustomPolicy.html",
                "description": "Grants permission to simulate whether an identity-based policy or resource-based policy provides permissions for specific API operations and resources",
                "accessLevel": "Read"
            },
            "SimulatePrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulatePrincipalPolicy.html",
                "description": "Grants permission to simulate whether an identity-based policy that is attached to a specified IAM entity (user or role) provides permissions for specific API operations and resources",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "role": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                }
            },
            "TagRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagRole.html",
                "description": "Grants permission to add tags to an IAM role.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "TagUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagUser.html",
                "description": "Grants permission to add tags to an IAM user.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UntagRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagRole.html",
                "description": "Grants permission to remove the specified tags from the role.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UntagUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagUser.html",
                "description": "Grants permission to remove the specified tags from the user.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateAccessKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccessKey.html",
                "description": "Grants permission to update the status of the specified access key as Active or Inactive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateAccountPasswordPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccountPasswordPolicy.html",
                "description": "Grants permission to update the password policy settings for the AWS account",
                "accessLevel": "Write"
            },
            "UpdateAssumeRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAssumeRolePolicy.html",
                "description": "Grants permission to update the policy that grants an IAM entity permission to assume a role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateGroup.html",
                "description": "Grants permission to update the name or path of the specified IAM group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateLoginProfile.html",
                "description": "Grants permission to change the password for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateOpenIDConnectProviderThumbprint": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateOpenIDConnectProviderThumbprint.html",
                "description": "Grants permission to update the entire list of server certificate thumbprints that are associated with an OpenID Connect (OIDC) provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "UpdateRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRole.html",
                "description": "Grants permission to update the description or maximum session duration setting of a role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UpdateRoleDescription": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRoleDescription.html",
                "description": "Grants permission to update only the description of a role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UpdateSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSAMLProvider.html",
                "description": "Grants permission to update the metadata document for an existing SAML provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "UpdateSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSSHPublicKey.html",
                "description": "Grants permission to update the status of an IAM user's SSH public key to active or inactive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServerCertificate.html",
                "description": "Grants permission to update the name or the path of the specified server certificate stored in IAM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "UpdateServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServiceSpecificCredential.html",
                "description": "Grants permission to update the status of a service-specific credential to active or inactive for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateSigningCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSigningCertificate.html",
                "description": "Grants permission to update the status of the specified user signing certificate to active or disabled",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateUser.html",
                "description": "Grants permission to update the name or the path of the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UploadSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSSHPublicKey.html",
                "description": "Grants permission to upload an SSH public key and associate it with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UploadServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadServerCertificate.html",
                "description": "Grants permission to upload a server certificate entity for the AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "UploadSigningCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSigningCertificate.html",
                "description": "Grants permission to upload an X.509 signing certificate and associate it with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "access-report": {
                "name": "access-report",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html",
                "arn": "arn:${Partition}:iam::${Account}:access-report/${EntityPath}",
                "conditionKeys": []
            },
            "assumed-role": {
                "name": "assumed-role",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_use-resources.html",
                "arn": "arn:${Partition}:iam::${Account}:assumed-role/${RoleName}/${RoleSessionName}",
                "conditionKeys": []
            },
            "federated-user": {
                "name": "federated-user",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html",
                "arn": "arn:${Partition}:iam::${Account}:federated-user/${UserName}",
                "conditionKeys": []
            },
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html",
                "arn": "arn:${Partition}:iam::${Account}:group/${GroupNameWithPath}",
                "conditionKeys": []
            },
            "instance-profile": {
                "name": "instance-profile",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_switch-role-ec2_instance-profiles.html",
                "arn": "arn:${Partition}:iam::${Account}:instance-profile/${InstanceProfileNameWithPath}",
                "conditionKeys": []
            },
            "mfa": {
                "name": "mfa",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html",
                "arn": "arn:${Partition}:iam::${Account}:mfa/${MfaTokenIdWithPath}",
                "conditionKeys": []
            },
            "oidc-provider": {
                "name": "oidc-provider",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html",
                "arn": "arn:${Partition}:iam::${Account}:oidc-provider/${OidcProviderName}",
                "conditionKeys": []
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_managed-vs-inline.html",
                "arn": "arn:${Partition}:iam::${Account}:policy/${PolicyNameWithPath}",
                "conditionKeys": []
            },
            "role": {
                "name": "role",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html",
                "arn": "arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}",
                "conditionKeys": [
                    "iam:ResourceTag/${TagKey}"
                ]
            },
            "saml-provider": {
                "name": "saml-provider",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html",
                "arn": "arn:${Partition}:iam::${Account}:saml-provider/${SamlProviderName}",
                "conditionKeys": []
            },
            "server-certificate": {
                "name": "server-certificate",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html",
                "arn": "arn:${Partition}:iam::${Account}:server-certificate/${CertificateNameWithPath}",
                "conditionKeys": []
            },
            "sms-mfa": {
                "name": "sms-mfa",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa_enable_sms.html",
                "arn": "arn:${Partition}:iam::${Account}:sms-mfa/${MfaTokenIdWithPath}",
                "conditionKeys": []
            },
            "user": {
                "name": "user",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html",
                "arn": "arn:${Partition}:iam::${Account}:user/${UserNameWithPath}",
                "conditionKeys": [
                    "iam:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to add a new client ID (audience) to the list of registered IDs for the specified IAM OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddClientIDToOpenIDConnectProvider.html
     */
    toAddClientIDToOpenIDConnectProvider() {
        this.add('iam:AddClientIDToOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to add an IAM role to the specified instance profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddRoleToInstanceProfile.html
     */
    toAddRoleToInstanceProfile() {
        this.add('iam:AddRoleToInstanceProfile');
        return this;
    }
    /**
     * Grants permission to add an IAM user to the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddUserToGroup.html
     */
    toAddUserToGroup() {
        this.add('iam:AddUserToGroup');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM group
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachGroupPolicy.html
     */
    toAttachGroupPolicy() {
        this.add('iam:AttachGroupPolicy');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachRolePolicy.html
     */
    toAttachRolePolicy() {
        this.add('iam:AttachRolePolicy');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachUserPolicy.html
     */
    toAttachUserPolicy() {
        this.add('iam:AttachUserPolicy');
        return this;
    }
    /**
     * Grants permission for an IAM user to to change their own password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ChangePassword.html
     */
    toChangePassword() {
        this.add('iam:ChangePassword');
        return this;
    }
    /**
     * Grants permission to create access key and secret access key for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccessKey.html
     */
    toCreateAccessKey() {
        this.add('iam:CreateAccessKey');
        return this;
    }
    /**
     * Grants permission to create an alias for your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccountAlias.html
     */
    toCreateAccountAlias() {
        this.add('iam:CreateAccountAlias');
        return this;
    }
    /**
     * Grants permission to create a new group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.add('iam:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a new instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateInstanceProfile.html
     */
    toCreateInstanceProfile() {
        this.add('iam:CreateInstanceProfile');
        return this;
    }
    /**
     * Grants permission to create a password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateLoginProfile.html
     */
    toCreateLoginProfile() {
        this.add('iam:CreateLoginProfile');
        return this;
    }
    /**
     * Grants permission to create an IAM resource that describes an identity provider (IdP) that supports OpenID Connect (OIDC)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateOpenIDConnectProvider.html
     */
    toCreateOpenIDConnectProvider() {
        this.add('iam:CreateOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to create a new managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        this.add('iam:CreatePolicy');
        return this;
    }
    /**
     * Grants permission to create a new version of the specified managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicyVersion.html
     */
    toCreatePolicyVersion() {
        this.add('iam:CreatePolicyVersion');
        return this;
    }
    /**
     * Grants permission to create a new role
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html
     */
    toCreateRole() {
        this.add('iam:CreateRole');
        return this;
    }
    /**
     * Grants permission to create an IAM resource that describes an identity provider (IdP) that supports SAML 2.0
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateSAMLProvider.html
     */
    toCreateSAMLProvider() {
        this.add('iam:CreateSAMLProvider');
        return this;
    }
    /**
     * Grants permission to create an IAM role that allows an AWS service to perform actions on your behalf
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAWSServiceName()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceLinkedRole.html
     */
    toCreateServiceLinkedRole() {
        this.add('iam:CreateServiceLinkedRole');
        return this;
    }
    /**
     * Grants permission to create a new service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceSpecificCredential.html
     */
    toCreateServiceSpecificCredential() {
        this.add('iam:CreateServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to create a new IAM user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.add('iam:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a new virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateVirtualMFADevice.html
     */
    toCreateVirtualMFADevice() {
        this.add('iam:CreateVirtualMFADevice');
        return this;
    }
    /**
     * Grants permission to deactivate the specified MFA device and remove its association with the IAM user for which it was originally enabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeactivateMFADevice.html
     */
    toDeactivateMFADevice() {
        this.add('iam:DeactivateMFADevice');
        return this;
    }
    /**
     * Grants permission to delete the access key pair that is associated with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccessKey.html
     */
    toDeleteAccessKey() {
        this.add('iam:DeleteAccessKey');
        return this;
    }
    /**
     * Grants permission to delete the specified AWS account alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountAlias.html
     */
    toDeleteAccountAlias() {
        this.add('iam:DeleteAccountAlias');
        return this;
    }
    /**
     * Grants permission to delete the password policy for the AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountPasswordPolicy.html
     */
    toDeleteAccountPasswordPolicy() {
        this.add('iam:DeleteAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.add('iam:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from its group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroupPolicy.html
     */
    toDeleteGroupPolicy() {
        this.add('iam:DeleteGroupPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteInstanceProfile.html
     */
    toDeleteInstanceProfile() {
        this.add('iam:DeleteInstanceProfile');
        return this;
    }
    /**
     * Grants permission to delete the password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteLoginProfile.html
     */
    toDeleteLoginProfile() {
        this.add('iam:DeleteLoginProfile');
        return this;
    }
    /**
     * Grants permission to delete an OpenID Connect identity provider (IdP) resource object in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteOpenIDConnectProvider.html
     */
    toDeleteOpenIDConnectProvider() {
        this.add('iam:DeleteOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to delete the specified managed policy and remove it from any IAM entities (users, groups, or roles) to which it is attached
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.add('iam:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to delete a version from the specified managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicyVersion.html
     */
    toDeletePolicyVersion() {
        this.add('iam:DeletePolicyVersion');
        return this;
    }
    /**
     * Grants permission to delete the specified role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRole.html
     */
    toDeleteRole() {
        this.add('iam:DeleteRole');
        return this;
    }
    /**
     * Grants permission to remove the permissions boundary from a role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePermissionsBoundary.html
     */
    toDeleteRolePermissionsBoundary() {
        this.add('iam:DeleteRolePermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from the specified role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePolicy.html
     */
    toDeleteRolePolicy() {
        this.add('iam:DeleteRolePolicy');
        return this;
    }
    /**
     * Grants permission to delete a SAML provider resource in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSAMLProvider.html
     */
    toDeleteSAMLProvider() {
        this.add('iam:DeleteSAMLProvider');
        return this;
    }
    /**
     * Grants permission to delete the specified SSH public key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSSHPublicKey.html
     */
    toDeleteSSHPublicKey() {
        this.add('iam:DeleteSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to delete the specified server certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServerCertificate.html
     */
    toDeleteServerCertificate() {
        this.add('iam:DeleteServerCertificate');
        return this;
    }
    /**
     * Grants permission to delete an IAM role that is linked to a specific AWS service, if the service is no longer using it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceLinkedRole.html
     */
    toDeleteServiceLinkedRole() {
        this.add('iam:DeleteServiceLinkedRole');
        return this;
    }
    /**
     * Grants permission to delete the specified service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceSpecificCredential.html
     */
    toDeleteServiceSpecificCredential() {
        this.add('iam:DeleteServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to delete a signing certificate that is associated with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSigningCertificate.html
     */
    toDeleteSigningCertificate() {
        this.add('iam:DeleteSigningCertificate');
        return this;
    }
    /**
     * Grants permission to delete the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.add('iam:DeleteUser');
        return this;
    }
    /**
     * Grants permission to remove the permissions boundary from the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPermissionsBoundary.html
     */
    toDeleteUserPermissionsBoundary() {
        this.add('iam:DeleteUserPermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from an IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPolicy.html
     */
    toDeleteUserPolicy() {
        this.add('iam:DeleteUserPolicy');
        return this;
    }
    /**
     * Grants permission to delete a virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteVirtualMFADevice.html
     */
    toDeleteVirtualMFADevice() {
        this.add('iam:DeleteVirtualMFADevice');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified IAM group
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachGroupPolicy.html
     */
    toDetachGroupPolicy() {
        this.add('iam:DetachGroupPolicy');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachRolePolicy.html
     */
    toDetachRolePolicy() {
        this.add('iam:DetachRolePolicy');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachUserPolicy.html
     */
    toDetachUserPolicy() {
        this.add('iam:DetachUserPolicy');
        return this;
    }
    /**
     * Grants permission to enable an MFA device and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_EnableMFADevice.html
     */
    toEnableMFADevice() {
        this.add('iam:EnableMFADevice');
        return this;
    }
    /**
     * Grants permission to generate a credential report for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateCredentialReport.html
     */
    toGenerateCredentialReport() {
        this.add('iam:GenerateCredentialReport');
        return this;
    }
    /**
     * Grants permission to generate an access report for an AWS Organizations entity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifOrganizationsPolicyId()
     *
     * Dependent actions:
     * - organizations:DescribePolicy
     * - organizations:ListChildren
     * - organizations:ListParents
     * - organizations:ListPoliciesForTarget
     * - organizations:ListRoots
     * - organizations:ListTargetsForPolicy
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateOrganizationsAccessReport.html
     */
    toGenerateOrganizationsAccessReport() {
        this.add('iam:GenerateOrganizationsAccessReport');
        return this;
    }
    /**
     * Grants permission to generate a service last accessed data report for an IAM resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateServiceLastAccessedDetails.html
     */
    toGenerateServiceLastAccessedDetails() {
        this.add('iam:GenerateServiceLastAccessedDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about when the specified access key was last used
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccessKeyLastUsed.html
     */
    toGetAccessKeyLastUsed() {
        this.add('iam:GetAccessKeyLastUsed');
        return this;
    }
    /**
     * Grants permission to retrieve information about all IAM users, groups, roles, and policies in your AWS account, including their relationships to one another
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountAuthorizationDetails.html
     */
    toGetAccountAuthorizationDetails() {
        this.add('iam:GetAccountAuthorizationDetails');
        return this;
    }
    /**
     * Grants permission to retrieve the password policy for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountPasswordPolicy.html
     */
    toGetAccountPasswordPolicy() {
        this.add('iam:GetAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about IAM entity usage and IAM quotas in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountSummary.html
     */
    toGetAccountSummary() {
        this.add('iam:GetAccountSummary');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all of the context keys that are referenced in the specified policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForCustomPolicy.html
     */
    toGetContextKeysForCustomPolicy() {
        this.add('iam:GetContextKeysForCustomPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all context keys that are referenced in all IAM policies that are attached to the specified IAM identity (user, group, or role)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForPrincipalPolicy.html
     */
    toGetContextKeysForPrincipalPolicy() {
        this.add('iam:GetContextKeysForPrincipalPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a credential report for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetCredentialReport.html
     */
    toGetCredentialReport() {
        this.add('iam:GetCredentialReport');
        return this;
    }
    /**
     * Grants permission to retrieve a list of IAM users in the specified IAM group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroup.html
     */
    toGetGroup() {
        this.add('iam:GetGroup');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded in the specified IAM group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroupPolicy.html
     */
    toGetGroupPolicy() {
        this.add('iam:GetGroupPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified instance profile, including the instance profile's path, GUID, ARN, and role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetInstanceProfile.html
     */
    toGetInstanceProfile() {
        this.add('iam:GetInstanceProfile');
        return this;
    }
    /**
     * Grants permission to retrieve the user name and password creation date for the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetLoginProfile.html
     */
    toGetLoginProfile() {
        this.add('iam:GetLoginProfile');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified OpenID Connect (OIDC) provider resource in IAM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOpenIDConnectProvider.html
     */
    toGetOpenIDConnectProvider() {
        this.add('iam:GetOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to retrieve an AWS Organizations access report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOrganizationsAccessReport.html
     */
    toGetOrganizationsAccessReport() {
        this.add('iam:GetOrganizationsAccessReport');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified managed policy, including the policy's default version and the total number of identities to which the policy is attached
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        this.add('iam:GetPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about a version of the specified managed policy, including the policy document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicyVersion.html
     */
    toGetPolicyVersion() {
        this.add('iam:GetPolicyVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified role, including the role's path, GUID, ARN, and the role's trust policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRole.html
     */
    toGetRole() {
        this.add('iam:GetRole');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded with the specified IAM role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRolePolicy.html
     */
    toGetRolePolicy() {
        this.add('iam:GetRolePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the SAML provider metadocument that was uploaded when the IAM SAML provider resource was created or updated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSAMLProvider.html
     */
    toGetSAMLProvider() {
        this.add('iam:GetSAMLProvider');
        return this;
    }
    /**
     * Grants permission to retrieve the specified SSH public key, including metadata about the key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSSHPublicKey.html
     */
    toGetSSHPublicKey() {
        this.add('iam:GetSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified server certificate stored in IAM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServerCertificate.html
     */
    toGetServerCertificate() {
        this.add('iam:GetServerCertificate');
        return this;
    }
    /**
     * Grants permission to retrieve information about the service last accessed data report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetails.html
     */
    toGetServiceLastAccessedDetails() {
        this.add('iam:GetServiceLastAccessedDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about the entities from the service last accessed data report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetailsWithEntities.html
     */
    toGetServiceLastAccessedDetailsWithEntities() {
        this.add('iam:GetServiceLastAccessedDetailsWithEntities');
        return this;
    }
    /**
     * Grants permission to retrieve an IAM service-linked role deletion status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLinkedRoleDeletionStatus.html
     */
    toGetServiceLinkedRoleDeletionStatus() {
        this.add('iam:GetServiceLinkedRoleDeletionStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified IAM user, including the user's creation date, path, unique ID, and ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        this.add('iam:GetUser');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded in the specified IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUserPolicy.html
     */
    toGetUserPolicy() {
        this.add('iam:GetUserPolicy');
        return this;
    }
    /**
     * Grants permission to list information about the access key IDs that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccessKeys.html
     */
    toListAccessKeys() {
        this.add('iam:ListAccessKeys');
        return this;
    }
    /**
     * Grants permission to list the account alias that is associated with the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccountAliases.html
     */
    toListAccountAliases() {
        this.add('iam:ListAccountAliases');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedGroupPolicies.html
     */
    toListAttachedGroupPolicies() {
        this.add('iam:ListAttachedGroupPolicies');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedRolePolicies.html
     */
    toListAttachedRolePolicies() {
        this.add('iam:ListAttachedRolePolicies');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedUserPolicies.html
     */
    toListAttachedUserPolicies() {
        this.add('iam:ListAttachedUserPolicies');
        return this;
    }
    /**
     * Grants permission to list all IAM identities to which the specified managed policy is attached
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListEntitiesForPolicy.html
     */
    toListEntitiesForPolicy() {
        this.add('iam:ListEntitiesForPolicy');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupPolicies.html
     */
    toListGroupPolicies() {
        this.add('iam:ListGroupPolicies');
        return this;
    }
    /**
     * Grants permission to list the IAM groups that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.add('iam:ListGroups');
        return this;
    }
    /**
     * Grants permission to list the IAM groups that the specified IAM user belongs to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupsForUser.html
     */
    toListGroupsForUser() {
        this.add('iam:ListGroupsForUser');
        return this;
    }
    /**
     * Grants permission to list the instance profiles that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfiles.html
     */
    toListInstanceProfiles() {
        this.add('iam:ListInstanceProfiles');
        return this;
    }
    /**
     * Grants permission to list the instance profiles that have the specified associated IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfilesForRole.html
     */
    toListInstanceProfilesForRole() {
        this.add('iam:ListInstanceProfilesForRole');
        return this;
    }
    /**
     * Grants permission to list the MFA devices for an IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListMFADevices.html
     */
    toListMFADevices() {
        this.add('iam:ListMFADevices');
        return this;
    }
    /**
     * Grants permission to list information about the IAM OpenID Connect (OIDC) provider resource objects that are defined in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListOpenIDConnectProviders.html
     */
    toListOpenIDConnectProviders() {
        this.add('iam:ListOpenIDConnectProviders');
        return this;
    }
    /**
     * Grants permission to list all managed policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        this.add('iam:ListPolicies');
        return this;
    }
    /**
     * Grants permission to list information about the policies that grant an entity access to a specific service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPoliciesGrantingServiceAccess.html
     */
    toListPoliciesGrantingServiceAccess() {
        this.add('iam:ListPoliciesGrantingServiceAccess');
        return this;
    }
    /**
     * Grants permission to list information about the versions of the specified managed policy, including the version that is currently set as the policy's default version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicyVersions.html
     */
    toListPolicyVersions() {
        this.add('iam:ListPolicyVersions');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRolePolicies.html
     */
    toListRolePolicies() {
        this.add('iam:ListRolePolicies');
        return this;
    }
    /**
     * Grants permission to list the tags that are attached to the specified IAM role.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoleTags.html
     */
    toListRoleTags() {
        this.add('iam:ListRoleTags');
        return this;
    }
    /**
     * Grants permission to list the IAM roles that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoles.html
     */
    toListRoles() {
        this.add('iam:ListRoles');
        return this;
    }
    /**
     * Grants permission to list the SAML provider resources in IAM
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSAMLProviders.html
     */
    toListSAMLProviders() {
        this.add('iam:ListSAMLProviders');
        return this;
    }
    /**
     * Grants permission to list information about the SSH public keys that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSSHPublicKeys.html
     */
    toListSSHPublicKeys() {
        this.add('iam:ListSSHPublicKeys');
        return this;
    }
    /**
     * Grants permission to list the server certificates that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServerCertificates.html
     */
    toListServerCertificates() {
        this.add('iam:ListServerCertificates');
        return this;
    }
    /**
     * Grants permission to list the service-specific credentials that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServiceSpecificCredentials.html
     */
    toListServiceSpecificCredentials() {
        this.add('iam:ListServiceSpecificCredentials');
        return this;
    }
    /**
     * Grants permission to list information about the signing certificates that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSigningCertificates.html
     */
    toListSigningCertificates() {
        this.add('iam:ListSigningCertificates');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserPolicies.html
     */
    toListUserPolicies() {
        this.add('iam:ListUserPolicies');
        return this;
    }
    /**
     * Grants permission to list the tags that are attached to the specified IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserTags.html
     */
    toListUserTags() {
        this.add('iam:ListUserTags');
        return this;
    }
    /**
     * Grants permission to list the IAM users that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.add('iam:ListUsers');
        return this;
    }
    /**
     * Grants permission to list virtual MFA devices by assignment status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListVirtualMFADevices.html
     */
    toListVirtualMFADevices() {
        this.add('iam:ListVirtualMFADevices');
        return this;
    }
    /**
     * Grants permission to pass a role to a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAssociatedResourceArn()
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html
     */
    toPassRole() {
        this.add('iam:PassRole');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutGroupPolicy.html
     */
    toPutGroupPolicy() {
        this.add('iam:PutGroupPolicy');
        return this;
    }
    /**
     * Grants permission to set a managed policy as a permissions boundary for a role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePermissionsBoundary.html
     */
    toPutRolePermissionsBoundary() {
        this.add('iam:PutRolePermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePolicy.html
     */
    toPutRolePolicy() {
        this.add('iam:PutRolePolicy');
        return this;
    }
    /**
     * Grants permission to set a managed policy as a permissions boundary for an IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPermissionsBoundary.html
     */
    toPutUserPermissionsBoundary() {
        this.add('iam:PutUserPermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPolicy.html
     */
    toPutUserPolicy() {
        this.add('iam:PutUserPolicy');
        return this;
    }
    /**
     * Grants permission to remove the client ID (audience) from the list of client IDs in the specified IAM OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveClientIDFromOpenIDConnectProvider.html
     */
    toRemoveClientIDFromOpenIDConnectProvider() {
        this.add('iam:RemoveClientIDFromOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to remove an IAM role from the specified EC2 instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveRoleFromInstanceProfile.html
     */
    toRemoveRoleFromInstanceProfile() {
        this.add('iam:RemoveRoleFromInstanceProfile');
        return this;
    }
    /**
     * Grants permission to remove an IAM user from the specified group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveUserFromGroup.html
     */
    toRemoveUserFromGroup() {
        this.add('iam:RemoveUserFromGroup');
        return this;
    }
    /**
     * Grants permission to reset the password for an existing service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResetServiceSpecificCredential.html
     */
    toResetServiceSpecificCredential() {
        this.add('iam:ResetServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to synchronize the specified MFA device with its IAM entity (user or role)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResyncMFADevice.html
     */
    toResyncMFADevice() {
        this.add('iam:ResyncMFADevice');
        return this;
    }
    /**
     * Grants permission to set the version of the specified policy as the policy's default version
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetDefaultPolicyVersion.html
     */
    toSetDefaultPolicyVersion() {
        this.add('iam:SetDefaultPolicyVersion');
        return this;
    }
    /**
     * Grants permission to set the STS global endpoint token version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetSecurityTokenServicePreferences.html
     */
    toSetSecurityTokenServicePreferences() {
        this.add('iam:SetSecurityTokenServicePreferences');
        return this;
    }
    /**
     * Grants permission to simulate whether an identity-based policy or resource-based policy provides permissions for specific API operations and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulateCustomPolicy.html
     */
    toSimulateCustomPolicy() {
        this.add('iam:SimulateCustomPolicy');
        return this;
    }
    /**
     * Grants permission to simulate whether an identity-based policy that is attached to a specified IAM entity (user or role) provides permissions for specific API operations and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulatePrincipalPolicy.html
     */
    toSimulatePrincipalPolicy() {
        this.add('iam:SimulatePrincipalPolicy');
        return this;
    }
    /**
     * Grants permission to add tags to an IAM role.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagRole.html
     */
    toTagRole() {
        this.add('iam:TagRole');
        return this;
    }
    /**
     * Grants permission to add tags to an IAM user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagUser.html
     */
    toTagUser() {
        this.add('iam:TagUser');
        return this;
    }
    /**
     * Grants permission to remove the specified tags from the role.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagRole.html
     */
    toUntagRole() {
        this.add('iam:UntagRole');
        return this;
    }
    /**
     * Grants permission to remove the specified tags from the user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagUser.html
     */
    toUntagUser() {
        this.add('iam:UntagUser');
        return this;
    }
    /**
     * Grants permission to update the status of the specified access key as Active or Inactive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccessKey.html
     */
    toUpdateAccessKey() {
        this.add('iam:UpdateAccessKey');
        return this;
    }
    /**
     * Grants permission to update the password policy settings for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccountPasswordPolicy.html
     */
    toUpdateAccountPasswordPolicy() {
        this.add('iam:UpdateAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to update the policy that grants an IAM entity permission to assume a role
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAssumeRolePolicy.html
     */
    toUpdateAssumeRolePolicy() {
        this.add('iam:UpdateAssumeRolePolicy');
        return this;
    }
    /**
     * Grants permission to update the name or path of the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.add('iam:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to change the password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateLoginProfile.html
     */
    toUpdateLoginProfile() {
        this.add('iam:UpdateLoginProfile');
        return this;
    }
    /**
     * Grants permission to update the entire list of server certificate thumbprints that are associated with an OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateOpenIDConnectProviderThumbprint.html
     */
    toUpdateOpenIDConnectProviderThumbprint() {
        this.add('iam:UpdateOpenIDConnectProviderThumbprint');
        return this;
    }
    /**
     * Grants permission to update the description or maximum session duration setting of a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRole.html
     */
    toUpdateRole() {
        this.add('iam:UpdateRole');
        return this;
    }
    /**
     * Grants permission to update only the description of a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRoleDescription.html
     */
    toUpdateRoleDescription() {
        this.add('iam:UpdateRoleDescription');
        return this;
    }
    /**
     * Grants permission to update the metadata document for an existing SAML provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSAMLProvider.html
     */
    toUpdateSAMLProvider() {
        this.add('iam:UpdateSAMLProvider');
        return this;
    }
    /**
     * Grants permission to update the status of an IAM user's SSH public key to active or inactive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSSHPublicKey.html
     */
    toUpdateSSHPublicKey() {
        this.add('iam:UpdateSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to update the name or the path of the specified server certificate stored in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServerCertificate.html
     */
    toUpdateServerCertificate() {
        this.add('iam:UpdateServerCertificate');
        return this;
    }
    /**
     * Grants permission to update the status of a service-specific credential to active or inactive for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServiceSpecificCredential.html
     */
    toUpdateServiceSpecificCredential() {
        this.add('iam:UpdateServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to update the status of the specified user signing certificate to active or disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSigningCertificate.html
     */
    toUpdateSigningCertificate() {
        this.add('iam:UpdateSigningCertificate');
        return this;
    }
    /**
     * Grants permission to update the name or the path of the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        this.add('iam:UpdateUser');
        return this;
    }
    /**
     * Grants permission to upload an SSH public key and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSSHPublicKey.html
     */
    toUploadSSHPublicKey() {
        this.add('iam:UploadSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to upload a server certificate entity for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadServerCertificate.html
     */
    toUploadServerCertificate() {
        this.add('iam:UploadServerCertificate');
        return this;
    }
    /**
     * Grants permission to upload an X.509 signing certificate and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSigningCertificate.html
     */
    toUploadSigningCertificate() {
        this.add('iam:UploadSigningCertificate');
        return this;
    }
    /**
     * Adds a resource of type access-report to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html
     *
     * @param entityPath - Identifier for the entityPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccessReport(entityPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:access-report/${EntityPath}';
        arn = arn.replace('${EntityPath}', entityPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assumed-role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_use-resources.html
     *
     * @param roleName - Identifier for the roleName.
     * @param roleSessionName - Identifier for the roleSessionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssumedRole(roleName, roleSessionName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:assumed-role/${RoleName}/${RoleSessionName}';
        arn = arn.replace('${RoleName}', roleName);
        arn = arn.replace('${RoleSessionName}', roleSessionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type federated-user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html
     *
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFederatedUser(userName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:federated-user/${UserName}';
        arn = arn.replace('${UserName}', userName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html
     *
     * @param groupNameWithPath - Identifier for the groupNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(groupNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:group/${GroupNameWithPath}';
        arn = arn.replace('${GroupNameWithPath}', groupNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance-profile to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_switch-role-ec2_instance-profiles.html
     *
     * @param instanceProfileNameWithPath - Identifier for the instanceProfileNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstanceProfile(instanceProfileNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:instance-profile/${InstanceProfileNameWithPath}';
        arn = arn.replace('${InstanceProfileNameWithPath}', instanceProfileNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mfa to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html
     *
     * @param mfaTokenIdWithPath - Identifier for the mfaTokenIdWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMfa(mfaTokenIdWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:mfa/${MfaTokenIdWithPath}';
        arn = arn.replace('${MfaTokenIdWithPath}', mfaTokenIdWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type oidc-provider to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     *
     * @param oidcProviderName - Identifier for the oidcProviderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOidcProvider(oidcProviderName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:oidc-provider/${OidcProviderName}';
        arn = arn.replace('${OidcProviderName}', oidcProviderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_managed-vs-inline.html
     *
     * @param policyNameWithPath - Identifier for the policyNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(policyNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:policy/${PolicyNameWithPath}';
        arn = arn.replace('${PolicyNameWithPath}', policyNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html
     *
     * @param roleNameWithPath - Identifier for the roleNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onRole(roleNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}';
        arn = arn.replace('${RoleNameWithPath}', roleNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type saml-provider to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html
     *
     * @param samlProviderName - Identifier for the samlProviderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSamlProvider(samlProviderName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:saml-provider/${SamlProviderName}';
        arn = arn.replace('${SamlProviderName}', samlProviderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type server-certificate to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html
     *
     * @param certificateNameWithPath - Identifier for the certificateNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onServerCertificate(certificateNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:server-certificate/${CertificateNameWithPath}';
        arn = arn.replace('${CertificateNameWithPath}', certificateNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sms-mfa to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa_enable_sms.html
     *
     * @param mfaTokenIdWithPath - Identifier for the mfaTokenIdWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSmsMfa(mfaTokenIdWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:sms-mfa/${MfaTokenIdWithPath}';
        arn = arn.replace('${MfaTokenIdWithPath}', mfaTokenIdWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     *
     * @param userNameWithPath - Identifier for the userNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onUser(userNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:user/${UserNameWithPath}';
        arn = arn.replace('${UserNameWithPath}', userNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the AWS service to which this role is attached
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_AWSServiceName
     *
     * Applies to actions:
     * - .toCreateServiceLinkedRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSServiceName(value, operator) {
        return this.if(`iam:AWSServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters by the resource that the role will be used on behalf of
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_AssociatedResourceArn
     *
     * Applies to actions:
     * - .toPassRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAssociatedResourceArn(value, operator) {
        return this.if(`iam:AssociatedResourceArn`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ID of an AWS Organizations policy
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_OrganizationsPolicyId
     *
     * Applies to actions:
     * - .toGenerateOrganizationsAccessReport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOrganizationsPolicyId(value, operator) {
        return this.if(`iam:OrganizationsPolicyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service to which this role is passed
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PassedToService
     *
     * Applies to actions:
     * - .toPassRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`iam:PassedToService`, value, operator || 'StringLike');
    }
    /**
     * Filters access if the specified policy is set as the permissions boundary on the IAM entity (user or role)
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PermissionsBoundary
     *
     * Applies to actions:
     * - .toAttachRolePolicy()
     * - .toAttachUserPolicy()
     * - .toCreateRole()
     * - .toCreateUser()
     * - .toDeleteRolePermissionsBoundary()
     * - .toDeleteRolePolicy()
     * - .toDeleteUserPermissionsBoundary()
     * - .toDeleteUserPolicy()
     * - .toDetachRolePolicy()
     * - .toDetachUserPolicy()
     * - .toPutRolePermissionsBoundary()
     * - .toPutRolePolicy()
     * - .toPutUserPermissionsBoundary()
     * - .toPutUserPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissionsBoundary(value, operator) {
        return this.if(`iam:PermissionsBoundary`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an IAM policy
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PolicyARN
     *
     * Applies to actions:
     * - .toAttachGroupPolicy()
     * - .toAttachRolePolicy()
     * - .toAttachUserPolicy()
     * - .toDetachGroupPolicy()
     * - .toDetachRolePolicy()
     * - .toDetachUserPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPolicyARN(value, operator) {
        return this.if(`iam:PolicyARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the tags attached to an IAM entity (user or role).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_ResourceTag
     *
     * Applies to resource types:
     * - role
     * - user
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iam:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Iam = Iam;
//# sourceMappingURL=data:application/json;base64,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