"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithCondition = void 0;
const _1_base_1 = require("./1-base");
/**
 * Adds "condition" functionality to the Policy Statement
 */
class PolicyStatementWithCondition extends _1_base_1.PolicyStatementBase {
    constructor() {
        super(...arguments);
        this.conditions = {};
        this.cdkConditionsApplied = false;
    }
    /**
     * Injects conditions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyConditions();
            return super.toJSON();
        }
        const statement = super.toJSON();
        if (this.hasConditions()) {
            statement.Condition = this.conditions;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyConditions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyConditions() {
        if (this.hasConditions() && !this.cdkConditionsApplied) {
            Object.keys(this.conditions).forEach((operator) => {
                Object.keys(this.conditions[operator]).forEach((key) => {
                    const condition = {};
                    condition[key] = this.conditions[operator][key];
                    // @ts-ignore only available after swapping 1-base
                    this.addCondition(operator, condition);
                });
            });
            this.cdkConditionsApplied = true;
        }
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.conditions).length > 0;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (typeof operator === 'undefined') {
            operator = 'StringLike';
        }
        if (!(operator in this.conditions)) {
            this.conditions[operator] = {};
        }
        this.conditions[operator][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifAwsCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator || 'ForAnyValue:StringEquals');
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:CurrentTime', value, operator || 'DateLessThanEquals');
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:EpochTime', value, operator || 'DateLessThanEquals');
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator || 'NumericLessThan');
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if(`aws:MultiFactorAuthPresent`, typeof value !== 'undefined' ? value : true, 'Bool');
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator || 'ArnLike');
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if(`aws:SecureTransport`, typeof value !== 'undefined' ? value : true, 'Bool');
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator || 'ArnLike');
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator || 'IpAddress');
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifAwsTokenIssueTime(value, operator) {
        return this.if('aws:TokenIssueTime', dateToString(value), operator || 'DateGreaterThanEquals');
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifAwsViaAWSService(value) {
        return this.if(`aws:ViaAWSService`, typeof value !== 'undefined' ? value : true, 'Bool');
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator || 'IpAddress');
    }
}
exports.PolicyStatementWithCondition = PolicyStatementWithCondition;
function dateToString(value) {
    if (typeof value.getMonth === 'function') {
        value = value.toISOString();
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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