"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * Adds "action" functionality to the Policy Statement
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        this.actionList = {};
        this.useNotActions = false;
        this.actions = [];
        this.cdkActionsApplied = false;
    }
    /**
     * Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotActions ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            statement[mode] = this.actions.filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            });
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotActions ? 'addNotActions' : 'addActions';
            const self = this;
            const uniqueActions = this.actions.filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            });
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * @param action Actions that will be added to the statement.
     */
    add(action) {
        this.actions.push(action);
        return this;
    }
    /**
     * Adds all actions of the statement provider to the statement, e.g. `actions: 'ec2:*'`
     */
    allActions() {
        if (this.servicePrefix.length) {
            this.add(`${this.servicePrefix}:*`);
        }
        else {
            this.add('*');
        }
        return this;
    }
    /**
     * Adds all actions that match one of the given regular expressions.
     *
     * @param expressions One or more regular expressions. The regular expressions need to be in [Perl/JavaScript literal style](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions) and need to be passed as strings,
     * For example:
     * ```typescript
     * allMatchingActions('/vpn/i')
     * ```
     */
    allMatchingActions(...expressions) {
        expressions.forEach((expression) => {
            for (const [name] of Object.entries(this.actionList)) {
                if (name.match(RegexParser(expression))) {
                    this.add(`${this.servicePrefix}:${name}`);
                }
            }
        });
        return this;
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) LIST to the statement
     *
     * Permission to list resources within the service to determine whether an object exists.
     *
     * Actions with this level of access can list objects but cannot see the contents of a resource.
     *
     * For example, the Amazon S3 action `ListBucket` has the List access level.
     */
    allListActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.LIST);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) READ to the statement
     *
     * Permission to read but not edit the contents and attributes of resources in the service.
     *
     * For example, the Amazon S3 actions `GetObject` and `GetBucketLocation` have the Read access level.
     */
    allReadActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.READ);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) WRITE to the statement
     *
     * Permission to create, delete, or modify resources in the service.
     *
     * For example, the Amazon S3 actions `CreateBucket`, `DeleteBucket` and `PutObject` have the Write access level.
     *
     * Write actions might also allow modifying a resource tag. However, an action that allows only changes to tags has the Tagging access level.
     */
    allWriteActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.WRITE);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) PERMISSION MANAGEMENT to the statement
     *
     * Permission to grant or modify resource permissions in the service.
     *
     * For example, most IAM and AWS Organizations actions, as well as actions like the Amazon S3 actions `PutBucketPolicy` and `DeleteBucketPolicy` have the Permissions management access level.
     */
    allPermissionManagementActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.PERMISSION_MANAGEMENT);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) TAGGING to the statement
     *
     * Permission to perform actions that only change the state of resource tags.
     *
     * For example, the IAM actions `TagRole` and `UntagRole` have the Tagging access level because they allow only tagging or untagging a role. However, the `CreateRole` action allows tagging a role resource when you create that role. Because the action does not only add a tag, it has the Write access level.
     */
    allTaggingActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.TAGGING);
    }
    addAccessLevel(accessLevel) {
        for (const [name, action] of Object.entries(this.actionList)) {
            if (Object.values(access_level_1.AccessLevel).includes(accessLevel)) {
                if (accessLevel == action.accessLevel) {
                    this.add(`${this.servicePrefix}:${name}`);
                }
            }
        }
        return this;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
//# sourceMappingURL=data:application/json;base64,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