#!/usr/bin/env python3
# -*- coding: UTF-8 -*-
#
# Copyright 2018 黎慧剑
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

"""
文件处理工具模块

@module file_tool
@file file_tool.py

"""

import sys
import os
import re
from enum import Enum
import platform
import subprocess
import shutil
try:
    import chardet
except:
    pass
# 根据当前文件路径将包路径纳入，在非安装的情况下可以引用到
sys.path.append(os.path.abspath(os.path.join(
    os.path.dirname(__file__), os.path.pardir, os.path.pardir)))
import HiveNetLib.base_tools.myzipfile as zipfile


__MOUDLE__ = 'file_tool'  # 模块名
__DESCRIPT__ = u'文件处理工具'  # 模块描述
__VERSION__ = '0.1.0'  # 版本
__AUTHOR__ = u'黎慧剑'  # 作者
__PUBLISH__ = '2018.08.29'  # 发布日期


class EnumFileSizeUnit(Enum):
    """
    文件大小单位
    @enum {int}
    """
    B = 1
    KB = 1024
    MB = 1024 * 1024
    GB = 1024 * 1024 * 1024
    TB = 1024 * 1024 * 1024 * 1024


class FileTool(object):
    """
    文件处理工具
    提供各类文件、目录相关的常用工具函数（静态方法）

    """

    @staticmethod
    def get_exefile_fullname():
        """
        获得执行主程序文件名（含路径）

        @returns {string} - 执行主程序文件名（含路径）

        @example
            filepath = FileTool.get_exefile_fullname()

        """
        return os.path.realpath(sys.argv[0])

    @staticmethod
    def get_exefile_name():
        """
        获得执行主程序文件名（不含路径，含扩展名）

        @returns {string} - 文件名（不含路径，含扩展名）

        @example
            filepath = FileTool.get_exefile_name()

        """
        return os.path.split(os.path.realpath(sys.argv[0]))[1]

    @staticmethod
    def get_exefile_name_no_ext():
        """
        获得执行主程序文件名（不含路径，不含扩展名）

        @returns {string} - 文件名（不含路径，不含扩展名）

        @example
            filepath = FileTool.get_exefile_name_no_ext()

        """
        _filename = os.path.split(os.path.realpath(sys.argv[0]))[1]
        _dot_index = _filename.rfind(".")
        if _dot_index == -1:
            return _filename
        else:
            return _filename[0: _dot_index]

    @staticmethod
    def get_exefile_path():
        """
        获得执行主程序的路径（不含文件名）

        @returns {string} - 程序路径（不含文件名，最后一个字符不为路径分隔符）

        @example
            filepath = FileTool.get_exefile_path()

        """
        return os.path.split(os.path.realpath(sys.argv[0]))[0]

    @staticmethod
    def get_file_name(file):
        """
        获得执行指定文件的文件名（不含路径，含扩展名）

        @param {string} file - 文件路径

        @returns {string} - 文件名（不含路径，含扩展名）
        """
        return os.path.split(os.path.realpath(file))[1]

    @staticmethod
    def get_file_name_no_ext(file):
        """
        获得指定文件的文件名（不含路径，不含扩展名）

        @param {string} file - 文件路径

        @returns {string} - 文件名（不含路径，不含扩展名）
        """
        _filename = os.path.split(os.path.realpath(file))[1]
        _dot_index = _filename.rfind(".")
        if _dot_index == -1:
            return _filename
        else:
            return _filename[0: _dot_index]

    @staticmethod
    def get_file_ext(file):
        """
        获得指定文件的扩展名

        @param {string} file - 文件路径

        @returns {string} - 文件扩展名
        """
        _dot_index = file.rfind(".")
        if _dot_index == -1:
            return ''
        else:
            return file[_dot_index + 1:]

    @staticmethod
    def get_file_path(file):
        """
        获得指定文件的路径（不含文件名）

        @param {string} file - 文件路径

        @returns {string} - 程序路径（不含文件名，最后一个字符不为路径分隔符）

        """
        return os.path.split(os.path.realpath(file))[0]

    @staticmethod
    def get_dir_name(path):
        """
        获取指定目录的目录名

        @param {string} path - 要获取的目录路径

        @returns {string} - 目录名

        """
        _path = os.path.realpath(path).rstrip('\\/').replace('\\', '/')
        _index = _path.rfind('/')
        if _index >= 0:
            return _path[_index + 1:]

        return _path

    @staticmethod
    def create_dir(path, exist_ok=False):
        """
        创建指定的路径

        @param {string} path - 需要创建的路径
        @param {bool} exist_ok=False - 如果路径存在是否不抛异常

        @throws {FileExistsError} - 路径存在的情况抛出文件存在异常

        @example
            FileTool.create_dir("c:/test/")

        """
        os.makedirs(path, exist_ok=exist_ok)

    @staticmethod
    def get_filelist(path='', regex_str='', is_fullname=True):
        """
        获取指定目录下的文件清单

        @param {string} path='' - 需要获取文件的目录
        @param {string} regex_str='' - 需匹配文件名的正则表达式（''代表无需匹配）
        @param {bool} is_fullname=True - 结果的文件名是否包含路径

        @returns {string[]} - 文件清单数组

        @example
            filelist = FileTools.get_filelist(path='c:\\')

        """
        _filelist = []
        _file_names = os.listdir(path)
        _pattern = None
        if len(regex_str) > 0:
            _pattern = re.compile(regex_str)
        for fn in _file_names:
            _full_filename = os.path.join(path, fn)
            if os.path.isfile(_full_filename):
                _temp_filename = fn
                if is_fullname:
                    _temp_filename = _full_filename
                if _pattern is not None:
                    if _pattern.match(fn):
                        _filelist.append(_temp_filename)
                else:
                    _filelist.append(_temp_filename)
        return _filelist

    @staticmethod
    def get_dirlist(path='', regex_str='', is_fullpath=True):
        """
        获取指定目录下的子目录清单

        @param {string} path='' - 需要获取子目录的目录
        @param {string} regex_str='' - 需匹配目录名的正则表达式（''代表无需匹配）
        @param {bool} is_fullpath=True - 结果的目录名是否包含路径

        @returns {string[]} - 目录清单数组（不带最后的分隔符）

        @throws {FileNotFoundError} - 当path不存在的情况下，会抛出该异常

        """
        _dirlist = []
        _file_names = os.listdir(path)
        _pattern = None
        if regex_str != "":
            _pattern = re.compile(regex_str)
        for fn in _file_names:
            _full_filename = os.path.join(path, fn)
            if not os.path.isfile(_full_filename):
                _temp_filename = fn
                if is_fullpath:
                    _temp_filename = _full_filename
                if _pattern is not None:
                    if _pattern.match(fn):
                        _dirlist.append(_temp_filename)
                else:
                    _dirlist.append(_temp_filename)
        return _dirlist

    @staticmethod
    def get_parent_dir(path):
        """
        获取指定路径或文件的上级路径

        @param {str} path - 路径或文件

        @retrun {str} - 上级路径
        """
        return os.path.abspath(os.path.join(path, '..'))

    @staticmethod
    def remove_dir(path):
        """
        删除指定目录（及目录下的所有文件及目录）
        由于Windows平台在处理删除目录时会存在一些权限异常，因此采用命令执行方式删除

        @param {string} path - 要删除的目录

        @throws {FileNotFoundError} - 找不到指定的路径时抛出该异常
        @throws {PermissionError} - 没有权限时抛出该异常
        @throws {NotADirectoryError} - 如果给出的路径不是目录而是文件时抛出

        """
        if platform.system() == 'Windows':
            real_path = os.path.realpath(path)
            if not os.path.exists(real_path):
                raise FileNotFoundError
            elif os.path.isfile(real_path):
                raise NotADirectoryError
            if subprocess.run('rmdir /S /Q %s' % (real_path.replace('/', '\\')), shell=True).returncode != 0:
                raise PermissionError
        else:
            shutil.rmtree(path=path, ignore_errors=False)

    @staticmethod
    def remove_file(filename):
        """
        删除指定文件

        @param {string} filename - 需要删除的文件路径

        @throws {FileNotFoundError} - 路径不是文件或文件不存在时抛出该异常
        @throws {PermissionError} - 没有权限时抛出该异常

        """
        if os.path.isfile(filename):
            os.remove(filename)
        else:
            raise FileNotFoundError

    @staticmethod
    def remove_sub_dirs(path='', regex_str=''):
        """
        删除指定目录下的子目录（及子目录下的文件和目录）

        @param {string} path='' - 需要删除的子目录的目录
        @param {string} regex_str='' - 需匹配目录名的正则表达式（''代表无需匹配）

        @throws {FileNotFoundError} - 当path不存在的情况下，会抛出该异常
        @throws {PermissionError} - 没有权限时抛出该异常
        @throws {NotADirectoryError} - 如果给出的路径不是目录而是文件时抛出

        """
        _dirs = FileTool.get_dirlist(path=path, regex_str=regex_str, is_fullpath=True)
        for _dir in _dirs:
            FileTool.remove_dir(_dir)

    @staticmethod
    def remove_files(path='', regex_str=''):
        """
        删除指定目录下的文件

        @param {string} path='' - 需要删除的文件的目录
        @param {string} regex_str='' - 需匹配文件名的正则表达式（''代表无需匹配）

        @throws {FileNotFoundError} - 当path不存在的情况下，会抛出该异常
        @throws {PermissionError} - 没有权限时抛出该异常
        @throws {NotADirectoryError} - 如果给出的路径不是目录而是文件时抛出

        """
        _files = FileTool.get_filelist(path=path, regex_str=regex_str, is_fullname=True)
        for _file in _files:
            FileTool.remove_file(_file)

    @staticmethod
    def remove_all_with_path(path='', regex_str='', with_sub_path=False):
        """
        删除指定文件夹下的所有文件及文件夹

        @param {string} path='' - 要处理的文件夹
        @param {string} regex_str='' - 文件名匹配的正则表达式
        @param {bool} with_sub_path=False - 是否包含子目录
        """
        _pattern = None
        if regex_str != "":
            _pattern = re.compile(regex_str)

        _path = os.path.realpath(path)
        _file_names = os.listdir(_path)

        for fn in _file_names:
            _match = True
            if _pattern is not None:
                if not _pattern.match(fn):
                    # 没有匹配上
                    _match = False

            _full_filename = os.path.join(_path, fn)
            if os.path.isfile(_full_filename):
                if _match:
                    # 匹配上删除文件
                    FileTool.remove_file(_full_filename)
            else:
                if _match:
                    # 匹配上，直接删除文件夹
                    FileTool.remove_dir(_full_filename)
                elif with_sub_path:
                    # 处理子文件夹
                    FileTool.remove_all_with_path(
                        path=_full_filename, regex_str=regex_str, with_sub_path=with_sub_path
                    )

    @staticmethod
    def copy_all_with_path(src_path='', dest_path='', regex_str=''):
        """
        复制指定文件夹下的所有文件及文件夹到目标文件夹

        @param {string} src_path='' - 源文件夹
        @param {string} dest_path='' - 目标文件夹
        @param {string} regex_str='' - 文件名匹配的正则表达式

        """
        _pattern = None
        _dest_path = os.path.realpath(dest_path)
        _src_path = os.path.realpath(src_path)
        _file_names = os.listdir(_src_path)
        if regex_str != "":
            _pattern = re.compile(regex_str)
        for fn in _file_names:
            if _pattern is not None:
                if not _pattern.match(fn):
                    # 获取下一个
                    continue

            _full_filename = os.path.join(_src_path, fn)
            _full_destname = os.path.join(_dest_path, fn)
            if os.path.isfile(_full_filename):
                # 复制文件
                shutil.copyfile(_full_filename, _full_destname)
            else:
                # 复制文件夹
                shutil.copytree(_full_filename, _full_destname)

    #############################
    # 文件内容处理
    #############################
    @staticmethod
    def get_file_text(filename, encoding='utf-8'):
        """
        获取文件文本

        @param {string} filename - 要获取的文件名（含路径）
        @param {string} encoding='utf-8' - 文件内容的编码
            注：如果将encoding设置为None，将使用chardet判断编码

        @return {string} - 返回全部文件内容
        """
        if encoding is not None:
            with open(filename, 'rt', encoding=encoding) as f:
                return f.read()
        else:
            # 使用chardet判断编码
            with open(filename, 'rb') as f:
                _bytes = f.read()
                _encoding = chardet.detect(_bytes)['encoding']
                if _encoding is None:
                    _encoding = 'utf-8'
                elif _encoding.startswith('ISO-8859'):
                    _encoding = 'gbk'
                return str(_bytes, encoding=_encoding)

    @staticmethod
    def create_fix_size_file(filename: str, size: int, unit=EnumFileSizeUnit.B):
        """
        生成指定大小的文件

        @param {str} filename - 要生成的文件
        @param {int} size - 大小
        @param {EnumFileSizeUnit} unit=EnumFileSizeUnit.B - 单位
        """
        with open(filename, 'w') as _file:
            _file.seek(size * unit.value - 1)   # 跳到指定位置
            _file.write('\x00')  # 一定要写入一个字符，否则无效
            _file.flush()

    @staticmethod
    def write_bytes_to_file(filename: str, data: bytes, position=0, file_obj=None):
        """
        在文件指定位置写入字节数组

        @param {str} filename - 要写入的文件
        @param {bytes} data - 写入数据
        @param {int} position=0 - 写入位置
        @param {object} file_obj=None - 已打开的文件对象，如果传入则代表使用该文件对象执行文件处理
        """
        if file_obj is not None:
            # 使用文件对象处理
            file_obj.seek(position)
            file_obj.write(bytes)
            file_obj.flush()
        else:
            with open(filename, 'wb') as _file:
                _file.seek(position)
                _file.write(bytes)
                _file.flush()

    #############################
    # zip文件处理
    #############################
    @staticmethod
    def zip(src_path, dest_path=None, dest_filename=None, mode='w',
            compression=zipfile.ZIP_DEFLATED, allowZip64=True, **kwargs):
        """
        压缩指定文件或路径

        @param {string} src_path - 要压缩的文件或目录
        @param {string} dest_path=None - 处理后的压缩包存放路径, None代表存放在src_path所在的目录下
        @param {string} dest_filename=None - 处理后的压缩包文件名, None代表使用对应的文件或目录名(增加.zip)
        @param {string} mode='w' - 打开zip文件的模式
            'w' - 表示新建一个zip文档或覆盖一个已经存在的zip文档
            'a' - 表示将数据附加到一个现存的zip文档中
        @param {int} compression=zipfile.ZIP_DEFLATED - 压缩方法，可以选的值包括：
            zipfile.ZIP_STORED = 0 - 仅打包存储（不压缩）
            zipfile.ZIP_DEFLATED = 8 - 压缩存储
        @param {bool} allowZip64=True - 当要处理的压缩包大于2G时，建议打开该开关
        @param {kwargs} - 动态参数，支持后续兼容性的扩展

        @throws {FileNotFoundError} - src_path指定的文件或目录不存在时抛出该异常
        """
        if not os.path.exists(src_path):
            raise FileNotFoundError('file or dir [%s] not found!' % src_path)

        # 处理目标文件路径
        _src_realpath = os.path.realpath(src_path)
        _src_path, _src_file = os.path.split(_src_realpath)
        if dest_path is None:
            dest_path = _src_path
        if dest_filename is None:
            dest_filename = _src_file + '.zip'

        # 创建zip文件
        _zip = zipfile.ZipFile(os.path.join(dest_path, dest_filename), mode=mode,
                               compression=compression, allowZip64=compression)

        if os.path.isfile(src_path):
            # 文件
            _zip.write(os.path.realpath(src_path), _src_file)
        else:
            # 遍历目录并写入文件
            _src_realpath = _src_realpath.replace('\\', '/')
            for root, dirs, files in os.walk(src_path):
                # 获取相对路径
                _abs_path = root.replace('\\', '/').replace(_src_realpath, '', 1)

                # 确保相对路径不能是'/'开头，否则解压检索的文件信息会有问题
                if len(_abs_path) > 0 and _abs_path[0] == '/':
                    _abs_path = _abs_path[1:]

                if len(files) == 0:
                    # 空目录，写入目录信息
                    _zip.writestr(_abs_path + '/', '')

                # 写入文件
                for filename in files:
                    _zip.write(os.path.join(root, filename), _abs_path + '/' + filename)

        # 保存压缩包
        _zip.close()

    @staticmethod
    def unzip(filename, dest_path=None, members=None, pwd=None, **kwargs):
        """
        解压缩文件到指定路径

        @param {string} filename - 要解压缩的文件
        @param {string} dest_path=None - 解压后目标路径
            注：为None时解压至文件所在路径，放入与文件名（去掉扩展名）相同的目录中
        @param {tuple|list}} members=None - 指定单独解压的文件清单(注意路径分隔符为'/')，不支持解压缩指定目录
            示例: members=['a.txt', 'path/b.txt']
        @param {bytes} pwd=None - 解压密码
            示例：pwd='123456'.encode('utf-8')
        @param {kwargs} - 动态参数，支持后续兼容性的扩展

        @throws {FileNotFoundError} - src_path指定的文件或目录不存在时抛出该异常
        """
        if not os.path.exists(filename):
            raise FileNotFoundError('file [%s] not found!' % filename)

        # 处理路径
        _src_path, _src_file = os.path.split(os.path.realpath(filename))
        if dest_path is None:
            _file_no_ext = _src_file
            _dot_index = _file_no_ext.rfind(".")
            if _dot_index != -1:
                _file_no_ext = _file_no_ext[0: _dot_index]
            dest_path = _src_path + '/' + _file_no_ext

        # 读取文件
        _zip = zipfile.ZipFile(filename)
        _zip.extractall(dest_path, members=members, pwd=pwd)
        _zip.close()

    @staticmethod
    def read_zip_file(filename, member, pwd=None, **kwargs):
        """
        读取压缩包中的某个文件的二进制数据

        @param {string} filename - 要处理的压缩文件
        @param {string} member - 要读取的包内文件名
        @param {bytes} pwd=None - 解压密码
            示例：pwd='123456'.encode('utf-8')
        @param {kwargs} - 动态参数，支持后续兼容性的扩展

        @return {bytes} - 读取到文件的二进制数据

        @throws {FileNotFoundError} - src_path指定的文件或目录不存在时抛出该异常
        """
        if not os.path.exists(filename):
            raise FileNotFoundError('file [%s] not found!' % filename)

        # 读取文件
        _zip = zipfile.ZipFile(filename)
        _bytes = _zip.read(member, pwd=pwd)
        _zip.close()
        return _bytes


if __name__ == '__main__':
    # 当程序自己独立运行时执行的操作
    # 打印版本信息
    print(('模块名：%s  -  %s\n'
           '作者：%s\n'
           '发布日期：%s\n'
           '版本：%s' % (__MOUDLE__, __DESCRIPT__, __AUTHOR__, __PUBLISH__, __VERSION__)))
