import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { IResource, Resource } from '@aws-cdk/core';
/**
 * Represents an SQS queue.
 *
 * @stability stable
 */
export interface IQueue extends IResource {
    /**
     * The ARN of this queue.
     *
     * @stability stable
     * @attribute true
     */
    readonly queueArn: string;
    /**
     * The URL of this queue.
     *
     * @stability stable
     * @attribute true
     */
    readonly queueUrl: string;
    /**
     * The name of this queue.
     *
     * @stability stable
     * @attribute true
     */
    readonly queueName: string;
    /**
     * If this queue is server-side encrypted, this is the KMS encryption key.
     *
     * @stability stable
     */
    readonly encryptionMasterKey?: kms.IKey;
    /**
     * Whether this queue is an Amazon SQS FIFO queue.
     *
     * If false, this is a standard queue.
     *
     * @stability stable
     */
    readonly fifo: boolean;
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is imported (`Queue.import`), then this is a no-op.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Grant permissions to consume messages from a queue.
     *
     * This will grant the following permissions:
     *
     *    - sqs:ChangeMessageVisibility
     *    - sqs:DeleteMessage
     *    - sqs:ReceiveMessage
     *    - sqs:GetQueueAttributes
     *    - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to.
     * @stability stable
     */
    grantConsumeMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *   - sqs:SendMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to.
     * @stability stable
     */
    grantSendMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *   - sqs:PurgeQueue
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to.
     * @stability stable
     */
    grantPurge(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the actions defined in queueActions to the identity Principal given on this SQS queue resource.
     *
     * @param grantee Principal to grant right to.
     * @param queueActions The actions to grant.
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...queueActions: string[]): iam.Grant;
}
/**
 * Reference to a new or existing Amazon SQS queue.
 *
 * @stability stable
 */
export declare abstract class QueueBase extends Resource implements IQueue {
    /**
     * The ARN of this queue.
     *
     * @stability stable
     */
    abstract readonly queueArn: string;
    /**
     * The URL of this queue.
     *
     * @stability stable
     */
    abstract readonly queueUrl: string;
    /**
     * The name of this queue.
     *
     * @stability stable
     */
    abstract readonly queueName: string;
    /**
     * If this queue is server-side encrypted, this is the KMS encryption key.
     *
     * @stability stable
     */
    abstract readonly encryptionMasterKey?: kms.IKey;
    /**
     * Whether this queue is an Amazon SQS FIFO queue.
     *
     * If false, this is a standard queue.
     *
     * @stability stable
     */
    abstract readonly fifo: boolean;
    /**
     * Controls automatic creation of policy objects.
     *
     * Set by subclasses.
     *
     * @stability stable
     */
    protected abstract readonly autoCreatePolicy: boolean;
    private policy?;
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is imported (`Queue.import`), then this is a no-op.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Grant permissions to consume messages from a queue.
     *
     * This will grant the following permissions:
     *
     *    - sqs:ChangeMessageVisibility
     *    - sqs:DeleteMessage
     *    - sqs:ReceiveMessage
     *    - sqs:GetQueueAttributes
     *    - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to.
     * @stability stable
     */
    grantConsumeMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *   - sqs:SendMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to.
     * @stability stable
     */
    grantSendMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *   - sqs:PurgeQueue
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to.
     * @stability stable
     */
    grantPurge(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the actions defined in queueActions to the identity Principal given on this SQS queue resource.
     *
     * @param grantee Principal to grant right to.
     * @param actions The actions to grant.
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * Reference to a queue.
 *
 * @stability stable
 */
export interface QueueAttributes {
    /**
     * The ARN of the queue.
     *
     * @stability stable
     */
    readonly queueArn: string;
    /**
     * The URL of the queue.
     *
     * @default - 'https://sqs.<region-endpoint>/<account-ID>/<queue-name>'
     * @see https://docs.aws.amazon.com/sdk-for-net/v2/developer-guide/QueueURL.html
     * @stability stable
     */
    readonly queueUrl?: string;
    /**
     * The name of the queue.
     *
     * @default if queue name is not specified, the name will be derived from the queue ARN
     * @stability stable
     */
    readonly queueName?: string;
    /**
     * KMS encryption key, if this queue is server-side encrypted by a KMS key.
     *
     * @default - None
     * @stability stable
     */
    readonly keyArn?: string;
}
