/// @file
/// @author David Pilger <dpilger26@gmail.com>
/// [GitHub Repository](https://github.com/dpilger26/NumCpp)
///
/// License
/// Copyright 2018-2022 David Pilger
///
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this
/// software and associated documentation files(the "Software"), to deal in the Software
/// without restriction, including without limitation the rights to use, copy, modify,
/// merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
/// permit persons to whom the Software is furnished to do so, subject to the following
/// conditions :
///
/// The above copyright notice and this permission notice shall be included in all copies
/// or substantial portions of the Software.
///
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
/// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
/// PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
/// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
/// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
/// DEALINGS IN THE SOFTWARE.
///
/// Description
/// matrix psuedo-inverse
///
#pragma once

#include <string>

#include "NumCpp/Core/Internal/StaticAsserts.hpp"
#include "NumCpp/Core/Types.hpp"
#include "NumCpp/Functions/zeros.hpp"
#include "NumCpp/Linalg/svd.hpp"
#include "NumCpp/NdArray.hpp"

namespace nc
{
    namespace linalg
    {
        //============================================================================
        // Method Description:
        /// matrix psuedo-inverse
        ///
        /// NumPy Reference: https://numpy.org/doc/stable/reference/generated/numpy.linalg.pinv.html
        ///
        /// @param inArray
        /// @return NdArray
        ///
        template<typename dtype>
        NdArray<double> pinv(const NdArray<dtype>& inArray)
        {
            STATIC_ASSERT_ARITHMETIC_OR_COMPLEX(dtype);

            NdArray<double> u;
            NdArray<double> d;
            NdArray<double> v;
            svd(inArray, u, d, v);

            const auto inShape = inArray.shape();
            auto       dPlus   = nc::zeros<double>(inShape.cols, inShape.rows); // transpose

            for (uint32 i = 0; i < d.shape().rows; ++i)
            {
                dPlus(i, i) = 1.0 / d(i, i);
            }

            return v.transpose().dot(dPlus).dot(u.transpose());
        }
    } // namespace linalg
} // namespace nc