
from . import config
import logging
import requests
import os
import re
import json
import asgiref.sync
import datetime
import csv
from typing import Union

_LOGGER = logging.getLogger(__name__)

BOOL_TO_STR = { True: 'true', False: 'false'}
MAX_TO_CHECK = 100

DEFAULT_PIWIGO_CSV = 'piwigo-images.csv'
PIWIGO_BLOCK = 'Piwigo'


class GetPicUrlService:

    def __init__ ( self ):

        self.maxToCheck = config.Config.get("MaxPicturesToCheck", MAX_TO_CHECK, PIWIGO_BLOCK )
        self.baseUrl = config.Config.getMandatory( "PiwigoUrl", PIWIGO_BLOCK )

    @staticmethod
    def isDangerousDashes ( picName : str ) -> bool:
        subsplit = picName.split('-')
        tooShort = True
        for subterm in subsplit:
            if len(subterm) > 2:
                tooShort = False
                break
        return tooShort


    @staticmethod
    def cleanQuery ( picName : str ) -> str :

        # Don't split on a dash
        terms = re.split('[/,&\'. ;\\\\]', picName)
        query = ""
        for term in terms:
            if len(term) > 3 and not GetPicUrlService.isDangerousDashes(term):
                if len(query) > 1:
                    query += ' '
                query += term
        return query

    @staticmethod
    def parseResponse ( response : json, picName : str, picDate ) :
        paging = response['paging']
        count = int(paging['count'])
        if count == 0:
            return None
        if picDate and type(picDate) == datetime.datetime:
            compareDate = picDate.strftime('%Y-%m-%d %H:%M:%S')
        else:
            compareDate = picDate
        for image in response['images']:
            # url = image['element_url']
            picUrl = image['page_url']
            matches = True
            if 'file' in image:
                fileName = image['file']
                origName = picName[-len(fileName):]
                if origName != fileName:
                    matches = False
            if 'date_creation' in image:
                createDate = image['date_creation']
                if compareDate and compareDate != createDate:
                    matches = False
            if matches:
                return picUrl
        return None

    @staticmethod
    def clean ( picName : str ):
        return picName.replace( ' ', '%20').replace("'",'_').replace('@','_').replace('#','_').replace('!','_').replace(':','_').replace(';','_')

    @staticmethod
    def getDateBrackets ( picDate ) :
        if not picDate:
            return None, None
        try:
            if type(picDate) == str:
                theDate = datetime.datetime.strptime( picDate, '%Y-%m-%d %H:%M:%S')
            elif type(picDate) == datetime.datetime:
                theDate = picDate
            else:
                raise ValueError(f'Unknown type for expected picDate value: {picDate} type = {type(picDate)}')
            endDate = theDate + datetime.timedelta(seconds=1)
            startDate = theDate + datetime.timedelta(seconds=-1)
            return startDate.strftime('%Y-%m-%d %H:%M:%S'), endDate.strftime('%Y-%m-%d %H:%M:%S')
        except Exception as e:
            _LOGGER.info(f'Failed to convert date for search: {picDate} with error {e}')
            return None, None

    @staticmethod
    async def getInfoForId ( piwigoId : int ) :

        picInfo = None
        try:
            piwigo = Piwigo()
            await asgiref.sync.sync_to_async(piwigo.pwg.session.login)( username=piwigo.userId, password=piwigo.password )
            result = await asgiref.sync.sync_to_async(piwigo.pwg.images.getInfo)( image_id=piwigoId )
            name = config.dictread( result, 'name' )
            filesize = config.intread( result, 'filesize') * 1024
            width = config.intread( result, 'width' )
            height = config.intread( result, 'height')
            createDateStr = config.dictread( result, 'date_creation' )
            availableDateStr = config.dictread( result, 'date_available')
            try:
                createDate = datetime.datetime.strptime( createDateStr, '%Y-%m-%d %H:%M:%S')
            except ValueError:
                createDate = None
            try:
                availableDate = datetime.datetime.strptime( availableDateStr, '%Y-%m-%d %H:%M:%S')
            except ValueError:
                availableDate = None

            picInfo = PictureInfo( name, createDate, filesize, width, height, piwigoId )

            picInfo.comment = config.dictread( result, 'comment' )
            picInfo.availableDate = availableDate
            picInfo.author = config.dictread( result, 'author' )
            picInfo.rating_score = config.dictread( result, 'rating_score' )
            picInfo.page_url = config.dictread( result, 'page_url' )
            picInfo.element_url = config.dictread( result, 'element_url' )
            picInfo.thumb_basic = result['derivatives']['thumb']['url']
            picInfo.small = result['derivatives']['small']['url']

        except Exception as e:
            _LOGGER.warning( f'Unable to fetch picture info for {piwigoId} with error {e}', exc_info=True)

        return picInfo

    async def getUrls ( self, picName : str, picDate ) -> (str, str):

        try:
            piwigo = Piwigo( )
            await asgiref.sync.sync_to_async(piwigo.pwg.session.login)( username=piwigo.userId, password=piwigo.password )
            dateStart, dateEnd = self.getDateBrackets(picDate)
            if dateStart and dateEnd:
                response = await asgiref.sync.sync_to_async(piwigo.pwg.images.search)( query=self.cleanQuery(picName),
                                                                                       per_page=self.maxToCheck,
                                                                                       f_min_date_created=dateStart,
                                                                                       f_max_date_created=dateEnd )
            else:
                response = await asgiref.sync.sync_to_async(piwigo.pwg.images.search)( query=self.cleanQuery(picName), per_page=self.maxToCheck )
            picUrl = self.parseResponse(response, picName, picDate )
        except Exception as e:
            _LOGGER.warning( f'Unable to fetch picture URL for {picName} with error {e}', exc_info=True)
            picUrl = None
        galleryUrl = self.baseUrl + '/galleries/' + self.clean(picName)
        return picUrl, galleryUrl


class WsNotExistException(Exception):

    def __init__(self, method):
        self._method = method

    def __str__(self):
        return "Ws %s not exist" % self._method


class WsErrorException(Exception):

    def __init__(self, strerr):
        self._strerr = strerr

    def __str__(self):
        return self._strerr


class WsPiwigoException(Exception):

    def __init__(self, err, message):
        self.err = err
        self.message = message

    def __str__(self):
        return "%s : %s" % (self.err, self.message)


class Piwigo:

    def __init__ ( self ):

        self.url = config.Config.getMandatory( "PiwigoUrl", PIWIGO_BLOCK )
        self.userId = config.Config.getMandatory( "PiwigoUserId", PIWIGO_BLOCK )
        self.password = config.Config.getMandatory( "PiwigoPassword", PIWIGO_BLOCK )
        self.cookies = None

        if self.url[-1] == '/' :
            self.url = self.url[:-1]
        self.webServiceUrl = f'{self.url}/ws.php?'

    def __getattr__(self, name):
        return PiwigoWebService(self, name)

    def setCookies ( self, webService, request ):

        try:
            if webService.methodName == 'pwg.session.login' and request.json()['stat'] == 'ok':
                self.cookies = request.cookies
            elif webService.methodName == 'pwg.session.logout':
                self.cookies = None
        except Exception as e:
            _LOGGER.info( f'Failed to set cookies for request with error {e}', exc_info=True )


class PiwigoWebService:

    def __init__(self, piwigo : Piwigo, methodName : str):
        self.methodName = methodName
        self.piwigo = piwigo
        self.isPostOnly = False

    def getMethodDetail(self):
        try:
            methodDetail = self.piwigo.reflection.getMethodDetails(methodName=self.methodName)
            return methodDetail
        except WsPiwigoException as e:
            _LOGGER.info( f'getMethodDetail() error {e}' )
            raise WsNotExistException(self.methodName)

    def getPostOnly(self):
        try:
            response = self.getMethodDetail()
            options = response['options']
            if len(options) < 1:
                return False
            else:
                return options['post_only']
        except WsNotExistException as e:
            raise e
        except Exception as e:
            _LOGGER.info( f'Exception in checking for isPostOnly: {e}', exc_info=True)
            return False

    def getParams(self):
        return { param['name'] : param for param in self.getMethodDetail()['params'] }

    def __call__(self, *arg, **kw):
        if self.methodName != 'reflection.getMethodDetails':
            self.isPostOnly = self.getPostOnly()
        for i in kw:
            if type(kw[i]) == bool :
                kw[i] = BOOL_TO_STR[kw[i]]
        serviceUrl = self.piwigo.webServiceUrl
        kw["method"] = self.methodName
        kw["format"] = "json"
        params = kw
        data = {}
        p = None
        if 'image' in kw:
            p = open(kw['image'], 'rb')
            files = {'image': p}
            params = { i : params[i] for i in params if i != 'image'}
        else:
            files = {}
        if self.isPostOnly:
            data = { i : params[i] for i in params if i != 'format'}
            params = { i : params[i] for i in params if i == 'format'}
            r = requests.post(serviceUrl, params=params, data=data, files=files, cookies=self.piwigo.cookies)
        else:
            r = requests.get(serviceUrl, params=params, data=data, files=files, cookies=self.piwigo.cookies)
        if p :
            p.close()
        try:
            result = r.json()
            if result['stat'] == 'fail':
                raise WsPiwigoException(result['err'], result['message'])
            self.piwigo.setCookies(self, r)
            return result['result']
        except Exception as e:
            _LOGGER.info(f'__call__() error {e}')
            raise WsErrorException(r.text)

    def __getattr__(self, name):
        return PiwigoWebService(self.piwigo, '%s.%s' % (self.methodName, name))

    def __str__(self) -> str:
        try:
            return "%s : %s" % (self.methodName, self.getMethodDetail()['description'])
        except WsNotExistException as _e:
            return "%s : not exist" % self._method


class PictureInfo:

    translation = str.maketrans( { "'" : '_', '@' : '_', '#' : '_', '!' : '_', ':' : '_', ';' : '_'})

    def __init__ ( self, fileName : str, creationDate : datetime.datetime, fileSize : int, width : int, height : int, piwigoId : int = None ):

        self.fileName     = fileName.translate( PictureInfo.translation )
        self.creationDate = creationDate
        self.fileSize     = fileSize
        self.width        = width
        self.height       = height
        dateKey           = creationDate.strftime('%Y-%m-%d %H:%M:%S') if creationDate else 'nodate'
        self.key          = self.fileName + '#' + str(self.width) + '#' + str(self.height) + '#' + dateKey
        self.piwigoId     = piwigoId
        self.comment      = None
        self.availableDate = None
        self.author        = None
        self.comment       = None
        self.ratingScore   = None
        self.pageUrl       = None
        self.elementUrl    = None
        self.thumbUrl      = None
        self.smallImageUrl = None


class PictureInfos :

    def __init__ ( self ):

        self.pictures = {}

    def loadViaConfig ( self ):
        csvFile = config.Config.get( 'PiwigoCsvFile', DEFAULT_PIWIGO_CSV, PIWIGO_BLOCK )
        self.loadFromCsv( csvFile )

    def loadFromCsv ( self, csvFile : str ):

        with open( csvFile, newline='') as csvfile:
            reader = csv.reader(csvfile, quotechar='"')
            for row in reader:
                try:
                    piwigoId     = int(row[0])
                    fileName     = row[1]
                    try:
                        creationDate = datetime.datetime.strptime( row[3], '%Y-%m-%d %H:%M:%S' )
                    except ValueError:
                        creationDate = None
                    fileSize     = int(row[7])
                    width        = int(row[8])
                    height       = int(row[9])
                    picInfo = PictureInfo( fileName, creationDate, fileSize, width, height, piwigoId )
                    if picInfo.key in self.pictures:
                        oldPic = self.pictures[picInfo.key]
                        _LOGGER.debug( f'Picture key already exists: {picInfo.key} - create dates are {picInfo.creationDate} vs {oldPic.creationDate}')
                    self.pictures[ picInfo.key ] = picInfo
                except Exception as e:
                    _LOGGER.debug( f'Failed to read out row {", ".join(row)}')


    def getIdForPicture ( self, fileName : str, creationDate : datetime.datetime, width : int, height : int ) -> Union[int, None]:
        picInfo = PictureInfo( fileName, creationDate, -1, width, height )
        if picInfo.key in self.pictures:
            return self.pictures[picInfo.key].piwigoId
        else:
            return None


if __name__ == '__main__':

    if os.getenv('CONFIGFILE', None) is None:
        config.DEFAULT_CONFIG = 'piclock.yaml'
    config.initialize()
    getUrlService = GetPicUrlService()
    #url, gUrl = asgiref.sync.async_to_sync(getUrlService.getUrls)("2018 - Family Trip to Miami, Belize, Guatemala/IMG_2035.JPG", "2018-12-26 16:46:56")
    #print(url)
    #print(gUrl)
    url, gUrl = asgiref.sync.async_to_sync(getUrlService.getUrls)("2011-03-22 Patrick Crans-Montana/2011-03-22 14-50-50.JPG", "2011-03-22 14:50:50")
    print(url)
    print(gUrl)
