"""
AbuLang 3D Library - 3D graphics using Ursina/Panda3D
Provides 3D object creation and manipulation
"""

try:
    from ursina import *
    URSINA_AVAILABLE = True
except ImportError:
    URSINA_AVAILABLE = False
    print("[AbuLang 3D] Warning: Ursina not installed. Install with: pip install ursina")


class Scene3D:
    """3D Scene manager using Ursina"""
    
    def __init__(self, title="AbuLang 3D", width=1280, height=720):
        """Initialize 3D scene"""
        if not URSINA_AVAILABLE:
            raise ImportError("Ursina is required for 3D. Install with: pip install ursina")
        
        self.title = title
        self.width = width
        self.height = height
        self.app = Ursina(title=title, window_type='onscreen')
        self.camera.position = (0, 0, -20)
        self.objects = []
    
    def create_cube(self, position=(0, 0, 0), scale=1, color=color.white):
        """Create a 3D cube"""
        cube = Box(pos=position, scale=scale, color=color)
        self.objects.append(cube)
        return cube
    
    def create_sphere(self, position=(0, 0, 0), radius=1, color=color.white):
        """Create a 3D sphere"""
        sphere = Sphere(pos=position, radius=radius, color=color)
        self.objects.append(sphere)
        return sphere
    
    def create_model(self, model_name, position=(0, 0, 0), scale=1, color=color.white):
        """Create a 3D model from file"""
        model = Model(model=model_name, pos=position, scale=scale, color=color)
        self.objects.append(model)
        return model
    
    def set_camera(self, position, target=(0, 0, 0)):
        """Set camera position and target"""
        self.camera.position = position
        self.camera.look_at(target)
    
    def run(self):
        """Start 3D scene"""
        self.app.run()
    
    def close(self):
        """Close 3D scene"""
        self.app.destroy()


class Object3D:
    """3D Object wrapper"""
    
    def __init__(self, ursina_obj):
        """Initialize 3D object"""
        self.obj = ursina_obj
        self.velocity = (0, 0, 0)
    
    def move(self, dx, dy, dz):
        """Move object"""
        self.obj.x += dx
        self.obj.y += dy
        self.obj.z += dz
    
    def set_position(self, x, y, z):
        """Set absolute position"""
        self.obj.position = (x, y, z)
    
    def rotate(self, dx, dy=None, dz=None):
        """
        Rotate object
        Can be called as:
        - rotate(dx, dy) - rotate around X and Y axes
        - rotate(dx, dy, dz) - rotate around X, Y, and Z axes
        """
        self.obj.rotation_x += dx
        if dy is not None:
            self.obj.rotation_y += dy
        if dz is not None:
            self.obj.rotation_z += dz
    
    def set_color(self, color):
        """Set object color"""
        self.obj.color = color
    
    def set_velocity(self, vx, vy, vz):
        """Set velocity for movement"""
        self.velocity = (vx, vy, vz)
    
    def update(self):
        """Update object position based on velocity"""
        if self.velocity != (0, 0, 0):
            self.move(*self.velocity)


def create_3d_scene(title="AbuLang 3D", width=1280, height=720):
    """Create a new 3D scene"""
    return Scene3D(title, width, height)
