"""
AbuLangModule Syntax Fixer - Converts AbuLang syntax to Python
Enables Python 2 style show/ask syntax in the module
"""

import re


def fix_abulang_syntax(code: str) -> str:
    """
    Fix AbuLang syntax to work in Python
    - show "text" -> print("text")
    - ask "prompt" -> input("prompt")
    """
    lines = code.split('\n')
    fixed_lines = []
    
    for line in lines:
        # Skip comments
        if line.strip().startswith('#'):
            fixed_lines.append(line)
            continue
        
        # Fix show command
        if line.strip().startswith('show '):
            fixed_line = _fix_show_command(line)
            fixed_lines.append(fixed_line)
            continue
        
        # Fix ask command
        if line.strip().startswith('ask '):
            fixed_line = _fix_ask_command(line)
            fixed_lines.append(fixed_line)
            continue
        
        fixed_lines.append(line)
    
    return '\n'.join(fixed_lines)


def _fix_show_command(line: str) -> str:
    """Fix show command: show "text" -> print("text")"""
    # Get indentation
    indent = len(line) - len(line.lstrip())
    indent_str = line[:indent]
    
    # Get the content after 'show '
    content = line[indent:].replace('show ', '', 1).strip()
    
    # If it starts with a quote, find the matching closing quote
    if content.startswith('"'):
        i = 1
        while i < len(content):
            if content[i] == '"' and (i == 0 or content[i-1] != '\\'):
                quoted_part = content[:i+1]
                rest = content[i+1:].strip()
                
                if rest:
                    # There's more after the quote (like + something)
                    return f'{indent_str}print({quoted_part} {rest})'
                else:
                    # Just the quoted string
                    return f'{indent_str}print({quoted_part})'
            i += 1
    
    elif content.startswith("'"):
        i = 1
        while i < len(content):
            if content[i] == "'" and (i == 0 or content[i-1] != '\\'):
                quoted_part = content[:i+1]
                rest = content[i+1:].strip()
                
                if rest:
                    return f'{indent_str}print({quoted_part} {rest})'
                else:
                    return f'{indent_str}print({quoted_part})'
            i += 1
    
    # If no quotes, just wrap in print()
    return f'{indent_str}print({content})'


def _fix_ask_command(line: str) -> str:
    """Fix ask command: ask "prompt" -> input("prompt")"""
    # Get indentation
    indent = len(line) - len(line.lstrip())
    indent_str = line[:indent]
    
    # Get the content after 'ask '
    content = line[indent:].replace('ask ', '', 1).strip()
    
    # If it starts with a quote, find the matching closing quote
    if content.startswith('"'):
        i = 1
        while i < len(content):
            if content[i] == '"' and (i == 0 or content[i-1] != '\\'):
                quoted_part = content[:i+1]
                rest = content[i+1:].strip()
                
                if rest:
                    return f'{indent_str}input({quoted_part} {rest})'
                else:
                    return f'{indent_str}input({quoted_part})'
            i += 1
    
    elif content.startswith("'"):
        i = 1
        while i < len(content):
            if content[i] == "'" and (i == 0 or content[i-1] != '\\'):
                quoted_part = content[:i+1]
                rest = content[i+1:].strip()
                
                if rest:
                    return f'{indent_str}input({quoted_part} {rest})'
                else:
                    return f'{indent_str}input({quoted_part})'
            i += 1
    
    # If no quotes, just wrap in input()
    return f'{indent_str}input({content})'
