"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerService = exports.Database = exports.KeyCloak = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class KeyCloak extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : getOrCreateVpc(this);
        this.db = this.addDatabase();
        this.addKeyCloakContainerService({
            dbSecret: this.db.secret,
            dbHost: this.db.clusterEndpointHostname,
            dbUser: this.db.databaseUsername,
            vpc: this.vpc,
            keycloakSecret: this._generateKeycloakSecret(),
        });
    }
    /**
     * @stability stable
     */
    addDatabase() {
        return new Database(this, 'Database', {
            vpc: this.vpc,
        });
    }
    /**
     * @stability stable
     */
    addKeyCloakContainerService(props) {
        return new ContainerService(this, 'KeyCloakContainerSerivce', props);
    }
    _generateKeycloakSecret() {
        return new secretsmanager.Secret(this, 'KCSecret', {
            generateSecretString: {
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'keycloak' }),
            },
        });
    }
}
exports.KeyCloak = KeyCloak;
/**
 * @stability stable
 */
class Database extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this._autoraListenerPort = 3306;
        this.databaseUsername = (_a = props.databaseUsername) !== null && _a !== void 0 ? _a : 'admin';
        const dbcluster = new rds.DatabaseCluster(this, 'Database', {
            engine: (_b = props.engine) !== null && _b !== void 0 ? _b : rds.DatabaseClusterEngine.AURORA_MYSQL,
            credentials: {
                username: this.databaseUsername,
            },
            instanceProps: {
                instanceType: (_c = props.instanceType) !== null && _c !== void 0 ? _c : new ec2.InstanceType('r5.large'),
                vpc: props.vpc,
            },
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.secret = dbcluster.secret;
        // allow internally from the same security group
        dbcluster.connections.allowInternally(ec2.Port.tcp(this._autoraListenerPort));
        // allow from the whole vpc cidr
        dbcluster.connections.allowFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._autoraListenerPort));
        this.dbcluster = dbcluster;
        this.vpc = props.vpc;
        this.clusterEndpointHostname = dbcluster.clusterEndpoint.hostname;
        this.clusterReadEndpointHostname = dbcluster.clusterReadEndpoint.hostname;
        this.clusterIdentifier = dbcluster.clusterIdentifier;
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterReadEndpointHostname', this.clusterReadEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
        if (dbcluster.secret) {
            printOutput(this, 'DBSecretArn', dbcluster.secret.secretArn);
        }
    }
}
exports.Database = Database;
/**
 * @stability stable
 */
class ContainerService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const vpc = props.vpc;
        const cluster = new ecs.Cluster(this, 'Cluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDef', {
            cpu: 4096,
            memoryLimitMiB: 30720,
        });
        const kc = taskDefinition.addContainer('keycloak', {
            image: ecs.ContainerImage.fromRegistry('jboss/keycloak:12.0.2'),
            environment: {
                DB_ADDR: props.dbHost,
                DB_DATABASE: 'keycloak',
                DB_PORT: '3306',
                DB_USER: props.dbUser,
                DB_VENDOR: 'mysql',
                JDBC_PARAMS: 'useSSL=false',
            },
            secrets: {
                DB_PASSWORD: ecs.Secret.fromSecretsManager(props.dbSecret, 'password'),
                KEYCLOAK_PASSWORD: ecs.Secret.fromSecretsManager(props.keycloakSecret, 'password'),
            },
        });
        kc.addPortMappings({
            containerPort: 8080,
        });
        this.service = new ecs.FargateService(this, 'Service', {
            cluster,
            taskDefinition,
            circuitBreaker: {
                rollback: true,
            },
        });
    }
}
exports.ContainerService = ContainerService;
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1' ?
        ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,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