import os
import threading
import time
from multiprocessing import Manager, Process
from typing import List, Tuple

from tqdm import tqdm

from .logger import Logger

__all__ = ["multi_tasks", "make_new_thread", "open_pdf"]


def multi_tasks(
    func, data_list: List, return_results: bool = True, desc: str = "waiting", num_workers: int = 10, refresh_time: float = 0.1
) -> list:
    """多进程处理任务

    Args:
        func (_type_): 处理函数
        data_list (List): 需要处理的数据列表
        return_results (bool, optional): 当没有返回值时，设置False可以节约内存. Defaults to True.
        desc (str, optional): 进度条显示名称. Defaults to "waiting".
        num_workers (int, optional): 进程数量. Defaults to 10.
        refresh_time (float, optional): 进度条更新时间(秒). Defaults to 0.1.
    """
    results = Manager().list()
    process_list: List[Process] = []

    step = (len(data_list) + num_workers - 1) // num_workers
    all_idx = list(range(len(data_list)))
    for i in range(0, len(data_list), step):
        part_idxs = all_idx[i : i + step]
        process_list.append(Process(target=_batch_func, args=(func, data_list, part_idxs, results, return_results)))
    process_list.append(Process(target=_show_processon, args=(len(data_list), desc, refresh_time, results)))
    for p in process_list:
        p.start()
    for p in process_list:
        p.join()

    real_results = [None] * len(data_list)
    if return_results:
        for i, result in results:
            real_results[i] = result
    return real_results


def _show_processon(nums: int, desc: str, refresh_time: float, results):
    """显示进度条"""
    cnt = 0
    step = 0
    pbar = tqdm(range(nums), desc=desc)
    while cnt < nums:
        cnt = len(results)
        pbar.update(cnt - step)
        step = cnt
        time.sleep(refresh_time)


def _batch_func(func, data_list: List, part_idxs: List[int], results, return_results):
    n = func.__code__.co_argcount
    for idx in part_idxs:
        data = data_list[idx]
        if n == 1:
            result = func(data)
        else:
            result = func(*data)
        if not return_results:
            result = None
        results.append([idx, result])


thread_list = []


def make_new_thread(func, args: Tuple = None):
    """start a new thread"""

    thread = threading.Thread(target=func, args=args)
    thread_list.append(thread)
    thread.start()


def open_pdf(pdf_filename: str):
    """open pdf"""
    if Logger.exists(pdf_filename):
        make_new_thread(os.system, (f"evince \"{pdf_filename}\"",))


if __name__ == "__main__":
    # how to use
    multi_tasks(lambda x: print(x), [1, 2, 3, 4, 5], desc="print")
    make_new_thread(print, ("hello"))
