"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certbot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const oneTimeEvents = require("@renovosolutions/cdk-library-one-time-event");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class Certbot extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        // Create a bucket if one is not provided
        if (props.bucket === undefined) {
            bucket = new aws_cdk_lib_1.aws_s3.Bucket(this, 'bucket', {
                objectOwnership: aws_cdk_lib_1.aws_s3.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                removalPolicy: props.removalPolicy || aws_cdk_lib_1.RemovalPolicy.RETAIN,
                autoDeleteObjects: props.enableObjectDeletion ?? false,
                versioned: true,
                lifecycleRules: [{
                        enabled: true,
                        abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                    }],
                encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
                enforceSSL: true,
                blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            });
        }
        else {
            bucket = props.bucket;
        }
        const functionDir = path.join(__dirname, '../function');
        // Create an SNS topic if one is not provided and add the defined email to it
        let snsTopic = props.snsTopic ?? new aws_cdk_lib_1.aws_sns.Topic(this, 'topic');
        if (props.snsTopic === undefined) {
            snsTopic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.EmailSubscription(props.letsencryptEmail));
        }
        // Set property defaults
        let layers = props.layers ?? [];
        let runOnDeploy = props.runOnDeploy ?? true;
        let functionDescription = props.functionDescription ?? 'Certbot Renewal Lambda for domain ' + props.letsencryptDomains.split(',')[0];
        let enableInsights = props.enableInsights ?? false;
        let insightsARN = props.insightsARN ?? 'arn:aws:lambda:' + aws_cdk_lib_1.Stack.of(this).region + ':580247275435:layer:LambdaInsightsExtension:14';
        // Set up role policies
        let managedPolicies = [aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')];
        if (enableInsights) {
            managedPolicies.push(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
            layers.push(aws_cdk_lib_1.aws_lambda.LayerVersion.fromLayerVersionArn(this, 'insightsLayer', insightsARN));
        }
        const snsPolicy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'snsPolicy', {
            description: 'Allow the Certbot function to notify an SNS topic upon completion.',
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['sns:Publish'],
                    resources: [snsTopic.topicArn],
                }),
            ],
        });
        let hostedZones = [];
        props.hostedZoneNames.forEach((domainName) => {
            hostedZones.push(aws_cdk_lib_1.aws_route53.HostedZone.fromLookup(this, 'zone' + domainName, {
                domainName,
                privateZone: false,
            }).hostedZoneArn);
        });
        const r53Policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'r53Policy', {
            description: 'Allow the Certbot function to perform DNS verification.',
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['route53:ListHostedZones'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'route53:GetChange',
                        'route53:ChangeResourceRecordSets',
                    ],
                    resources: ['arn:aws:route53:::change/*'].concat(hostedZones),
                }),
            ],
        });
        const acmPolicy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'acmPolicy', {
            description: 'Allow the Certbot function to import and list certificates.',
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'acm:ListCertificates',
                        'acm:ImportCertificate',
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['acm:DescribeCertificate'],
                    resources: ['arn:aws:acm:' + aws_cdk_lib_1.Stack.of(this).region + ':' + aws_cdk_lib_1.Stack.of(this).account + ':certificate/*'],
                }),
            ],
        });
        managedPolicies.push(snsPolicy);
        managedPolicies.push(r53Policy);
        managedPolicies.push(acmPolicy);
        const role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        bucket.grantWrite(role);
        // Create the Lambda function
        this.handler = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_8,
            role,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(functionDir + '/function.zip'),
            handler: 'index.handler',
            functionName: props.functionName,
            description: functionDescription,
            environment: {
                LETSENCRYPT_DOMAINS: props.letsencryptDomains,
                LETSENCRYPT_EMAIL: props.letsencryptEmail,
                CERTIFICATE_BUCKET: bucket.bucketName,
                OBJECT_PREFIX: props.objectPrefix || '',
                REISSUE_DAYS: (props.reIssueDays === undefined) ? '30' : String(props.reIssueDays),
                PREFERRED_CHAIN: props.preferredChain || 'None',
                NOTIFICATION_SNS_ARN: snsTopic.topicArn,
            },
            layers,
            timeout: props.timeout || aws_cdk_lib_1.Duration.seconds(180),
        });
        // Add function triggers
        new aws_cdk_lib_1.aws_events.Rule(this, 'trigger', {
            schedule: props.schedule || aws_cdk_lib_1.aws_events.Schedule.cron({ minute: '0', hour: '0', weekDay: '1' }),
            targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler)],
        });
        if (runOnDeploy) {
            new aws_cdk_lib_1.aws_events.Rule(this, 'triggerImmediate', {
                schedule: new oneTimeEvents.OnDeploy(this, 'schedule', {
                    offsetMinutes: props.runOnDeployWaitMinutes || 10,
                }).schedule,
                targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler)],
            });
        }
    }
}
exports.Certbot = Certbot;
_a = JSII_RTTI_SYMBOL_1;
Certbot[_a] = { fqn: "@renovosolutions/cdk-library-certbot.Certbot", version: "2.2.176" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFFN0IsNkVBQTZFO0FBQzdFLDZDQVlxQjtBQUNyQiwyQ0FBdUM7QUF1R3ZDLE1BQWEsT0FBUSxTQUFRLHNCQUFTO0lBSXBDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBbUI7UUFDM0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLE1BQWlCLENBQUM7UUFFdEIseUNBQXlDO1FBQ3pDLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDOUIsTUFBTSxHQUFHLElBQUksb0JBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtnQkFDckMsZUFBZSxFQUFFLG9CQUFFLENBQUMsZUFBZSxDQUFDLHNCQUFzQjtnQkFDMUQsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhLElBQUksMkJBQWEsQ0FBQyxNQUFNO2dCQUMxRCxpQkFBaUIsRUFBRSxLQUFLLENBQUMsb0JBQW9CLElBQUksS0FBSztnQkFDdEQsU0FBUyxFQUFFLElBQUk7Z0JBQ2YsY0FBYyxFQUFFLENBQUM7d0JBQ2YsT0FBTyxFQUFFLElBQUk7d0JBQ2IsbUNBQW1DLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO3FCQUN0RCxDQUFDO2dCQUNGLFVBQVUsRUFBRSxvQkFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7Z0JBQzFDLFVBQVUsRUFBRSxJQUFJO2dCQUNoQixpQkFBaUIsRUFBRSxvQkFBRSxDQUFDLGlCQUFpQixDQUFDLFNBQVM7YUFDbEQsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1NBQ3ZCO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFeEQsNkVBQTZFO1FBQzdFLElBQUksUUFBUSxHQUFjLEtBQUssQ0FBQyxRQUFRLElBQUksSUFBSSxxQkFBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDekUsSUFBSSxLQUFLLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtZQUNoQyxRQUFRLENBQUMsZUFBZSxDQUFDLElBQUksbUNBQWEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1NBQ3ZGO1FBRUQsd0JBQXdCO1FBQ3hCLElBQUksTUFBTSxHQUEyQixLQUFLLENBQUMsTUFBTSxJQUFJLEVBQUUsQ0FBQztRQUN4RCxJQUFJLFdBQVcsR0FBWSxLQUFLLENBQUMsV0FBVyxJQUFJLElBQUksQ0FBQztRQUNyRCxJQUFJLG1CQUFtQixHQUFXLEtBQUssQ0FBQyxtQkFBbUIsSUFBSSxvQ0FBb0MsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzdJLElBQUksY0FBYyxHQUFZLEtBQUssQ0FBQyxjQUFjLElBQUksS0FBSyxDQUFDO1FBQzVELElBQUksV0FBVyxHQUFXLEtBQUssQ0FBQyxXQUFXLElBQUksaUJBQWlCLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxHQUFHLGdEQUFnRCxDQUFDO1FBRTVJLHVCQUF1QjtRQUN2QixJQUFJLGVBQWUsR0FBRyxDQUFDLHFCQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDBDQUEwQyxDQUFDLENBQUMsQ0FBQztRQUMvRyxJQUFJLGNBQWMsRUFBRTtZQUNsQixlQUFlLENBQUMsSUFBSSxDQUFDLHFCQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDZDQUE2QyxDQUFDLENBQUMsQ0FBQztZQUNoSCxNQUFNLENBQUMsSUFBSSxDQUFDLHdCQUFNLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztTQUMxRjtRQUVELE1BQU0sU0FBUyxHQUFHLElBQUkscUJBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUN6RCxXQUFXLEVBQUUsb0VBQW9FO1lBQ2pGLFVBQVUsRUFBRTtnQkFDVixJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixNQUFNLEVBQUUscUJBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztvQkFDeEIsT0FBTyxFQUFFLENBQUMsYUFBYSxDQUFDO29CQUN4QixTQUFTLEVBQUUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDO2lCQUMvQixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLFdBQVcsR0FBWSxFQUFFLENBQUM7UUFDOUIsS0FBSyxDQUFDLGVBQWUsQ0FBQyxPQUFPLENBQUUsQ0FBQyxVQUFVLEVBQUUsRUFBRTtZQUM1QyxXQUFXLENBQUMsSUFBSSxDQUFDLHlCQUFHLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsTUFBTSxHQUFHLFVBQVUsRUFBRTtnQkFDcEUsVUFBVTtnQkFDVixXQUFXLEVBQUUsS0FBSzthQUNuQixDQUFDLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDcEIsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxJQUFJLHFCQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDekQsV0FBVyxFQUFFLHlEQUF5RDtZQUN0RSxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLE9BQU8sRUFBRSxDQUFDLHlCQUF5QixDQUFDO29CQUNwQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2pCLENBQUM7Z0JBQ0YsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLE9BQU8sRUFBRTt3QkFDUCxtQkFBbUI7d0JBQ25CLGtDQUFrQztxQkFDbkM7b0JBQ0QsU0FBUyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDO2lCQUM5RCxDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxJQUFJLHFCQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDekQsV0FBVyxFQUFFLDZEQUE2RDtZQUMxRSxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLE9BQU8sRUFBRTt3QkFDUCxzQkFBc0I7d0JBQ3RCLHVCQUF1QjtxQkFDeEI7b0JBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNqQixDQUFDO2dCQUNGLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE1BQU0sRUFBRSxxQkFBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO29CQUN4QixPQUFPLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQztvQkFDcEMsU0FBUyxFQUFFLENBQUMsY0FBYyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxHQUFHLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxHQUFHLGdCQUFnQixDQUFDO2lCQUN0RyxDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxlQUFlLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2hDLGVBQWUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDaEMsZUFBZSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUVoQyxNQUFNLElBQUksR0FBRyxJQUFJLHFCQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDdEMsU0FBUyxFQUFFLElBQUkscUJBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxlQUFlO1NBQ2hCLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFeEIsNkJBQTZCO1FBQzdCLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSx3QkFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ2xELE9BQU8sRUFBRSx3QkFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1lBQ2xDLElBQUk7WUFDSixJQUFJLEVBQUUsd0JBQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsR0FBRyxlQUFlLENBQUM7WUFDMUQsT0FBTyxFQUFFLGVBQWU7WUFDeEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLFdBQVcsRUFBRSxtQkFBbUI7WUFDaEMsV0FBVyxFQUFFO2dCQUNYLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7Z0JBQzdDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7Z0JBQ3pDLGtCQUFrQixFQUFFLE1BQU0sQ0FBQyxVQUFVO2dCQUNyQyxhQUFhLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxFQUFFO2dCQUN2QyxZQUFZLEVBQUUsQ0FBQyxLQUFLLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDO2dCQUNsRixlQUFlLEVBQUUsS0FBSyxDQUFDLGNBQWMsSUFBSSxNQUFNO2dCQUMvQyxvQkFBb0IsRUFBRSxRQUFRLENBQUMsUUFBUTthQUN4QztZQUNELE1BQU07WUFDTixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sSUFBSSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7U0FDaEQsQ0FBQyxDQUFDO1FBRUgsd0JBQXdCO1FBQ3hCLElBQUksd0JBQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMvQixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVEsSUFBSSx3QkFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDO1lBQzFGLE9BQU8sRUFBRSxDQUFDLElBQUksZ0NBQU8sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ3BELENBQUMsQ0FBQztRQUVILElBQUksV0FBVyxFQUFFO1lBQ2YsSUFBSSx3QkFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ3hDLFFBQVEsRUFBRSxJQUFJLGFBQWEsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtvQkFDckQsYUFBYSxFQUFFLEtBQUssQ0FBQyxzQkFBc0IsSUFBSSxFQUFFO2lCQUNsRCxDQUFDLENBQUMsUUFBUTtnQkFDWCxPQUFPLEVBQUUsQ0FBQyxJQUFJLGdDQUFPLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQzthQUNwRCxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7O0FBekpILDBCQTBKQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5cbmltcG9ydCAqIGFzIG9uZVRpbWVFdmVudHMgZnJvbSAnQHJlbm92b3NvbHV0aW9ucy9jZGstbGlicmFyeS1vbmUtdGltZS1ldmVudCc7XG5pbXBvcnQge1xuICBhd3NfZXZlbnRzIGFzIGV2ZW50cyxcbiAgYXdzX2V2ZW50c190YXJnZXRzIGFzIHRhcmdldHMsXG4gIGF3c19pYW0gYXMgaWFtLFxuICBhd3NfbGFtYmRhIGFzIGxhbWJkYSxcbiAgYXdzX3MzIGFzIHMzLFxuICBhd3Nfcm91dGU1MyBhcyByNTMsXG4gIGF3c19zbnMgYXMgc25zLFxuICBhd3Nfc25zX3N1YnNjcmlwdGlvbnMgYXMgc3Vic2NyaXB0aW9ucyxcbiAgRHVyYXRpb24sXG4gIFJlbW92YWxQb2xpY3ksXG4gIFN0YWNrLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuZXhwb3J0IGludGVyZmFjZSBDZXJ0Ym90UHJvcHMge1xuICAvKipcbiAgICogVGhlIGNvbW1hIGRlbGltaXRlZCBsaXN0IG9mIGRvbWFpbnMgZm9yIHdoaWNoIHRoZSBMZXQncyBFbmNyeXB0IGNlcnRpZmljYXRlIHdpbGwgYmUgdmFsaWQuIFByaW1hcnkgZG9tYWluIHNob3VsZCBiZSBmaXJzdC5cbiAgICovXG4gIHJlYWRvbmx5IGxldHNlbmNyeXB0RG9tYWluczogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGVtYWlsIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBMZXQncyBFbmNyeXB0IGNlcnRpZmljYXRlIHJlcXVlc3QuXG4gICAqL1xuICByZWFkb25seSBsZXRzZW5jcnlwdEVtYWlsOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBBbnkgYWRkaXRpb25hbCBMYW1iZGEgbGF5ZXJzIHRvIHVzZSB3aXRoIHRoZSBjcmVhdGVkIGZ1bmN0aW9uLiBGb3IgZXhhbXBsZSBMYW1iZGEgRXh0ZW5zaW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgbGF5ZXJzPzogbGFtYmRhLklMYXllclZlcnNpb25bXTtcbiAgLyoqXG4gICAqIEhvc3RlZCB6b25lIG5hbWVzIHRoYXQgd2lsbCBiZSByZXF1aXJlZCBmb3IgRE5TIHZlcmlmaWNhdGlvbiB3aXRoIGNlcnRib3RcbiAgICovXG4gIHJlYWRvbmx5IGhvc3RlZFpvbmVOYW1lczogc3RyaW5nW107XG4gIC8qKlxuICAgKiBUaGUgUzMgYnVja2V0IHRvIHBsYWNlIHRoZSByZXN1bHRpbmcgY2VydGlmaWNhdGVzIGluLiBJZiBubyBidWNrZXQgaXMgZ2l2ZW4gb25lIHdpbGwgYmUgY3JlYXRlZCBhdXRvbWF0aWNhbGx5LlxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0PzogczMuQnVja2V0O1xuICAvKipcbiAgICogVGhlIHByZWZpeCB0byBhcHBseSB0byB0aGUgZmluYWwgUzMga2V5IG5hbWUgZm9yIHRoZSBjZXJ0aWZpY2F0ZXMuIERlZmF1bHQgaXMgbm8gcHJlZml4LlxuICAgKi9cbiAgcmVhZG9ubHkgb2JqZWN0UHJlZml4Pzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG51bWJlcnMgb2YgZGF5cyBsZWZ0IHVudGlsIHRoZSBwcmlvciBjZXJ0IGV4cGlyZXMgYmVmb3JlIGlzc3VpbmcgYSBuZXcgb25lLlxuICAgKlxuICAgKiBAZGVmYXVsdCAzMFxuICAgKi9cbiAgcmVhZG9ubHkgcmVJc3N1ZURheXM/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBTZXQgdGhlIHByZWZlcnJlZCBjZXJ0aWZpY2F0ZSBjaGFpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgJ05vbmUnXG4gICAqL1xuICByZWFkb25seSBwcmVmZXJyZWRDaGFpbj86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBTTlMgdG9waWMgdG8gbm90aWZ5IHdoZW4gYSBuZXcgY2VydCBpcyBpc3N1ZWQuIElmIG5vIHRvcGljIGlzIGdpdmVuIG9uZSB3aWxsIGJlIGNyZWF0ZWQgYXV0b21hdGljYWxseS5cbiAgICovXG4gIHJlYWRvbmx5IHNuc1RvcGljPzogc25zLlRvcGljO1xuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gZW5hYmxlIExhbWJkYSBJbnNpZ2h0c1xuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlSW5zaWdodHM/OiBib29sZWFuO1xuICAvKipcbiAgICogSW5zaWdodHMgbGF5ZXIgQVJOIGZvciB5b3VyIHJlZ2lvbi4gRGVmYXVsdHMgdG8gbGF5ZXIgZm9yIFVTLUVBU1QtMVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zaWdodHNBUk4/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgdGltZW91dCBkdXJhdGlvbiBmb3IgTGFtYmRhIGZ1bmN0aW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmFjdGlvbi5zZWNvbmRzKDE4MClcbiAgICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBEdXJhdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBzY2hlZHVsZSBmb3IgdGhlIGNlcnRpZmljYXRlIGNoZWNrIHRyaWdnZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IGV2ZW50cy5TY2hlZHVsZS5jcm9uKHsgbWludXRlOiAnMCcsIGhvdXI6ICcwJywgd2Vla0RheTogJzEnIH0pXG4gICAqL1xuICByZWFkb25seSBzY2hlZHVsZT86IGV2ZW50cy5TY2hlZHVsZTtcbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRvIHNjaGVkdWxlIGEgdHJpZ2dlciB0byBydW4gdGhlIGZ1bmN0aW9uIGFmdGVyIGVhY2ggZGVwbG95bWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBydW5PbkRlcGxveT86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBIb3cgbWFueSBtaW51dGVzIHRvIHdhaXQgYmVmb3JlIHJ1bm5pbmcgdGhlIHBvc3QgZGVwbG95bWVudCBMYW1iZGEgdHJpZ2dlclxuICAgKlxuICAgKiBAZGVmYXVsdCAxMFxuICAgKi9cbiAgcmVhZG9ubHkgcnVuT25EZXBsb3lXYWl0TWludXRlcz86IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBkZXNjcmlwdGlvbiBmb3IgdGhlIHJlc3VsdGluZyBMYW1iZGEgZnVuY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBmdW5jdGlvbkRlc2NyaXB0aW9uPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHJlc3VsdGluZyBMYW1iZGEgZnVuY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBmdW5jdGlvbk5hbWU/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgcmVtb3ZhbCBwb2xpY3kgZm9yIHRoZSBTMyBidWNrZXQgdGhhdCBpcyBhdXRvbWF0aWNhbGx5IGNyZWF0ZWQuXG4gICAqXG4gICAqIEhhcyBubyBlZmZlY3QgaWYgYSBidWNrZXQgaXMgZ2l2ZW4gYXMgYSBwcm9wZXJ0eVxuICAgKlxuICAgKiBAZGVmYXVsdCBSZW1vdmFsUG9saWN5LlJFVEFJTlxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBlbmFibGUgYXV0b21hdGljIG9iamVjdCBkZWxldGlvbiBpZiB0aGUgcHJvdmlkZWQgYnVja2V0IGlzIGRlbGV0ZWQuXG4gICAqXG4gICAqIEhhcyBubyBlZmZlY3QgaWYgYSBidWNrZXQgaXMgZ2l2ZW4gYXMgYSBwcm9wZXJ0eVxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlT2JqZWN0RGVsZXRpb24/OiBib29sZWFuO1xufVxuXG5leHBvcnQgY2xhc3MgQ2VydGJvdCBleHRlbmRzIENvbnN0cnVjdCB7XG5cbiAgcHVibGljIHJlYWRvbmx5IGhhbmRsZXI6IGxhbWJkYS5GdW5jdGlvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2VydGJvdFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGxldCBidWNrZXQ6IHMzLkJ1Y2tldDtcblxuICAgIC8vIENyZWF0ZSBhIGJ1Y2tldCBpZiBvbmUgaXMgbm90IHByb3ZpZGVkXG4gICAgaWYgKHByb3BzLmJ1Y2tldCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBidWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsICdidWNrZXQnLCB7XG4gICAgICAgIG9iamVjdE93bmVyc2hpcDogczMuT2JqZWN0T3duZXJzaGlwLkJVQ0tFVF9PV05FUl9QUkVGRVJSRUQsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IHByb3BzLnJlbW92YWxQb2xpY3kgfHwgUmVtb3ZhbFBvbGljeS5SRVRBSU4sXG4gICAgICAgIGF1dG9EZWxldGVPYmplY3RzOiBwcm9wcy5lbmFibGVPYmplY3REZWxldGlvbiA/PyBmYWxzZSxcbiAgICAgICAgdmVyc2lvbmVkOiB0cnVlLFxuICAgICAgICBsaWZlY3ljbGVSdWxlczogW3tcbiAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgIGFib3J0SW5jb21wbGV0ZU11bHRpcGFydFVwbG9hZEFmdGVyOiBEdXJhdGlvbi5kYXlzKDEpLFxuICAgICAgICB9XSxcbiAgICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgICBibG9ja1B1YmxpY0FjY2VzczogczMuQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGJ1Y2tldCA9IHByb3BzLmJ1Y2tldDtcbiAgICB9XG5cbiAgICBjb25zdCBmdW5jdGlvbkRpciA9IHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9mdW5jdGlvbicpO1xuXG4gICAgLy8gQ3JlYXRlIGFuIFNOUyB0b3BpYyBpZiBvbmUgaXMgbm90IHByb3ZpZGVkIGFuZCBhZGQgdGhlIGRlZmluZWQgZW1haWwgdG8gaXRcbiAgICBsZXQgc25zVG9waWM6IHNucy5Ub3BpYyA9IHByb3BzLnNuc1RvcGljID8/IG5ldyBzbnMuVG9waWModGhpcywgJ3RvcGljJyk7XG4gICAgaWYgKHByb3BzLnNuc1RvcGljID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHNuc1RvcGljLmFkZFN1YnNjcmlwdGlvbihuZXcgc3Vic2NyaXB0aW9ucy5FbWFpbFN1YnNjcmlwdGlvbihwcm9wcy5sZXRzZW5jcnlwdEVtYWlsKSk7XG4gICAgfVxuXG4gICAgLy8gU2V0IHByb3BlcnR5IGRlZmF1bHRzXG4gICAgbGV0IGxheWVyczogbGFtYmRhLklMYXllclZlcnNpb25bXSA9IHByb3BzLmxheWVycyA/PyBbXTtcbiAgICBsZXQgcnVuT25EZXBsb3k6IGJvb2xlYW4gPSBwcm9wcy5ydW5PbkRlcGxveSA/PyB0cnVlO1xuICAgIGxldCBmdW5jdGlvbkRlc2NyaXB0aW9uOiBzdHJpbmcgPSBwcm9wcy5mdW5jdGlvbkRlc2NyaXB0aW9uID8/ICdDZXJ0Ym90IFJlbmV3YWwgTGFtYmRhIGZvciBkb21haW4gJyArIHByb3BzLmxldHNlbmNyeXB0RG9tYWlucy5zcGxpdCgnLCcpWzBdO1xuICAgIGxldCBlbmFibGVJbnNpZ2h0czogYm9vbGVhbiA9IHByb3BzLmVuYWJsZUluc2lnaHRzID8/IGZhbHNlO1xuICAgIGxldCBpbnNpZ2h0c0FSTjogc3RyaW5nID0gcHJvcHMuaW5zaWdodHNBUk4gPz8gJ2Fybjphd3M6bGFtYmRhOicgKyBTdGFjay5vZih0aGlzKS5yZWdpb24gKyAnOjU4MDI0NzI3NTQzNTpsYXllcjpMYW1iZGFJbnNpZ2h0c0V4dGVuc2lvbjoxNCc7XG5cbiAgICAvLyBTZXQgdXAgcm9sZSBwb2xpY2llc1xuICAgIGxldCBtYW5hZ2VkUG9saWNpZXMgPSBbaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJyldO1xuICAgIGlmIChlbmFibGVJbnNpZ2h0cykge1xuICAgICAgbWFuYWdlZFBvbGljaWVzLnB1c2goaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdDbG91ZFdhdGNoTGFtYmRhSW5zaWdodHNFeGVjdXRpb25Sb2xlUG9saWN5JykpO1xuICAgICAgbGF5ZXJzLnB1c2gobGFtYmRhLkxheWVyVmVyc2lvbi5mcm9tTGF5ZXJWZXJzaW9uQXJuKHRoaXMsICdpbnNpZ2h0c0xheWVyJywgaW5zaWdodHNBUk4pKTtcbiAgICB9XG5cbiAgICBjb25zdCBzbnNQb2xpY3kgPSBuZXcgaWFtLk1hbmFnZWRQb2xpY3kodGhpcywgJ3Nuc1BvbGljeScsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiAnQWxsb3cgdGhlIENlcnRib3QgZnVuY3Rpb24gdG8gbm90aWZ5IGFuIFNOUyB0b3BpYyB1cG9uIGNvbXBsZXRpb24uJyxcbiAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3NuczpQdWJsaXNoJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbc25zVG9waWMudG9waWNBcm5dLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBsZXQgaG9zdGVkWm9uZXM6c3RyaW5nW10gPSBbXTtcbiAgICBwcm9wcy5ob3N0ZWRab25lTmFtZXMuZm9yRWFjaCggKGRvbWFpbk5hbWUpID0+IHtcbiAgICAgIGhvc3RlZFpvbmVzLnB1c2gocjUzLkhvc3RlZFpvbmUuZnJvbUxvb2t1cCh0aGlzLCAnem9uZScgKyBkb21haW5OYW1lLCB7XG4gICAgICAgIGRvbWFpbk5hbWUsXG4gICAgICAgIHByaXZhdGVab25lOiBmYWxzZSxcbiAgICAgIH0pLmhvc3RlZFpvbmVBcm4pO1xuICAgIH0pO1xuXG4gICAgY29uc3QgcjUzUG9saWN5ID0gbmV3IGlhbS5NYW5hZ2VkUG9saWN5KHRoaXMsICdyNTNQb2xpY3knLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ0FsbG93IHRoZSBDZXJ0Ym90IGZ1bmN0aW9uIHRvIHBlcmZvcm0gRE5TIHZlcmlmaWNhdGlvbi4nLFxuICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFsncm91dGU1MzpMaXN0SG9zdGVkWm9uZXMnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAncm91dGU1MzpHZXRDaGFuZ2UnLFxuICAgICAgICAgICAgJ3JvdXRlNTM6Q2hhbmdlUmVzb3VyY2VSZWNvcmRTZXRzJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogWydhcm46YXdzOnJvdXRlNTM6OjpjaGFuZ2UvKiddLmNvbmNhdChob3N0ZWRab25lcyksXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGFjbVBvbGljeSA9IG5ldyBpYW0uTWFuYWdlZFBvbGljeSh0aGlzLCAnYWNtUG9saWN5Jywge1xuICAgICAgZGVzY3JpcHRpb246ICdBbGxvdyB0aGUgQ2VydGJvdCBmdW5jdGlvbiB0byBpbXBvcnQgYW5kIGxpc3QgY2VydGlmaWNhdGVzLicsXG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgJ2FjbTpMaXN0Q2VydGlmaWNhdGVzJyxcbiAgICAgICAgICAgICdhY206SW1wb3J0Q2VydGlmaWNhdGUnLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogWydhY206RGVzY3JpYmVDZXJ0aWZpY2F0ZSddLFxuICAgICAgICAgIHJlc291cmNlczogWydhcm46YXdzOmFjbTonICsgU3RhY2sub2YodGhpcykucmVnaW9uICsgJzonICsgU3RhY2sub2YodGhpcykuYWNjb3VudCArICc6Y2VydGlmaWNhdGUvKiddLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBtYW5hZ2VkUG9saWNpZXMucHVzaChzbnNQb2xpY3kpO1xuICAgIG1hbmFnZWRQb2xpY2llcy5wdXNoKHI1M1BvbGljeSk7XG4gICAgbWFuYWdlZFBvbGljaWVzLnB1c2goYWNtUG9saWN5KTtcblxuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ3JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llcyxcbiAgICB9KTtcblxuICAgIGJ1Y2tldC5ncmFudFdyaXRlKHJvbGUpO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICB0aGlzLmhhbmRsZXIgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdoYW5kbGVyJywge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfOCxcbiAgICAgIHJvbGUsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoZnVuY3Rpb25EaXIgKyAnL2Z1bmN0aW9uLnppcCcpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgZnVuY3Rpb25OYW1lOiBwcm9wcy5mdW5jdGlvbk5hbWUsXG4gICAgICBkZXNjcmlwdGlvbjogZnVuY3Rpb25EZXNjcmlwdGlvbixcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIExFVFNFTkNSWVBUX0RPTUFJTlM6IHByb3BzLmxldHNlbmNyeXB0RG9tYWlucyxcbiAgICAgICAgTEVUU0VOQ1JZUFRfRU1BSUw6IHByb3BzLmxldHNlbmNyeXB0RW1haWwsXG4gICAgICAgIENFUlRJRklDQVRFX0JVQ0tFVDogYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIE9CSkVDVF9QUkVGSVg6IHByb3BzLm9iamVjdFByZWZpeCB8fCAnJyxcbiAgICAgICAgUkVJU1NVRV9EQVlTOiAocHJvcHMucmVJc3N1ZURheXMgPT09IHVuZGVmaW5lZCkgPyAnMzAnIDogU3RyaW5nKHByb3BzLnJlSXNzdWVEYXlzKSxcbiAgICAgICAgUFJFRkVSUkVEX0NIQUlOOiBwcm9wcy5wcmVmZXJyZWRDaGFpbiB8fCAnTm9uZScsXG4gICAgICAgIE5PVElGSUNBVElPTl9TTlNfQVJOOiBzbnNUb3BpYy50b3BpY0FybixcbiAgICAgIH0sXG4gICAgICBsYXllcnMsXG4gICAgICB0aW1lb3V0OiBwcm9wcy50aW1lb3V0IHx8IER1cmF0aW9uLnNlY29uZHMoMTgwKSxcbiAgICB9KTtcblxuICAgIC8vIEFkZCBmdW5jdGlvbiB0cmlnZ2Vyc1xuICAgIG5ldyBldmVudHMuUnVsZSh0aGlzLCAndHJpZ2dlcicsIHtcbiAgICAgIHNjaGVkdWxlOiBwcm9wcy5zY2hlZHVsZSB8fCBldmVudHMuU2NoZWR1bGUuY3Jvbih7IG1pbnV0ZTogJzAnLCBob3VyOiAnMCcsIHdlZWtEYXk6ICcxJyB9KSxcbiAgICAgIHRhcmdldHM6IFtuZXcgdGFyZ2V0cy5MYW1iZGFGdW5jdGlvbih0aGlzLmhhbmRsZXIpXSxcbiAgICB9KTtcblxuICAgIGlmIChydW5PbkRlcGxveSkge1xuICAgICAgbmV3IGV2ZW50cy5SdWxlKHRoaXMsICd0cmlnZ2VySW1tZWRpYXRlJywge1xuICAgICAgICBzY2hlZHVsZTogbmV3IG9uZVRpbWVFdmVudHMuT25EZXBsb3kodGhpcywgJ3NjaGVkdWxlJywge1xuICAgICAgICAgIG9mZnNldE1pbnV0ZXM6IHByb3BzLnJ1bk9uRGVwbG95V2FpdE1pbnV0ZXMgfHwgMTAsXG4gICAgICAgIH0pLnNjaGVkdWxlLFxuICAgICAgICB0YXJnZXRzOiBbbmV3IHRhcmdldHMuTGFtYmRhRnVuY3Rpb24odGhpcy5oYW5kbGVyKV0sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==