"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
/**
 * Define a CloudWatch EventBus
 *
 * @resource AWS::Events::EventBus
 */
class EventBus extends core_1.Resource {
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.stringValue({ produce: () => this.node.uniqueId }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: eventBusName,
            eventSourceName
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param eventBusArn ARN of imported event bus
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = eventBusArn;
                this.eventBusName = parts.resourceName || '';
                this.eventBusPolicy = '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing event bus resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported event bus properties
     */
    static fromEventBusAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = attrs.eventBusArn;
                this.eventBusName = attrs.eventBusName;
                this.eventBusPolicy = attrs.eventBusPolicy;
                this.eventSourceName = attrs.eventSourceName;
            }
        }
        return new Import(scope, id);
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error(`'eventBusName' and 'eventSourceName' cannot both be provided`);
            }
            else if (eventBusName !== undefined) {
                if (eventBusName === 'default') {
                    throw new Error(`'eventBusName' must not be 'default'`);
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error(`'eventBusName' must not contain '/'`);
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnQtYnVzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZXZlbnQtYnVzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQTRFO0FBQzVFLHlEQUFpRDtBQStGakQ7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLGVBQVE7SUFvR3BDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBcUI7UUFDN0QsTUFBTSxFQUFFLFlBQVksRUFBRSxlQUFlLEVBQUUsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUM5RCxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsRUFDdkQsS0FBSyxDQUNOLENBQUM7UUFFRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFlBQVksRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBRWpELE1BQU0sUUFBUSxHQUFHLElBQUksOEJBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2pELElBQUksRUFBRSxZQUFZO1lBQ2xCLGVBQWU7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtZQUNoRSxPQUFPLEVBQUUsUUFBUTtZQUNqQixRQUFRLEVBQUUsV0FBVztZQUNyQixZQUFZLEVBQUUsUUFBUSxDQUFDLElBQUk7U0FDNUIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2hFLElBQUksQ0FBQyxjQUFjLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQztRQUMxQyxJQUFJLENBQUMsZUFBZSxHQUFHLFFBQVEsQ0FBQyxlQUFlLENBQUM7SUFDbEQsQ0FBQztJQXhIRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsV0FBbUI7UUFDN0UsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFcEQsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLGdCQUFXLEdBQUcsV0FBVyxDQUFDO2dCQUMxQixpQkFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDO2dCQUN4QyxtQkFBYyxHQUFHLEVBQUUsQ0FBQztZQUN0QyxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsc0JBQXNCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDMUYsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLGdCQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztnQkFDaEMsaUJBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUNsQyxtQkFBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7Z0JBQ3RDLG9CQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztZQUMxRCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRU8sTUFBTSxDQUFDLGFBQWEsQ0FBQyxtQkFBMkIsRUFBRSxLQUFxQjtRQUM3RSxJQUFJLEtBQUssRUFBRTtZQUNULE1BQU0sRUFBRSxZQUFZLEVBQUUsZUFBZSxFQUFFLEdBQUcsS0FBSyxDQUFDO1lBQ2hELE1BQU0saUJBQWlCLEdBQUcsNkJBQTZCLENBQUM7WUFFeEQsSUFBSSxZQUFZLEtBQUssU0FBUyxJQUFJLGVBQWUsS0FBSyxTQUFTLEVBQUU7Z0JBQy9ELE1BQU0sSUFBSSxLQUFLLENBQ2IsOERBQThELENBQy9ELENBQUM7YUFDSDtpQkFBTSxJQUFJLFlBQVksS0FBSyxTQUFTLEVBQUU7Z0JBQ3JDLElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtvQkFDOUIsTUFBTSxJQUFJLEtBQUssQ0FDYixzQ0FBc0MsQ0FDdkMsQ0FBQztpQkFDSDtxQkFBTSxJQUFJLFlBQVksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUU7b0JBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQ2IscUNBQXFDLENBQ3RDLENBQUM7aUJBQ0g7cUJBQU0sSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRTtvQkFDaEQsTUFBTSxJQUFJLEtBQUssQ0FDYixnQ0FBZ0MsaUJBQWlCLEVBQUUsQ0FDcEQsQ0FBQztpQkFDSDtnQkFDRCxPQUFPLEVBQUUsWUFBWSxFQUFFLENBQUM7YUFDekI7aUJBQU0sSUFBSSxlQUFlLEtBQUssU0FBUyxFQUFFO2dCQUN4QywwQ0FBMEM7Z0JBQzFDLE1BQU0sb0JBQW9CLEdBQUcseUNBQXlDLENBQUM7Z0JBQ3ZFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUU7b0JBQy9DLE1BQU0sSUFBSSxLQUFLLENBQ2IsbUNBQW1DLG9CQUFvQixFQUFFLENBQzFELENBQUM7aUJBQ0g7cUJBQU0sSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRTtvQkFDbkQsTUFBTSxJQUFJLEtBQUssQ0FDYixtQ0FBbUMsaUJBQWlCLEVBQUUsQ0FDdkQsQ0FBQztpQkFDSDtnQkFDRCxPQUFPLEVBQUUsWUFBWSxFQUFFLGVBQWUsRUFBRSxlQUFlLEVBQUUsQ0FBQzthQUMzRDtTQUNGO1FBQ0QsT0FBTyxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxDQUFDO0lBQy9DLENBQUM7Q0E4Q0Y7QUEzSEQsNEJBMkhDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuRXZlbnRCdXMgfSBmcm9tICcuL2V2ZW50cy5nZW5lcmF0ZWQnO1xuXG4vKipcbiAqIEludGVyZmFjZSB3aGljaCBhbGwgRXZlbnRCdXMgYmFzZWQgY2xhc3NlcyBNVVNUIGltcGxlbWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElFdmVudEJ1cyBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgcGh5c2ljYWwgSUQgb2YgdGhpcyBldmVudCBidXMgcmVzb3VyY2VcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtZXZlbnRzLWV2ZW50YnVzLmh0bWwjY2ZuLWV2ZW50cy1ldmVudGJ1cy1uYW1lXG4gICAqL1xuICByZWFkb25seSBldmVudEJ1c05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGlzIGV2ZW50IGJ1cyByZXNvdXJjZVxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1ldmVudHMtZXZlbnRidXMuaHRtbCNBcm4tZm46OmdldGF0dFxuICAgKi9cbiAgcmVhZG9ubHkgZXZlbnRCdXNBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEpTT04gcG9saWN5IG9mIHRoaXMgZXZlbnQgYnVzIHJlc291cmNlXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI1BvbGljeS1mbjo6Z2V0YXR0XG4gICAqL1xuICByZWFkb25seSBldmVudEJ1c1BvbGljeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFydG5lciBldmVudCBzb3VyY2UgdG8gYXNzb2NpYXRlIHdpdGggdGhpcyBldmVudCBidXMgcmVzb3VyY2VcbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI2Nmbi1ldmVudHMtZXZlbnRidXMtZXZlbnRzb3VyY2VuYW1lXG4gICAqL1xuICByZWFkb25seSBldmVudFNvdXJjZU5hbWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyB0byBkZWZpbmUgYW4gZXZlbnQgYnVzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXZlbnRCdXNQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZXZlbnQgYnVzIHlvdSBhcmUgY3JlYXRpbmdcbiAgICogTm90ZTogSWYgJ2V2ZW50U291cmNlTmFtZScgaXMgcGFzc2VkIGluLCB5b3UgY2Fubm90IHNldCB0aGlzXG4gICAqXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1ldmVudHMtZXZlbnRidXMuaHRtbCNjZm4tZXZlbnRzLWV2ZW50YnVzLW5hbWVcbiAgICogQGRlZmF1bHQgLSBhdXRvbWF0aWNhbGx5IGdlbmVyYXRlZCBuYW1lXG4gICAqL1xuICByZWFkb25seSBldmVudEJ1c05hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXJ0bmVyIGV2ZW50IHNvdXJjZSB0byBhc3NvY2lhdGUgd2l0aCB0aGlzIGV2ZW50IGJ1cyByZXNvdXJjZVxuICAgKiBOb3RlOiBJZiAnZXZlbnRCdXNOYW1lJyBpcyBwYXNzZWQgaW4sIHlvdSBjYW5ub3Qgc2V0IHRoaXNcbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI2Nmbi1ldmVudHMtZXZlbnRidXMtZXZlbnRzb3VyY2VuYW1lXG4gICAqIEBkZWZhdWx0IC0gbm8gcGFydG5lciBldmVudCBzb3VyY2VcbiAgICovXG4gIHJlYWRvbmx5IGV2ZW50U291cmNlTmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2Ugd2l0aCBwcm9wZXJ0aWVzIG5lY2Vzc2FyeSB0byBpbXBvcnQgYSByZXVzYWJsZSBFdmVudEJ1c1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEV2ZW50QnVzQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBUaGUgcGh5c2ljYWwgSUQgb2YgdGhpcyBldmVudCBidXMgcmVzb3VyY2VcbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI2Nmbi1ldmVudHMtZXZlbnRidXMtbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgZXZlbnRCdXNOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhpcyBldmVudCBidXMgcmVzb3VyY2VcbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI0Fybi1mbjo6Z2V0YXR0XG4gICAqL1xuICByZWFkb25seSBldmVudEJ1c0Fybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSlNPTiBwb2xpY3kgb2YgdGhpcyBldmVudCBidXMgcmVzb3VyY2VcbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI1BvbGljeS1mbjo6Z2V0YXR0XG4gICAqL1xuICByZWFkb25seSBldmVudEJ1c1BvbGljeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFydG5lciBldmVudCBzb3VyY2UgdG8gYXNzb2NpYXRlIHdpdGggdGhpcyBldmVudCBidXMgcmVzb3VyY2VcbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWV2ZW50cy1ldmVudGJ1cy5odG1sI2Nmbi1ldmVudHMtZXZlbnRidXMtZXZlbnRzb3VyY2VuYW1lXG4gICAqIEBkZWZhdWx0IC0gbm8gcGFydG5lciBldmVudCBzb3VyY2VcbiAgICovXG4gIHJlYWRvbmx5IGV2ZW50U291cmNlTmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBEZWZpbmUgYSBDbG91ZFdhdGNoIEV2ZW50QnVzXG4gKlxuICogQHJlc291cmNlIEFXUzo6RXZlbnRzOjpFdmVudEJ1c1xuICovXG5leHBvcnQgY2xhc3MgRXZlbnRCdXMgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElFdmVudEJ1cyB7XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyBldmVudCBidXMgcmVzb3VyY2VcbiAgICogQHBhcmFtIHNjb3BlIFBhcmVudCBjb25zdHJ1Y3RcbiAgICogQHBhcmFtIGlkIENvbnN0cnVjdCBJRFxuICAgKiBAcGFyYW0gZXZlbnRCdXNBcm4gQVJOIG9mIGltcG9ydGVkIGV2ZW50IGJ1c1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tRXZlbnRCdXNBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgZXZlbnRCdXNBcm46IHN0cmluZyk6IElFdmVudEJ1cyB7XG4gICAgY29uc3QgcGFydHMgPSBTdGFjay5vZihzY29wZSkucGFyc2VBcm4oZXZlbnRCdXNBcm4pO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRXZlbnRCdXMge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzQXJuID0gZXZlbnRCdXNBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZXZlbnRCdXNOYW1lID0gcGFydHMucmVzb3VyY2VOYW1lIHx8ICcnO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzUG9saWN5ID0gJyc7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgZXZlbnQgYnVzIHJlc291cmNlXG4gICAqIEBwYXJhbSBzY29wZSBQYXJlbnQgY29uc3RydWN0XG4gICAqIEBwYXJhbSBpZCBDb25zdHJ1Y3QgSURcbiAgICogQHBhcmFtIGF0dHJzIEltcG9ydGVkIGV2ZW50IGJ1cyBwcm9wZXJ0aWVzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21FdmVudEJ1c0F0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEV2ZW50QnVzQXR0cmlidXRlcyk6IElFdmVudEJ1cyB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRXZlbnRCdXMge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzQXJuID0gYXR0cnMuZXZlbnRCdXNBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZXZlbnRCdXNOYW1lID0gYXR0cnMuZXZlbnRCdXNOYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzUG9saWN5ID0gYXR0cnMuZXZlbnRCdXNQb2xpY3k7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZXZlbnRTb3VyY2VOYW1lID0gYXR0cnMuZXZlbnRTb3VyY2VOYW1lO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBldmVudEJ1c1Byb3BzKGRlZmF1bHRFdmVudEJ1c05hbWU6IHN0cmluZywgcHJvcHM/OiBFdmVudEJ1c1Byb3BzKSB7XG4gICAgaWYgKHByb3BzKSB7XG4gICAgICBjb25zdCB7IGV2ZW50QnVzTmFtZSwgZXZlbnRTb3VyY2VOYW1lIH0gPSBwcm9wcztcbiAgICAgIGNvbnN0IGV2ZW50QnVzTmFtZVJlZ2V4ID0gL15bXFwvXFwuXFwtX0EtWmEtejAtOV17MSwyNTZ9JC87XG5cbiAgICAgIGlmIChldmVudEJ1c05hbWUgIT09IHVuZGVmaW5lZCAmJiBldmVudFNvdXJjZU5hbWUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYCdldmVudEJ1c05hbWUnIGFuZCAnZXZlbnRTb3VyY2VOYW1lJyBjYW5ub3QgYm90aCBiZSBwcm92aWRlZGBcbiAgICAgICAgKTtcbiAgICAgIH0gZWxzZSBpZiAoZXZlbnRCdXNOYW1lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKGV2ZW50QnVzTmFtZSA9PT0gJ2RlZmF1bHQnKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgYCdldmVudEJ1c05hbWUnIG11c3Qgbm90IGJlICdkZWZhdWx0J2BcbiAgICAgICAgICApO1xuICAgICAgICB9IGVsc2UgaWYgKGV2ZW50QnVzTmFtZS5pbmRleE9mKCcvJykgPiAtMSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgIGAnZXZlbnRCdXNOYW1lJyBtdXN0IG5vdCBjb250YWluICcvJ2BcbiAgICAgICAgICApO1xuICAgICAgICB9IGVsc2UgaWYgKCFldmVudEJ1c05hbWVSZWdleC50ZXN0KGV2ZW50QnVzTmFtZSkpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgICBgJ2V2ZW50QnVzTmFtZScgbXVzdCBzYXRpc2Z5OiAke2V2ZW50QnVzTmFtZVJlZ2V4fWBcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7IGV2ZW50QnVzTmFtZSB9O1xuICAgICAgfSBlbHNlIGlmIChldmVudFNvdXJjZU5hbWUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAvLyBFeDogYXdzLnBhcnRuZXIvUGFydG5lck5hbWUvYWNjdDEvcmVwbzFcbiAgICAgICAgY29uc3QgZXZlbnRTb3VyY2VOYW1lUmVnZXggPSAvXmF3c1xcLnBhcnRuZXIoXFwvW1xcLlxcLV9BLVphLXowLTldKyl7Mix9JC87XG4gICAgICAgIGlmICghZXZlbnRTb3VyY2VOYW1lUmVnZXgudGVzdChldmVudFNvdXJjZU5hbWUpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgYCdldmVudFNvdXJjZU5hbWUnIG11c3Qgc2F0aXNmeTogJHtldmVudFNvdXJjZU5hbWVSZWdleH1gXG4gICAgICAgICAgKTtcbiAgICAgICAgfSBlbHNlIGlmICghZXZlbnRCdXNOYW1lUmVnZXgudGVzdChldmVudFNvdXJjZU5hbWUpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgYCdldmVudFNvdXJjZU5hbWUnIG11c3Qgc2F0aXNmeTogJHtldmVudEJ1c05hbWVSZWdleH1gXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4geyBldmVudEJ1c05hbWU6IGV2ZW50U291cmNlTmFtZSwgZXZlbnRTb3VyY2VOYW1lIH07XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiB7IGV2ZW50QnVzTmFtZTogZGVmYXVsdEV2ZW50QnVzTmFtZSB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBwaHlzaWNhbCBJRCBvZiB0aGlzIGV2ZW50IGJ1cyByZXNvdXJjZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBldmVudCBidXMsIHN1Y2ggYXM6XG4gICAqIGFybjphd3M6ZXZlbnRzOnVzLWVhc3QtMjoxMjM0NTY3ODkwMTI6ZXZlbnQtYnVzL2F3cy5wYXJ0bmVyL1BhcnRuZXJOYW1lL2FjY3QxL3JlcG8xLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwb2xpY3kgZm9yIHRoZSBldmVudCBidXMgaW4gSlNPTiBmb3JtLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGV2ZW50QnVzUG9saWN5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBwYXJ0bmVyIGV2ZW50IHNvdXJjZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGV2ZW50U291cmNlTmFtZT86IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wcz86IEV2ZW50QnVzUHJvcHMpIHtcbiAgICBjb25zdCB7IGV2ZW50QnVzTmFtZSwgZXZlbnRTb3VyY2VOYW1lIH0gPSBFdmVudEJ1cy5ldmVudEJ1c1Byb3BzKFxuICAgICAgTGF6eS5zdHJpbmdWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMubm9kZS51bmlxdWVJZCB9KSxcbiAgICAgIHByb3BzXG4gICAgKTtcblxuICAgIHN1cGVyKHNjb3BlLCBpZCwgeyBwaHlzaWNhbE5hbWU6IGV2ZW50QnVzTmFtZSB9KTtcblxuICAgIGNvbnN0IGV2ZW50QnVzID0gbmV3IENmbkV2ZW50QnVzKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG5hbWU6IGV2ZW50QnVzTmFtZSxcbiAgICAgIGV2ZW50U291cmNlTmFtZVxuICAgIH0pO1xuXG4gICAgdGhpcy5ldmVudEJ1c0FybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoZXZlbnRCdXMuYXR0ckFybiwge1xuICAgICAgc2VydmljZTogJ2V2ZW50cycsXG4gICAgICByZXNvdXJjZTogJ2V2ZW50LWJ1cycsXG4gICAgICByZXNvdXJjZU5hbWU6IGV2ZW50QnVzLm5hbWVcbiAgICB9KTtcblxuICAgIHRoaXMuZXZlbnRCdXNOYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUoZXZlbnRCdXMucmVmKTtcbiAgICB0aGlzLmV2ZW50QnVzUG9saWN5ID0gZXZlbnRCdXMuYXR0clBvbGljeTtcbiAgICB0aGlzLmV2ZW50U291cmNlTmFtZSA9IGV2ZW50QnVzLmV2ZW50U291cmNlTmFtZTtcbiAgfVxufVxuIl19