"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'Bus'
        }));
        test.done();
    },
    'named event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'myEventBus'
        }));
        test.done();
    },
    'partner event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: 'aws.partner/PartnerName/acct1/repo1'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'aws.partner/PartnerName/acct1/repo1',
            EventSourceName: 'aws.partner/PartnerName/acct1/repo1'
        }));
        test.done();
    },
    'can get bus name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const bus = new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus'
        });
        // WHEN
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                EventBusName: bus.eventBusName
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            EventBusName: { Ref: 'BusEA82B648' }
        }));
        test.done();
    },
    'can get bus arn'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const bus = new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus'
        });
        // WHEN
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                EventBusArn: bus.eventBusArn
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            EventBusArn: { 'Fn::GetAtt': ['BusEA82B648', 'Arn'] }
        }));
        test.done();
    },
    'event bus name cannot be default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'default'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must not be 'default'/);
        test.done();
    },
    'event bus name cannot contain slash'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'my/bus'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must not contain '\/'/);
        test.done();
    },
    'event bus cannot have name and source name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myBus',
            eventSourceName: 'myBus'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' and 'eventSourceName' cannot both be provided/);
        test.done();
    },
    'event bus name cannot be empty string'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: ''
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must satisfy: /);
        test.done();
    },
    'event bus source name must follow pattern'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: 'invalid-partner'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventSourceName' must satisfy: \/\^aws/);
        test.done();
    },
    'event bus source name cannot be empty string'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: ''
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventSourceName' must satisfy: /);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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