"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(id, resource) {
        this.id = id;
        this.resource = resource;
    }
    bind() {
        return {
            id: this.id || '',
            arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
            targetResource: this.resource,
        };
    }
}
module.exports = {
    'default rule'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(10 minutes)",
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'can get rule name'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                RuleName: rule.ruleName
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            RuleName: { Ref: 'MyRuleA44AB831' }
        }));
        test.done();
    },
    'rule with physical name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            ruleName: 'PhysicalName',
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Name: 'PhysicalName'
        }));
        test.done();
    },
    'eventPattern is rendered properly'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                account: ['account1', 'account2'],
                detail: {
                    foo: [1, 2],
                },
                detailType: ['detailType1'],
                id: ['id1', 'id2'],
                region: ['region1', 'region2', 'region3'],
                resources: ['r1'],
                source: ['src1', 'src2'],
                time: ['t1'],
                version: ['0']
            }
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "EventPattern": {
                            account: ['account1', 'account2'],
                            detail: { foo: [1, 2] },
                            'detail-type': ['detailType1'],
                            id: ['id1', 'id2'],
                            region: ['region1', 'region2', 'region3'],
                            resources: ['r1'],
                            source: ['src1', 'src2'],
                            time: ['t1'],
                            version: ['0']
                        },
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'fails synthesis if neither eventPattern nor scheudleExpression are specified'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        new rule_1.Rule(stack, 'Rule');
        test.throws(() => app.synth(), /Either 'eventPattern' or 'schedule' must be defined/);
        test.done();
    },
    'addEventPattern can be used to add filters'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule');
        rule.addEventPattern({
            account: ['12345'],
            detail: {
                foo: ['hello']
            }
        });
        rule.addEventPattern({
            source: ['aws.source'],
            detail: {
                foo: ['bar'],
                goo: {
                    hello: ['world']
                }
            }
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "EventPattern": {
                            "account": [
                                "12345"
                            ],
                            "detail": {
                                "foo": [
                                    "hello",
                                    "bar"
                                ],
                                "goo": {
                                    "hello": [
                                        "world"
                                    ]
                                }
                            },
                            "source": [
                                "aws.source"
                            ]
                        },
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'targets can be added via props or addTarget with input transformer'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
            })
        };
        const t2 = {
            bind: () => ({
                id: '',
                arn: 'ARN2',
                input: lib_1.RuleTargetInput.fromText(`This is ${lib_1.EventField.fromPath('$.detail.bla')}`),
            })
        };
        const rule = new rule_1.Rule(stack, 'EventRule', {
            targets: [t1],
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
        });
        rule.addTarget(t2);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "EventRule5A491D2C": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(5 minutes)",
                        "State": "ENABLED",
                        "Targets": [
                            {
                                "Arn": "ARN1",
                                "Id": "Target0",
                                "KinesisParameters": {
                                    "PartitionKeyPath": "partitionKeyPath"
                                }
                            },
                            {
                                "Arn": "ARN2",
                                "Id": "Target1",
                                "InputTransformer": {
                                    "InputPathsMap": {
                                        "detail-bla": "$.detail.bla"
                                    },
                                    "InputTemplate": "\"This is <detail-bla>\""
                                },
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'input template can contain tokens'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '', arn: 'ARN2', input: lib_1.RuleTargetInput.fromText('Hello, "world"')
            })
        });
        // tokens are used here (FnConcat), but this is a text template so we
        // expect it to be wrapped in double quotes automatically for us.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join('', ['a', 'b']).toString()),
            })
        });
        // jsonTemplate can be used to format JSON documents with replacements
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN3',
                input: lib_1.RuleTargetInput.fromObject({ foo: lib_1.EventField.fromPath('$.detail.bar') }),
            })
        });
        // tokens can also used for JSON templates.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN4',
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join(' ', ['hello', '"world"']).toString()),
            })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "EventRule5A491D2C": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "State": "ENABLED",
                        "ScheduleExpression": "rate(1 minute)",
                        "Targets": [
                            {
                                "Arn": "ARN2",
                                "Id": "Target0",
                                "Input": '"Hello, \\"world\\""',
                            },
                            {
                                "Arn": "ARN1",
                                "Id": "Target1",
                                "Input": "\"ab\"",
                                "KinesisParameters": {
                                    "PartitionKeyPath": "partitionKeyPath"
                                }
                            },
                            {
                                "Arn": "ARN3",
                                "Id": "Target2",
                                "InputTransformer": {
                                    "InputPathsMap": {
                                        "detail-bar": "$.detail.bar"
                                    },
                                    "InputTemplate": "{\"foo\":<detail-bar>}"
                                }
                            },
                            {
                                "Arn": "ARN4",
                                "Id": "Target3",
                                "Input": '"hello \\"world\\""'
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'target can declare role which will be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        const role = new iam.Role(stack, 'SomeRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('nobody')
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN2',
                role,
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            "Targets": [
                {
                    "Arn": "ARN2",
                    "Id": "Target0",
                    "RoleArn": { "Fn::GetAtt": ["SomeRole6DDC54DD", "Arn"] }
                }
            ]
        }));
        test.done();
    },
    'asEventRuleTarget can use the ruleArn and a uniqueId of the rule'(test) {
        const stack = new cdk.Stack();
        let receivedRuleArn = 'FAIL';
        let receivedRuleId = 'FAIL';
        const t1 = {
            bind: (eventRule) => {
                receivedRuleArn = eventRule.ruleArn;
                receivedRuleId = eventRule.node.uniqueId;
                return {
                    id: '',
                    arn: 'ARN1',
                    kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
                };
            }
        };
        const rule = new rule_1.Rule(stack, 'EventRule');
        rule.addTarget(t1);
        test.deepEqual(stack.resolve(receivedRuleArn), stack.resolve(rule.ruleArn));
        test.deepEqual(receivedRuleId, rule.node.uniqueId);
        test.done();
    },
    'fromEventRuleArn'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const importedRule = rule_1.Rule.fromEventRuleArn(stack, 'ImportedRule', 'arn:aws:events:us-east-2:123456789012:rule/example');
        // THEN
        test.deepEqual(importedRule.ruleArn, 'arn:aws:events:us-east-2:123456789012:rule/example');
        test.deepEqual(importedRule.ruleName, 'example');
        test.done();
    },
    'rule can be disabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "State": "DISABLED"
        }));
        test.done();
    },
    'can add multiple targets with the same id'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        rule.addTarget(new SomeTarget());
        rule.addTarget(new SomeTarget());
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    "Arn": "ARN1",
                    "Id": "Target0",
                    "KinesisParameters": {
                        "PartitionKeyPath": "partitionKeyPath"
                    }
                },
                {
                    "Arn": "ARN1",
                    "Id": "Target1",
                    "KinesisParameters": {
                        "PartitionKeyPath": "partitionKeyPath"
                    }
                }
            ]
        }));
        test.done();
    },
    'rule and target must be in the same region'(test) {
        const app = new cdk.App();
        const sourceStack = new cdk.Stack(app, 'SourceStack');
        const rule = new rule_1.Rule(sourceStack, 'Rule');
        const targetStack = new cdk.Stack(app, 'TargetStack', { env: { region: 'us-west-2' } });
        const resource = new cdk.Construct(targetStack, 'Resource');
        test.throws(() => {
            rule.addTarget(new SomeTarget('T', resource));
        }, /Rule and target must be in the same region/);
        test.done();
    },
    'sqsParameters are generated when they are specified in target props'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                sqsParameters: { messageGroupId: 'messageGroupId' }
            })
        };
        new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
            targets: [t1],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    "Arn": "ARN1",
                    "Id": "Target0",
                    "SqsParameters": {
                        "MessageGroupId": "messageGroupId"
                    }
                }
            ]
        }));
        test.done();
    },
    'associate rule with event bus'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                detail: ['detail']
            },
            eventBus,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventBusName: {
                Ref: 'EventBus7B8748AA'
            }
        }));
        test.done();
    },
    'throws with eventBus and schedule'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // THEN
        test.throws(() => new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
            eventBus,
        }), /Cannot associate rule with 'eventBus' when using 'schedule'/);
        test.done();
    },
    'for cross-account targets': {
        'requires that the source stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceStack = new cdk.Stack(app, 'SourceStack');
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the source stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetStack = new cdk.Stack(app, 'TargetStack');
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete region'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete region for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the source stack be part of an App'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(undefined, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack which uses cross-account targets must be part of a CDK app/);
            test.done();
        },
        'requires that the target stack be part of an App'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(undefined, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Target stack which uses cross-account event targets must be part of a CDK app/);
            test.done();
        },
        'requires that the source and target stacks be part of the same App'(test) {
            const sourceApp = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(sourceApp, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetApp = new cdk.App();
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(targetApp, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack and target stack must belong to the same CDK app/);
            test.done();
        },
        'generates the correct rules in the source and target stacks when eventPattern is passed in the constructor'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule', {
                eventPattern: {
                    source: ['some-event'],
                },
            });
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource1 = new cdk.Construct(targetStack, 'Resource1');
            const resource2 = new cdk.Construct(targetStack, 'Resource2');
            rule.addTarget(new SomeTarget('T1', resource1));
            rule.addTarget(new SomeTarget('T2', resource2));
            assert_1.expect(sourceStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T1",
                        "Arn": {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    { "Ref": "AWS::Partition" },
                                    `:events:us-west-2:${targetAccount}:event-bus/default`,
                                ],
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T1",
                        "Arn": "ARN1",
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T2",
                        "Arn": "ARN1",
                    },
                ],
            }));
            const eventBusPolicyStack = app.node.findChild(`EventBusPolicy-${sourceAccount}-us-west-2-${targetAccount}`);
            assert_1.expect(eventBusPolicyStack).to(assert_1.haveResourceLike('AWS::Events::EventBusPolicy', {
                "Action": "events:PutEvents",
                "StatementId": "MySid",
                "Principal": sourceAccount,
            }));
            test.done();
        },
        'generates the correct rule in the target stack when addEventPattern in the source rule is used'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource = new cdk.Construct(targetStack, 'Resource1');
            rule.addTarget(new SomeTarget('T', resource));
            rule.addEventPattern({
                source: ['some-event'],
            });
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T",
                        "Arn": "ARN1",
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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