# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pricehist',
 'pricehist.beanprice',
 'pricehist.outputs',
 'pricehist.resources',
 'pricehist.sources']

package_data = \
{'': ['*']}

install_requires = \
['cssselect>=1.1.0,<2.0.0',
 'curlify>=2.2.1,<3.0.0',
 'lxml>=4.6.2,<5.0.0',
 'requests>=2.25.1,<3.0.0']

entry_points = \
{'console_scripts': ['pricehist = pricehist.cli:cli']}

setup_kwargs = {
    'name': 'pricehist',
    'version': '1.2.5',
    'description': 'Fetch and format historical price data',
    'long_description': '# pricehist\n\nA command-line tool for fetching and formatting historical price data, with\nsupport for multiple data sources and output formats.\n\n[![Pipeline status](https://gitlab.com/chrisberkhout/pricehist/badges/master/pipeline.svg)](https://gitlab.com/chrisberkhout/pricehist/-/commits/master)\n[![Coverage report](https://gitlab.com/chrisberkhout/pricehist/badges/master/coverage.svg)](https://gitlab.com/chrisberkhout/pricehist/-/commits/master)\n[![PyPI version](https://badge.fury.io/py/pricehist.svg)](https://badge.fury.io/py/pricehist)\n[![Downloads](https://pepy.tech/badge/pricehist)](https://pepy.tech/project/pricehist)\n[![License](https://img.shields.io/pypi/l/pricehist)](https://gitlab.com/chrisberkhout/pricehist/-/blob/master/LICENSE)\n[![Code style](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n## Installation\n\nInstall via pip or\n[pipx](https://pypa.github.io/pipx/).\n\n```\npipx install pricehist\n```\n\n## Sources\n\n- **`alphavantage`**: [Alpha Vantage](https://www.alphavantage.co/)\n- **`coinbasepro`**: [Coinbase Pro](https://pro.coinbase.com/)\n- **`coindesk`**: [CoinDesk Bitcoin Price Index](https://www.coindesk.com/coindesk-api)\n- **`coinmarketcap`**: [CoinMarketCap](https://coinmarketcap.com/)\n- **`ecb`**: [European Central Bank Euro foreign exchange reference rates](https://www.ecb.europa.eu/stats/exchange/eurofxref/html/index.en.html)\n- **`yahoo`**: [Yahoo! Finance](https://finance.yahoo.com/)\n\n## Output formats\n\n- **`beancount`**: [Beancount](http://furius.ca/beancount/)\n- **`csv`**: [Comma-separated values](https://en.wikipedia.org/wiki/Comma-separated_values)\n- **`gnucash-sql`**: [GnuCash](https://www.gnucash.org/) SQL\n- **`ledger`**: [Ledger](https://www.ledger-cli.org/) and [hledger](https://hledger.org/)\n\n## How to\n\n### Fetch prices\n\nFetch prices by choosing a source, a pair and, optionally, a time interval.\n\n```\npricehist fetch ecb EUR/AUD -s 2021-01-04 -e 2021-01-08\n```\n```\ndate,base,quote,amount,source,type\n2021-01-04,EUR,AUD,1.5928,ecb,reference\n2021-01-05,EUR,AUD,1.5927,ecb,reference\n2021-01-06,EUR,AUD,1.5824,ecb,reference\n2021-01-07,EUR,AUD,1.5836,ecb,reference\n2021-01-08,EUR,AUD,1.5758,ecb,reference\n```\n\nThe default output format is CSV, which is suitable for use in spreadsheets and\nwith other tools. For example, you can generate a price chart from the command\nline as follows (or using [an alias](https://gitlab.com/-/snippets/2163031)).\n\n```\npricehist fetch coindesk BTC/USD -s 2021-01-01 | \\\n  sed 1d | \\\n  cut -d, -f1,4 | \\\n  gnuplot -p -e \'\n    set datafile separator ",";\n    set xdata time;\n    set timefmt "%Y-%m-%d";\n    set format x "%b\\n%Y";\n    plot "/dev/stdin" using 1:2 with lines title "BTC/USD"\n  \'\n```\n\n![BTC/USD prices](https://gitlab.com/chrisberkhout/pricehist/-/raw/master/example-gnuplot.png)\n\n### Show usage information\n\nAdd `-h` to any command to see usage information.\n\n```\npricehist fetch -h\n```\n```\nusage: pricehist fetch SOURCE PAIR [-h] [-vvv] [-t TYPE] [-s DATE | -sx DATE] [-e DATE | -ex DATE]\n[-o beancount|csv|gnucash-sql|ledger] [--invert] [--quantize INT]\n[--fmt-base SYM] [--fmt-quote SYM] [--fmt-time TIME] [--fmt-decimal CHAR] [--fmt-thousands CHAR]\n[--fmt-symbol rightspace|right|leftspace|left] [--fmt-datesep CHAR] [--fmt-csvdelim CHAR]\n\npositional arguments:\n  SOURCE                   the source identifier\n  PAIR                     pair, usually BASE/QUOTE, e.g. BTC/USD\n\noptional arguments:\n  -h, --help               show this help message and exit\n  -vvv, --verbose          show all log messages\n  -t TYPE, --type TYPE     price type, e.g. close\n  -s DATE, --start DATE    start date, inclusive (default: source start)\n  -sx DATE, --startx DATE  start date, exclusive\n  -e DATE, --end DATE      end date, inclusive (default: today)\n  -ex DATE, --endx DATE    end date, exclusive\n  -o FMT, --output FMT     output format (default: csv)\n  --invert                 invert the price, swapping base and quote\n  --quantize INT           round to the given number of decimal places\n  --fmt-base SYM           rename the base symbol in output\n  --fmt-quote SYM          rename the quote symbol in output\n  --fmt-time TIME          set a particular time of day in output (default: 00:00:00)\n  --fmt-decimal CHAR       decimal point in output (default: \'.\')\n  --fmt-thousands CHAR     thousands separator in output (default: \'\')\n  --fmt-symbol LOCATION    commodity symbol placement in output (default: rightspace)\n  --fmt-datesep CHAR       date separator in output (default: \'-\')\n  --fmt-csvdelim CHAR      field delimiter for CSV output (default: \',\')\n```\n\n### Choose and customize the output format\n\nAs the output format you can choose one of `beancount`, `csv`, `ledger` or\n`gnucash-sql`.\n\n```\npricehist fetch ecb EUR/AUD -s 2021-01-04 -e 2021-01-08 -o ledger\n```\n```\nP 2021-01-04 00:00:00 EUR 1.5928 AUD\nP 2021-01-05 00:00:00 EUR 1.5927 AUD\nP 2021-01-06 00:00:00 EUR 1.5824 AUD\nP 2021-01-07 00:00:00 EUR 1.5836 AUD\nP 2021-01-08 00:00:00 EUR 1.5758 AUD\n```\n\nFormatting options let you control certain details of the output.\n\n```\npricehist fetch ecb EUR/AUD -s 2021-01-04 -e 2021-01-08 -o ledger \\\n  --fmt-time \'\' --fmt-datesep / --fmt-base € --fmt-quote $ --fmt-symbol left\n```\n```\nP 2021/01/04 € $1.5928\nP 2021/01/05 € $1.5927\nP 2021/01/06 € $1.5824\nP 2021/01/07 € $1.5836\nP 2021/01/08 € $1.5758\n```\n\n### Load prices into GnuCash\n\nYou can generate SQL for a GnuCash database and apply it immediately with one\nof the following commands.\n\n```\npricehist fetch ecb EUR/AUD -s 2021-01-01 -o gnucash-sql | sqlite3 Accounts.gnucash\npricehist fetch ecb EUR/AUD -s 2021-01-01 -o gnucash-sql | mysql -u username -p -D databasename\npricehist fetch ecb EUR/AUD -s 2021-01-01 -o gnucash-sql | psql -U username -d databasename -v ON_ERROR_STOP=1\n```\n\nBeware that the GnuCash project itself does not support integration at the\ndatabase level, so there is a risk that the SQL generated by `pricehist` will\nbe ineffective or even damaging for some version of GnuCash. In practice, this\nstrategy has been used successfully by other projects. Reading the SQL and\nkeeping regular database backups is recommended.\n\nThe GnuCash database must already contain commodities with mnemonics matching\nthe base and quote of new prices, otherwise the SQL will fail without making\nchanges.\n\nEach price entry is given a GUID based on its content (date, base, quote,\nsource, type and amount) and existing GUIDs are skipped in the final insert, so\nyou can apply identical or overlapping SQL files multiple times without\ncreating duplicate entries in the database.\n\n### Show source information\n\nThe `source` command shows information about a source.\n\n```\npricehist source alphavantage\n```\n```\nID          : alphavantage\nName        : Alpha Vantage\nDescription : Provider of market data for stocks, forex and cryptocurrencies\nURL         : https://www.alphavantage.co/\nStart       : 1995-01-01\nTypes       : close, open, high, low, adjclose, mid\nNotes       : Alpha Vantage has data on...\n```\n\nAvailable symbols can be listed for most sources, either as full pairs or as\nseparate base and quote symbols that will work in certain combinations.\n\n```\npricehist source ecb --symbols\n```\n```\nEUR/AUD    Euro against Australian Dollar\nEUR/BGN    Euro against Bulgarian Lev\nEUR/BRL    Euro against Brazilian Real\nEUR/CAD    Euro against Canadian Dollar\nEUR/CHF    Euro against Swiss Franc\n...\n```\n\nIt may also be possible to search for symbols.\n\n```\npricehist source alphavantage --search Tesla\n```\n```\nTL0.DEX       Tesla, Equity, XETRA, EUR\nTL0.FRK       Tesla, Equity, Frankfurt, EUR\nTSLA34.SAO    Tesla, Equity, Brazil/Sao Paolo, BRL\nTSLA          Tesla Inc, Equity, United States, USD\nTXLZF         Tesla Exploration Ltd, Equity, United States, USD\n```\n\n### Inspect source interactions\n\nYou can see extra information by adding the verbose option (`--verbose` or\n`-vvv`), including `curl` commands that reproduce each request to a source.\n\n```\npricehist fetch coindesk BTC/USD -s 2021-01-01 -e 2021-01-05 -vvv\n```\n```\nDEBUG Began pricehist run at 2021-08-12 14:38:26.630357.\nDEBUG Starting new HTTPS connection (1): api.coindesk.com:443\nDEBUG https://api.coindesk.com:443 "GET /v1/bpi/historical/close.json?currency=USD&start=2021-01-01&end=2021-01-05 HTTP/1.1" 200 319\nDEBUG curl -X GET -H \'Accept: */*\' -H \'Accept-Encoding: gzip, deflate\' -H \'Connection: keep-alive\' -H \'User-Agent: python-requests/2.25.1\' --compressed \'https://api.coindesk.com/v1/bpi/historical/close.json?currency=USD&start=2021-01-01&end=2021-01-05\'\nDEBUG Available data covers the interval [2021-01-01--2021-01-05], as requested.\ndate,base,quote,amount,source,type\n2021-01-01,BTC,USD,29391.775,coindesk,close\n2021-01-02,BTC,USD,32198.48,coindesk,close\n2021-01-03,BTC,USD,33033.62,coindesk,close\n2021-01-04,BTC,USD,32017.565,coindesk,close\n2021-01-05,BTC,USD,34035.0067,coindesk,close\nDEBUG Ended pricehist run at 2021-08-12 14:38:26.709428.\n```\n\nRunning a logged `curl` command shows exactly what data is returned by the\nsource.\n\n```\npricehist fetch coindesk BTC/USD -s 2021-01-01 -e 2021-01-05 -vvv 2>&1 \\\n  | grep \'^DEBUG curl\' | sed \'s/^DEBUG //\' | bash | jq .\n```\n```json\n{\n  "bpi": {\n    "2021-01-01": 29391.775,\n    "2021-01-02": 32198.48,\n    "2021-01-03": 33033.62,\n    "2021-01-04": 32017.565,\n    "2021-01-05": 34035.0067\n  },\n  "disclaimer": "This data was produced from the CoinDesk Bitcoin Price Index. BPI value data returned as USD.",\n  "time": {\n    "updated": "Jan 6, 2021 00:03:00 UTC",\n    "updatedISO": "2021-01-06T00:03:00+00:00"\n  }\n}\n```\n\n### Use via `bean-price`\n\nBeancount users may wish to use `pricehist` sources via `bean-price`. To do so,\nensure the `pricehist` package is installed in an accessible location.\n\nYou can fetch the latest price directly from the command line.\n\n```\nbean-price -e "USD:pricehist.beanprice.coindesk/BTC:USD"\n```\n```\n2021-08-18 price BTC:USD                          44725.12 USD\n```\n\nYou can fetch a series of prices by providing a Beancount file as input.\n\n```\n; input.beancount\n2021-08-14 commodity BTC\n  price: "USD:pricehist.beanprice.coindesk/BTC:USD:close"\n```\n\n```\nbean-price input.beancount --update --update-rate daily --inactive --clear-cache\n```\n```\n2021-08-14 price BTC                            47098.2633 USD\n2021-08-15 price BTC                            47018.9017 USD\n2021-08-16 price BTC                             45927.405 USD\n2021-08-17 price BTC                            44686.3333 USD\n2021-08-18 price BTC                              44725.12 USD\n```\n\nAdding `-v` will print progress information, `-vv` will print debug information,\nincluding that from `pricehist`.\n\nA source map specification for `bean-price` has the form\n`<currency>:<module>/[^]<ticker>`. Additional `<module>/[^]<ticker>` parts can\nbe appended, separated by commas.\n\nThe module name will be of the form `pricehist.beanprice.<source_id>`.\n\nThe ticker symbol will be of the form `BASE:QUOTE:TYPE`.\n\nAny non-alphanumeric characters except the equals sign (`=`), hyphen (`-`),\nperiod (`.`), or parentheses (`(` or `)`) are special characters that need to\nbe encoded as their a two-digit hexadecimal code prefixed with an underscore,\nbecause `bean-price` ticker symbols don\'t allow all the characters used by\n`pricehist` pairs.\n[This page](https://replit.com/@chrisberkhout/bpticker) will do it for you.\n\nFor example, the Yahoo! Finance symbol for the Dow Jones Industrial Average is\n`^DJI`, and would have the source map specification\n`USD:pricehist.beanprice.yahoo/_5eDJI`, or for the daily high price\n`USD:pricehist.beanprice.yahoo/_5eDJI::high`.\n\n### Use as a library\n\nYou may find `pricehist`\'s source classes useful in your own scripts.\n\n```\n$\xa0python\nPython 3.9.6 (default, Jun 30 2021, 10:22:16)\n[GCC 11.1.0] on linux\nType "help", "copyright", "credits" or "license" for more information.\n>>> from pricehist.series import Series\n>>> from pricehist.sources.ecb import ECB\n>>> series = ECB().fetch(Series("EUR", "AUD", "reference", "2021-01-04", "2021-01-08"))\n>>> series.prices\n[Price(date=\'2021-01-04\', amount=Decimal(\'1.5928\')), Price(date=\'2021-01-05\', amount=Decimal(\'1.5927\')), Price(date=\'2021-01-06\', amount=Decimal(\'1.5824\')), Price(date=\'2021-01-07\', amount=Decimal(\'1.5836\')), Price(date=\'2021-01-08\', amount=Decimal(\'1.5758\'))]\n```\n\nA subclass of `pricehist.exceptions.SourceError` will be raised for any error.\n\n### Contribute\n\nContributions are welcome! If you discover a bug or want to work on a\nnon-trivial change, please open a\n[GitLab issue](https://gitlab.com/chrisberkhout/pricehist/-/issues)\nto discuss it.\n\nRun `make install-pre-commit-hook` set up local pre-commit checks.\nSet up your editor to run\n[isort](https://pycqa.github.io/isort/),\n[Black](https://black.readthedocs.io/en/stable/) and\n[Flake8](https://flake8.pycqa.org/en/latest/),\nor run them manually via `make format lint`.\n\n## Terminology\n\nA **source** is an upstream service that can provide a series of prices.\n\nEach **series** of prices is for one pair and price type.\n\nThe [**pair**](https://en.wikipedia.org/wiki/Currency_pair) is made up of a\nbase and a quote, each given as a symbol. Sometimes you will give the base\nonly, and the quote will be determined with information from the source. The\navailable pairs, the symbols used in them and the available price types all\ndepend on the particular source used.\n\nThe **base** is the currency or commodity being valued. Each price expresses\nthe value of one unit of the base.\n\nThe **quote** is the unit used to express the value of the base.\n\nA **symbol** is a code or abbreviation for a currency or commodity.\n\nThe **prices** in a series each have a date and an amount.\n\nThe **amount** is the number of units of the quote that are equal to one unit\nof the base.\n\nConsider the following command.\n\n```\npricehist fetch coindesk BTC/USD --type close\n```\n\n- **`coindesk`** is the ID of the CoinDesk Bitcoin Price Index source.\n- **`BTC`** is the symbol for Bitcoin, used here as the base.\n- **`USD`** is the symbol for the United States Dollar, used here as the quote.\n- **`BTC/USD`** is the pair Bitcoin against United States Dollar.\n- **`close`** is the price type for the last price of each day.\n\nA BTC/USD price of the amount 29,391.775 can be written as\n"BTC/USD = 29391.78" or "BTC 29391.78 USD", and means that one Bitcoin is\nworth 29,391.78 United States Dollars.\n\n## Initial design choices\n\nTo keep things simple, `pricehist` provides only univariate time series of\ndaily historical prices. It doesn\'t provide other types of market, financial or\neconomic data, real-time prices, or other temporal resolutions. Multiple or\nmultivariate series require multiple invocations.\n\n## Potential features\n\nIn the future, `pricehist` may be extended to cover some of the following\nfeatures:\n\n- **Time of day**: Sources sometimes provide specific times for each day\'s\n  high/low prices and these could be preserved for output. This would require\n  changes to how dates are handled internally, clarification of time zone\n  handling and extension of the time formatting option.\n- **Alternate resolutions**: Some sources can provide higher or lower\n  resolution data, such as hourly or weekly. These could be supported where\n  available. For other cases an option could be provided for downsampling data\n  before output.\n- **Real-time prices**: These generally come from different source endpoints\n  than the historical data. Real-time prices will usually have a different\n  price type, such as `last`, `bid` or `ask`. Support for real-time prices\n  would allow adding sources that don\'t provide historical data. Start and end\n  times are irrelevant when requesting real-time prices. A "follow" option\n  could continuously poll for new prices.\n- **Related non-price data**: Trading volume, spreads, split and dividend\n  events and other related data could be supported. The base/quote/type model\n  used for prices would work for some of this. Other things may require\n  extending the model.\n- **Multivariate series**: Would allow, for example, fetching\n  high/low/open/close prices in a single invocation.\n- **`format` command**: A command for rewriting existing CSV data into one of\n  the other output formats.\n\n## Alternatives\n\nBeancount\'s [`bean-price`](https://github.com/beancount/beanprice) tool fetches\nprices and addresses other workflow concerns in a Beancount-specific manner,\ngenerally requiring a Beancount file as input.\n\nThe [Piecash](https://piecash.readthedocs.io/) library is a pythonic interface\nto GnuCash files stored in SQL which has a\n[`Commodity.update_prices`](https://piecash.readthedocs.io/en/master/api/piecash.core.commodity.html?highlight=update_prices#piecash.core.commodity.Commodity.update_prices)\nmethod for fetching historical prices.\nThe GnuCash wiki documents [wrapper scripts](https://wiki.gnucash.org/wiki/Stocks/get_prices)\nfor the [Finance::QuoteHist](https://metacpan.org/pod/Finance::QuoteHist) Perl\nmodule.\n',
    'author': 'Chris Berkhout',
    'author_email': 'chris@chrisberkhout.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/chrisberkhout/pricehist',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
