"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudwatch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudwatch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudwatch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudwatch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudwatch';
        this.accessLevelList = {
            "Write": [
                "DeleteAlarms",
                "DeleteAnomalyDetector",
                "DeleteDashboards",
                "DeleteInsightRules",
                "DisableAlarmActions",
                "DisableInsightRules",
                "EnableAlarmActions",
                "EnableInsightRules",
                "PutAnomalyDetector",
                "PutDashboard",
                "PutInsightRule",
                "PutMetricAlarm",
                "PutMetricData",
                "SetAlarmState"
            ],
            "Read": [
                "DescribeAlarmHistory",
                "DescribeAlarms",
                "DescribeAlarmsForMetric",
                "DescribeAnomalyDetectors",
                "DescribeInsightRules",
                "GetDashboard",
                "GetInsightRuleReport",
                "GetMetricData",
                "GetMetricStatistics",
                "GetMetricWidgetImage"
            ],
            "List": [
                "ListDashboards",
                "ListMetrics",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Deletes all specified alarms. In the event of an error, no alarms are deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteAlarms.html
     */
    toDeleteAlarms() {
        this.to('cloudwatch:DeleteAlarms');
        return this;
    }
    /**
     * Deletes the specified anomaly detection model from your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteAnomalyDetector.html
     */
    toDeleteAnomalyDetector() {
        this.to('cloudwatch:DeleteAnomalyDetector');
        return this;
    }
    /**
     * Deletes all CloudWatch dashboards that you specify
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteDashboards.html
     */
    toDeleteDashboards() {
        this.to('cloudwatch:DeleteDashboards');
        return this;
    }
    /**
     * Grants permission to delete a collection of insight rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteInsightRules.html
     */
    toDeleteInsightRules() {
        this.to('cloudwatch:DeleteInsightRules');
        return this;
    }
    /**
     * Retrieves history for the specified alarm
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmHistory.html
     */
    toDescribeAlarmHistory() {
        this.to('cloudwatch:DescribeAlarmHistory');
        return this;
    }
    /**
     * Retrieves alarms with the specified names
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html
     */
    toDescribeAlarms() {
        this.to('cloudwatch:DescribeAlarms');
        return this;
    }
    /**
     * Retrieves all alarms for a single metric
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmsForMetric.html
     */
    toDescribeAlarmsForMetric() {
        this.to('cloudwatch:DescribeAlarmsForMetric');
        return this;
    }
    /**
     * Lists the anomaly detection models that you have created in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAnomalyDetectors.html
     */
    toDescribeAnomalyDetectors() {
        this.to('cloudwatch:DescribeAnomalyDetectors');
        return this;
    }
    /**
     * Grants permission to describe all insight rules, currently owned by the user's account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeInsightRules.html
     */
    toDescribeInsightRules() {
        this.to('cloudwatch:DescribeInsightRules');
        return this;
    }
    /**
     * Disables actions for the specified alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DisableAlarmActions.html
     */
    toDisableAlarmActions() {
        this.to('cloudwatch:DisableAlarmActions');
        return this;
    }
    /**
     * Grants permission to disable a collection of insight rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DisableInsightRules.html
     */
    toDisableInsightRules() {
        this.to('cloudwatch:DisableInsightRules');
        return this;
    }
    /**
     * Enables actions for the specified alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_EnableAlarmActions.html
     */
    toEnableAlarmActions() {
        this.to('cloudwatch:EnableAlarmActions');
        return this;
    }
    /**
     * Grants permission to enable a collection of insight rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_EnableInsightRules.html
     */
    toEnableInsightRules() {
        this.to('cloudwatch:EnableInsightRules');
        return this;
    }
    /**
     * Displays the details of the CloudWatch dashboard you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetDashboard.html
     */
    toGetDashboard() {
        this.to('cloudwatch:GetDashboard');
        return this;
    }
    /**
     * Grants permission to return the top-N report of unique contributors over a time range for a given insight rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetInsightRuleReport.html
     */
    toGetInsightRuleReport() {
        this.to('cloudwatch:GetInsightRuleReport');
        return this;
    }
    /**
     * Required to retrieve batch amounts of CloudWatch metric data and perform metric math on retrieved data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html
     */
    toGetMetricData() {
        this.to('cloudwatch:GetMetricData');
        return this;
    }
    /**
     * Gets statistics for the specified metric
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html
     */
    toGetMetricStatistics() {
        this.to('cloudwatch:GetMetricStatistics');
        return this;
    }
    /**
     * Required to retrieve snapshots of metric widgets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricWidgetImage.html
     */
    toGetMetricWidgetImage() {
        this.to('cloudwatch:GetMetricWidgetImage');
        return this;
    }
    /**
     * Returns a list of all CloudWatch dashboards in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        this.to('cloudwatch:ListDashboards');
        return this;
    }
    /**
     * Returns a list of valid metrics stored for the AWS account owner
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html
     */
    toListMetrics() {
        this.to('cloudwatch:ListMetrics');
        return this;
    }
    /**
     * This action lists tags for an Amazon CloudWatch resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('cloudwatch:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates an anomaly detection model for a CloudWatch metric.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutAnomalyDetector.html
     */
    toPutAnomalyDetector() {
        this.to('cloudwatch:PutAnomalyDetector');
        return this;
    }
    /**
     * Creates a CloudWatch dashboard, or updates an existing dashboard if it already exists
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutDashboard.html
     */
    toPutDashboard() {
        this.to('cloudwatch:PutDashboard');
        return this;
    }
    /**
     * Grants permission to create a new insight rule or replace an existing insight rule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutInsightRule.html
     */
    toPutInsightRule() {
        this.to('cloudwatch:PutInsightRule');
        return this;
    }
    /**
     * Creates or updates an alarm and associates it with the specified Amazon CloudWatch metric
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricAlarm.html
     */
    toPutMetricAlarm() {
        this.to('cloudwatch:PutMetricAlarm');
        return this;
    }
    /**
     * Publishes metric data points to Amazon CloudWatch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricData.html
     */
    toPutMetricData() {
        this.to('cloudwatch:PutMetricData');
        return this;
    }
    /**
     * Temporarily sets the state of an alarm for testing purposes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_SetAlarmState.html
     */
    toSetAlarmState() {
        this.to('cloudwatch:SetAlarmState');
        return this;
    }
    /**
     * This action tags an Amazon CloudWatch resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('cloudwatch:TagResource');
        return this;
    }
    /**
     * This action removes a tag from an Amazon CloudWatch resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('cloudwatch:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type alarm to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param alarmName - Identifier for the alarmName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlarm(alarmName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudwatch:${Region}:${Account}:alarm:${AlarmName}';
        arn = arn.replace('${AlarmName}', alarmName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param dashboardName - Identifier for the dashboardName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDashboard(dashboardName, account, partition) {
        var arn = 'arn:${Partition}:cloudwatch::${Account}:dashboard/${DashboardName}';
        arn = arn.replace('${DashboardName}', dashboardName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type insight-rule to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param insightRuleName - Identifier for the insightRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInsightRule(insightRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudwatch:${Region}:${Account}:insight-rule/${InsightRuleName}';
        arn = arn.replace('${InsightRuleName}', insightRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of optional namespace values
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/iam-cw-condition-keys-namespace.html
     *
     * Applies to actions:
     * - .toPutMetricData()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespace(value, operator) {
        return this.if(`cloudwatch:namespace`, value, operator || 'StringLike');
    }
}
exports.Cloudwatch = Cloudwatch;
//# sourceMappingURL=data:application/json;base64,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