"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeguruReviewer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codeguru-reviewer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodegurureviewer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodeguruReviewer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codeguru-reviewer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodegurureviewer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codeguru-reviewer';
        this.accessLevelList = {
            "Write": [
                "AssociateRepository",
                "DisassociateRepository",
                "PutRecommendationFeedback"
            ],
            "Read": [
                "CreateConnectionToken",
                "DescribeCodeReview",
                "DescribeRecommendationFeedback",
                "DescribeRepositoryAssociation",
                "GetMetricsData",
                "ListThirdPartyRepositories"
            ],
            "List": [
                "ListCodeReviews",
                "ListRecommendationFeedback",
                "ListRecommendations",
                "ListRepositoryAssociations"
            ]
        };
    }
    /**
     * Grants permission to associates a repository with Amazon CodeGuru Reviewer.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - codecommit:ListRepositories
     * - codecommit:TagResource
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     */
    toAssociateRepository() {
        this.to('codeguru-reviewer:AssociateRepository');
        return this;
    }
    /**
     * Grants permission to perform webbased oauth handshake for 3rd party providers.
     *
     * Access Level: Read
     */
    toCreateConnectionToken() {
        this.to('codeguru-reviewer:CreateConnectionToken');
        return this;
    }
    /**
     * Grants permission to describe a code review.
     *
     * Access Level: Read
     */
    toDescribeCodeReview() {
        this.to('codeguru-reviewer:DescribeCodeReview');
        return this;
    }
    /**
     * Grants permission to describe a recommendation feedback on a code review.
     *
     * Access Level: Read
     */
    toDescribeRecommendationFeedback() {
        this.to('codeguru-reviewer:DescribeRecommendationFeedback');
        return this;
    }
    /**
     * Grants permission to describe a repository association.
     *
     * Access Level: Read
     */
    toDescribeRepositoryAssociation() {
        this.to('codeguru-reviewer:DescribeRepositoryAssociation');
        return this;
    }
    /**
     * Grants permission to disassociate a repository with Amazon CodeGuru Reviewer.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - codecommit:UntagResource
     * - events:DeleteRule
     * - events:RemoveTargets
     */
    toDisassociateRepository() {
        this.to('codeguru-reviewer:DisassociateRepository');
        return this;
    }
    /**
     * Grants permission to view pull request metrics in console.
     *
     * Access Level: Read
     */
    toGetMetricsData() {
        this.to('codeguru-reviewer:GetMetricsData');
        return this;
    }
    /**
     * Grants permission to list summary of code reviews.
     *
     * Access Level: List
     */
    toListCodeReviews() {
        this.to('codeguru-reviewer:ListCodeReviews');
        return this;
    }
    /**
     * Grants permission to list summary of recommendation feedback on a code review.
     *
     * Access Level: List
     */
    toListRecommendationFeedback() {
        this.to('codeguru-reviewer:ListRecommendationFeedback');
        return this;
    }
    /**
     * Grants permission to list summary of recommendations on a code review
     *
     * Access Level: List
     */
    toListRecommendations() {
        this.to('codeguru-reviewer:ListRecommendations');
        return this;
    }
    /**
     * Grants permission to list summary of repository associations.
     *
     * Access Level: List
     */
    toListRepositoryAssociations() {
        this.to('codeguru-reviewer:ListRepositoryAssociations');
        return this;
    }
    /**
     * Grants permission to list 3rd party providers repositories in console.
     *
     * Access Level: Read
     */
    toListThirdPartyRepositories() {
        this.to('codeguru-reviewer:ListThirdPartyRepositories');
        return this;
    }
    /**
     * Grants permission to put feedback for a recommendation on a code review.
     *
     * Access Level: Write
     */
    toPutRecommendationFeedback() {
        this.to('codeguru-reviewer:PutRecommendationFeedback');
        return this;
    }
    /**
     * Adds a resource of type association to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssociation(resourceId, account, partition) {
        var arn = 'arn:${Partition}:codeguru-reviewer::${Account}:association:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type codereview to the statement
     *
     * @param codeReviewUuid - Identifier for the codeReviewUuid.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCodereview(codeReviewUuid, account, region, partition) {
        var arn = 'arn:${Partition}:codeguru-reviewer:${Region}:${Account}:code-review:${CodeReviewUuid}';
        arn = arn.replace('${CodeReviewUuid}', codeReviewUuid);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepository(repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}';
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CodeguruReviewer = CodeguruReviewer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uY29kZWd1cnVyZXZpZXdlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFtYXpvbmNvZGVndXJ1cmV2aWV3ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esc0NBQTRDO0FBRTVDOzs7O0dBSUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLHdCQUFlO0lBR25EOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxtQkFBbUIsQ0FBQztRQXlKakMsb0JBQWUsR0FBb0I7WUFDM0MsT0FBTyxFQUFFO2dCQUNQLHFCQUFxQjtnQkFDckIsd0JBQXdCO2dCQUN4QiwyQkFBMkI7YUFDNUI7WUFDRCxNQUFNLEVBQUU7Z0JBQ04sdUJBQXVCO2dCQUN2QixvQkFBb0I7Z0JBQ3BCLGdDQUFnQztnQkFDaEMsK0JBQStCO2dCQUMvQixnQkFBZ0I7Z0JBQ2hCLDRCQUE0QjthQUM3QjtZQUNELE1BQU0sRUFBRTtnQkFDTixpQkFBaUI7Z0JBQ2pCLDRCQUE0QjtnQkFDNUIscUJBQXFCO2dCQUNyQiw0QkFBNEI7YUFDN0I7U0FDRixDQUFDO0lBcEtGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFDakQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGdDQUFnQztRQUNyQyxJQUFJLENBQUMsRUFBRSxDQUFDLGtEQUFrRCxDQUFDLENBQUM7UUFDNUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLCtCQUErQjtRQUNwQyxJQUFJLENBQUMsRUFBRSxDQUFDLGlEQUFpRCxDQUFDLENBQUM7UUFDM0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxFQUFFLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUF3QkQ7Ozs7OztPQU1HO0lBQ0ksYUFBYSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUMzRSxJQUFJLEdBQUcsR0FBRywwRUFBMEUsQ0FBQztRQUNyRixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLFlBQVksQ0FBQyxjQUFzQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9GLElBQUksR0FBRyxHQUFHLHVGQUF1RixDQUFDO1FBQ2xHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3ZELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0YsSUFBSSxHQUFHLEdBQUcsb0VBQW9FLENBQUM7UUFDL0UsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsbUJBQW1CLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDdkQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7Q0FDRjtBQXJPRCw0Q0FxT0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tIFwiLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbFwiO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50IH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbY29kZWd1cnUtcmV2aWV3ZXJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmNvZGVndXJ1cmV2aWV3ZXIuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBDb2RlZ3VydVJldmlld2VyIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnY29kZWd1cnUtcmV2aWV3ZXInO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2NvZGVndXJ1LXJldmlld2VyXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jb2RlZ3VydXJldmlld2VyLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZXMgYSByZXBvc2l0b3J5IHdpdGggQW1hem9uIENvZGVHdXJ1IFJldmlld2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGNvZGVjb21taXQ6TGlzdFJlcG9zaXRvcmllc1xuICAgKiAtIGNvZGVjb21taXQ6VGFnUmVzb3VyY2VcbiAgICogLSBldmVudHM6UHV0UnVsZVxuICAgKiAtIGV2ZW50czpQdXRUYXJnZXRzXG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVSZXBvc2l0b3J5KCkge1xuICAgIHRoaXMudG8oJ2NvZGVndXJ1LXJldmlld2VyOkFzc29jaWF0ZVJlcG9zaXRvcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBwZXJmb3JtIHdlYmJhc2VkIG9hdXRoIGhhbmRzaGFrZSBmb3IgM3JkIHBhcnR5IHByb3ZpZGVycy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDb25uZWN0aW9uVG9rZW4oKSB7XG4gICAgdGhpcy50bygnY29kZWd1cnUtcmV2aWV3ZXI6Q3JlYXRlQ29ubmVjdGlvblRva2VuJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBjb2RlIHJldmlldy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUNvZGVSZXZpZXcoKSB7XG4gICAgdGhpcy50bygnY29kZWd1cnUtcmV2aWV3ZXI6RGVzY3JpYmVDb2RlUmV2aWV3Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSByZWNvbW1lbmRhdGlvbiBmZWVkYmFjayBvbiBhIGNvZGUgcmV2aWV3LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUmVjb21tZW5kYXRpb25GZWVkYmFjaygpIHtcbiAgICB0aGlzLnRvKCdjb2RlZ3VydS1yZXZpZXdlcjpEZXNjcmliZVJlY29tbWVuZGF0aW9uRmVlZGJhY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhIHJlcG9zaXRvcnkgYXNzb2NpYXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVSZXBvc2l0b3J5QXNzb2NpYXRpb24oKSB7XG4gICAgdGhpcy50bygnY29kZWd1cnUtcmV2aWV3ZXI6RGVzY3JpYmVSZXBvc2l0b3J5QXNzb2NpYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYSByZXBvc2l0b3J5IHdpdGggQW1hem9uIENvZGVHdXJ1IFJldmlld2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGNvZGVjb21taXQ6VW50YWdSZXNvdXJjZVxuICAgKiAtIGV2ZW50czpEZWxldGVSdWxlXG4gICAqIC0gZXZlbnRzOlJlbW92ZVRhcmdldHNcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZVJlcG9zaXRvcnkoKSB7XG4gICAgdGhpcy50bygnY29kZWd1cnUtcmV2aWV3ZXI6RGlzYXNzb2NpYXRlUmVwb3NpdG9yeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgcHVsbCByZXF1ZXN0IG1ldHJpY3MgaW4gY29uc29sZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNZXRyaWNzRGF0YSgpIHtcbiAgICB0aGlzLnRvKCdjb2RlZ3VydS1yZXZpZXdlcjpHZXRNZXRyaWNzRGF0YScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3Qgc3VtbWFyeSBvZiBjb2RlIHJldmlld3MuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENvZGVSZXZpZXdzKCkge1xuICAgIHRoaXMudG8oJ2NvZGVndXJ1LXJldmlld2VyOkxpc3RDb2RlUmV2aWV3cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3Qgc3VtbWFyeSBvZiByZWNvbW1lbmRhdGlvbiBmZWVkYmFjayBvbiBhIGNvZGUgcmV2aWV3LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZWNvbW1lbmRhdGlvbkZlZWRiYWNrKCkge1xuICAgIHRoaXMudG8oJ2NvZGVndXJ1LXJldmlld2VyOkxpc3RSZWNvbW1lbmRhdGlvbkZlZWRiYWNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBzdW1tYXJ5IG9mIHJlY29tbWVuZGF0aW9ucyBvbiBhIGNvZGUgcmV2aWV3XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFJlY29tbWVuZGF0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdjb2RlZ3VydS1yZXZpZXdlcjpMaXN0UmVjb21tZW5kYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBzdW1tYXJ5IG9mIHJlcG9zaXRvcnkgYXNzb2NpYXRpb25zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZXBvc2l0b3J5QXNzb2NpYXRpb25zKCkge1xuICAgIHRoaXMudG8oJ2NvZGVndXJ1LXJldmlld2VyOkxpc3RSZXBvc2l0b3J5QXNzb2NpYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCAzcmQgcGFydHkgcHJvdmlkZXJzIHJlcG9zaXRvcmllcyBpbiBjb25zb2xlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUaGlyZFBhcnR5UmVwb3NpdG9yaWVzKCkge1xuICAgIHRoaXMudG8oJ2NvZGVndXJ1LXJldmlld2VyOkxpc3RUaGlyZFBhcnR5UmVwb3NpdG9yaWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcHV0IGZlZWRiYWNrIGZvciBhIHJlY29tbWVuZGF0aW9uIG9uIGEgY29kZSByZXZpZXcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICovXG4gIHB1YmxpYyB0b1B1dFJlY29tbWVuZGF0aW9uRmVlZGJhY2soKSB7XG4gICAgdGhpcy50bygnY29kZWd1cnUtcmV2aWV3ZXI6UHV0UmVjb21tZW5kYXRpb25GZWVkYmFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFwiV3JpdGVcIjogW1xuICAgICAgXCJBc3NvY2lhdGVSZXBvc2l0b3J5XCIsXG4gICAgICBcIkRpc2Fzc29jaWF0ZVJlcG9zaXRvcnlcIixcbiAgICAgIFwiUHV0UmVjb21tZW5kYXRpb25GZWVkYmFja1wiXG4gICAgXSxcbiAgICBcIlJlYWRcIjogW1xuICAgICAgXCJDcmVhdGVDb25uZWN0aW9uVG9rZW5cIixcbiAgICAgIFwiRGVzY3JpYmVDb2RlUmV2aWV3XCIsXG4gICAgICBcIkRlc2NyaWJlUmVjb21tZW5kYXRpb25GZWVkYmFja1wiLFxuICAgICAgXCJEZXNjcmliZVJlcG9zaXRvcnlBc3NvY2lhdGlvblwiLFxuICAgICAgXCJHZXRNZXRyaWNzRGF0YVwiLFxuICAgICAgXCJMaXN0VGhpcmRQYXJ0eVJlcG9zaXRvcmllc1wiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0Q29kZVJldmlld3NcIixcbiAgICAgIFwiTGlzdFJlY29tbWVuZGF0aW9uRmVlZGJhY2tcIixcbiAgICAgIFwiTGlzdFJlY29tbWVuZGF0aW9uc1wiLFxuICAgICAgXCJMaXN0UmVwb3NpdG9yeUFzc29jaWF0aW9uc1wiXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBhc3NvY2lhdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkFzc29jaWF0aW9uKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmNvZGVndXJ1LXJldmlld2VyOjoke0FjY291bnR9OmFzc29jaWF0aW9uOiR7UmVzb3VyY2VJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlSWR9JywgcmVzb3VyY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjb2RlcmV2aWV3IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGNvZGVSZXZpZXdVdWlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNvZGVSZXZpZXdVdWlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uQ29kZXJldmlldyhjb2RlUmV2aWV3VXVpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpjb2RlZ3VydS1yZXZpZXdlcjoke1JlZ2lvbn06JHtBY2NvdW50fTpjb2RlLXJldmlldzoke0NvZGVSZXZpZXdVdWlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Q29kZVJldmlld1V1aWR9JywgY29kZVJldmlld1V1aWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJlcG9zaXRvcnkgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Jldmlld2VyLXVnL2F1dGgtYW5kLWFjY2Vzcy1jb250cm9sLWlhbS1hY2Nlc3MtY29udHJvbC1pZGVudGl0eS1iYXNlZC5odG1sI2Fybi1mb3JtYXRzXG4gICAqXG4gICAqIEBwYXJhbSByZXBvc2l0b3J5TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSByZXBvc2l0b3J5TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXBvc2l0b3J5KHJlcG9zaXRvcnlOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmNvZGVjb21taXQ6JHtSZWdpb259OiR7QWNjb3VudH06JHtSZXBvc2l0b3J5TmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlcG9zaXRvcnlOYW1lfScsIHJlcG9zaXRvcnlOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19