"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Comprehend = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [comprehend](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Comprehend extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [comprehend](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'comprehend';
        this.accessLevelList = {
            "Read": [
                "BatchDetectDominantLanguage",
                "BatchDetectEntities",
                "BatchDetectKeyPhrases",
                "BatchDetectSentiment",
                "BatchDetectSyntax",
                "ClassifyDocument",
                "DescribeDocumentClassificationJob",
                "DescribeDocumentClassifier",
                "DescribeDominantLanguageDetectionJob",
                "DescribeEndpoint",
                "DescribeEntitiesDetectionJob",
                "DescribeEntityRecognizer",
                "DescribeKeyPhrasesDetectionJob",
                "DescribePiiEntitiesDetectionJob",
                "DescribeSentimentDetectionJob",
                "DescribeTopicsDetectionJob",
                "DetectDominantLanguage",
                "DetectEntities",
                "DetectKeyPhrases",
                "DetectPiiEntities",
                "DetectSentiment",
                "DetectSyntax"
            ],
            "Write": [
                "CreateDocumentClassifier",
                "CreateEndpoint",
                "CreateEntityRecognizer",
                "DeleteDocumentClassifier",
                "DeleteEndpoint",
                "DeleteEntityRecognizer",
                "StartDocumentClassificationJob",
                "StartDominantLanguageDetectionJob",
                "StartEntitiesDetectionJob",
                "StartKeyPhrasesDetectionJob",
                "StartPiiEntitiesDetectionJob",
                "StartSentimentDetectionJob",
                "StartTopicsDetectionJob",
                "StopDominantLanguageDetectionJob",
                "StopEntitiesDetectionJob",
                "StopKeyPhrasesDetectionJob",
                "StopPiiEntitiesDetectionJob",
                "StopSentimentDetectionJob",
                "StopTrainingDocumentClassifier",
                "StopTrainingEntityRecognizer",
                "UpdateEndpoint"
            ],
            "List": [
                "ListDocumentClassificationJobs",
                "ListDocumentClassifiers",
                "ListDominantLanguageDetectionJobs",
                "ListEndpoints",
                "ListEntitiesDetectionJobs",
                "ListEntityRecognizers",
                "ListKeyPhrasesDetectionJobs",
                "ListPiiEntitiesDetectionJobs",
                "ListSentimentDetectionJobs",
                "ListTagsForResource",
                "ListTopicsDetectionJobs"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to detect the language or languages present in the list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectDominantLanguage.html
     */
    toBatchDetectDominantLanguage() {
        this.to('comprehend:BatchDetectDominantLanguage');
        return this;
    }
    /**
     * Grants permission to detect the named entities ("People", "Places", "Locations", etc) within the given list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectEntities.html
     */
    toBatchDetectEntities() {
        this.to('comprehend:BatchDetectEntities');
        return this;
    }
    /**
     * Grants permission to detect the phrases in the list of text documents that are most indicative of the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectKeyPhrases.html
     */
    toBatchDetectKeyPhrases() {
        this.to('comprehend:BatchDetectKeyPhrases');
        return this;
    }
    /**
     * Grants permission to detect the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSentiment.html
     */
    toBatchDetectSentiment() {
        this.to('comprehend:BatchDetectSentiment');
        return this;
    }
    /**
     * Grants permission to detect syntactic information (like Part of Speech, Tokens) in a list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSyntax.html
     */
    toBatchDetectSyntax() {
        this.to('comprehend:BatchDetectSyntax');
        return this;
    }
    /**
     * Grants permission to create a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ClassifyDocument.html
     */
    toClassifyDocument() {
        this.to('comprehend:ClassifyDocument');
        return this;
    }
    /**
     * Grants permission to create a new document classifier that you can use to categorize documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateDocumentClassifier.html
     */
    toCreateDocumentClassifier() {
        this.to('comprehend:CreateDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to create a model-specific endpoint for synchronous inference for a previously trained custom model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        this.to('comprehend:CreateEndpoint');
        return this;
    }
    /**
     * Grants permission to create an entity recognizer using submitted files
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEntityRecognizer.html
     */
    toCreateEntityRecognizer() {
        this.to('comprehend:CreateEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to delete a previously created document classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteDocumentClassifier.html
     */
    toDeleteDocumentClassifier() {
        this.to('comprehend:DeleteDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to delete a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.to('comprehend:DeleteEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a submitted entity recognizer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEntityRecognizer.html
     */
    toDeleteEntityRecognizer() {
        this.to('comprehend:DeleteEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a document classification job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassificationJob.html
     */
    toDescribeDocumentClassificationJob() {
        this.to('comprehend:DescribeDocumentClassificationJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a document classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassifier.html
     */
    toDescribeDocumentClassifier() {
        this.to('comprehend:DescribeDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a dominant language detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDominantLanguageDetectionJob.html
     */
    toDescribeDominantLanguageDetectionJob() {
        this.to('comprehend:DescribeDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a specific endpoint. Use this operation to get the status of an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        this.to('comprehend:DescribeEndpoint');
        return this;
    }
    /**
     * Grants permission to get the properties associated with an entities detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntitiesDetectionJob.html
     */
    toDescribeEntitiesDetectionJob() {
        this.to('comprehend:DescribeEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to provide details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntityRecognizer.html
     */
    toDescribeEntityRecognizer() {
        this.to('comprehend:DescribeEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a key phrases detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeKeyPhrasesDetectionJob.html
     */
    toDescribeKeyPhrasesDetectionJob() {
        this.to('comprehend:DescribeKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a PII entities detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribePiiEntitiesDetectionJob.html
     */
    toDescribePiiEntitiesDetectionJob() {
        this.to('comprehend:DescribePiiEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a sentiment detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeSentimentDetectionJob.html
     */
    toDescribeSentimentDetectionJob() {
        this.to('comprehend:DescribeSentimentDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a topic detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeTopicsDetectionJob.html
     */
    toDescribeTopicsDetectionJob() {
        this.to('comprehend:DescribeTopicsDetectionJob');
        return this;
    }
    /**
     * Grants permission to detect the language or languages present in the text
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html
     */
    toDetectDominantLanguage() {
        this.to('comprehend:DetectDominantLanguage');
        return this;
    }
    /**
     * Grants permission to detect the named entities ("People", "Places", "Locations", etc) within the given text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectEntities.html
     */
    toDetectEntities() {
        this.to('comprehend:DetectEntities');
        return this;
    }
    /**
     * Grants permission to detect the phrases in the text that are most indicative of the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectKeyPhrases.html
     */
    toDetectKeyPhrases() {
        this.to('comprehend:DetectKeyPhrases');
        return this;
    }
    /**
     * Grants permission to detect the personally identifiable information entities ("Name", "SSN", "PIN", etc) within the given text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectPiiEntities.html
     */
    toDetectPiiEntities() {
        this.to('comprehend:DetectPiiEntities');
        return this;
    }
    /**
     * Grants permission to detect the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSentiment.html
     */
    toDetectSentiment() {
        this.to('comprehend:DetectSentiment');
        return this;
    }
    /**
     * Grants permission to detect syntactic information (like Part of Speech, Tokens) in a text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSyntax.html
     */
    toDetectSyntax() {
        this.to('comprehend:DetectSyntax');
        return this;
    }
    /**
     * Grants permission to get a list of the document classification jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassificationJobs.html
     */
    toListDocumentClassificationJobs() {
        this.to('comprehend:ListDocumentClassificationJobs');
        return this;
    }
    /**
     * Grants permission to get a list of the document classifiers that you have created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassifiers.html
     */
    toListDocumentClassifiers() {
        this.to('comprehend:ListDocumentClassifiers');
        return this;
    }
    /**
     * Grants permission to get a list of the dominant language detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDominantLanguageDetectionJobs.html
     */
    toListDominantLanguageDetectionJobs() {
        this.to('comprehend:ListDominantLanguageDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of all existing endpoints that you've created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEndpoints.html
     */
    toListEndpoints() {
        this.to('comprehend:ListEndpoints');
        return this;
    }
    /**
     * Grants permission to get a list of the entity detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntitiesDetectionJobs.html
     */
    toListEntitiesDetectionJobs() {
        this.to('comprehend:ListEntitiesDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of the properties of all entity recognizers that you created, including recognizers currently in training
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntityRecognizers.html
     */
    toListEntityRecognizers() {
        this.to('comprehend:ListEntityRecognizers');
        return this;
    }
    /**
     * Grants permission to get a list of key phrase detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListKeyPhrasesDetectionJobs.html
     */
    toListKeyPhrasesDetectionJobs() {
        this.to('comprehend:ListKeyPhrasesDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of PII entities detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListPiiEntitiesDetectionJobs.html
     */
    toListPiiEntitiesDetectionJobs() {
        this.to('comprehend:ListPiiEntitiesDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of sentiment detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListSentimentDetectionJobs.html
     */
    toListSentimentDetectionJobs() {
        this.to('comprehend:ListSentimentDetectionJobs');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('comprehend:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to get a list of the topic detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTopicsDetectionJobs.html
     */
    toListTopicsDetectionJobs() {
        this.to('comprehend:ListTopicsDetectionJobs');
        return this;
    }
    /**
     * Grants permission to start an asynchronous document classification job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDocumentClassificationJob.html
     */
    toStartDocumentClassificationJob() {
        this.to('comprehend:StartDocumentClassificationJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous dominant language detection job for a collection of documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDominantLanguageDetectionJob.html
     */
    toStartDominantLanguageDetectionJob() {
        this.to('comprehend:StartDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous entity detection job for a collection of documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartEntitiesDetectionJob.html
     */
    toStartEntitiesDetectionJob() {
        this.to('comprehend:StartEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous key phrase detection job for a collection of documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartKeyPhrasesDetectionJob.html
     */
    toStartKeyPhrasesDetectionJob() {
        this.to('comprehend:StartKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous PII entities detection job for a collection of documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartPiiEntitiesDetectionJob.html
     */
    toStartPiiEntitiesDetectionJob() {
        this.to('comprehend:StartPiiEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous sentiment detection job for a collection of documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartSentimentDetectionJob.html
     */
    toStartSentimentDetectionJob() {
        this.to('comprehend:StartSentimentDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartTopicsDetectionJob.html
     */
    toStartTopicsDetectionJob() {
        this.to('comprehend:StartTopicsDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a dominant language detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopDominantLanguageDetectionJob.html
     */
    toStopDominantLanguageDetectionJob() {
        this.to('comprehend:StopDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop an entity detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopEntitiesDetectionJob.html
     */
    toStopEntitiesDetectionJob() {
        this.to('comprehend:StopEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a key phrase detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopKeyPhrasesDetectionJob.html
     */
    toStopKeyPhrasesDetectionJob() {
        this.to('comprehend:StopKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a PII entities detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopPiiEntitiesDetectionJob.html
     */
    toStopPiiEntitiesDetectionJob() {
        this.to('comprehend:StopPiiEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a sentiment detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopSentimentDetectionJob.html
     */
    toStopSentimentDetectionJob() {
        this.to('comprehend:StopSentimentDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a previously created document classifier training job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingDocumentClassifier.html
     */
    toStopTrainingDocumentClassifier() {
        this.to('comprehend:StopTrainingDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to stop a previously created entity recognizer training job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingEntityRecognizer.html
     */
    toStopTrainingEntityRecognizer() {
        this.to('comprehend:StopTrainingEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('comprehend:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a resource with given key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('comprehend:UntagResource');
        return this;
    }
    /**
     * Grants permission to update information about the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UpdateEndpoint.html
     */
    toUpdateEndpoint() {
        this.to('comprehend:UpdateEndpoint');
        return this;
    }
    /**
     * Adds a resource of type document-classifier to the statement
     *
     * @param documentClassifierName - Identifier for the documentClassifierName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifier(documentClassifierName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier/${DocumentClassifierName}';
        arn = arn.replace('${DocumentClassifierName}', documentClassifierName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type document-classifier-endpoint to the statement
     *
     * @param documentClassifierEndpointName - Identifier for the documentClassifierEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifierEndpoint(documentClassifierEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier-endpoint/${DocumentClassifierEndpointName}';
        arn = arn.replace('${DocumentClassifierEndpointName}', documentClassifierEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer to the statement
     *
     * @param entityRecognizerName - Identifier for the entityRecognizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizer(entityRecognizerName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer/${EntityRecognizerName}';
        arn = arn.replace('${EntityRecognizerName}', entityRecognizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer-endpoint to the statement
     *
     * @param entityRecognizerEndpointName - Identifier for the entityRecognizerEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizerEndpoint(entityRecognizerEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer-endpoint/${EntityRecognizerEndpointName}';
        arn = arn.replace('${EntityRecognizerEndpointName}', entityRecognizerEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Comprehend = Comprehend;
//# sourceMappingURL=data:application/json;base64,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