"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dynamodb = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dynamodb](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondynamodb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dynamodb extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dynamodb](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondynamodb.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dynamodb';
        this.accessLevelList = {
            "Read": [
                "BatchGetItem",
                "ConditionCheckItem",
                "DescribeBackup",
                "DescribeContinuousBackups",
                "DescribeContributorInsights",
                "DescribeGlobalTable",
                "DescribeGlobalTableSettings",
                "DescribeLimits",
                "DescribeReservedCapacity",
                "DescribeReservedCapacityOfferings",
                "DescribeStream",
                "DescribeTable",
                "DescribeTableReplicaAutoScaling",
                "DescribeTimeToLive",
                "GetItem",
                "GetRecords",
                "GetShardIterator",
                "ListStreams",
                "ListTagsOfResource",
                "Query",
                "Scan"
            ],
            "Write": [
                "BatchWriteItem",
                "CreateBackup",
                "CreateGlobalTable",
                "CreateTable",
                "CreateTableReplica",
                "DeleteBackup",
                "DeleteItem",
                "DeleteTable",
                "DeleteTableReplica",
                "PurchaseReservedCapacityOfferings",
                "PutItem",
                "RestoreTableFromBackup",
                "RestoreTableToPointInTime",
                "UpdateContinuousBackups",
                "UpdateContributorInsights",
                "UpdateGlobalTable",
                "UpdateGlobalTableSettings",
                "UpdateItem",
                "UpdateTable",
                "UpdateTableReplicaAutoScaling",
                "UpdateTimeToLive"
            ],
            "List": [
                "ListBackups",
                "ListContributorInsights",
                "ListGlobalTables",
                "ListTables"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Returns the attributes of one or more items from one or more tables
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    toBatchGetItem() {
        this.to('dynamodb:BatchGetItem');
        return this;
    }
    /**
     * Puts or deletes multiple items in one or more tables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    toBatchWriteItem() {
        this.to('dynamodb:BatchWriteItem');
        return this;
    }
    /**
     * The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html
     */
    toConditionCheckItem() {
        this.to('dynamodb:ConditionCheckItem');
        return this;
    }
    /**
     * Creates a backup for an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        this.to('dynamodb:CreateBackup');
        return this;
    }
    /**
     * Enables the user to create a global table from an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateGlobalTable.html
     */
    toCreateGlobalTable() {
        this.to('dynamodb:CreateGlobalTable');
        return this;
    }
    /**
     * The CreateTable operation adds a new table to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html
     */
    toCreateTable() {
        this.to('dynamodb:CreateTable');
        return this;
    }
    /**
     * Adds a new replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html
     */
    toCreateTableReplica() {
        this.to('dynamodb:CreateTableReplica');
        return this;
    }
    /**
     * Deletes an existing backup of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        this.to('dynamodb:DeleteBackup');
        return this;
    }
    /**
     * Deletes a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    toDeleteItem() {
        this.to('dynamodb:DeleteItem');
        return this;
    }
    /**
     * The DeleteTable operation deletes a table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteTable.html
     */
    toDeleteTable() {
        this.to('dynamodb:DeleteTable');
        return this;
    }
    /**
     * Deletes a replica table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html
     */
    toDeleteTableReplica() {
        this.to('dynamodb:DeleteTableReplica');
        return this;
    }
    /**
     * Describes an existing backup of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeBackup.html
     */
    toDescribeBackup() {
        this.to('dynamodb:DescribeBackup');
        return this;
    }
    /**
     * Checks the status of the backup restore settings on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContinuousBackups.html
     */
    toDescribeContinuousBackups() {
        this.to('dynamodb:DescribeContinuousBackups');
        return this;
    }
    /**
     * Describes the contributor insights status and related details for a given table or global secondary index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContributorInsights.html
     */
    toDescribeContributorInsights() {
        this.to('dynamodb:DescribeContributorInsights');
        return this;
    }
    /**
     * Returns information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTable.html
     */
    toDescribeGlobalTable() {
        this.to('dynamodb:DescribeGlobalTable');
        return this;
    }
    /**
     * Returns settings information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTableSettings.html
     */
    toDescribeGlobalTableSettings() {
        this.to('dynamodb:DescribeGlobalTableSettings');
        return this;
    }
    /**
     * Returns the current provisioned-capacity limits for your AWS account in a region, both for the region as a whole and for any one DynamoDB table that you create there
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        this.to('dynamodb:DescribeLimits');
        return this;
    }
    /**
     * Describes one or more of the Reserved Capacity purchased
     *
     * Access Level: Read
     */
    toDescribeReservedCapacity() {
        this.to('dynamodb:DescribeReservedCapacity');
        return this;
    }
    /**
     * Describes Reserved Capacity offerings that are available for purchase
     *
     * Access Level: Read
     */
    toDescribeReservedCapacityOfferings() {
        this.to('dynamodb:DescribeReservedCapacityOfferings');
        return this;
    }
    /**
     * Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeStream.html
     */
    toDescribeStream() {
        this.to('dynamodb:DescribeStream');
        return this;
    }
    /**
     * Returns information about the table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html
     */
    toDescribeTable() {
        this.to('dynamodb:DescribeTable');
        return this;
    }
    /**
     * Describes the auto scaling settings across all replicas of the global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTableReplicaAutoScaling.html
     */
    toDescribeTableReplicaAutoScaling() {
        this.to('dynamodb:DescribeTableReplicaAutoScaling');
        return this;
    }
    /**
     * Gives a description of the Time to Live (TTL) status on the specified table.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTimeToLive.html
     */
    toDescribeTimeToLive() {
        this.to('dynamodb:DescribeTimeToLive');
        return this;
    }
    /**
     * The GetItem operation returns a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    toGetItem() {
        this.to('dynamodb:GetItem');
        return this;
    }
    /**
     * Retrieves the stream records from a given shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetRecords.html
     */
    toGetRecords() {
        this.to('dynamodb:GetRecords');
        return this;
    }
    /**
     * Returns a shard iterator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetShardIterator.html
     */
    toGetShardIterator() {
        this.to('dynamodb:GetShardIterator');
        return this;
    }
    /**
     * List backups associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListBackups.html
     */
    toListBackups() {
        this.to('dynamodb:ListBackups');
        return this;
    }
    /**
     * Lists the ContributorInsightsSummary for all tables and global secondary indexes associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListContributorInsights.html
     */
    toListContributorInsights() {
        this.to('dynamodb:ListContributorInsights');
        return this;
    }
    /**
     * Lists all global tables that have a replica in the specified region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListGlobalTables.html
     */
    toListGlobalTables() {
        this.to('dynamodb:ListGlobalTables');
        return this;
    }
    /**
     * Returns an array of stream ARNs associated with the current account and endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        this.to('dynamodb:ListStreams');
        return this;
    }
    /**
     * Returns an array of table names associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTables.html
     */
    toListTables() {
        this.to('dynamodb:ListTables');
        return this;
    }
    /**
     * List all tags on an Amazon DynamoDB resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTagsOfResource.html
     */
    toListTagsOfResource() {
        this.to('dynamodb:ListTagsOfResource');
        return this;
    }
    /**
     * Purchases Reserved Capacity for use with your account
     *
     * Access Level: Write
     */
    toPurchaseReservedCapacityOfferings() {
        this.to('dynamodb:PurchaseReservedCapacityOfferings');
        return this;
    }
    /**
     * Creates a new item, or replaces an old item with a new item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    toPutItem() {
        this.to('dynamodb:PutItem');
        return this;
    }
    /**
     * Uses the primary key of a table or a secondary index to directly access items from that table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    toQuery() {
        this.to('dynamodb:Query');
        return this;
    }
    /**
     * Creates a new table from an existing backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableFromBackup.html
     */
    toRestoreTableFromBackup() {
        this.to('dynamodb:RestoreTableFromBackup');
        return this;
    }
    /**
     * Restores a table to a point in time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableToPointInTime.html
     */
    toRestoreTableToPointInTime() {
        this.to('dynamodb:RestoreTableToPointInTime');
        return this;
    }
    /**
     * Returns one or more items and item attributes by accessing every item in a table or a secondary index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    toScan() {
        this.to('dynamodb:Scan');
        return this;
    }
    /**
     * Associate a set of tags with an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('dynamodb:TagResource');
        return this;
    }
    /**
     * Removes the association of tags from an Amazon DynamoDB resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('dynamodb:UntagResource');
        return this;
    }
    /**
     * Enables or disables continuous backups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContinuousBackups.html
     */
    toUpdateContinuousBackups() {
        this.to('dynamodb:UpdateContinuousBackups');
        return this;
    }
    /**
     * Updates the status for contributor insights for a specific table or global secondary index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContributorInsights.html
     */
    toUpdateContributorInsights() {
        this.to('dynamodb:UpdateContributorInsights');
        return this;
    }
    /**
     * Enables the user to add or remove replicas in the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTable.html
     */
    toUpdateGlobalTable() {
        this.to('dynamodb:UpdateGlobalTable');
        return this;
    }
    /**
     * Enables the user to update settings of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTableSettings.html
     */
    toUpdateGlobalTableSettings() {
        this.to('dynamodb:UpdateGlobalTableSettings');
        return this;
    }
    /**
     * Edits an existing item's attributes, or adds a new item to the table if it does not already exist
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    toUpdateItem() {
        this.to('dynamodb:UpdateItem');
        return this;
    }
    /**
     * Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html
     */
    toUpdateTable() {
        this.to('dynamodb:UpdateTable');
        return this;
    }
    /**
     * Updates auto scaling settings on your replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTableReplicaAutoScaling.html
     */
    toUpdateTableReplicaAutoScaling() {
        this.to('dynamodb:UpdateTableReplicaAutoScaling');
        return this;
    }
    /**
     * Enables or disables TTL for the specified table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTimeToLive.html
     */
    toUpdateTimeToLive() {
        this.to('dynamodb:UpdateTimeToLive');
        return this;
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.PrimaryKey
     *
     * @param tableName - Identifier for the tableName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIndex(tableName, indexName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/index/${IndexName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${IndexName}', indexName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.Streams
     *
     * @param tableName - Identifier for the tableName.
     * @param streamLabel - Identifier for the streamLabel.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStream(tableName, streamLabel, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/stream/${StreamLabel}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${StreamLabel}', streamLabel);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.TablesItemsAttributes
     *
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(tableName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param backupName - Identifier for the backupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackup(tableName, backupName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/backup/${BackupName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${BackupName}', backupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type global-table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables_HowItWorks.html
     *
     * @param globalTableName - Identifier for the globalTableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalTable(globalTableName, account, partition) {
        var arn = 'arn:${Partition}:dynamodb::${Account}:global-table/${GlobalTableName}';
        arn = arn.replace('${GlobalTableName}', globalTableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filter based on the attribute (field or column) names of the table.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributes(value, operator) {
        return this.if(`dynamodb:Attributes`, value, operator || 'StringLike');
    }
    /**
     * Used to block Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`dynamodb:EnclosingOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters based on the partition key of the table.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLeadingKeys(value, operator) {
        return this.if(`dynamodb:LeadingKeys`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the ReturnConsumedCapacity parameter of a request. Contains either "TOTAL" or "NONE".
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnConsumedCapacity(value, operator) {
        return this.if(`dynamodb:ReturnConsumedCapacity`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the ReturnValues parameter of request. Contains one of the following: "ALL_OLD", "UPDATED_OLD","ALL_NEW","UPDATED_NEW", or "NONE".
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnValues(value, operator) {
        return this.if(`dynamodb:ReturnValues`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the Select parameter of a Query or Scan request.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toGetItem()
     * - .toQuery()
     * - .toScan()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSelect(value, operator) {
        return this.if(`dynamodb:Select`, value, operator || 'StringLike');
    }
}
exports.Dynamodb = Dynamodb;
//# sourceMappingURL=data:application/json;base64,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