"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticache = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticache](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticache.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticache extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticache](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticache.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticache';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Write": [
                "AuthorizeCacheSecurityGroupIngress",
                "BatchApplyUpdateAction",
                "BatchStopUpdateAction",
                "CompleteMigration",
                "CopySnapshot",
                "CreateCacheCluster",
                "CreateCacheParameterGroup",
                "CreateCacheSecurityGroup",
                "CreateCacheSubnetGroup",
                "CreateGlobalReplicationGroup",
                "CreateReplicationGroup",
                "CreateSnapshot",
                "CreateUser",
                "CreateUserGroup",
                "DecreaseNodeGroupsInGlobalReplicationGroup",
                "DecreaseReplicaCount",
                "DeleteCacheCluster",
                "DeleteCacheParameterGroup",
                "DeleteCacheSecurityGroup",
                "DeleteCacheSubnetGroup",
                "DeleteGlobalReplicationGroup",
                "DeleteReplicationGroup",
                "DeleteSnapshot",
                "DeleteUser",
                "DeleteUserGroup",
                "DisassociateGlobalReplicationGroup",
                "FailoverGlobalReplicationGroup",
                "IncreaseNodeGroupsInGlobalReplicationGroup",
                "IncreaseReplicaCount",
                "ModifyCacheCluster",
                "ModifyCacheParameterGroup",
                "ModifyCacheSubnetGroup",
                "ModifyGlobalReplicationGroup",
                "ModifyReplicationGroup",
                "ModifyReplicationGroupShardConfiguration",
                "ModifyUser",
                "ModifyUserGroup",
                "PurchaseReservedCacheNodesOffering",
                "RebalanceSlotsInGlobalReplicationGroup",
                "RebootCacheCluster",
                "ResetCacheParameterGroup",
                "RevokeCacheSecurityGroupIngress",
                "StartMigration",
                "TestFailover"
            ],
            "List": [
                "DescribeCacheClusters",
                "DescribeCacheEngineVersions",
                "DescribeCacheParameterGroups",
                "DescribeCacheParameters",
                "DescribeCacheSecurityGroups",
                "DescribeCacheSubnetGroups",
                "DescribeEngineDefaultParameters",
                "DescribeEvents",
                "DescribeGlobalReplicationGroups",
                "DescribeReplicationGroups",
                "DescribeReservedCacheNodes",
                "DescribeReservedCacheNodesOfferings",
                "DescribeServiceUpdates",
                "DescribeSnapshots",
                "DescribeUpdateActions",
                "DescribeUserGroups",
                "DescribeUsers",
                "ListAllowedNodeTypeModifications"
            ],
            "Read": [
                "ListTagsForResource"
            ]
        };
    }
    /**
     * The AddTagsToResource action adds up to 10 cost allocation tags to the named resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('elasticache:AddTagsToResource');
        return this;
    }
    /**
     * The AuthorizeCacheSecurityGroupIngress action allows network ingress to a cache security group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AuthorizeCacheSecurityGroupIngress.html
     */
    toAuthorizeCacheSecurityGroupIngress() {
        this.to('elasticache:AuthorizeCacheSecurityGroupIngress');
        return this;
    }
    /**
     * Apply the service update.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchApplyUpdateAction.html
     */
    toBatchApplyUpdateAction() {
        this.to('elasticache:BatchApplyUpdateAction');
        return this;
    }
    /**
     * Stop the service update.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchStopUpdateAction.html
     */
    toBatchStopUpdateAction() {
        this.to('elasticache:BatchStopUpdateAction');
        return this;
    }
    /**
     * Stop the service update.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CompleteMigration.html
     */
    toCompleteMigration() {
        this.to('elasticache:CompleteMigration');
        return this;
    }
    /**
     * The CopySnapshot action makes a copy of an existing snapshot.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.to('elasticache:CopySnapshot');
        return this;
    }
    /**
     * The CreateCacheCluster action creates a cache cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheCluster.html
     */
    toCreateCacheCluster() {
        this.to('elasticache:CreateCacheCluster');
        return this;
    }
    /**
     * The CreateCacheParameterGroup action creates a new cache parameter group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheParameterGroup.html
     */
    toCreateCacheParameterGroup() {
        this.to('elasticache:CreateCacheParameterGroup');
        return this;
    }
    /**
     * The CreateCacheSecurityGroup action creates a new cache security group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSecurityGroup.html
     */
    toCreateCacheSecurityGroup() {
        this.to('elasticache:CreateCacheSecurityGroup');
        return this;
    }
    /**
     * The CreateCacheSubnetGroup action creates a new cache subnet group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSubnetGroup.html
     */
    toCreateCacheSubnetGroup() {
        this.to('elasticache:CreateCacheSubnetGroup');
        return this;
    }
    /**
     * The CreateGlobalReplicationGroup action creates a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateGlobalReplicationGroup.html
     */
    toCreateGlobalReplicationGroup() {
        this.to('elasticache:CreateGlobalReplicationGroup');
        return this;
    }
    /**
     * The CreateReplicationGroup action creates a replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateReplicationGroup.html
     */
    toCreateReplicationGroup() {
        this.to('elasticache:CreateReplicationGroup');
        return this;
    }
    /**
     * The CreateSnapshot action creates a copy of an entire cache cluster at a specific moment in time.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.to('elasticache:CreateSnapshot');
        return this;
    }
    /**
     * The CreateUser action creates a new user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('elasticache:CreateUser');
        return this;
    }
    /**
     * The CreateUserGroup action creates a new user group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUserGroup.html
     */
    toCreateUserGroup() {
        this.to('elasticache:CreateUserGroup');
        return this;
    }
    /**
     * The DecreaseNodeGroupsInGlobalReplicationGroup action dec a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toDecreaseNodeGroupsInGlobalReplicationGroup() {
        this.to('elasticache:DecreaseNodeGroupsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The DecreaseReplicaCount action decreases the number of replicas in a Redis replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseReplicaCount.html
     */
    toDecreaseReplicaCount() {
        this.to('elasticache:DecreaseReplicaCount');
        return this;
    }
    /**
     * The DeleteCacheCluster action deletes a previously provisioned cache cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheCluster.html
     */
    toDeleteCacheCluster() {
        this.to('elasticache:DeleteCacheCluster');
        return this;
    }
    /**
     * The DeleteCacheParameterGroup action deletes the specified cache parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheParameterGroup.html
     */
    toDeleteCacheParameterGroup() {
        this.to('elasticache:DeleteCacheParameterGroup');
        return this;
    }
    /**
     * The DeleteCacheSecurityGroup action deletes a cache security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSecurityGroup.html
     */
    toDeleteCacheSecurityGroup() {
        this.to('elasticache:DeleteCacheSecurityGroup');
        return this;
    }
    /**
     * The DeleteCacheSubnetGroup action deletes a cache subnet group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSubnetGroup.html
     */
    toDeleteCacheSubnetGroup() {
        this.to('elasticache:DeleteCacheSubnetGroup');
        return this;
    }
    /**
     * The DeleteGlobalReplicationGroup action deletes a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteGlobalReplicationGroup.html
     */
    toDeleteGlobalReplicationGroup() {
        this.to('elasticache:DeleteGlobalReplicationGroup');
        return this;
    }
    /**
     * The DeleteReplicationGroup action deletes an existing replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteReplicationGroup.html
     */
    toDeleteReplicationGroup() {
        this.to('elasticache:DeleteReplicationGroup');
        return this;
    }
    /**
     * The DeleteSnapshot action deletes an existing snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.to('elasticache:DeleteSnapshot');
        return this;
    }
    /**
     * The DeleteUser action deletes an existing user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('elasticache:DeleteUser');
        return this;
    }
    /**
     * The DeleteUserGroup action deletes an existing user group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUserGroup.html
     */
    toDeleteUserGroup() {
        this.to('elasticache:DeleteUserGroup');
        return this;
    }
    /**
     * The DescribeCacheClusters action returns information about all provisioned cache clusters if no cache cluster identifier is specified, or about a specific cache cluster if a cache cluster identifier is supplied.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheClusters.html
     */
    toDescribeCacheClusters() {
        this.to('elasticache:DescribeCacheClusters');
        return this;
    }
    /**
     * The DescribeCacheEngineVersions action returns a list of the available cache engines and their versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheEngineVersions.html
     */
    toDescribeCacheEngineVersions() {
        this.to('elasticache:DescribeCacheEngineVersions');
        return this;
    }
    /**
     * The DescribeCacheParameterGroups action returns information about parameter groups for this account, or a particular parameter group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameterGroups.html
     */
    toDescribeCacheParameterGroups() {
        this.to('elasticache:DescribeCacheParameterGroups');
        return this;
    }
    /**
     * The DescribeCacheParameters action returns the detailed parameter list for a particular cache parameter group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameters.html
     */
    toDescribeCacheParameters() {
        this.to('elasticache:DescribeCacheParameters');
        return this;
    }
    /**
     * The DescribeCacheSecurityGroups action returns a list of cache security group descriptions, or the description of the specified security group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSecurityGroups.html
     */
    toDescribeCacheSecurityGroups() {
        this.to('elasticache:DescribeCacheSecurityGroups');
        return this;
    }
    /**
     * The DescribeCacheSubnetGroups action returns a list of cache subnet group descriptions, or the description of the specified subnet group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSubnetGroups.html
     */
    toDescribeCacheSubnetGroups() {
        this.to('elasticache:DescribeCacheSubnetGroups');
        return this;
    }
    /**
     * The DescribeEngineDefaultParameters action returns the default engine and system parameter information for the specified cache engine.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters() {
        this.to('elasticache:DescribeEngineDefaultParameters');
        return this;
    }
    /**
     * The DescribeEvents action returns events related to cache clusters, cache security groups, and cache parameter groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('elasticache:DescribeEvents');
        return this;
    }
    /**
     * The DescribeGlobalReplicationGroups action returns information about global datastores for this account, or a particular global datastore.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeGlobalReplicationGroups.html
     */
    toDescribeGlobalReplicationGroups() {
        this.to('elasticache:DescribeGlobalReplicationGroups');
        return this;
    }
    /**
     * The DescribeReplicationGroups action returns information about replication groups for this account, or a particular replication group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReplicationGroups.html
     */
    toDescribeReplicationGroups() {
        this.to('elasticache:DescribeReplicationGroups');
        return this;
    }
    /**
     * The DescribeReservedCacheNodes action returns information about reserved cache nodes for this account, or a particular reserved cache node.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodes.html
     */
    toDescribeReservedCacheNodes() {
        this.to('elasticache:DescribeReservedCacheNodes');
        return this;
    }
    /**
     * The DescribeReservedCacheNodesOfferings action lists available reserved cache node offerings.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodesOfferings.html
     */
    toDescribeReservedCacheNodesOfferings() {
        this.to('elasticache:DescribeReservedCacheNodesOfferings');
        return this;
    }
    /**
     * Returns details of the service updates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates() {
        this.to('elasticache:DescribeServiceUpdates');
        return this;
    }
    /**
     * The DescribeSnapshots action returns information about cache cluster snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.to('elasticache:DescribeSnapshots');
        return this;
    }
    /**
     * Returns details of the update actions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUpdateActions.html
     */
    toDescribeUpdateActions() {
        this.to('elasticache:DescribeUpdateActions');
        return this;
    }
    /**
     * The DescribeUserGroups action returns information about all user groups for this account, or a particular user group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUserGroups.html
     */
    toDescribeUserGroups() {
        this.to('elasticache:DescribeUserGroups');
        return this;
    }
    /**
     * The DescribeUsers action returns information about all users for this account, or a particular user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        this.to('elasticache:DescribeUsers');
        return this;
    }
    /**
     * The DisassociateGlobalReplicationGroup action removes a secondary Replication Group from the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DisassociateGlobalReplicationGroup.html
     */
    toDisassociateGlobalReplicationGroup() {
        this.to('elasticache:DisassociateGlobalReplicationGroup');
        return this;
    }
    /**
     * The FailoverGlobalReplicationGroup action removes a secondary Replication Group from the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_FailoverGlobalReplicationGroup.html
     */
    toFailoverGlobalReplicationGroup() {
        this.to('elasticache:FailoverGlobalReplicationGroup');
        return this;
    }
    /**
     * The IncreaseNodeGroupsInGlobalReplicationGroup action increases the number of node groups in the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toIncreaseNodeGroupsInGlobalReplicationGroup() {
        this.to('elasticache:IncreaseNodeGroupsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The IncreaseReplicaCount action increases the number of replicas in a Redis replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseReplicaCount.html
     */
    toIncreaseReplicaCount() {
        this.to('elasticache:IncreaseReplicaCount');
        return this;
    }
    /**
     * List Allowed Node Type Modifications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListAllowedNodeTypeModifications.html
     */
    toListAllowedNodeTypeModifications() {
        this.to('elasticache:ListAllowedNodeTypeModifications');
        return this;
    }
    /**
     * The ListTagsForResource action lists all cost allocation tags currently on the named resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('elasticache:ListTagsForResource');
        return this;
    }
    /**
     * The ModifyCacheCluster action modifies the settings for a cache cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html
     */
    toModifyCacheCluster() {
        this.to('elasticache:ModifyCacheCluster');
        return this;
    }
    /**
     * The ModifyCacheParameterGroup action modifies the parameters of a cache parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheParameterGroup.html
     */
    toModifyCacheParameterGroup() {
        this.to('elasticache:ModifyCacheParameterGroup');
        return this;
    }
    /**
     * The ModifyCacheSubnetGroup action modifies an existing cache subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheSubnetGroup.html
     */
    toModifyCacheSubnetGroup() {
        this.to('elasticache:ModifyCacheSubnetGroup');
        return this;
    }
    /**
     * The ModifyGlobalReplicationGroup action modifies the settings for a Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyGlobalReplicationGroup.html
     */
    toModifyGlobalReplicationGroup() {
        this.to('elasticache:ModifyGlobalReplicationGroup');
        return this;
    }
    /**
     * The ModifyReplicationGroup action modifies the settings for a replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroup.html
     */
    toModifyReplicationGroup() {
        this.to('elasticache:ModifyReplicationGroup');
        return this;
    }
    /**
     * The ModifyReplicationGroupShardConfiguration action allows you to add shards, remove shards, or rebalance the keyspaces among exisiting shards.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroupShardConfiguration.html
     */
    toModifyReplicationGroupShardConfiguration() {
        this.to('elasticache:ModifyReplicationGroupShardConfiguration');
        return this;
    }
    /**
     * The ModifyUser action modifies an existing user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUser.html
     */
    toModifyUser() {
        this.to('elasticache:ModifyUser');
        return this;
    }
    /**
     * The ModifyUserGroup action modifies an existing user group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUserGroup.html
     */
    toModifyUserGroup() {
        this.to('elasticache:ModifyUserGroup');
        return this;
    }
    /**
     * The PurchaseReservedCacheNodesOffering action allows you to purchase a reserved cache node offering.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_PurchaseReservedCacheNodesOffering.html
     */
    toPurchaseReservedCacheNodesOffering() {
        this.to('elasticache:PurchaseReservedCacheNodesOffering');
        return this;
    }
    /**
     * The RebalanceSlotsInGlobalReplicationGroup action redistributes slots to ensure uniform distribution across existing shards in the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebalanceSlotsInGlobalReplicationGroup.html
     */
    toRebalanceSlotsInGlobalReplicationGroup() {
        this.to('elasticache:RebalanceSlotsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The RebootCacheCluster action reboots some, or all, of the cache nodes within a provisioned cache cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebootCacheCluster.html
     */
    toRebootCacheCluster() {
        this.to('elasticache:RebootCacheCluster');
        return this;
    }
    /**
     * The RemoveTagsFromResource action removes the tags identified by the TagKeys list from the named resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('elasticache:RemoveTagsFromResource');
        return this;
    }
    /**
     * The ResetCacheParameterGroup action modifies the parameters of a cache parameter group to the engine or system default value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ResetCacheParameterGroup.html
     */
    toResetCacheParameterGroup() {
        this.to('elasticache:ResetCacheParameterGroup');
        return this;
    }
    /**
     * The RevokeCacheSecurityGroupIngress action revokes ingress from a cache security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RevokeCacheSecurityGroupIngress.html
     */
    toRevokeCacheSecurityGroupIngress() {
        this.to('elasticache:RevokeCacheSecurityGroupIngress');
        return this;
    }
    /**
     * Start the migration of data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_StartMigration.html
     */
    toStartMigration() {
        this.to('elasticache:StartMigration');
        return this;
    }
    /**
     * The TestFailover action allows you to test automatic failover on a specified node group in a replication group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_TestFailover.html
     */
    toTestFailover() {
        this.to('elasticache:TestFailover');
        return this;
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * @param cacheParameterGroupName - Identifier for the cacheParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onParametergroup(cacheParameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:parametergroup:${CacheParameterGroupName}';
        arn = arn.replace('${CacheParameterGroupName}', cacheParameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * @param cacheSecurityGroupName - Identifier for the cacheSecurityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSecuritygroup(cacheSecurityGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:securitygroup:${CacheSecurityGroupName}';
        arn = arn.replace('${CacheSecurityGroupName}', cacheSecurityGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * @param cacheSubnetGroupName - Identifier for the cacheSubnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSubnetgroup(cacheSubnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:subnetgroup:${CacheSubnetGroupName}';
        arn = arn.replace('${CacheSubnetGroupName}', cacheSubnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type replicationgroup to the statement
     *
     * @param replicationGroupId - Identifier for the replicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReplicationgroup(replicationGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:replicationgroup:${ReplicationGroupId}';
        arn = arn.replace('${ReplicationGroupId}', replicationGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * @param cacheClusterId - Identifier for the cacheClusterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCluster(cacheClusterId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:cluster:${CacheClusterId}';
        arn = arn.replace('${CacheClusterId}', cacheClusterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instance to the statement
     *
     * @param reservedCacheNodeId - Identifier for the reservedCacheNodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReservedInstance(reservedCacheNodeId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:reserved-instance:${ReservedCacheNodeId}';
        arn = arn.replace('${ReservedCacheNodeId}', reservedCacheNodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSnapshot(snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:snapshot:${SnapshotName}';
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type globalreplicationgroup to the statement
     *
     * @param globalReplicationGroupId - Identifier for the globalReplicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalreplicationgroup(globalReplicationGroupId, account, partition) {
        var arn = 'arn:${Partition}:elasticache::${Account}:globalreplicationgroup:${GlobalReplicationGroupId}';
        arn = arn.replace('${GlobalReplicationGroupId}', globalReplicationGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUser(userId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:user:${UserId}';
        arn = arn.replace('${UserId}', userId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type usergroup to the statement
     *
     * @param userGroupId - Identifier for the userGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUsergroup(userGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:usergroup:${UserGroupId}';
        arn = arn.replace('${UserGroupId}', userGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Elasticache = Elasticache;
//# sourceMappingURL=data:application/json;base64,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