"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticfilesystem = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticfilesystem](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticfilesystem extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticfilesystem](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticfilesystem';
        this.accessLevelList = {
            "Write": [
                "Backup",
                "ClientRootAccess",
                "ClientWrite",
                "CreateAccessPoint",
                "CreateMountTarget",
                "DeleteAccessPoint",
                "DeleteFileSystem",
                "DeleteFileSystemPolicy",
                "DeleteMountTarget",
                "ModifyMountTargetSecurityGroups",
                "PutBackupPolicy",
                "PutFileSystemPolicy",
                "PutLifecycleConfiguration",
                "Restore",
                "UpdateFileSystem"
            ],
            "Read": [
                "ClientMount",
                "DescribeBackupPolicy",
                "DescribeFileSystemPolicy",
                "DescribeLifecycleConfiguration",
                "DescribeMountTargetSecurityGroups",
                "DescribeMountTargets",
                "DescribeTags",
                "ListTagsForResource"
            ],
            "Tagging": [
                "CreateFileSystem",
                "CreateTags",
                "DeleteTags",
                "TagResource",
                "UntagResource"
            ],
            "List": [
                "DescribeAccessPoints",
                "DescribeFileSystems"
            ]
        };
    }
    /**
     * Starts a backup job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    toBackup() {
        this.to('elasticfilesystem:Backup');
        return this;
    }
    /**
     * Permission for allowing read-access to a file system.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccessPointArn()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    toClientMount() {
        this.to('elasticfilesystem:ClientMount');
        return this;
    }
    /**
     * Permission for allowing root-access to a file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessPointArn()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    toClientRootAccess() {
        this.to('elasticfilesystem:ClientRootAccess');
        return this;
    }
    /**
     * Permission for allowing write-access to a file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessPointArn()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    toClientWrite() {
        this.to('elasticfilesystem:ClientWrite');
        return this;
    }
    /**
     * Creates an access point for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        this.to('elasticfilesystem:CreateAccessPoint');
        return this;
    }
    /**
     * Creates a new, empty file system.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifEncrypted()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html
     */
    toCreateFileSystem() {
        this.to('elasticfilesystem:CreateFileSystem');
        return this;
    }
    /**
     * Creates a mount target for a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html
     */
    toCreateMountTarget() {
        this.to('elasticfilesystem:CreateMountTarget');
        return this;
    }
    /**
     * Creates or overwrites tags associated with a file system.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html
     */
    toCreateTags() {
        this.to('elasticfilesystem:CreateTags');
        return this;
    }
    /**
     * Deletes the specified access point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        this.to('elasticfilesystem:DeleteAccessPoint');
        return this;
    }
    /**
     * Deletes a file system, permanently severing access to its contents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html
     */
    toDeleteFileSystem() {
        this.to('elasticfilesystem:DeleteFileSystem');
        return this;
    }
    /**
     * Clears the resource-level policy for a given file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html
     */
    toDeleteFileSystemPolicy() {
        this.to('elasticfilesystem:DeleteFileSystemPolicy');
        return this;
    }
    /**
     * Deletes the specified mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html
     */
    toDeleteMountTarget() {
        this.to('elasticfilesystem:DeleteMountTarget');
        return this;
    }
    /**
     * Deletes the specified tags from a file system.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html
     */
    toDeleteTags() {
        this.to('elasticfilesystem:DeleteTags');
        return this;
    }
    /**
     * Returns the descriptions of Amazon EFS access points.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html
     */
    toDescribeAccessPoints() {
        this.to('elasticfilesystem:DescribeAccessPoints');
        return this;
    }
    /**
     * Returns the current BackupPolicy object for the specified Amazon EFS file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeBackupPolicy.html
     */
    toDescribeBackupPolicy() {
        this.to('elasticfilesystem:DescribeBackupPolicy');
        return this;
    }
    /**
     * Returns the current resource-level policy for a given file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html
     */
    toDescribeFileSystemPolicy() {
        this.to('elasticfilesystem:DescribeFileSystemPolicy');
        return this;
    }
    /**
     * Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html
     */
    toDescribeFileSystems() {
        this.to('elasticfilesystem:DescribeFileSystems');
        return this;
    }
    /**
     * Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html
     */
    toDescribeLifecycleConfiguration() {
        this.to('elasticfilesystem:DescribeLifecycleConfiguration');
        return this;
    }
    /**
     * Returns the security groups currently in effect for a mount target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html
     */
    toDescribeMountTargetSecurityGroups() {
        this.to('elasticfilesystem:DescribeMountTargetSecurityGroups');
        return this;
    }
    /**
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html
     */
    toDescribeMountTargets() {
        this.to('elasticfilesystem:DescribeMountTargets');
        return this;
    }
    /**
     * Returns the tags associated with a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('elasticfilesystem:DescribeTags');
        return this;
    }
    /**
     * Returns the tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('elasticfilesystem:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the set of security groups in effect for a mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html
     */
    toModifyMountTargetSecurityGroups() {
        this.to('elasticfilesystem:ModifyMountTargetSecurityGroups');
        return this;
    }
    /**
     * Enables automatic backups with AWS Backup by creating a new BackupPolicy object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutBackupPolicy.html
     */
    toPutBackupPolicy() {
        this.to('elasticfilesystem:PutBackupPolicy');
        return this;
    }
    /**
     * Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html
     */
    toPutFileSystemPolicy() {
        this.to('elasticfilesystem:PutFileSystemPolicy');
        return this;
    }
    /**
     * Enables lifecycle management by creating a new LifecycleConfiguration object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        this.to('elasticfilesystem:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Starts a restore job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    toRestore() {
        this.to('elasticfilesystem:Restore');
        return this;
    }
    /**
     * Creates or overwrites tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html
     */
    toTagResource() {
        this.to('elasticfilesystem:TagResource');
        return this;
    }
    /**
     * Deletes the specified tags from a specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html
     */
    toUntagResource() {
        this.to('elasticfilesystem:UntagResource');
        return this;
    }
    /**
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html
     */
    toUpdateFileSystem() {
        this.to('elasticfilesystem:UpdateFileSystem');
        return this;
    }
    /**
     * Adds a resource of type file-system to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFileSystem(fileSystemId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}';
        arn = arn.replace('${FileSystemId}', fileSystemId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type access-point to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param accessPointId - Identifier for the accessPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccessPoint(accessPointId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}';
        arn = arn.replace('${AccessPointId}', accessPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ARN of the access point used to mount the file system
     *
     * Applies to actions:
     * - .toClientMount()
     * - .toClientRootAccess()
     * - .toClientWrite()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointArn(value, operator) {
        return this.if(`elasticfilesystem:AccessPointArn`, value, operator || 'StringLike');
    }
    /**
     * Control encryption behavior for new EFS file systems
     *
     * Applies to actions:
     * - .toCreateFileSystem()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEncrypted(value) {
        return this.if(`elasticfilesystem:Encrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
}
exports.Elasticfilesystem = Elasticfilesystem;
//# sourceMappingURL=data:application/json;base64,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