"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticmapreduce = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticmapreduce](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticmapreduce.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticmapreduce extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticmapreduce](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticmapreduce.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticmapreduce';
        this.accessLevelList = {
            "Write": [
                "AddInstanceFleet",
                "AddInstanceGroups",
                "AddJobFlowSteps",
                "CancelSteps",
                "CreateEditor",
                "CreateSecurityConfiguration",
                "DeleteEditor",
                "DeleteSecurityConfiguration",
                "ModifyCluster",
                "ModifyInstanceFleet",
                "ModifyInstanceGroups",
                "OpenEditorInConsole",
                "PutAutoScalingPolicy",
                "PutManagedScalingPolicy",
                "RemoveAutoScalingPolicy",
                "RemoveManagedScalingPolicy",
                "RunJobFlow",
                "SetTerminationProtection",
                "StartEditor",
                "StopEditor",
                "TerminateJobFlows"
            ],
            "Tagging": [
                "AddTags",
                "RemoveTags"
            ],
            "Read": [
                "DescribeCluster",
                "DescribeEditor",
                "DescribeJobFlows",
                "DescribeSecurityConfiguration",
                "DescribeStep",
                "GetBlockPublicAccessConfiguration",
                "GetManagedScalingPolicy",
                "ListBootstrapActions",
                "ListInstanceFleets",
                "ListInstanceGroups",
                "ListInstances",
                "ListSteps"
            ],
            "List": [
                "ListClusters",
                "ListEditors",
                "ListSecurityConfigurations",
                "ViewEventsFromAllClustersInConsole"
            ],
            "Permissions management": [
                "PutBlockPublicAccessConfiguration"
            ]
        };
    }
    /**
     * Grants permission to add an instance fleet to a running cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddInstanceFleet.html
     */
    toAddInstanceFleet() {
        this.to('elasticmapreduce:AddInstanceFleet');
        return this;
    }
    /**
     * Grants permission to add instance groups to a running cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddInstanceGroups.html
     */
    toAddInstanceGroups() {
        this.to('elasticmapreduce:AddInstanceGroups');
        return this;
    }
    /**
     * Grants permission to add new steps to a running cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html
     */
    toAddJobFlowSteps() {
        this.to('elasticmapreduce:AddJobFlowSteps');
        return this;
    }
    /**
     * Grants permission to add tags to an Amazon EMR resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        this.to('elasticmapreduce:AddTags');
        return this;
    }
    /**
     * Grants permission to cancel a pending step or steps in a running cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html
     */
    toCancelSteps() {
        this.to('elasticmapreduce:CancelSteps');
        return this;
    }
    /**
     * Grants permission to create an EMR notebook.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-create.html
     */
    toCreateEditor() {
        this.to('elasticmapreduce:CreateEditor');
        return this;
    }
    /**
     * Grants permission to create a security configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_CreateSecurityConfiguration.html
     */
    toCreateSecurityConfiguration() {
        this.to('elasticmapreduce:CreateSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to delete an EMR notebook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-deleting
     */
    toDeleteEditor() {
        this.to('elasticmapreduce:DeleteEditor');
        return this;
    }
    /**
     * Grants permission to delete a security configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DeleteSecurityConfiguration.html
     */
    toDeleteSecurityConfiguration() {
        this.to('elasticmapreduce:DeleteSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to get details about a cluster, including status, hardware and software configuration, VPC settings, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeCluster.html
     */
    toDescribeCluster() {
        this.to('elasticmapreduce:DescribeCluster');
        return this;
    }
    /**
     * Grants permission to view information about a notebook, including status, user, role, tags, location, and more.
     *
     * Access Level: Read
     */
    toDescribeEditor() {
        this.to('elasticmapreduce:DescribeEditor');
        return this;
    }
    /**
     * This API is deprecated and will eventually be removed. We recommend you use ListClusters, DescribeCluster, ListSteps, ListInstanceGroups and ListBootstrapActions instead.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeJobFlows.html
     */
    toDescribeJobFlows() {
        this.to('elasticmapreduce:DescribeJobFlows');
        return this;
    }
    /**
     * Grants permission to get details of a security configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeSecurityConfiguration.html
     */
    toDescribeSecurityConfiguration() {
        this.to('elasticmapreduce:DescribeSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to get details about a cluster step.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeStep.html
     */
    toDescribeStep() {
        this.to('elasticmapreduce:DescribeStep');
        return this;
    }
    /**
     * Grants permission to retrieve the EMR block public access configuration for the AWS account in the Region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetBlockPublicAccessConfiguration.html
     */
    toGetBlockPublicAccessConfiguration() {
        this.to('elasticmapreduce:GetBlockPublicAccessConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve the managed scaling policy associated with a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetManagedScalingPolicy.html
     */
    toGetManagedScalingPolicy() {
        this.to('elasticmapreduce:GetManagedScalingPolicy');
        return this;
    }
    /**
     * Grants permission to get details about the bootstrap actions associated with a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListBootstrapActions.html
     */
    toListBootstrapActions() {
        this.to('elasticmapreduce:ListBootstrapActions');
        return this;
    }
    /**
     * Grants permission to get the status of accessible clusters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListClusters.html
     */
    toListClusters() {
        this.to('elasticmapreduce:ListClusters');
        return this;
    }
    /**
     * Grants permission to list summary information for accessible EMR notebooks.
     *
     * Access Level: List
     */
    toListEditors() {
        this.to('elasticmapreduce:ListEditors');
        return this;
    }
    /**
     * Grants permission to get details of instance fleets in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstanceFleets.html
     */
    toListInstanceFleets() {
        this.to('elasticmapreduce:ListInstanceFleets');
        return this;
    }
    /**
     * Grants permission to get details of instance groups in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstanceGroups.html
     */
    toListInstanceGroups() {
        this.to('elasticmapreduce:ListInstanceGroups');
        return this;
    }
    /**
     * Grants permission to get details about the Amazon EC2 instances in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstances.html
     */
    toListInstances() {
        this.to('elasticmapreduce:ListInstances');
        return this;
    }
    /**
     * Grants permission to list available security configurations in this account by name, along with creation dates and times.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSecurityConfigurations.html
     */
    toListSecurityConfigurations() {
        this.to('elasticmapreduce:ListSecurityConfigurations');
        return this;
    }
    /**
     * Grants permission to list steps associated with a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSteps.html
     */
    toListSteps() {
        this.to('elasticmapreduce:ListSteps');
        return this;
    }
    /**
     * Grants permission to change cluster settings such as number of steps that can be executed concurrently for a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyCluster.html
     */
    toModifyCluster() {
        this.to('elasticmapreduce:ModifyCluster');
        return this;
    }
    /**
     * Grants permission to change the target On-Demand and target Spot capacities for a instance fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html
     */
    toModifyInstanceFleet() {
        this.to('elasticmapreduce:ModifyInstanceFleet');
        return this;
    }
    /**
     * Grants permission to change the number and configuration of EC2 instances for an instance group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html
     */
    toModifyInstanceGroups() {
        this.to('elasticmapreduce:ModifyInstanceGroups');
        return this;
    }
    /**
     * Grants permission to launch the Jupyter notebook editor for an EMR notebook from within the console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toOpenEditorInConsole() {
        this.to('elasticmapreduce:OpenEditorInConsole');
        return this;
    }
    /**
     * Grants permission to create or update an automatic scaling policy for a core instance group or task instance group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutAutoScalingPolicy.html
     */
    toPutAutoScalingPolicy() {
        this.to('elasticmapreduce:PutAutoScalingPolicy');
        return this;
    }
    /**
     * Grants permission to create or update the EMR block public access configuration for the AWS account in the Region.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutBlockPublicAccessConfiguration.html
     */
    toPutBlockPublicAccessConfiguration() {
        this.to('elasticmapreduce:PutBlockPublicAccessConfiguration');
        return this;
    }
    /**
     * Grants permission to create or update the managed scaling policy associated with a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutManagedScalingPolicy.html
     */
    toPutManagedScalingPolicy() {
        this.to('elasticmapreduce:PutManagedScalingPolicy');
        return this;
    }
    /**
     * Grants permission to remove an automatic scaling policy from an instance group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveAutoScalingPolicy.html
     */
    toRemoveAutoScalingPolicy() {
        this.to('elasticmapreduce:RemoveAutoScalingPolicy');
        return this;
    }
    /**
     * Grants permission to remove the managed scaling policy associated with a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveManagedScalingPolicy.html
     */
    toRemoveManagedScalingPolicy() {
        this.to('elasticmapreduce:RemoveManagedScalingPolicy');
        return this;
    }
    /**
     * Grants permission to remove tags from an Amazon EMR resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        this.to('elasticmapreduce:RemoveTags');
        return this;
    }
    /**
     * Grants permission to create and launch a cluster (job flow).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html
     */
    toRunJobFlow() {
        this.to('elasticmapreduce:RunJobFlow');
        return this;
    }
    /**
     * Grants permission to add and remove termination protection for a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetTerminationProtection.html
     */
    toSetTerminationProtection() {
        this.to('elasticmapreduce:SetTerminationProtection');
        return this;
    }
    /**
     * Grants permission to start an EMR notebook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toStartEditor() {
        this.to('elasticmapreduce:StartEditor');
        return this;
    }
    /**
     * Grants permission to shut down an EMR notebook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html
     */
    toStopEditor() {
        this.to('elasticmapreduce:StopEditor');
        return this;
    }
    /**
     * Grants permission to terminate a cluster (job flow).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html
     */
    toTerminateJobFlows() {
        this.to('elasticmapreduce:TerminateJobFlows');
        return this;
    }
    /**
     * Grants permission to use the EMR management console to view events from all clusters.
     *
     * Access Level: List
     */
    toViewEventsFromAllClustersInConsole() {
        this.to('elasticmapreduce:ViewEventsFromAllClustersInConsole');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-overview.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onCluster(clusterId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticmapreduce:${Region}:${Account}:cluster/${ClusterId}';
        arn = arn.replace('${ClusterId}', clusterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type editor to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html
     *
     * @param editorId - Identifier for the editorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEditor(editorId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticmapreduce:${Region}:${Account}:editor/${EditorId}';
        arn = arn.replace('${EditorId}', editorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on whether the tag and value pair is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toRunJobFlow()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestTag(tagKey, value, operator) {
        return this.if(`elasticmapreduce:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag and value pair associated with an Amazon EMR resource
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to resource types:
     * - cluster
     * - editor
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`elasticmapreduce:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Elasticmapreduce = Elasticmapreduce;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uZWxhc3RpY21hcHJlZHVjZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFtYXpvbmVsYXN0aWNtYXByZWR1Y2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esc0NBQXNEO0FBRXREOzs7O0dBSUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLHdCQUFlO0lBR25EOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxrQkFBa0IsQ0FBQztRQXVmaEMsb0JBQWUsR0FBb0I7WUFDM0MsT0FBTyxFQUFFO2dCQUNQLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixpQkFBaUI7Z0JBQ2pCLGFBQWE7Z0JBQ2IsY0FBYztnQkFDZCw2QkFBNkI7Z0JBQzdCLGNBQWM7Z0JBQ2QsNkJBQTZCO2dCQUM3QixlQUFlO2dCQUNmLHFCQUFxQjtnQkFDckIsc0JBQXNCO2dCQUN0QixxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIseUJBQXlCO2dCQUN6Qix5QkFBeUI7Z0JBQ3pCLDRCQUE0QjtnQkFDNUIsWUFBWTtnQkFDWiwwQkFBMEI7Z0JBQzFCLGFBQWE7Z0JBQ2IsWUFBWTtnQkFDWixtQkFBbUI7YUFDcEI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsU0FBUztnQkFDVCxZQUFZO2FBQ2I7WUFDRCxNQUFNLEVBQUU7Z0JBQ04saUJBQWlCO2dCQUNqQixnQkFBZ0I7Z0JBQ2hCLGtCQUFrQjtnQkFDbEIsK0JBQStCO2dCQUMvQixjQUFjO2dCQUNkLG1DQUFtQztnQkFDbkMseUJBQXlCO2dCQUN6QixzQkFBc0I7Z0JBQ3RCLG9CQUFvQjtnQkFDcEIsb0JBQW9CO2dCQUNwQixlQUFlO2dCQUNmLFdBQVc7YUFDWjtZQUNELE1BQU0sRUFBRTtnQkFDTixjQUFjO2dCQUNkLGFBQWE7Z0JBQ2IsNEJBQTRCO2dCQUM1QixvQ0FBb0M7YUFDckM7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsbUNBQW1DO2FBQ3BDO1NBQ0YsQ0FBQztJQWppQkYsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxJQUFJLENBQUMsRUFBRSxDQUFDLDhDQUE4QyxDQUFDLENBQUM7UUFDeEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxFQUFFLENBQUMsZ0RBQWdELENBQUMsQ0FBQztRQUMxRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQ0FBbUM7UUFDeEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1FBQzlELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLDBDQUEwQyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLDZDQUE2QyxDQUFDLENBQUM7UUFDdkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQ0FBbUM7UUFDeEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1FBQzlELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLDBDQUEwQyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsMkNBQTJDLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxvQ0FBb0M7UUFDekMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO1FBQy9ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQXVERDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkYsSUFBSSxHQUFHLEdBQUcsNkVBQTZFLENBQUM7UUFDeEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFFBQVEsQ0FBQyxRQUFnQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JGLElBQUksR0FBRyxHQUFHLDJFQUEyRSxDQUFDO1FBQ3RGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMzQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxZQUFZLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUFnQyxNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzdGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxhQUFhLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDekYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFpQyxNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzlGLENBQUM7Q0FDRjtBQTduQkQsNENBNm5CQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gXCIuLi9zaGFyZWQvYWNjZXNzLWxldmVsXCI7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZWxhc3RpY21hcHJlZHVjZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uZWxhc3RpY21hcHJlZHVjZS5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEVsYXN0aWNtYXByZWR1Y2UgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdlbGFzdGljbWFwcmVkdWNlJztcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtlbGFzdGljbWFwcmVkdWNlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25lbGFzdGljbWFwcmVkdWNlLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhbiBpbnN0YW5jZSBmbGVldCB0byBhIHJ1bm5pbmcgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FkZEluc3RhbmNlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWRkSW5zdGFuY2VGbGVldCgpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkFkZEluc3RhbmNlRmxlZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgaW5zdGFuY2UgZ3JvdXBzIHRvIGEgcnVubmluZyBjbHVzdGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQWRkSW5zdGFuY2VHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWRkSW5zdGFuY2VHcm91cHMoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpBZGRJbnN0YW5jZUdyb3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBuZXcgc3RlcHMgdG8gYSBydW5uaW5nIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BZGRKb2JGbG93U3RlcHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWRkSm9iRmxvd1N0ZXBzKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6QWRkSm9iRmxvd1N0ZXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIHRhZ3MgdG8gYW4gQW1hem9uIEVNUiByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQWRkVGFncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BZGRUYWdzKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6QWRkVGFncycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNhbmNlbCBhIHBlbmRpbmcgc3RlcCBvciBzdGVwcyBpbiBhIHJ1bm5pbmcgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NhbmNlbFN0ZXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbFN0ZXBzKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6Q2FuY2VsU3RlcHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gRU1SIG5vdGVib29rLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L01hbmFnZW1lbnRHdWlkZS9lbXItbWFuYWdlZC1ub3RlYm9va3MtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUVkaXRvcigpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkNyZWF0ZUVkaXRvcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTZWN1cml0eUNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2VjdXJpdHlDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6Q3JlYXRlU2VjdXJpdHlDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIEVNUiBub3RlYm9vay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9NYW5hZ2VtZW50R3VpZGUvZW1yLW1hbmFnZWQtbm90ZWJvb2tzLmh0bWwjZW1yLW1hbmFnZWQtbm90ZWJvb2tzLWRlbGV0aW5nXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVFZGl0b3IoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpEZWxldGVFZGl0b3InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBzZWN1cml0eSBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlU2VjdXJpdHlDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNlY3VyaXR5Q29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkRlbGV0ZVNlY3VyaXR5Q29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBkZXRhaWxzIGFib3V0IGEgY2x1c3RlciwgaW5jbHVkaW5nIHN0YXR1cywgaGFyZHdhcmUgYW5kIHNvZnR3YXJlIGNvbmZpZ3VyYXRpb24sIFZQQyBzZXR0aW5ncywgYW5kIHNvIG9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUNsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVDbHVzdGVyKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6RGVzY3JpYmVDbHVzdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBpbmZvcm1hdGlvbiBhYm91dCBhIG5vdGVib29rLCBpbmNsdWRpbmcgc3RhdHVzLCB1c2VyLCByb2xlLCB0YWdzLCBsb2NhdGlvbiwgYW5kIG1vcmUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFZGl0b3IoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpEZXNjcmliZUVkaXRvcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgQVBJIGlzIGRlcHJlY2F0ZWQgYW5kIHdpbGwgZXZlbnR1YWxseSBiZSByZW1vdmVkLiBXZSByZWNvbW1lbmQgeW91IHVzZSBMaXN0Q2x1c3RlcnMsIERlc2NyaWJlQ2x1c3RlciwgTGlzdFN0ZXBzLCBMaXN0SW5zdGFuY2VHcm91cHMgYW5kIExpc3RCb290c3RyYXBBY3Rpb25zIGluc3RlYWQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlSm9iRmxvd3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVKb2JGbG93cygpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkRlc2NyaWJlSm9iRmxvd3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgZGV0YWlscyBvZiBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU2VjdXJpdHlDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU2VjdXJpdHlDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6RGVzY3JpYmVTZWN1cml0eUNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgZGV0YWlscyBhYm91dCBhIGNsdXN0ZXIgc3RlcC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGVwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU3RlcCgpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkRlc2NyaWJlU3RlcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBFTVIgYmxvY2sgcHVibGljIGFjY2VzcyBjb25maWd1cmF0aW9uIGZvciB0aGUgQVdTIGFjY291bnQgaW4gdGhlIFJlZ2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0QmxvY2tQdWJsaWNBY2Nlc3NDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJsb2NrUHVibGljQWNjZXNzQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkdldEJsb2NrUHVibGljQWNjZXNzQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBtYW5hZ2VkIHNjYWxpbmcgcG9saWN5IGFzc29jaWF0ZWQgd2l0aCBhIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldE1hbmFnZWRTY2FsaW5nUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1hbmFnZWRTY2FsaW5nUG9saWN5KCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6R2V0TWFuYWdlZFNjYWxpbmdQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgZGV0YWlscyBhYm91dCB0aGUgYm9vdHN0cmFwIGFjdGlvbnMgYXNzb2NpYXRlZCB3aXRoIGEgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEJvb3RzdHJhcEFjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEJvb3RzdHJhcEFjdGlvbnMoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpMaXN0Qm9vdHN0cmFwQWN0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgc3RhdHVzIG9mIGFjY2Vzc2libGUgY2x1c3RlcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RDbHVzdGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Q2x1c3RlcnMoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpMaXN0Q2x1c3RlcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHN1bW1hcnkgaW5mb3JtYXRpb24gZm9yIGFjY2Vzc2libGUgRU1SIG5vdGVib29rcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RWRpdG9ycygpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkxpc3RFZGl0b3JzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGRldGFpbHMgb2YgaW5zdGFuY2UgZmxlZXRzIGluIGEgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEluc3RhbmNlRmxlZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnN0YW5jZUZsZWV0cygpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkxpc3RJbnN0YW5jZUZsZWV0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBkZXRhaWxzIG9mIGluc3RhbmNlIGdyb3VwcyBpbiBhIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RJbnN0YW5jZUdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SW5zdGFuY2VHcm91cHMoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpMaXN0SW5zdGFuY2VHcm91cHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgZGV0YWlscyBhYm91dCB0aGUgQW1hem9uIEVDMiBpbnN0YW5jZXMgaW4gYSBjbHVzdGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0SW5zdGFuY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnN0YW5jZXMoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpMaXN0SW5zdGFuY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhdmFpbGFibGUgc2VjdXJpdHkgY29uZmlndXJhdGlvbnMgaW4gdGhpcyBhY2NvdW50IGJ5IG5hbWUsIGFsb25nIHdpdGggY3JlYXRpb24gZGF0ZXMgYW5kIHRpbWVzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U2VjdXJpdHlDb25maWd1cmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2VjdXJpdHlDb25maWd1cmF0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOkxpc3RTZWN1cml0eUNvbmZpZ3VyYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBzdGVwcyBhc3NvY2lhdGVkIHdpdGggYSBjbHVzdGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U3RlcHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFN0ZXBzKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6TGlzdFN0ZXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2hhbmdlIGNsdXN0ZXIgc2V0dGluZ3Mgc3VjaCBhcyBudW1iZXIgb2Ygc3RlcHMgdGhhdCBjYW4gYmUgZXhlY3V0ZWQgY29uY3VycmVudGx5IGZvciBhIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeUNsdXN0ZXIoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpNb2RpZnlDbHVzdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2hhbmdlIHRoZSB0YXJnZXQgT24tRGVtYW5kIGFuZCB0YXJnZXQgU3BvdCBjYXBhY2l0aWVzIGZvciBhIGluc3RhbmNlIGZsZWV0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5SW5zdGFuY2VGbGVldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlJbnN0YW5jZUZsZWV0KCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6TW9kaWZ5SW5zdGFuY2VGbGVldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNoYW5nZSB0aGUgbnVtYmVyIGFuZCBjb25maWd1cmF0aW9uIG9mIEVDMiBpbnN0YW5jZXMgZm9yIGFuIGluc3RhbmNlIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5SW5zdGFuY2VHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTW9kaWZ5SW5zdGFuY2VHcm91cHMoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpNb2RpZnlJbnN0YW5jZUdyb3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxhdW5jaCB0aGUgSnVweXRlciBub3RlYm9vayBlZGl0b3IgZm9yIGFuIEVNUiBub3RlYm9vayBmcm9tIHdpdGhpbiB0aGUgY29uc29sZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9NYW5hZ2VtZW50R3VpZGUvZW1yLW1hbmFnZWQtbm90ZWJvb2tzLmh0bWwjZW1yLW1hbmFnZWQtbm90ZWJvb2tzLWVkaXRvclxuICAgKi9cbiAgcHVibGljIHRvT3BlbkVkaXRvckluQ29uc29sZSgpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOk9wZW5FZGl0b3JJbkNvbnNvbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgb3IgdXBkYXRlIGFuIGF1dG9tYXRpYyBzY2FsaW5nIHBvbGljeSBmb3IgYSBjb3JlIGluc3RhbmNlIGdyb3VwIG9yIHRhc2sgaW5zdGFuY2UgZ3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRBdXRvU2NhbGluZ1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRBdXRvU2NhbGluZ1BvbGljeSgpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOlB1dEF1dG9TY2FsaW5nUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIG9yIHVwZGF0ZSB0aGUgRU1SIGJsb2NrIHB1YmxpYyBhY2Nlc3MgY29uZmlndXJhdGlvbiBmb3IgdGhlIEFXUyBhY2NvdW50IGluIHRoZSBSZWdpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dEJsb2NrUHVibGljQWNjZXNzQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRCbG9ja1B1YmxpY0FjY2Vzc0NvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpQdXRCbG9ja1B1YmxpY0FjY2Vzc0NvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgb3IgdXBkYXRlIHRoZSBtYW5hZ2VkIHNjYWxpbmcgcG9saWN5IGFzc29jaWF0ZWQgd2l0aCBhIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRNYW5hZ2VkU2NhbGluZ1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRNYW5hZ2VkU2NhbGluZ1BvbGljeSgpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOlB1dE1hbmFnZWRTY2FsaW5nUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIGFuIGF1dG9tYXRpYyBzY2FsaW5nIHBvbGljeSBmcm9tIGFuIGluc3RhbmNlIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVtb3ZlQXV0b1NjYWxpbmdQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVtb3ZlQXV0b1NjYWxpbmdQb2xpY3koKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpSZW1vdmVBdXRvU2NhbGluZ1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgbWFuYWdlZCBzY2FsaW5nIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggYSBjbHVzdGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVtb3ZlTWFuYWdlZFNjYWxpbmdQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVtb3ZlTWFuYWdlZFNjYWxpbmdQb2xpY3koKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpSZW1vdmVNYW5hZ2VkU2NhbGluZ1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gYW4gQW1hem9uIEVNUiByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVtb3ZlVGFncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZW1vdmVUYWdzKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6UmVtb3ZlVGFncycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbmQgbGF1bmNoIGEgY2x1c3RlciAoam9iIGZsb3cpLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUnVuSm9iRmxvdy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SdW5Kb2JGbG93KCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6UnVuSm9iRmxvdycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhbmQgcmVtb3ZlIHRlcm1pbmF0aW9uIHByb3RlY3Rpb24gZm9yIGEgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldFRlcm1pbmF0aW9uUHJvdGVjdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRUZXJtaW5hdGlvblByb3RlY3Rpb24oKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpTZXRUZXJtaW5hdGlvblByb3RlY3Rpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhbiBFTVIgbm90ZWJvb2suXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvTWFuYWdlbWVudEd1aWRlL2Vtci1tYW5hZ2VkLW5vdGVib29rcy13b3JraW5nLXdpdGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRFZGl0b3IoKSB7XG4gICAgdGhpcy50bygnZWxhc3RpY21hcHJlZHVjZTpTdGFydEVkaXRvcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNodXQgZG93biBhbiBFTVIgbm90ZWJvb2suXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvTWFuYWdlbWVudEd1aWRlL2Vtci1tYW5hZ2VkLW5vdGVib29rcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wRWRpdG9yKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6U3RvcEVkaXRvcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHRlcm1pbmF0ZSBhIGNsdXN0ZXIgKGpvYiBmbG93KS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Rlcm1pbmF0ZUpvYkZsb3dzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Rlcm1pbmF0ZUpvYkZsb3dzKCkge1xuICAgIHRoaXMudG8oJ2VsYXN0aWNtYXByZWR1Y2U6VGVybWluYXRlSm9iRmxvd3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1c2UgdGhlIEVNUiBtYW5hZ2VtZW50IGNvbnNvbGUgdG8gdmlldyBldmVudHMgZnJvbSBhbGwgY2x1c3RlcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKi9cbiAgcHVibGljIHRvVmlld0V2ZW50c0Zyb21BbGxDbHVzdGVyc0luQ29uc29sZSgpIHtcbiAgICB0aGlzLnRvKCdlbGFzdGljbWFwcmVkdWNlOlZpZXdFdmVudHNGcm9tQWxsQ2x1c3RlcnNJbkNvbnNvbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQWRkSW5zdGFuY2VGbGVldFwiLFxuICAgICAgXCJBZGRJbnN0YW5jZUdyb3Vwc1wiLFxuICAgICAgXCJBZGRKb2JGbG93U3RlcHNcIixcbiAgICAgIFwiQ2FuY2VsU3RlcHNcIixcbiAgICAgIFwiQ3JlYXRlRWRpdG9yXCIsXG4gICAgICBcIkNyZWF0ZVNlY3VyaXR5Q29uZmlndXJhdGlvblwiLFxuICAgICAgXCJEZWxldGVFZGl0b3JcIixcbiAgICAgIFwiRGVsZXRlU2VjdXJpdHlDb25maWd1cmF0aW9uXCIsXG4gICAgICBcIk1vZGlmeUNsdXN0ZXJcIixcbiAgICAgIFwiTW9kaWZ5SW5zdGFuY2VGbGVldFwiLFxuICAgICAgXCJNb2RpZnlJbnN0YW5jZUdyb3Vwc1wiLFxuICAgICAgXCJPcGVuRWRpdG9ySW5Db25zb2xlXCIsXG4gICAgICBcIlB1dEF1dG9TY2FsaW5nUG9saWN5XCIsXG4gICAgICBcIlB1dE1hbmFnZWRTY2FsaW5nUG9saWN5XCIsXG4gICAgICBcIlJlbW92ZUF1dG9TY2FsaW5nUG9saWN5XCIsXG4gICAgICBcIlJlbW92ZU1hbmFnZWRTY2FsaW5nUG9saWN5XCIsXG4gICAgICBcIlJ1bkpvYkZsb3dcIixcbiAgICAgIFwiU2V0VGVybWluYXRpb25Qcm90ZWN0aW9uXCIsXG4gICAgICBcIlN0YXJ0RWRpdG9yXCIsXG4gICAgICBcIlN0b3BFZGl0b3JcIixcbiAgICAgIFwiVGVybWluYXRlSm9iRmxvd3NcIlxuICAgIF0sXG4gICAgXCJUYWdnaW5nXCI6IFtcbiAgICAgIFwiQWRkVGFnc1wiLFxuICAgICAgXCJSZW1vdmVUYWdzXCJcbiAgICBdLFxuICAgIFwiUmVhZFwiOiBbXG4gICAgICBcIkRlc2NyaWJlQ2x1c3RlclwiLFxuICAgICAgXCJEZXNjcmliZUVkaXRvclwiLFxuICAgICAgXCJEZXNjcmliZUpvYkZsb3dzXCIsXG4gICAgICBcIkRlc2NyaWJlU2VjdXJpdHlDb25maWd1cmF0aW9uXCIsXG4gICAgICBcIkRlc2NyaWJlU3RlcFwiLFxuICAgICAgXCJHZXRCbG9ja1B1YmxpY0FjY2Vzc0NvbmZpZ3VyYXRpb25cIixcbiAgICAgIFwiR2V0TWFuYWdlZFNjYWxpbmdQb2xpY3lcIixcbiAgICAgIFwiTGlzdEJvb3RzdHJhcEFjdGlvbnNcIixcbiAgICAgIFwiTGlzdEluc3RhbmNlRmxlZXRzXCIsXG4gICAgICBcIkxpc3RJbnN0YW5jZUdyb3Vwc1wiLFxuICAgICAgXCJMaXN0SW5zdGFuY2VzXCIsXG4gICAgICBcIkxpc3RTdGVwc1wiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0Q2x1c3RlcnNcIixcbiAgICAgIFwiTGlzdEVkaXRvcnNcIixcbiAgICAgIFwiTGlzdFNlY3VyaXR5Q29uZmlndXJhdGlvbnNcIixcbiAgICAgIFwiVmlld0V2ZW50c0Zyb21BbGxDbHVzdGVyc0luQ29uc29sZVwiXG4gICAgXSxcbiAgICBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIjogW1xuICAgICAgXCJQdXRCbG9ja1B1YmxpY0FjY2Vzc0NvbmZpZ3VyYXRpb25cIlxuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2x1c3RlciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L01hbmFnZW1lbnRHdWlkZS9lbXItb3ZlcnZpZXcuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY2x1c3RlcklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2x1c3RlcihjbHVzdGVySWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZWxhc3RpY21hcHJlZHVjZToke1JlZ2lvbn06JHtBY2NvdW50fTpjbHVzdGVyLyR7Q2x1c3RlcklkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Q2x1c3RlcklkfScsIGNsdXN0ZXJJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZWRpdG9yIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvTWFuYWdlbWVudEd1aWRlL2Vtci1tYW5hZ2VkLW5vdGVib29rcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBlZGl0b3JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBlZGl0b3JJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRWRpdG9yKGVkaXRvcklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmVsYXN0aWNtYXByZWR1Y2U6JHtSZWdpb259OiR7QWNjb3VudH06ZWRpdG9yLyR7RWRpdG9ySWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtFZGl0b3JJZH0nLCBlZGl0b3JJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHdoZXRoZXIgdGhlIHRhZyBhbmQgdmFsdWUgcGFpciBpcyBwcm92aWRlZCB3aXRoIHRoZSBhY3Rpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvTWFuYWdlbWVudEd1aWRlL2Vtci1wbGFuLWFjY2Vzcy1pYW0uaHRtbCNlbXItZmluZS1ncmFpbmVkLWNsdXN0ZXItYWNjZXNzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BZGRUYWdzKClcbiAgICogLSAudG9DcmVhdGVFZGl0b3IoKVxuICAgKiAtIC50b1J1bkpvYkZsb3coKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGVsYXN0aWNtYXByZWR1Y2U6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgdGFnIGFuZCB2YWx1ZSBwYWlyIGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gRU1SIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L01hbmFnZW1lbnRHdWlkZS9lbXItcGxhbi1hY2Nlc3MtaWFtLmh0bWwjZW1yLWZpbmUtZ3JhaW5lZC1jbHVzdGVyLWFjY2Vzc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGNsdXN0ZXJcbiAgICogLSBlZGl0b3JcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZWxhc3RpY21hcHJlZHVjZTpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG59XG4iXX0=