"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [macie2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacie.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [macie2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacie.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'macie2';
        this.accessLevelList = {
            "Write": [
                "AcceptInvitation",
                "ArchiveFindings",
                "CreateClassificationJob",
                "CreateCustomDataIdentifier",
                "CreateFindingsFilter",
                "CreateInvitations",
                "CreateMember",
                "CreateSampleFindings",
                "DeclineInvitations",
                "DeleteCustomDataIdentifier",
                "DeleteFindingsFilter",
                "DeleteInvitations",
                "DeleteMember",
                "DisableMacie",
                "DisableOrganizationAdminAccount",
                "DisassociateFromMasterAccount",
                "DisassociateMember",
                "EnableMacie",
                "EnableOrganizationAdminAccount",
                "PutClassificationExportConfiguration",
                "TestCustomDataIdentifier",
                "UnarchiveFindings",
                "UpdateClassificationJob",
                "UpdateFindingsFilter",
                "UpdateMacieSession",
                "UpdateMemberSession",
                "UpdateOrganizationConfiguration"
            ],
            "Read": [
                "BatchGetCustomDataIdentifiers",
                "DescribeBuckets",
                "DescribeClassificationJob",
                "DescribeOrganizationConfiguration",
                "GetBucketStatistics",
                "GetClassificationExportConfiguration",
                "GetCustomDataIdentifier",
                "GetFindingStatistics",
                "GetFindings",
                "GetFindingsFilter",
                "GetInvitationsCount",
                "GetMacieSession",
                "GetMasterAccount",
                "GetMember",
                "GetUsageStatistics",
                "GetUsageTotals"
            ],
            "List": [
                "ListClassificationJobs",
                "ListCustomDataIdentifiers",
                "ListFindings",
                "ListFindingsFilters",
                "ListInvitations",
                "ListMembers",
                "ListOrganizationAdminAccounts",
                "ListTagsForResources"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to accept an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-accept.html
     */
    toAcceptInvitation() {
        this.to('macie2:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to archive one or more findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-archive.html
     */
    toArchiveFindings() {
        this.to('macie2:ArchiveFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about one or more custom data identifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-get.html
     */
    toBatchGetCustomDataIdentifiers() {
        this.to('macie2:BatchGetCustomDataIdentifiers');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a classification job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs.html
     */
    toCreateClassificationJob() {
        this.to('macie2:CreateClassificationJob');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a custom data identifier
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers.html
     */
    toCreateCustomDataIdentifier() {
        this.to('macie2:CreateCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    toCreateFindingsFilter() {
        this.to('macie2:CreateFindingsFilter');
        return this;
    }
    /**
     * Grants permission to send an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    toCreateInvitations() {
        this.to('macie2:CreateInvitations');
        return this;
    }
    /**
     * Grants permission to associate an account with an Amazon Macie master account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    toCreateMember() {
        this.to('macie2:CreateMember');
        return this;
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-sample.html
     */
    toCreateSampleFindings() {
        this.to('macie2:CreateSampleFindings');
        return this;
    }
    /**
     * Grants permission to decline Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-decline.html
     */
    toDeclineInvitations() {
        this.to('macie2:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    toDeleteCustomDataIdentifier() {
        this.to('macie2:DeleteCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to delete a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    toDeleteFindingsFilter() {
        this.to('macie2:DeleteFindingsFilter');
        return this;
    }
    /**
     * Grants permission to delete Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-delete.html
     */
    toDeleteInvitations() {
        this.to('macie2:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete the association between an Amazon Macie master account and an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    toDeleteMember() {
        this.to('macie2:DeleteMember');
        return this;
    }
    /**
     * Grants permission to retrieve statistical and other data about S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3.html
     */
    toDescribeBuckets() {
        this.to('macie2:DescribeBuckets');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and settings for a classification job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    toDescribeClassificationJob() {
        this.to('macie2:DescribeClassificationJob');
        return this;
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    toDescribeOrganizationConfiguration() {
        this.to('macie2:DescribeOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to disable an Amazon Macie account, which also deletes Macie resources for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toDisableMacie() {
        this.to('macie2:DisableMacie');
        return this;
    }
    /**
     * Grants permission to disable an account as a delegated administrator of Amazon Macie for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    toDisableOrganizationAdminAccount() {
        this.to('macie2:DisableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants an Amazon Macie member account with permission to disassociate from its master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master-disassociate.html
     */
    toDisassociateFromMasterAccount() {
        this.to('macie2:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants an Amazon Macie master account with permission to disassociate from a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-disassociate-id.html
     */
    toDisassociateMember() {
        this.to('macie2:DisassociateMember');
        return this;
    }
    /**
     * Grants permission to enable and specify the configuration settings for a new Amazon Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toEnableMacie() {
        this.to('macie2:EnableMacie');
        return this;
    }
    /**
     * Grants permission to enable an account as a delegated administrator of Amazon Macie for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    toEnableOrganizationAdminAccount() {
        this.to('macie2:EnableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated statistical data for all the S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3-statistics.html
     */
    toGetBucketStatistics() {
        this.to('macie2:GetBucketStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve the settings for exporting data classification results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    toGetClassificationExportConfiguration() {
        this.to('macie2:GetClassificationExportConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve information about the settings for a custom data identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    toGetCustomDataIdentifier() {
        this.to('macie2:GetCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated statistical data about findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-statistics.html
     */
    toGetFindingStatistics() {
        this.to('macie2:GetFindingStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve information about one or more findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-describe.html
     */
    toGetFindings() {
        this.to('macie2:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about the settings for a findings filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    toGetFindingsFilter() {
        this.to('macie2:GetFindingsFilter');
        return this;
    }
    /**
     * Grants permission to retrieve the count of Amazon Macie membership invitations that were received by an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-count.html
     */
    toGetInvitationsCount() {
        this.to('macie2:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and configuration settings for an Amazon Macie account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toGetMacieSession() {
        this.to('macie2:GetMacieSession');
        return this;
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie master account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master.html
     */
    toGetMasterAccount() {
        this.to('macie2:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information about an account that's associated with an Amazon Macie master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    toGetMember() {
        this.to('macie2:GetMember');
        return this;
    }
    /**
     * Grants permission to retrieve quotas and aggregated usage data for one or more accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage-statistics.html
     */
    toGetUsageStatistics() {
        this.to('macie2:GetUsageStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated usage data for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage.html
     */
    toGetUsageTotals() {
        this.to('macie2:GetUsageTotals');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and settings for one or more classification jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-list.html
     */
    toListClassificationJobs() {
        this.to('macie2:ListClassificationJobs');
        return this;
    }
    /**
     * Grants permission to retrieve information about all custom data identifiers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-list.html
     */
    toListCustomDataIdentifiers() {
        this.to('macie2:ListCustomDataIdentifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a subset of information about one or more findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings.html
     */
    toListFindings() {
        this.to('macie2:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about all findings filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    toListFindingsFilters() {
        this.to('macie2:ListFindingsFilters');
        return this;
    }
    /**
     * Grants permission to retrieve information about all the Amazon Macie membership invitations that were received by an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    toListInvitations() {
        this.to('macie2:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve information about all the accounts that are associated with an Amazon Macie master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    toListMembers() {
        this.to('macie2:ListMembers');
        return this;
    }
    /**
     * Grants permission to retrieve information about the delegated, Amazon Macie administrator account for an AWS organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    toListOrganizationAdminAccounts() {
        this.to('macie2:ListOrganizationAdminAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for an Amazon Macie resource or member account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    toListTagsForResources() {
        this.to('macie2:ListTagsForResources');
        return this;
    }
    /**
     * Grants permission to create or update the settings for exporting data classification results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    toPutClassificationExportConfiguration() {
        this.to('macie2:PutClassificationExportConfiguration');
        return this;
    }
    /**
     * Grants permission to add or update the tags for an Amazon Macie resource or member account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    toTagResource() {
        this.to('macie2:TagResource');
        return this;
    }
    /**
     * Grants permission to test a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-test.html
     */
    toTestCustomDataIdentifier() {
        this.to('macie2:TestCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to reactivate (unarchive) one or more findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-unarchive.html
     */
    toUnarchiveFindings() {
        this.to('macie2:UnarchiveFindings');
        return this;
    }
    /**
     * Grants permission to remove tags from an Amazon Macie resource or member account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    toUntagResource() {
        this.to('macie2:UntagResource');
        return this;
    }
    /**
     * Grants permission to cancel a classification job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    toUpdateClassificationJob() {
        this.to('macie2:UpdateClassificationJob');
        return this;
    }
    /**
     * Grants permission to update the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    toUpdateFindingsFilter() {
        this.to('macie2:UpdateFindingsFilter');
        return this;
    }
    /**
     * Grants permission to suspend or re-enable an Amazon Macie account, or update the configuration settings for a Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toUpdateMacieSession() {
        this.to('macie2:UpdateMacieSession');
        return this;
    }
    /**
     * Grants an Amazon Macie master account with permission to suspend or re-enable a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie-members-id.html
     */
    toUpdateMemberSession() {
        this.to('macie2:UpdateMemberSession');
        return this;
    }
    /**
     * Grants permission to update Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    toUpdateOrganizationConfiguration() {
        this.to('macie2:UpdateOrganizationConfiguration');
        return this;
    }
    /**
     * Adds a resource of type ClassificationJob to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClassificationJob(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:classification-job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CustomDataIdentifier to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomDataIdentifier(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:custom-data-identifier/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Member to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMember(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:member/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type FindingsFilter to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFindingsFilter(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:findings-filter/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Macie2 = Macie2;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9ubWFjaWUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b25tYWNpZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxzQ0FBNEM7QUFFNUM7Ozs7R0FJRztBQUNILE1BQWEsTUFBTyxTQUFRLHdCQUFlO0lBR3pDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxRQUFRLENBQUM7UUFzcUJ0QixvQkFBZSxHQUFvQjtZQUMzQyxPQUFPLEVBQUU7Z0JBQ1Asa0JBQWtCO2dCQUNsQixpQkFBaUI7Z0JBQ2pCLHlCQUF5QjtnQkFDekIsNEJBQTRCO2dCQUM1QixzQkFBc0I7Z0JBQ3RCLG1CQUFtQjtnQkFDbkIsY0FBYztnQkFDZCxzQkFBc0I7Z0JBQ3RCLG9CQUFvQjtnQkFDcEIsNEJBQTRCO2dCQUM1QixzQkFBc0I7Z0JBQ3RCLG1CQUFtQjtnQkFDbkIsY0FBYztnQkFDZCxjQUFjO2dCQUNkLGlDQUFpQztnQkFDakMsK0JBQStCO2dCQUMvQixvQkFBb0I7Z0JBQ3BCLGFBQWE7Z0JBQ2IsZ0NBQWdDO2dCQUNoQyxzQ0FBc0M7Z0JBQ3RDLDBCQUEwQjtnQkFDMUIsbUJBQW1CO2dCQUNuQix5QkFBeUI7Z0JBQ3pCLHNCQUFzQjtnQkFDdEIsb0JBQW9CO2dCQUNwQixxQkFBcUI7Z0JBQ3JCLGlDQUFpQzthQUNsQztZQUNELE1BQU0sRUFBRTtnQkFDTiwrQkFBK0I7Z0JBQy9CLGlCQUFpQjtnQkFDakIsMkJBQTJCO2dCQUMzQixtQ0FBbUM7Z0JBQ25DLHFCQUFxQjtnQkFDckIsc0NBQXNDO2dCQUN0Qyx5QkFBeUI7Z0JBQ3pCLHNCQUFzQjtnQkFDdEIsYUFBYTtnQkFDYixtQkFBbUI7Z0JBQ25CLHFCQUFxQjtnQkFDckIsaUJBQWlCO2dCQUNqQixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsb0JBQW9CO2dCQUNwQixnQkFBZ0I7YUFDakI7WUFDRCxNQUFNLEVBQUU7Z0JBQ04sd0JBQXdCO2dCQUN4QiwyQkFBMkI7Z0JBQzNCLGNBQWM7Z0JBQ2QscUJBQXFCO2dCQUNyQixpQkFBaUI7Z0JBQ2pCLGFBQWE7Z0JBQ2IsK0JBQStCO2dCQUMvQixzQkFBc0I7YUFDdkI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQTN0QkYsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLElBQUksQ0FBQyxFQUFFLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLElBQUksQ0FBQyxFQUFFLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQWtFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1CQUFtQixDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUNqRixJQUFJLEdBQUcsR0FBRyxzRUFBc0UsQ0FBQztRQUNqRixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxzQkFBc0IsQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDcEYsSUFBSSxHQUFHLEdBQUcsMEVBQTBFLENBQUM7UUFDckYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksUUFBUSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUN0RSxJQUFJLEdBQUcsR0FBRywwREFBMEQsQ0FBQztRQUNyRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxnQkFBZ0IsQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDOUUsSUFBSSxHQUFHLEdBQUcsbUVBQW1FLENBQUM7UUFDOUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBdHpCRCx3QkFzekJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW21hY2llMl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9ubWFjaWUuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBNYWNpZTIgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdtYWNpZTInO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW21hY2llMl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9ubWFjaWUuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWNjZXB0IGFuIEFtYXpvbiBNYWNpZSBtZW1iZXJzaGlwIGludml0YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9pbnZpdGF0aW9ucy1hY2NlcHQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWNjZXB0SW52aXRhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6QWNjZXB0SW52aXRhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFyY2hpdmUgb25lIG9yIG1vcmUgZmluZGluZ3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5ncy1hcmNoaXZlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FyY2hpdmVGaW5kaW5ncygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6QXJjaGl2ZUZpbmRpbmdzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgb25lIG9yIG1vcmUgY3VzdG9tIGRhdGEgaWRlbnRpZmllcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2N1c3RvbS1kYXRhLWlkZW50aWZpZXJzLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaEdldEN1c3RvbURhdGFJZGVudGlmaWVycygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6QmF0Y2hHZXRDdXN0b21EYXRhSWRlbnRpZmllcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW5kIGRlZmluZSB0aGUgc2V0dGluZ3MgZm9yIGEgY2xhc3NpZmljYXRpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2Uvam9icy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDbGFzc2lmaWNhdGlvbkpvYigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6Q3JlYXRlQ2xhc3NpZmljYXRpb25Kb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW5kIGRlZmluZSB0aGUgc2V0dGluZ3MgZm9yIGEgY3VzdG9tIGRhdGEgaWRlbnRpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2N1c3RvbS1kYXRhLWlkZW50aWZpZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUN1c3RvbURhdGFJZGVudGlmaWVyKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpDcmVhdGVDdXN0b21EYXRhSWRlbnRpZmllcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbmQgZGVmaW5lIHRoZSBzZXR0aW5ncyBmb3IgYSBmaW5kaW5ncyBmaWx0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5nc2ZpbHRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRmluZGluZ3NGaWx0ZXIoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkNyZWF0ZUZpbmRpbmdzRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2VuZCBhbiBBbWF6b24gTWFjaWUgbWVtYmVyc2hpcCBpbnZpdGF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvaW52aXRhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW52aXRhdGlvbnMoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkNyZWF0ZUludml0YXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGFuIGFjY291bnQgd2l0aCBhbiBBbWF6b24gTWFjaWUgbWFzdGVyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9tZW1iZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU1lbWJlcigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6Q3JlYXRlTWVtYmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIHNhbXBsZSBmaW5kaW5nc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2ZpbmRpbmdzLXNhbXBsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTYW1wbGVGaW5kaW5ncygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6Q3JlYXRlU2FtcGxlRmluZGluZ3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWNsaW5lIEFtYXpvbiBNYWNpZSBtZW1iZXJzaGlwIGludml0YXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvaW52aXRhdGlvbnMtZGVjbGluZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWNsaW5lSW52aXRhdGlvbnMoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkRlY2xpbmVJbnZpdGF0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGN1c3RvbSBkYXRhIGlkZW50aWZpZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9jdXN0b20tZGF0YS1pZGVudGlmaWVycy1pZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVDdXN0b21EYXRhSWRlbnRpZmllcigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6RGVsZXRlQ3VzdG9tRGF0YUlkZW50aWZpZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBmaW5kaW5ncyBmaWx0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5nc2ZpbHRlcnMtaWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRmluZGluZ3NGaWx0ZXIoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkRlbGV0ZUZpbmRpbmdzRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIEFtYXpvbiBNYWNpZSBtZW1iZXJzaGlwIGludml0YXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvaW52aXRhdGlvbnMtZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUludml0YXRpb25zKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpEZWxldGVJbnZpdGF0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgYXNzb2NpYXRpb24gYmV0d2VlbiBhbiBBbWF6b24gTWFjaWUgbWFzdGVyIGFjY291bnQgYW5kIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9tZW1iZXJzLWlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU1lbWJlcigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6RGVsZXRlTWVtYmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgc3RhdGlzdGljYWwgYW5kIG90aGVyIGRhdGEgYWJvdXQgUzMgYnVja2V0cyB0aGF0IEFtYXpvbiBNYWNpZSBtb25pdG9ycyBhbmQgYW5hbHl6ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2RhdGFzb3VyY2VzLXMzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQnVja2V0cygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6RGVzY3JpYmVCdWNrZXRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHN0YXR1cyBhbmQgc2V0dGluZ3MgZm9yIGEgY2xhc3NpZmljYXRpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9qb2JzLWpvYmlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ2xhc3NpZmljYXRpb25Kb2IoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkRlc2NyaWJlQ2xhc3NpZmljYXRpb25Kb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgQW1hem9uIE1hY2llIGNvbmZpZ3VyYXRpb24gc2V0dGluZ3MgZm9yIGFuIEFXUyBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2FkbWluLWNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVPcmdhbml6YXRpb25Db25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpEZXNjcmliZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhYmxlIGFuIEFtYXpvbiBNYWNpZSBhY2NvdW50LCB3aGljaCBhbHNvIGRlbGV0ZXMgTWFjaWUgcmVzb3VyY2VzIGZvciB0aGUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21hY2llLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2FibGVNYWNpZSgpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6RGlzYWJsZU1hY2llJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYWJsZSBhbiBhY2NvdW50IGFzIGEgZGVsZWdhdGVkIGFkbWluaXN0cmF0b3Igb2YgQW1hem9uIE1hY2llIGZvciBhbiBBV1Mgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvYWRtaW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudCgpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6RGlzYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBhbiBBbWF6b24gTWFjaWUgbWVtYmVyIGFjY291bnQgd2l0aCBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBmcm9tIGl0cyBtYXN0ZXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21hc3Rlci1kaXNhc3NvY2lhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlRnJvbU1hc3RlckFjY291bnQoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkRpc2Fzc29jaWF0ZUZyb21NYXN0ZXJBY2NvdW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIGFuIEFtYXpvbiBNYWNpZSBtYXN0ZXIgYWNjb3VudCB3aXRoIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGZyb20gYSBtZW1iZXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21lbWJlcnMtZGlzYXNzb2NpYXRlLWlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZU1lbWJlcigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6RGlzYXNzb2NpYXRlTWVtYmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZW5hYmxlIGFuZCBzcGVjaWZ5IHRoZSBjb25maWd1cmF0aW9uIHNldHRpbmdzIGZvciBhIG5ldyBBbWF6b24gTWFjaWUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21hY2llLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0VuYWJsZU1hY2llKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpFbmFibGVNYWNpZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGVuYWJsZSBhbiBhY2NvdW50IGFzIGEgZGVsZWdhdGVkIGFkbWluaXN0cmF0b3Igb2YgQW1hem9uIE1hY2llIGZvciBhbiBBV1Mgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvYWRtaW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRW5hYmxlT3JnYW5pemF0aW9uQWRtaW5BY2NvdW50KCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpFbmFibGVPcmdhbml6YXRpb25BZG1pbkFjY291bnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhZ2dyZWdhdGVkIHN0YXRpc3RpY2FsIGRhdGEgZm9yIGFsbCB0aGUgUzMgYnVja2V0cyB0aGF0IEFtYXpvbiBNYWNpZSBtb25pdG9ycyBhbmQgYW5hbHl6ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2RhdGFzb3VyY2VzLXMzLXN0YXRpc3RpY3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QnVja2V0U3RhdGlzdGljcygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6R2V0QnVja2V0U3RhdGlzdGljcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBzZXR0aW5ncyBmb3IgZXhwb3J0aW5nIGRhdGEgY2xhc3NpZmljYXRpb24gcmVzdWx0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvY2xhc3NpZmljYXRpb24tZXhwb3J0LWNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q2xhc3NpZmljYXRpb25FeHBvcnRDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpHZXRDbGFzc2lmaWNhdGlvbkV4cG9ydENvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc2V0dGluZ3MgZm9yIGEgY3VzdG9tIGRhdGEgaWRlbnRpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvY3VzdG9tLWRhdGEtaWRlbnRpZmllcnMtaWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q3VzdG9tRGF0YUlkZW50aWZpZXIoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkdldEN1c3RvbURhdGFJZGVudGlmaWVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYWdncmVnYXRlZCBzdGF0aXN0aWNhbCBkYXRhIGFib3V0IGZpbmRpbmdzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5ncy1zdGF0aXN0aWNzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEZpbmRpbmdTdGF0aXN0aWNzKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpHZXRGaW5kaW5nU3RhdGlzdGljcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IG9uZSBvciBtb3JlIGZpbmRpbmdzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5ncy1kZXNjcmliZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRGaW5kaW5ncygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6R2V0RmluZGluZ3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc2V0dGluZ3MgZm9yIGEgZmluZGluZ3MgZmlsdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5nc2ZpbHRlcnMtaWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RmluZGluZ3NGaWx0ZXIoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkdldEZpbmRpbmdzRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGNvdW50IG9mIEFtYXpvbiBNYWNpZSBtZW1iZXJzaGlwIGludml0YXRpb25zIHRoYXQgd2VyZSByZWNlaXZlZCBieSBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9pbnZpdGF0aW9ucy1jb3VudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnZpdGF0aW9uc0NvdW50KCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpHZXRJbnZpdGF0aW9uc0NvdW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHN0YXR1cyBhbmQgY29uZmlndXJhdGlvbiBzZXR0aW5ncyBmb3IgYW4gQW1hem9uIE1hY2llIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21hY2llLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1hY2llU2Vzc2lvbigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6R2V0TWFjaWVTZXNzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgdGhlIEFtYXpvbiBNYWNpZSBtYXN0ZXIgYWNjb3VudCBmb3IgYW4gYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvbWFzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1hc3RlckFjY291bnQoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkdldE1hc3RlckFjY291bnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhbiBhY2NvdW50IHRoYXQncyBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIE1hY2llIG1hc3RlciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9tZW1iZXJzLWlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1lbWJlcigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6R2V0TWVtYmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgcXVvdGFzIGFuZCBhZ2dyZWdhdGVkIHVzYWdlIGRhdGEgZm9yIG9uZSBvciBtb3JlIGFjY291bnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS91c2FnZS1zdGF0aXN0aWNzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFVzYWdlU3RhdGlzdGljcygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6R2V0VXNhZ2VTdGF0aXN0aWNzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYWdncmVnYXRlZCB1c2FnZSBkYXRhIGZvciBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS91c2FnZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRVc2FnZVRvdGFscygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6R2V0VXNhZ2VUb3RhbHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc3RhdHVzIGFuZCBzZXR0aW5ncyBmb3Igb25lIG9yIG1vcmUgY2xhc3NpZmljYXRpb24gam9ic1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2Uvam9icy1saXN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDbGFzc2lmaWNhdGlvbkpvYnMoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkxpc3RDbGFzc2lmaWNhdGlvbkpvYnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhbGwgY3VzdG9tIGRhdGEgaWRlbnRpZmllcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2N1c3RvbS1kYXRhLWlkZW50aWZpZXJzLWxpc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEN1c3RvbURhdGFJZGVudGlmaWVycygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6TGlzdEN1c3RvbURhdGFJZGVudGlmaWVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgc3Vic2V0IG9mIGluZm9ybWF0aW9uIGFib3V0IG9uZSBvciBtb3JlIGZpbmRpbmdzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5ncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RmluZGluZ3MoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkxpc3RGaW5kaW5ncycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGFsbCBmaW5kaW5ncyBmaWx0ZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9maW5kaW5nc2ZpbHRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEZpbmRpbmdzRmlsdGVycygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6TGlzdEZpbmRpbmdzRmlsdGVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGFsbCB0aGUgQW1hem9uIE1hY2llIG1lbWJlcnNoaXAgaW52aXRhdGlvbnMgdGhhdCB3ZXJlIHJlY2VpdmVkIGJ5IGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2ludml0YXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnZpdGF0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6TGlzdEludml0YXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgYWxsIHRoZSBhY2NvdW50cyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIE1hY2llIG1hc3RlciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9tZW1iZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNZW1iZXJzKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpMaXN0TWVtYmVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHRoZSBkZWxlZ2F0ZWQsIEFtYXpvbiBNYWNpZSBhZG1pbmlzdHJhdG9yIGFjY291bnQgZm9yIGFuIEFXUyBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2FkbWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RPcmdhbml6YXRpb25BZG1pbkFjY291bnRzKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpMaXN0T3JnYW5pemF0aW9uQWRtaW5BY2NvdW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSB0YWdzIGZvciBhbiBBbWF6b24gTWFjaWUgcmVzb3VyY2Ugb3IgbWVtYmVyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL3RhZ3MtcmVzb3VyY2Vhcm4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZXMoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOkxpc3RUYWdzRm9yUmVzb3VyY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIG9yIHVwZGF0ZSB0aGUgc2V0dGluZ3MgZm9yIGV4cG9ydGluZyBkYXRhIGNsYXNzaWZpY2F0aW9uIHJlc3VsdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9jbGFzc2lmaWNhdGlvbi1leHBvcnQtY29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRDbGFzc2lmaWNhdGlvbkV4cG9ydENvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOlB1dENsYXNzaWZpY2F0aW9uRXhwb3J0Q29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciB1cGRhdGUgdGhlIHRhZ3MgZm9yIGFuIEFtYXpvbiBNYWNpZSByZXNvdXJjZSBvciBtZW1iZXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvdGFncy1yZXNvdXJjZWFybi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB0ZXN0IGEgY3VzdG9tIGRhdGEgaWRlbnRpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2N1c3RvbS1kYXRhLWlkZW50aWZpZXJzLXRlc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGVzdEN1c3RvbURhdGFJZGVudGlmaWVyKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpUZXN0Q3VzdG9tRGF0YUlkZW50aWZpZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZWFjdGl2YXRlICh1bmFyY2hpdmUpIG9uZSBvciBtb3JlIGZpbmRpbmdzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvZmluZGluZ3MtdW5hcmNoaXZlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VuYXJjaGl2ZUZpbmRpbmdzKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpVbmFyY2hpdmVGaW5kaW5ncycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gYW4gQW1hem9uIE1hY2llIHJlc291cmNlIG9yIG1lbWJlciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS90YWdzLXJlc291cmNlYXJuLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnbWFjaWUyOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjYW5jZWwgYSBjbGFzc2lmaWNhdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWFjaWUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9qb2JzLWpvYmlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNsYXNzaWZpY2F0aW9uSm9iKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpVcGRhdGVDbGFzc2lmaWNhdGlvbkpvYicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgc2V0dGluZ3MgZm9yIGEgZmluZGluZ3MgZmlsdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC9BUElSZWZlcmVuY2UvZmluZGluZ3NmaWx0ZXJzLWlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUZpbmRpbmdzRmlsdGVyKCkge1xuICAgIHRoaXMudG8oJ21hY2llMjpVcGRhdGVGaW5kaW5nc0ZpbHRlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN1c3BlbmQgb3IgcmUtZW5hYmxlIGFuIEFtYXpvbiBNYWNpZSBhY2NvdW50LCBvciB1cGRhdGUgdGhlIGNvbmZpZ3VyYXRpb24gc2V0dGluZ3MgZm9yIGEgTWFjaWUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21hY2llLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU1hY2llU2Vzc2lvbigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6VXBkYXRlTWFjaWVTZXNzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIGFuIEFtYXpvbiBNYWNpZSBtYXN0ZXIgYWNjb3VudCB3aXRoIHBlcm1pc3Npb24gdG8gc3VzcGVuZCBvciByZS1lbmFibGUgYSBtZW1iZXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL21hY2llLW1lbWJlcnMtaWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTWVtYmVyU2Vzc2lvbigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6VXBkYXRlTWVtYmVyU2Vzc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBBbWF6b24gTWFjaWUgY29uZmlndXJhdGlvbiBzZXR0aW5ncyBmb3IgYW4gQVdTIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvQVBJUmVmZXJlbmNlL2FkbWluLWNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdtYWNpZTI6VXBkYXRlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFwiV3JpdGVcIjogW1xuICAgICAgXCJBY2NlcHRJbnZpdGF0aW9uXCIsXG4gICAgICBcIkFyY2hpdmVGaW5kaW5nc1wiLFxuICAgICAgXCJDcmVhdGVDbGFzc2lmaWNhdGlvbkpvYlwiLFxuICAgICAgXCJDcmVhdGVDdXN0b21EYXRhSWRlbnRpZmllclwiLFxuICAgICAgXCJDcmVhdGVGaW5kaW5nc0ZpbHRlclwiLFxuICAgICAgXCJDcmVhdGVJbnZpdGF0aW9uc1wiLFxuICAgICAgXCJDcmVhdGVNZW1iZXJcIixcbiAgICAgIFwiQ3JlYXRlU2FtcGxlRmluZGluZ3NcIixcbiAgICAgIFwiRGVjbGluZUludml0YXRpb25zXCIsXG4gICAgICBcIkRlbGV0ZUN1c3RvbURhdGFJZGVudGlmaWVyXCIsXG4gICAgICBcIkRlbGV0ZUZpbmRpbmdzRmlsdGVyXCIsXG4gICAgICBcIkRlbGV0ZUludml0YXRpb25zXCIsXG4gICAgICBcIkRlbGV0ZU1lbWJlclwiLFxuICAgICAgXCJEaXNhYmxlTWFjaWVcIixcbiAgICAgIFwiRGlzYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudFwiLFxuICAgICAgXCJEaXNhc3NvY2lhdGVGcm9tTWFzdGVyQWNjb3VudFwiLFxuICAgICAgXCJEaXNhc3NvY2lhdGVNZW1iZXJcIixcbiAgICAgIFwiRW5hYmxlTWFjaWVcIixcbiAgICAgIFwiRW5hYmxlT3JnYW5pemF0aW9uQWRtaW5BY2NvdW50XCIsXG4gICAgICBcIlB1dENsYXNzaWZpY2F0aW9uRXhwb3J0Q29uZmlndXJhdGlvblwiLFxuICAgICAgXCJUZXN0Q3VzdG9tRGF0YUlkZW50aWZpZXJcIixcbiAgICAgIFwiVW5hcmNoaXZlRmluZGluZ3NcIixcbiAgICAgIFwiVXBkYXRlQ2xhc3NpZmljYXRpb25Kb2JcIixcbiAgICAgIFwiVXBkYXRlRmluZGluZ3NGaWx0ZXJcIixcbiAgICAgIFwiVXBkYXRlTWFjaWVTZXNzaW9uXCIsXG4gICAgICBcIlVwZGF0ZU1lbWJlclNlc3Npb25cIixcbiAgICAgIFwiVXBkYXRlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvblwiXG4gICAgXSxcbiAgICBcIlJlYWRcIjogW1xuICAgICAgXCJCYXRjaEdldEN1c3RvbURhdGFJZGVudGlmaWVyc1wiLFxuICAgICAgXCJEZXNjcmliZUJ1Y2tldHNcIixcbiAgICAgIFwiRGVzY3JpYmVDbGFzc2lmaWNhdGlvbkpvYlwiLFxuICAgICAgXCJEZXNjcmliZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb25cIixcbiAgICAgIFwiR2V0QnVja2V0U3RhdGlzdGljc1wiLFxuICAgICAgXCJHZXRDbGFzc2lmaWNhdGlvbkV4cG9ydENvbmZpZ3VyYXRpb25cIixcbiAgICAgIFwiR2V0Q3VzdG9tRGF0YUlkZW50aWZpZXJcIixcbiAgICAgIFwiR2V0RmluZGluZ1N0YXRpc3RpY3NcIixcbiAgICAgIFwiR2V0RmluZGluZ3NcIixcbiAgICAgIFwiR2V0RmluZGluZ3NGaWx0ZXJcIixcbiAgICAgIFwiR2V0SW52aXRhdGlvbnNDb3VudFwiLFxuICAgICAgXCJHZXRNYWNpZVNlc3Npb25cIixcbiAgICAgIFwiR2V0TWFzdGVyQWNjb3VudFwiLFxuICAgICAgXCJHZXRNZW1iZXJcIixcbiAgICAgIFwiR2V0VXNhZ2VTdGF0aXN0aWNzXCIsXG4gICAgICBcIkdldFVzYWdlVG90YWxzXCJcbiAgICBdLFxuICAgIFwiTGlzdFwiOiBbXG4gICAgICBcIkxpc3RDbGFzc2lmaWNhdGlvbkpvYnNcIixcbiAgICAgIFwiTGlzdEN1c3RvbURhdGFJZGVudGlmaWVyc1wiLFxuICAgICAgXCJMaXN0RmluZGluZ3NcIixcbiAgICAgIFwiTGlzdEZpbmRpbmdzRmlsdGVyc1wiLFxuICAgICAgXCJMaXN0SW52aXRhdGlvbnNcIixcbiAgICAgIFwiTGlzdE1lbWJlcnNcIixcbiAgICAgIFwiTGlzdE9yZ2FuaXphdGlvbkFkbWluQWNjb3VudHNcIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZXNcIlxuICAgIF0sXG4gICAgXCJUYWdnaW5nXCI6IFtcbiAgICAgIFwiVGFnUmVzb3VyY2VcIixcbiAgICAgIFwiVW50YWdSZXNvdXJjZVwiXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBDbGFzc2lmaWNhdGlvbkpvYiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tYWNpZS9sYXRlc3QvdXNlci93aGF0LWlzLW1hY2llLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNsYXNzaWZpY2F0aW9uSm9iKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259Om1hY2llMjo6JHtBY2NvdW50fTpjbGFzc2lmaWNhdGlvbi1qb2IvJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEN1c3RvbURhdGFJZGVudGlmaWVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC91c2VyL3doYXQtaXMtbWFjaWUuaHRtbC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25DdXN0b21EYXRhSWRlbnRpZmllcihyZXNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTptYWNpZTI6OiR7QWNjb3VudH06Y3VzdG9tLWRhdGEtaWRlbnRpZmllci8ke1Jlc291cmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZUlkfScsIHJlc291cmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTWVtYmVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC91c2VyL3doYXQtaXMtbWFjaWUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTWVtYmVyKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259Om1hY2llMjo6JHtBY2NvdW50fTptZW1iZXIvJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEZpbmRpbmdzRmlsdGVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21hY2llL2xhdGVzdC91c2VyL3doYXQtaXMtbWFjaWUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRmluZGluZ3NGaWx0ZXIocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bWFjaWUyOjoke0FjY291bnR9OmZpbmRpbmdzLWZpbHRlci8ke1Jlc291cmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZUlkfScsIHJlc291cmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==