"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rds = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [rds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrds.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rds extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [rds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrds.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'rds';
        this.accessLevelList = {
            "Write": [
                "AddRoleToDBCluster",
                "AddRoleToDBInstance",
                "AddSourceIdentifierToSubscription",
                "ApplyPendingMaintenanceAction",
                "BacktrackDBCluster",
                "CancelExportTask",
                "CopyDBClusterParameterGroup",
                "CopyDBClusterSnapshot",
                "CopyDBParameterGroup",
                "CopyDBSnapshot",
                "CopyOptionGroup",
                "CreateDBCluster",
                "CreateDBClusterEndpoint",
                "CreateDBClusterParameterGroup",
                "CreateDBClusterSnapshot",
                "CreateDBInstance",
                "CreateDBInstanceReadReplica",
                "CreateDBParameterGroup",
                "CreateDBProxy",
                "CreateDBSecurityGroup",
                "CreateDBSnapshot",
                "CreateDBSubnetGroup",
                "CreateEventSubscription",
                "CreateGlobalCluster",
                "CreateOptionGroup",
                "DeleteDBCluster",
                "DeleteDBClusterEndpoint",
                "DeleteDBClusterParameterGroup",
                "DeleteDBClusterSnapshot",
                "DeleteDBInstance",
                "DeleteDBInstanceAutomatedBackup",
                "DeleteDBParameterGroup",
                "DeleteDBProxy",
                "DeleteDBSecurityGroup",
                "DeleteDBSnapshot",
                "DeleteDBSubnetGroup",
                "DeleteEventSubscription",
                "DeleteGlobalCluster",
                "DeleteOptionGroup",
                "DeregisterDBProxyTargets",
                "FailoverDBCluster",
                "ModifyCurrentDBClusterCapacity",
                "ModifyDBCluster",
                "ModifyDBClusterEndpoint",
                "ModifyDBClusterParameterGroup",
                "ModifyDBClusterSnapshotAttribute",
                "ModifyDBInstance",
                "ModifyDBParameterGroup",
                "ModifyDBProxy",
                "ModifyDBProxyTargetGroup",
                "ModifyDBSnapshot",
                "ModifyDBSnapshotAttribute",
                "ModifyDBSubnetGroup",
                "ModifyEventSubscription",
                "ModifyGlobalCluster",
                "ModifyOptionGroup",
                "PromoteReadReplica",
                "PromoteReadReplicaDBCluster",
                "PurchaseReservedDBInstancesOffering",
                "RebootDBInstance",
                "RegisterDBProxyTargets",
                "RemoveFromGlobalCluster",
                "RemoveRoleFromDBCluster",
                "RemoveRoleFromDBInstance",
                "RemoveSourceIdentifierFromSubscription",
                "ResetDBClusterParameterGroup",
                "ResetDBParameterGroup",
                "RestoreDBClusterFromS3",
                "RestoreDBClusterFromSnapshot",
                "RestoreDBClusterToPointInTime",
                "RestoreDBInstanceFromDBSnapshot",
                "RestoreDBInstanceFromS3",
                "RestoreDBInstanceToPointInTime",
                "RevokeDBSecurityGroupIngress",
                "StartActivityStream",
                "StartDBCluster",
                "StartDBInstance",
                "StartExportTask",
                "StopActivityStream",
                "StopDBCluster",
                "StopDBInstance"
            ],
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Permissions management": [
                "AuthorizeDBSecurityGroupIngress"
            ],
            "List": [
                "DescribeAccountAttributes",
                "DescribeCertificates",
                "DescribeDBClusterBacktracks",
                "DescribeDBClusterEndpoints",
                "DescribeDBClusterParameterGroups",
                "DescribeDBClusterParameters",
                "DescribeDBClusterSnapshotAttributes",
                "DescribeDBClusterSnapshots",
                "DescribeDBClusters",
                "DescribeDBEngineVersions",
                "DescribeDBInstanceAutomatedBackups",
                "DescribeDBInstances",
                "DescribeDBLogFiles",
                "DescribeDBParameterGroups",
                "DescribeDBParameters",
                "DescribeDBProxies",
                "DescribeDBProxyTargetGroups",
                "DescribeDBProxyTargets",
                "DescribeDBSecurityGroups",
                "DescribeDBSnapshotAttributes",
                "DescribeDBSnapshots",
                "DescribeDBSubnetGroups",
                "DescribeEngineDefaultClusterParameters",
                "DescribeEngineDefaultParameters",
                "DescribeEventCategories",
                "DescribeEventSubscriptions",
                "DescribeEvents",
                "DescribeExportTasks",
                "DescribeGlobalClusters",
                "DescribeOptionGroupOptions",
                "DescribeOptionGroups",
                "DescribeOrderableDBInstanceOptions",
                "DescribePendingMaintenanceActions",
                "DescribeReservedDBInstances",
                "DescribeReservedDBInstancesOfferings",
                "DescribeSourceRegions",
                "DescribeValidDBInstanceModifications"
            ],
            "Read": [
                "DownloadDBLogFilePortion",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Grants permission to associate an Identity and Access Management (IAM) role from an Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBCluster.html
     */
    toAddRoleToDBCluster() {
        this.to('rds:AddRoleToDBCluster');
        return this;
    }
    /**
     * Grants permission to associate an AWS Identity and Access Management (IAM) role with a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBInstance.html
     */
    toAddRoleToDBInstance() {
        this.to('rds:AddRoleToDBInstance');
        return this;
    }
    /**
     * Grants permission to add a source identifier to an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddSourceIdentifierToSubscription.html
     */
    toAddSourceIdentifierToSubscription() {
        this.to('rds:AddSourceIdentifierToSubscription');
        return this;
    }
    /**
     * Grants permission to add metadata tags to an Amazon RDS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('rds:AddTagsToResource');
        return this;
    }
    /**
     * Grants permission to apply a pending maintenance action to a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction() {
        this.to('rds:ApplyPendingMaintenanceAction');
        return this;
    }
    /**
     * Grants permission to enable ingress to a DBSecurityGroup using one of two forms of authorization
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html
     */
    toAuthorizeDBSecurityGroupIngress() {
        this.to('rds:AuthorizeDBSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to backtrack a DB cluster to a specific time, without creating a new DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_BacktrackDBCluster.html
     */
    toBacktrackDBCluster() {
        this.to('rds:BacktrackDBCluster');
        return this;
    }
    /**
     * Grants permission to cancel an export task in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        this.to('rds:CancelExportTask');
        return this;
    }
    /**
     * Grants permission to copy the specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterParameterGroup.html
     */
    toCopyDBClusterParameterGroup() {
        this.to('rds:CopyDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterSnapshot.html
     */
    toCopyDBClusterSnapshot() {
        this.to('rds:CopyDBClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to copy the specified DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBParameterGroup.html
     */
    toCopyDBParameterGroup() {
        this.to('rds:CopyDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to copy the specified DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBSnapshot.html
     */
    toCopyDBSnapshot() {
        this.to('rds:CopyDBSnapshot');
        return this;
    }
    /**
     * Grants permission to copy the specified option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyOptionGroup.html
     */
    toCopyOptionGroup() {
        this.to('rds:CopyOptionGroup');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html
     */
    toCreateDBCluster() {
        this.to('rds:CreateDBCluster');
        return this;
    }
    /**
     * Grants permission to create a new custom endpoint and associates it with an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpointType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterEndpoint.html
     */
    toCreateDBClusterEndpoint() {
        this.to('rds:CreateDBClusterEndpoint');
        return this;
    }
    /**
     * Grants permission to create a new DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html
     */
    toCreateDBClusterParameterGroup() {
        this.to('rds:CreateDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterSnapshot.html
     */
    toCreateDBClusterSnapshot() {
        this.to('rds:CreateDBClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create a new DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html
     */
    toCreateDBInstance() {
        this.to('rds:CreateDBInstance');
        return this;
    }
    /**
     * Grants permission to create a DB instance that acts as a Read Replica of a source DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html
     */
    toCreateDBInstanceReadReplica() {
        this.to('rds:CreateDBInstanceReadReplica');
        return this;
    }
    /**
     * Grants permission to create a new DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html
     */
    toCreateDBParameterGroup() {
        this.to('rds:CreateDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to create a database proxy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxy.html
     */
    toCreateDBProxy() {
        this.to('rds:CreateDBProxy');
        return this;
    }
    /**
     * Grants permission to create a new DB security group. DB security groups control access to a DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSecurityGroup.html
     */
    toCreateDBSecurityGroup() {
        this.to('rds:CreateDBSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a DBSnapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSnapshot.html
     */
    toCreateDBSnapshot() {
        this.to('rds:CreateDBSnapshot');
        return this;
    }
    /**
     * Grants permission to create a new DB subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSubnetGroup.html
     */
    toCreateDBSubnetGroup() {
        this.to('rds:CreateDBSubnetGroup');
        return this;
    }
    /**
     * Grants permission to create an RDS event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription() {
        this.to('rds:CreateEventSubscription');
        return this;
    }
    /**
     * Grants permission to create an Aurora global database spread across multiple regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateGlobalCluster.html
     */
    toCreateGlobalCluster() {
        this.to('rds:CreateGlobalCluster');
        return this;
    }
    /**
     * Grants permission to create a new option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateOptionGroup.html
     */
    toCreateOptionGroup() {
        this.to('rds:CreateOptionGroup');
        return this;
    }
    /**
     * Grants permission to delete a previously provisioned DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBCluster.html
     */
    toDeleteDBCluster() {
        this.to('rds:DeleteDBCluster');
        return this;
    }
    /**
     * Grants permission to delete a custom endpoint and removes it from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterEndpoint.html
     */
    toDeleteDBClusterEndpoint() {
        this.to('rds:DeleteDBClusterEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterParameterGroup.html
     */
    toDeleteDBClusterParameterGroup() {
        this.to('rds:DeleteDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to delete a DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterSnapshot.html
     */
    toDeleteDBClusterSnapshot() {
        this.to('rds:DeleteDBClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a previously provisioned DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstance.html
     */
    toDeleteDBInstance() {
        this.to('rds:DeleteDBInstance');
        return this;
    }
    /**
     * Grants permission to deletes automated backups based on the source instance's DbiResourceId value or the restorable instance's resource ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstanceAutomatedBackup.html
     */
    toDeleteDBInstanceAutomatedBackup() {
        this.to('rds:DeleteDBInstanceAutomatedBackup');
        return this;
    }
    /**
     * Grants permission to delete a specified DBParameterGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBParameterGroup.html
     */
    toDeleteDBParameterGroup() {
        this.to('rds:DeleteDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to delete a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxy.html
     */
    toDeleteDBProxy() {
        this.to('rds:DeleteDBProxy');
        return this;
    }
    /**
     * Grants permission to delete a DB security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSecurityGroup.html
     */
    toDeleteDBSecurityGroup() {
        this.to('rds:DeleteDBSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a DBSnapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSnapshot.html
     */
    toDeleteDBSnapshot() {
        this.to('rds:DeleteDBSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSubnetGroup.html
     */
    toDeleteDBSubnetGroup() {
        this.to('rds:DeleteDBSubnetGroup');
        return this;
    }
    /**
     * Grants permission to delete an RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription() {
        this.to('rds:DeleteEventSubscription');
        return this;
    }
    /**
     * Grants permission to delete a global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteGlobalCluster.html
     */
    toDeleteGlobalCluster() {
        this.to('rds:DeleteGlobalCluster');
        return this;
    }
    /**
     * Grants permission to delete an existing option group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteOptionGroup.html
     */
    toDeleteOptionGroup() {
        this.to('rds:DeleteOptionGroup');
        return this;
    }
    /**
     * Grants permission to remove targets from a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeregisterDBProxyTargets.html
     */
    toDeregisterDBProxyTargets() {
        this.to('rds:DeregisterDBProxyTargets');
        return this;
    }
    /**
     * Grants permission to list all of the attributes for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.to('rds:DescribeAccountAttributes');
        return this;
    }
    /**
     * Lists the set of CA certificates provided by Amazon RDS for this AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates() {
        this.to('rds:DescribeCertificates');
        return this;
    }
    /**
     * Grants permission to return information about backtracks for a DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterBacktracks.html
     */
    toDescribeDBClusterBacktracks() {
        this.to('rds:DescribeDBClusterBacktracks');
        return this;
    }
    /**
     * Grants permission to return information about endpoints for an Amazon Aurora DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterEndpoints.html
     */
    toDescribeDBClusterEndpoints() {
        this.to('rds:DescribeDBClusterEndpoints');
        return this;
    }
    /**
     * Grants permission to return a list of DBClusterParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameterGroups.html
     */
    toDescribeDBClusterParameterGroups() {
        this.to('rds:DescribeDBClusterParameterGroups');
        return this;
    }
    /**
     * Grants permission to return the detailed parameter list for a particular DB cluster parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameters.html
     */
    toDescribeDBClusterParameters() {
        this.to('rds:DescribeDBClusterParameters');
        return this;
    }
    /**
     * Grants permission to return a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshotAttributes.html
     */
    toDescribeDBClusterSnapshotAttributes() {
        this.to('rds:DescribeDBClusterSnapshotAttributes');
        return this;
    }
    /**
     * Grants permission to return information about DB cluster snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshots.html
     */
    toDescribeDBClusterSnapshots() {
        this.to('rds:DescribeDBClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to return information about provisioned Aurora DB clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html
     */
    toDescribeDBClusters() {
        this.to('rds:DescribeDBClusters');
        return this;
    }
    /**
     * Grants permission to return a list of the available DB engines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBEngineVersions.html
     */
    toDescribeDBEngineVersions() {
        this.to('rds:DescribeDBEngineVersions');
        return this;
    }
    /**
     * Grants permission to return a list of automated backups for both current and deleted instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstanceAutomatedBackups.html
     */
    toDescribeDBInstanceAutomatedBackups() {
        this.to('rds:DescribeDBInstanceAutomatedBackups');
        return this;
    }
    /**
     * Grants permission to return information about provisioned RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html
     */
    toDescribeDBInstances() {
        this.to('rds:DescribeDBInstances');
        return this;
    }
    /**
     * Grants permission to return a list of DB log files for the DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBLogFiles.html
     */
    toDescribeDBLogFiles() {
        this.to('rds:DescribeDBLogFiles');
        return this;
    }
    /**
     * Grants permission to return a list of DBParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameterGroups.html
     */
    toDescribeDBParameterGroups() {
        this.to('rds:DescribeDBParameterGroups');
        return this;
    }
    /**
     * Grants permission to return the detailed parameter list for a particular DB parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameters.html
     */
    toDescribeDBParameters() {
        this.to('rds:DescribeDBParameters');
        return this;
    }
    /**
     * Grants permission to view proxies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxies.html
     */
    toDescribeDBProxies() {
        this.to('rds:DescribeDBProxies');
        return this;
    }
    /**
     * Grants permission to view database proxy target group details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargetGroups.html
     */
    toDescribeDBProxyTargetGroups() {
        this.to('rds:DescribeDBProxyTargetGroups');
        return this;
    }
    /**
     * Grants permission to view database proxy target details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargets.html
     */
    toDescribeDBProxyTargets() {
        this.to('rds:DescribeDBProxyTargets');
        return this;
    }
    /**
     * Grants permission to return a list of DBSecurityGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSecurityGroups.html
     */
    toDescribeDBSecurityGroups() {
        this.to('rds:DescribeDBSecurityGroups');
        return this;
    }
    /**
     * Grants permission to return a list of DB snapshot attribute names and values for a manual DB snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshotAttributes.html
     */
    toDescribeDBSnapshotAttributes() {
        this.to('rds:DescribeDBSnapshotAttributes');
        return this;
    }
    /**
     * Grants permission to return information about DB snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshots.html
     */
    toDescribeDBSnapshots() {
        this.to('rds:DescribeDBSnapshots');
        return this;
    }
    /**
     * Grants permission to return a list of DBSubnetGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSubnetGroups.html
     */
    toDescribeDBSubnetGroups() {
        this.to('rds:DescribeDBSubnetGroups');
        return this;
    }
    /**
     * Grants permission to return the default engine and system parameter information for the cluster database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultClusterParameters.html
     */
    toDescribeEngineDefaultClusterParameters() {
        this.to('rds:DescribeEngineDefaultClusterParameters');
        return this;
    }
    /**
     * Grants permission to return the default engine and system parameter information for the specified database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters() {
        this.to('rds:DescribeEngineDefaultParameters');
        return this;
    }
    /**
     * Grants permission to display a list of categories for all event source types, or, if specified, for a specified source type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories() {
        this.to('rds:DescribeEventCategories');
        return this;
    }
    /**
     * Grants permission to list all the subscription descriptions for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions() {
        this.to('rds:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Grants permission to return events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('rds:DescribeEvents');
        return this;
    }
    /**
     * Grants permission to return information about the export tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        this.to('rds:DescribeExportTasks');
        return this;
    }
    /**
     * Grants permission to return information about Aurora global database clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeGlobalClusters.html
     */
    toDescribeGlobalClusters() {
        this.to('rds:DescribeGlobalClusters');
        return this;
    }
    /**
     * Grants permission to describe all available options
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroupOptions.html
     */
    toDescribeOptionGroupOptions() {
        this.to('rds:DescribeOptionGroupOptions');
        return this;
    }
    /**
     * Grants permission to describe the available option groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroups.html
     */
    toDescribeOptionGroups() {
        this.to('rds:DescribeOptionGroups');
        return this;
    }
    /**
     * Grants permission to return a list of orderable DB instance options for the specified engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOrderableDBInstanceOptions.html
     */
    toDescribeOrderableDBInstanceOptions() {
        this.to('rds:DescribeOrderableDBInstanceOptions');
        return this;
    }
    /**
     * Grants permission to return a list of resources (for example, DB instances) that have at least one pending maintenance action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribePendingMaintenanceActions.html
     */
    toDescribePendingMaintenanceActions() {
        this.to('rds:DescribePendingMaintenanceActions');
        return this;
    }
    /**
     * Grants permission to return information about reserved DB instances for this account, or about a specified reserved DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstances.html
     */
    toDescribeReservedDBInstances() {
        this.to('rds:DescribeReservedDBInstances');
        return this;
    }
    /**
     * Grants permission to list available reserved DB instance offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstancesOfferings.html
     */
    toDescribeReservedDBInstancesOfferings() {
        this.to('rds:DescribeReservedDBInstancesOfferings');
        return this;
    }
    /**
     * Grants permission to return a list of the source AWS Regions where the current AWS Region can create a Read Replica or copy a DB snapshot from
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeSourceRegions.html
     */
    toDescribeSourceRegions() {
        this.to('rds:DescribeSourceRegions');
        return this;
    }
    /**
     * Grants permission to list available modifications you can make to your DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeValidDBInstanceModifications.html
     */
    toDescribeValidDBInstanceModifications() {
        this.to('rds:DescribeValidDBInstanceModifications');
        return this;
    }
    /**
     * Grants permission to download all or a portion of the specified log file, up to 1 MB in size
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DownloadDBLogFilePortion.html
     */
    toDownloadDBLogFilePortion() {
        this.to('rds:DownloadDBLogFilePortion');
        return this;
    }
    /**
     * Grants permission to force a failover for a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverDBCluster.html
     */
    toFailoverDBCluster() {
        this.to('rds:FailoverDBCluster');
        return this;
    }
    /**
     * Grants permission to list all tags on an Amazon RDS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('rds:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to modify current cluster capacity for an Amazon Aurora Severless DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCurrentDBClusterCapacity.html
     */
    toModifyCurrentDBClusterCapacity() {
        this.to('rds:ModifyCurrentDBClusterCapacity');
        return this;
    }
    /**
     * Grants permission to modify a setting for an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBCluster.html
     */
    toModifyDBCluster() {
        this.to('rds:ModifyDBCluster');
        return this;
    }
    /**
     * Grants permission to modify the properties of an endpoint in an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterEndpoint.html
     */
    toModifyDBClusterEndpoint() {
        this.to('rds:ModifyDBClusterEndpoint');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterParameterGroup.html
     */
    toModifyDBClusterParameterGroup() {
        this.to('rds:ModifyDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterSnapshotAttribute.html
     */
    toModifyDBClusterSnapshotAttribute() {
        this.to('rds:ModifyDBClusterSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify settings for a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html
     */
    toModifyDBInstance() {
        this.to('rds:ModifyDBInstance');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBParameterGroup.html
     */
    toModifyDBParameterGroup() {
        this.to('rds:ModifyDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to modify database proxy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxy.html
     */
    toModifyDBProxy() {
        this.to('rds:ModifyDBProxy');
        return this;
    }
    /**
     * Grants permission to modify target group for a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyTargetGroup.html
     */
    toModifyDBProxyTargetGroup() {
        this.to('rds:ModifyDBProxyTargetGroup');
        return this;
    }
    /**
     * Grants permission to update a manual DB snapshot, which can be encrypted or not encrypted, with a new engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshot.html
     */
    toModifyDBSnapshot() {
        this.to('rds:ModifyDBSnapshot');
        return this;
    }
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshotAttribute.html
     */
    toModifyDBSnapshotAttribute() {
        this.to('rds:ModifyDBSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify an existing DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSubnetGroup.html
     */
    toModifyDBSubnetGroup() {
        this.to('rds:ModifyDBSubnetGroup');
        return this;
    }
    /**
     * Grants permission to modify an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription() {
        this.to('rds:ModifyEventSubscription');
        return this;
    }
    /**
     * Grants permission to modify a setting for an Amazon Aurora global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyGlobalCluster.html
     */
    toModifyGlobalCluster() {
        this.to('rds:ModifyGlobalCluster');
        return this;
    }
    /**
     * Grants permission to modify an existing option group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyOptionGroup.html
     */
    toModifyOptionGroup() {
        this.to('rds:ModifyOptionGroup');
        return this;
    }
    /**
     * Grants permission to promote a Read Replica DB instance to a standalone DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplica.html
     */
    toPromoteReadReplica() {
        this.to('rds:PromoteReadReplica');
        return this;
    }
    /**
     * Grants permission to promote a Read Replica DB cluster to a standalone DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplicaDBCluster.html
     */
    toPromoteReadReplicaDBCluster() {
        this.to('rds:PromoteReadReplicaDBCluster');
        return this;
    }
    /**
     * Grants permission to purchase a reserved DB instance offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PurchaseReservedDBInstancesOffering.html
     */
    toPurchaseReservedDBInstancesOffering() {
        this.to('rds:PurchaseReservedDBInstancesOffering');
        return this;
    }
    /**
     * Grants permission to restart the database engine service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBInstance.html
     */
    toRebootDBInstance() {
        this.to('rds:RebootDBInstance');
        return this;
    }
    /**
     * Grants permission to add targets to a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RegisterDBProxyTargets.html
     */
    toRegisterDBProxyTargets() {
        this.to('rds:RegisterDBProxyTargets');
        return this;
    }
    /**
     * Grants permission to detach an Aurora secondary cluster from an Aurora global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveFromGlobalCluster.html
     */
    toRemoveFromGlobalCluster() {
        this.to('rds:RemoveFromGlobalCluster');
        return this;
    }
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBCluster.html
     */
    toRemoveRoleFromDBCluster() {
        this.to('rds:RemoveRoleFromDBCluster');
        return this;
    }
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBInstance.html
     */
    toRemoveRoleFromDBInstance() {
        this.to('rds:RemoveRoleFromDBInstance');
        return this;
    }
    /**
     * Grants permission to remove a source identifier from an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveSourceIdentifierFromSubscription.html
     */
    toRemoveSourceIdentifierFromSubscription() {
        this.to('rds:RemoveSourceIdentifierFromSubscription');
        return this;
    }
    /**
     * Grants permission to remove metadata tags from an Amazon RDS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('rds:RemoveTagsFromResource');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBClusterParameterGroup.html
     */
    toResetDBClusterParameterGroup() {
        this.to('rds:ResetDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB parameter group to the engine/system default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBParameterGroup.html
     */
    toResetDBParameterGroup() {
        this.to('rds:ResetDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to create an Amazon Aurora DB cluster from data stored in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromS3.html
     */
    toRestoreDBClusterFromS3() {
        this.to('rds:RestoreDBClusterFromS3');
        return this;
    }
    /**
     * Grants permission to create a new DB cluster from a DB cluster snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromSnapshot.html
     */
    toRestoreDBClusterFromSnapshot() {
        this.to('rds:RestoreDBClusterFromSnapshot');
        return this;
    }
    /**
     * Grants permission to restore a DB cluster to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterToPointInTime.html
     */
    toRestoreDBClusterToPointInTime() {
        this.to('rds:RestoreDBClusterToPointInTime');
        return this;
    }
    /**
     * Grants permission to create a new DB instance from a DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromDBSnapshot.html
     */
    toRestoreDBInstanceFromDBSnapshot() {
        this.to('rds:RestoreDBInstanceFromDBSnapshot');
        return this;
    }
    /**
     * Grants permission to create a new DB instance from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromS3.html
     */
    toRestoreDBInstanceFromS3() {
        this.to('rds:RestoreDBInstanceFromS3');
        return this;
    }
    /**
     * Grants permission to restore a DB instance to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceToPointInTime.html
     */
    toRestoreDBInstanceToPointInTime() {
        this.to('rds:RestoreDBInstanceToPointInTime');
        return this;
    }
    /**
     * Grants permission to revoke ingress from a DBSecurityGroup for previously authorized IP ranges or EC2 or VPC Security Groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RevokeDBSecurityGroupIngress.html
     */
    toRevokeDBSecurityGroupIngress() {
        this.to('rds:RevokeDBSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to start Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartActivityStream.html
     */
    toStartActivityStream() {
        this.to('rds:StartActivityStream');
        return this;
    }
    /**
     * Starts the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBCluster.html
     */
    toStartDBCluster() {
        this.to('rds:StartDBCluster');
        return this;
    }
    /**
     * Grants permission to start the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstance.html
     */
    toStartDBInstance() {
        this.to('rds:StartDBInstance');
        return this;
    }
    /**
     * Grants permission to start a new Export task for a DB snapshot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartExportTask.html
     */
    toStartExportTask() {
        this.to('rds:StartExportTask');
        return this;
    }
    /**
     * Grants permission to stop Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopActivityStream.html
     */
    toStopActivityStream() {
        this.to('rds:StopActivityStream');
        return this;
    }
    /**
     * Grants permission to stop the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBCluster.html
     */
    toStopDBCluster() {
        this.to('rds:StopDBCluster');
        return this;
    }
    /**
     * Grants permission to stop the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstance.html
     */
    toStopDBInstance() {
        this.to('rds:StopDBInstance');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Aurora.Managing.html
     *
     * @param dbClusterInstanceName - Identifier for the dbClusterInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterTag()
     */
    onCluster(dbClusterInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster:${DbClusterInstanceName}';
        arn = arn.replace('${DbClusterInstanceName}', dbClusterInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster-endpoint to the statement
     *
     * @param dbClusterEndpoint - Identifier for the dbClusterEndpoint.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClusterEndpoint(dbClusterEndpoint, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster-endpoint:${DbClusterEndpoint}';
        arn = arn.replace('${DbClusterEndpoint}', dbClusterEndpoint);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster-pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html
     *
     * @param clusterParameterGroupName - Identifier for the clusterParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterPgTag()
     */
    onClusterPg(clusterParameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster-pg:${ClusterParameterGroupName}';
        arn = arn.replace('${ClusterParameterGroupName}', clusterParameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster-snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param clusterSnapshotName - Identifier for the clusterSnapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterSnapshotTag()
     */
    onClusterSnapshot(clusterSnapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster-snapshot:${ClusterSnapshotName}';
        arn = arn.replace('${ClusterSnapshotName}', clusterSnapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type db to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.html
     *
     * @param dbInstanceName - Identifier for the dbInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDatabaseClass()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifMultiAz()
     * - .ifPiops()
     * - .ifStorageEncrypted()
     * - .ifStorageSize()
     * - .ifVpc()
     * - .ifDbTag()
     */
    onDb(dbInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:db:${DbInstanceName}';
        arn = arn.replace('${DbInstanceName}', dbInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type es to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html
     *
     * @param subscriptionName - Identifier for the subscriptionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEs(subscriptionName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:es:${SubscriptionName}';
        arn = arn.replace('${SubscriptionName}', subscriptionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type global-cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html
     *
     * @param globalCluster - Identifier for the globalCluster.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalCluster(globalCluster, account, partition) {
        var arn = 'arn:${Partition}:rds::${Account}:global-cluster:${GlobalCluster}';
        arn = arn.replace('${GlobalCluster}', globalCluster);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type og to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithOptionGroups.html
     *
     * @param optionGroupName - Identifier for the optionGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifOgTag()
     */
    onOg(optionGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:og:${OptionGroupName}';
        arn = arn.replace('${OptionGroupName}', optionGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPgTag()
     */
    onPg(parameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:pg:${ParameterGroupName}';
        arn = arn.replace('${ParameterGroupName}', parameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type proxy to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html
     *
     * @param dbProxyId - Identifier for the dbProxyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProxy(dbProxyId, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:db-proxy:${DbProxyId}';
        arn = arn.replace('${DbProxyId}', dbProxyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ri to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithReservedDBInstances.html
     *
     * @param reservedDbInstanceName - Identifier for the reservedDbInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRiTag()
     */
    onRi(reservedDbInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:ri:${ReservedDbInstanceName}';
        arn = arn.replace('${ReservedDbInstanceName}', reservedDbInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type secgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithSecurityGroups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSecgrpTag()
     */
    onSecgrp(securityGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:secgrp:${SecurityGroupName}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSnapshotTag()
     */
    onSnapshot(snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:snapshot:${SnapshotName}';
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Scenarios.html#USER_VPC.Scenario1
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onSubgrp(subnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:subgrp:${SubnetGroupName}';
        arn = arn.replace('${SubnetGroupName}', subnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type target to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html
     *
     * @param targetId - Identifier for the targetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTarget(targetId, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:target:${TargetId}';
        arn = arn.replace('${TargetId}', targetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type target-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html
     *
     * @param targetGroupId - Identifier for the targetGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTargetGroup(targetGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:target-group:${TargetGroupId}';
        arn = arn.replace('${TargetGroupId}', targetGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the type of DB instance class
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseClass(value, operator) {
        return this.if(`rds:DatabaseClass`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the database engine. For possible values refer to the engine parameter in CreateDBInstance API
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseEngine(value, operator) {
        return this.if(`rds:DatabaseEngine`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the user-defined name of the database on the DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseName(value, operator) {
        return this.if(`rds:DatabaseName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of the endpoint. One of: READER, WRITER, CUSTOM
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBClusterEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointType(value, operator) {
        return this.if(`rds:EndpointType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the value that specifies whether the DB instance runs in multiple Availability Zones. To indicate that the DB instance is using Multi-AZ, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiAz(value) {
        return this.if(`rds:MultiAz`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the value that contains the number of Provisioned IOPS (PIOPS) that the instance supports. To indicate a DB instance that does not have PIOPS enabled, specify 0
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPiops(value, operator) {
        return this.if(`rds:Piops`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the value that specifies whether the DB instance storage should be encrypted. To enforce storage encryption, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifStorageEncrypted(value) {
        return this.if(`rds:StorageEncrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the storage volume size (in GB)
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifStorageSize(value, operator) {
        return this.if(`rds:StorageSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the value that specifies whether the DB instance runs in an Amazon Virtual Private Cloud (Amazon VPC). To indicate that the DB instance runs in an Amazon VPC, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifVpc(value) {
        return this.if(`rds:Vpc`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the tag attached to a DB cluster parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster-pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterPgTag(tagKey, value, operator) {
        return this.if(`rds:cluster-pg-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB cluster snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster-snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterSnapshotTag(tagKey, value, operator) {
        return this.if(`rds:cluster-snapshot-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB cluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterTag(tagKey, value, operator) {
        return this.if(`rds:cluster-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbTag(tagKey, value, operator) {
        return this.if(`rds:db-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to an event subscription
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - es
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`rds:es-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB option group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - og
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOgTag(tagKey, value, operator) {
        return this.if(`rds:og-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPgTag(tagKey, value, operator) {
        return this.if(`rds:pg-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the set of tag keys and values that can be used to tag a resource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDBCluster()
     * - .toCreateDBClusterParameterGroup()
     * - .toCreateDBClusterSnapshot()
     * - .toCreateDBInstance()
     * - .toCreateDBInstanceReadReplica()
     * - .toCreateDBParameterGroup()
     * - .toCreateDBSecurityGroup()
     * - .toCreateDBSnapshot()
     * - .toCreateDBSubnetGroup()
     * - .toCreateEventSubscription()
     * - .toCreateOptionGroup()
     * - .toRemoveTagsFromResource()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceFromS3()
     * - .toRestoreDBInstanceToPointInTime()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`rds:req-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a reserved DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - ri
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRiTag(tagKey, value, operator) {
        return this.if(`rds:ri-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB security group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - secgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecgrpTag(tagKey, value, operator) {
        return this.if(`rds:secgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSnapshotTag(tagKey, value, operator) {
        return this.if(`rds:snapshot-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB subnet group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - subgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`rds:subgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Rds = Rds;
//# sourceMappingURL=data:application/json;base64,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