"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rekognition = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rekognition extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'rekognition';
        this.accessLevelList = {
            "Read": [
                "CompareFaces",
                "DescribeCollection",
                "DescribeProjectVersions",
                "DescribeProjects",
                "DescribeStreamProcessor",
                "DetectCustomLabels",
                "DetectFaces",
                "DetectLabels",
                "DetectModerationLabels",
                "DetectProtectiveEquipment",
                "DetectText",
                "GetCelebrityInfo",
                "GetCelebrityRecognition",
                "GetContentModeration",
                "GetFaceDetection",
                "GetFaceSearch",
                "GetLabelDetection",
                "GetPersonTracking",
                "GetSegmentDetection",
                "GetTextDetection",
                "ListCollections",
                "ListFaces",
                "RecognizeCelebrities",
                "SearchFaces",
                "SearchFacesByImage"
            ],
            "Write": [
                "CreateCollection",
                "CreateProject",
                "CreateProjectVersion",
                "CreateStreamProcessor",
                "DeleteCollection",
                "DeleteFaces",
                "DeleteProject",
                "DeleteProjectVersion",
                "DeleteStreamProcessor",
                "IndexFaces",
                "StartCelebrityRecognition",
                "StartContentModeration",
                "StartFaceDetection",
                "StartFaceSearch",
                "StartLabelDetection",
                "StartPersonTracking",
                "StartProjectVersion",
                "StartSegmentDetection",
                "StartStreamProcessor",
                "StartTextDetection",
                "StopProjectVersion",
                "StopStreamProcessor"
            ],
            "List": [
                "ListStreamProcessors"
            ]
        };
    }
    /**
     * Compares a face in source input image with each face detected in the target input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html
     */
    toCompareFaces() {
        this.to('rekognition:CompareFaces');
        return this;
    }
    /**
     * Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html
     */
    toCreateCollection() {
        this.to('rekognition:CreateCollection');
        return this;
    }
    /**
     * Creates a new Amazon Rekognition Custom Labels project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html
     */
    toCreateProject() {
        this.to('rekognition:CreateProject');
        return this;
    }
    /**
     * Creates a new version of a model and begins training.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html
     */
    toCreateProjectVersion() {
        this.to('rekognition:CreateProjectVersion');
        return this;
    }
    /**
     * Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html
     */
    toCreateStreamProcessor() {
        this.to('rekognition:CreateStreamProcessor');
        return this;
    }
    /**
     * Deletes the specified collection. Note that this operation removes all faces in the collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html
     */
    toDeleteCollection() {
        this.to('rekognition:DeleteCollection');
        return this;
    }
    /**
     * Deletes faces from a collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html
     */
    toDeleteFaces() {
        this.to('rekognition:DeleteFaces');
        return this;
    }
    /**
     * Deletes a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html
     */
    toDeleteProject() {
        this.to('rekognition:DeleteProject');
        return this;
    }
    /**
     * Deletes a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html
     */
    toDeleteProjectVersion() {
        this.to('rekognition:DeleteProjectVersion');
        return this;
    }
    /**
     * Deletes the stream processor identified by Name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html
     */
    toDeleteStreamProcessor() {
        this.to('rekognition:DeleteStreamProcessor');
        return this;
    }
    /**
     * Describes the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html
     */
    toDescribeCollection() {
        this.to('rekognition:DescribeCollection');
        return this;
    }
    /**
     * Lists and describes the model versions in an Amazon Rekognition Custom Labels project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html
     */
    toDescribeProjectVersions() {
        this.to('rekognition:DescribeProjectVersions');
        return this;
    }
    /**
     * Lists and gets information about your Amazon Rekognition Custom Labels projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html
     */
    toDescribeProjects() {
        this.to('rekognition:DescribeProjects');
        return this;
    }
    /**
     * Provides information about a stream processor created by CreateStreamProcessor.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html
     */
    toDescribeStreamProcessor() {
        this.to('rekognition:DescribeStreamProcessor');
        return this;
    }
    /**
     * Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html
     */
    toDetectCustomLabels() {
        this.to('rekognition:DetectCustomLabels');
        return this;
    }
    /**
     * Detects human faces within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html
     */
    toDetectFaces() {
        this.to('rekognition:DetectFaces');
        return this;
    }
    /**
     * Detects instances of real-world labels within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html
     */
    toDetectLabels() {
        this.to('rekognition:DetectLabels');
        return this;
    }
    /**
     * Detects moderation labels within input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html
     */
    toDetectModerationLabels() {
        this.to('rekognition:DetectModerationLabels');
        return this;
    }
    /**
     * Detects Protective Equipment in the input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectProtectiveEquipment.html
     */
    toDetectProtectiveEquipment() {
        this.to('rekognition:DetectProtectiveEquipment');
        return this;
    }
    /**
     * Detects text in the input image and converts it into machine-readable text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html
     */
    toDetectText() {
        this.to('rekognition:DetectText');
        return this;
    }
    /**
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html
     */
    toGetCelebrityInfo() {
        this.to('rekognition:GetCelebrityInfo');
        return this;
    }
    /**
     * Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html
     */
    toGetCelebrityRecognition() {
        this.to('rekognition:GetCelebrityRecognition');
        return this;
    }
    /**
     * Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html
     */
    toGetContentModeration() {
        this.to('rekognition:GetContentModeration');
        return this;
    }
    /**
     * Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html
     */
    toGetFaceDetection() {
        this.to('rekognition:GetFaceDetection');
        return this;
    }
    /**
     * Gets the face search results for Rekognition Video face search started by StartFaceSearch.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html
     */
    toGetFaceSearch() {
        this.to('rekognition:GetFaceSearch');
        return this;
    }
    /**
     * Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html
     */
    toGetLabelDetection() {
        this.to('rekognition:GetLabelDetection');
        return this;
    }
    /**
     * Gets information about people detected within a video.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html
     */
    toGetPersonTracking() {
        this.to('rekognition:GetPersonTracking');
        return this;
    }
    /**
     * Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html
     */
    toGetSegmentDetection() {
        this.to('rekognition:GetSegmentDetection');
        return this;
    }
    /**
     * Gets text detection results for a Rekognition Video analysis started by StartTextDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html
     */
    toGetTextDetection() {
        this.to('rekognition:GetTextDetection');
        return this;
    }
    /**
     * Detects faces in the input image and adds them to the specified collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html
     */
    toIndexFaces() {
        this.to('rekognition:IndexFaces');
        return this;
    }
    /**
     * Returns a list of collection IDs in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html
     */
    toListCollections() {
        this.to('rekognition:ListCollections');
        return this;
    }
    /**
     * Returns metadata for faces in the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html
     */
    toListFaces() {
        this.to('rekognition:ListFaces');
        return this;
    }
    /**
     * Gets a list of stream processors that you have created with CreateStreamProcessor.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html
     */
    toListStreamProcessors() {
        this.to('rekognition:ListStreamProcessors');
        return this;
    }
    /**
     * Returns an array of celebrities recognized in the input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html
     */
    toRecognizeCelebrities() {
        this.to('rekognition:RecognizeCelebrities');
        return this;
    }
    /**
     * For a given input face ID, searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html
     */
    toSearchFaces() {
        this.to('rekognition:SearchFaces');
        return this;
    }
    /**
     * For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html
     */
    toSearchFacesByImage() {
        this.to('rekognition:SearchFacesByImage');
        return this;
    }
    /**
     * Starts asynchronous recognition of celebrities in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html
     */
    toStartCelebrityRecognition() {
        this.to('rekognition:StartCelebrityRecognition');
        return this;
    }
    /**
     * Starts asynchronous detection of explicit or suggestive adult content in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html
     */
    toStartContentModeration() {
        this.to('rekognition:StartContentModeration');
        return this;
    }
    /**
     * Starts asynchronous detection of faces in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html
     */
    toStartFaceDetection() {
        this.to('rekognition:StartFaceDetection');
        return this;
    }
    /**
     * Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html
     */
    toStartFaceSearch() {
        this.to('rekognition:StartFaceSearch');
        return this;
    }
    /**
     * Starts asynchronous detection of labels in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html
     */
    toStartLabelDetection() {
        this.to('rekognition:StartLabelDetection');
        return this;
    }
    /**
     * Starts the asynchronous tracking of persons in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html
     */
    toStartPersonTracking() {
        this.to('rekognition:StartPersonTracking');
        return this;
    }
    /**
     * Starts the deployment of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html
     */
    toStartProjectVersion() {
        this.to('rekognition:StartProjectVersion');
        return this;
    }
    /**
     * Starts asynchronous detection of segments in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html
     */
    toStartSegmentDetection() {
        this.to('rekognition:StartSegmentDetection');
        return this;
    }
    /**
     * Starts processing a stream processor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html
     */
    toStartStreamProcessor() {
        this.to('rekognition:StartStreamProcessor');
        return this;
    }
    /**
     * Starts asynchronous detection of text in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html
     */
    toStartTextDetection() {
        this.to('rekognition:StartTextDetection');
        return this;
    }
    /**
     * Stops a deployed model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html
     */
    toStopProjectVersion() {
        this.to('rekognition:StopProjectVersion');
        return this;
    }
    /**
     * Stops a running stream processor that was created by CreateStreamProcessor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html
     */
    toStopStreamProcessor() {
        this.to('rekognition:StopStreamProcessor');
        return this;
    }
    /**
     * Adds a resource of type collection to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html
     *
     * @param collectionId - Identifier for the collectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCollection(collectionId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}';
        arn = arn.replace('${CollectionId}', collectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streamprocessor to the statement
     *
     * @param streamprocessorId - Identifier for the streamprocessorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStreamprocessor(streamprocessorId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}';
        arn = arn.replace('${StreamprocessorId}', streamprocessorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type projectversion to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param versionName - Identifier for the versionName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProjectversion(projectName, versionName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${VersionName}', versionName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Rekognition = Rekognition;
//# sourceMappingURL=data:application/json;base64,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