"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sns = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sns](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsns.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sns extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sns](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsns.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sns';
        this.accessLevelList = {
            "Permissions management": [
                "AddPermission",
                "RemovePermission"
            ],
            "Read": [
                "CheckIfPhoneNumberIsOptedOut",
                "GetEndpointAttributes",
                "GetPlatformApplicationAttributes",
                "GetSMSAttributes",
                "GetSubscriptionAttributes",
                "GetTopicAttributes",
                "ListPhoneNumbersOptedOut",
                "ListTagsForResource"
            ],
            "Write": [
                "ConfirmSubscription",
                "CreatePlatformApplication",
                "CreatePlatformEndpoint",
                "CreateTopic",
                "DeleteEndpoint",
                "DeletePlatformApplication",
                "DeleteTopic",
                "OptInPhoneNumber",
                "Publish",
                "SetEndpointAttributes",
                "SetPlatformApplicationAttributes",
                "SetSMSAttributes",
                "SetSubscriptionAttributes",
                "SetTopicAttributes",
                "Subscribe",
                "Unsubscribe"
            ],
            "List": [
                "ListEndpointsByPlatformApplication",
                "ListPlatformApplications",
                "ListSubscriptions",
                "ListSubscriptionsByTopic",
                "ListTopics"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html
     */
    toAddPermission() {
        this.to('sns:AddPermission');
        return this;
    }
    /**
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html
     */
    toCheckIfPhoneNumberIsOptedOut() {
        this.to('sns:CheckIfPhoneNumberIsOptedOut');
        return this;
    }
    /**
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html
     */
    toConfirmSubscription() {
        this.to('sns:ConfirmSubscription');
        return this;
    }
    /**
     * Creates a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html
     */
    toCreatePlatformApplication() {
        this.to('sns:CreatePlatformApplication');
        return this;
    }
    /**
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html
     */
    toCreatePlatformEndpoint() {
        this.to('sns:CreatePlatformEndpoint');
        return this;
    }
    /**
     * Creates a topic to which notifications can be published.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html
     */
    toCreateTopic() {
        this.to('sns:CreateTopic');
        return this;
    }
    /**
     * Deletes the endpoint for a device and mobile app from Amazon SNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.to('sns:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes a platform application object for one of the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html
     */
    toDeletePlatformApplication() {
        this.to('sns:DeletePlatformApplication');
        return this;
    }
    /**
     * Deletes a topic and all its subscriptions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html
     */
    toDeleteTopic() {
        this.to('sns:DeleteTopic');
        return this;
    }
    /**
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html
     */
    toGetEndpointAttributes() {
        this.to('sns:GetEndpointAttributes');
        return this;
    }
    /**
     * Retrieves the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html
     */
    toGetPlatformApplicationAttributes() {
        this.to('sns:GetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Returns the settings for sending SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html
     */
    toGetSMSAttributes() {
        this.to('sns:GetSMSAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a subscription.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html
     */
    toGetSubscriptionAttributes() {
        this.to('sns:GetSubscriptionAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html
     */
    toGetTopicAttributes() {
        this.to('sns:GetTopicAttributes');
        return this;
    }
    /**
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html
     */
    toListEndpointsByPlatformApplication() {
        this.to('sns:ListEndpointsByPlatformApplication');
        return this;
    }
    /**
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html
     */
    toListPhoneNumbersOptedOut() {
        this.to('sns:ListPhoneNumbersOptedOut');
        return this;
    }
    /**
     * Lists the platform application objects for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html
     */
    toListPlatformApplications() {
        this.to('sns:ListPlatformApplications');
        return this;
    }
    /**
     * Returns a list of the requester's subscriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html
     */
    toListSubscriptions() {
        this.to('sns:ListSubscriptions');
        return this;
    }
    /**
     * Returns a list of the subscriptions to a specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html
     */
    toListSubscriptionsByTopic() {
        this.to('sns:ListSubscriptionsByTopic');
        return this;
    }
    /**
     * List all tags added to the specified Amazon SNS topic.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('sns:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html
     */
    toListTopics() {
        this.to('sns:ListTopics');
        return this;
    }
    /**
     * Opts in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html
     */
    toOptInPhoneNumber() {
        this.to('sns:OptInPhoneNumber');
        return this;
    }
    /**
     * Sends a message to all of a topic's subscribed endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Publish.html
     */
    toPublish() {
        this.to('sns:Publish');
        return this;
    }
    /**
     * Removes a statement from a topic's access control policy.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('sns:RemovePermission');
        return this;
    }
    /**
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html
     */
    toSetEndpointAttributes() {
        this.to('sns:SetEndpointAttributes');
        return this;
    }
    /**
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html
     */
    toSetPlatformApplicationAttributes() {
        this.to('sns:SetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Set the default settings for sending SMS messages and receiving daily SMS usage reports.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html
     */
    toSetSMSAttributes() {
        this.to('sns:SetSMSAttributes');
        return this;
    }
    /**
     * Allows a subscription owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html
     */
    toSetSubscriptionAttributes() {
        this.to('sns:SetSubscriptionAttributes');
        return this;
    }
    /**
     * Allows a topic owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html
     */
    toSetTopicAttributes() {
        this.to('sns:SetTopicAttributes');
        return this;
    }
    /**
     * Prepares to subscribe an endpoint by sending the endpoint a confirmation message.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpoint()
     * - .ifProtocol()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html
     */
    toSubscribe() {
        this.to('sns:Subscribe');
        return this;
    }
    /**
     * Add tags to the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('sns:TagResource');
        return this;
    }
    /**
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the topic's owner can unsubscribe, and an AWS signature is required.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html
     */
    toUnsubscribe() {
        this.to('sns:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('sns:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:sns:${Region}:${Account}:${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The URL, email address, or ARN from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpoint(value, operator) {
        return this.if(`sns:Endpoint`, value, operator || 'StringLike');
    }
    /**
     * The protocol value from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtocol(value, operator) {
        return this.if(`sns:Protocol`, value, operator || 'StringLike');
    }
}
exports.Sns = Sns;
//# sourceMappingURL=data:application/json;base64,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