"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodestarConnections = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codestar-connections](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestarconnections.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodestarConnections extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codestar-connections](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestarconnections.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codestar-connections';
        this.accessLevelList = {
            "Write": [
                "CreateConnection",
                "CreateHost",
                "DeleteConnection",
                "DeleteHost",
                "UpdateConnectionInstallation"
            ],
            "Read": [
                "GetConnection",
                "GetHost",
                "GetIndividualAccessToken",
                "GetInstallationUrl",
                "PassConnection",
                "RegisterAppCode",
                "StartAppRegistrationHandshake",
                "StartOAuthHandshake",
                "UseConnection"
            ],
            "List": [
                "ListConnections",
                "ListHosts",
                "ListInstallationTargets",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create a Connection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        this.to('codestar-connections:CreateConnection');
        return this;
    }
    /**
     * Grants permission to create a host resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateHost.html
     */
    toCreateHost() {
        this.to('codestar-connections:CreateHost');
        return this;
    }
    /**
     * Grants permission to delete a Connection resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        this.to('codestar-connections:DeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete a host resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteHost.html
     */
    toDeleteHost() {
        this.to('codestar-connections:DeleteHost');
        return this;
    }
    /**
     * Grants permission to get details about a Connection resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetConnection.html
     */
    toGetConnection() {
        this.to('codestar-connections:GetConnection');
        return this;
    }
    /**
     * Grants permission to get details about a host resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetHost.html
     */
    toGetHost() {
        this.to('codestar-connections:GetHost');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * Dependent actions:
     * - codestar-connections:StartOAuthHandshake
     */
    toGetIndividualAccessToken() {
        this.to('codestar-connections:GetIndividualAccessToken');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     */
    toGetInstallationUrl() {
        this.to('codestar-connections:GetInstallationUrl');
        return this;
    }
    /**
     * Grants permission to list Connection resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListConnections.html
     */
    toListConnections() {
        this.to('codestar-connections:ListConnections');
        return this;
    }
    /**
     * Grants permission to list host resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListHosts.html
     */
    toListHosts() {
        this.to('codestar-connections:ListHosts');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: List
     *
     * Dependent actions:
     * - codestar-connections:GetIndividualAccessToken
     * - codestar-connections:StartOAuthHandshake
     */
    toListInstallationTargets() {
        this.to('codestar-connections:ListInstallationTargets');
        return this;
    }
    /**
     * Gets the set of key-value pairs that are used to manage the resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('codestar-connections:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to pass a Connection resource to an AWS service that accepts a Connection ARN as input, such as codepipeline:CreatePipeline
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPassedToService()
     */
    toPassConnection() {
        this.to('codestar-connections:PassConnection');
        return this;
    }
    /**
     * Grants permission to associate a third party server, such as a GitHub Enterprise Server instance, with a Host
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifHostArn()
     */
    toRegisterAppCode() {
        this.to('codestar-connections:RegisterAppCode');
        return this;
    }
    /**
     * Grants permission to associate a third party server, such as a GitHub Enterprise Server instance, with a Host
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifHostArn()
     */
    toStartAppRegistrationHandshake() {
        this.to('codestar-connections:StartAppRegistrationHandshake');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     */
    toStartOAuthHandshake() {
        this.to('codestar-connections:StartOAuthHandshake');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('codestar-connections:TagResource');
        return this;
    }
    /**
     * Removes tags from an AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('codestar-connections:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a Connection resource with an installation of the CodeStar Connections App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstallationId()
     *
     * Dependent actions:
     * - codestar-connections:GetIndividualAccessToken
     * - codestar-connections:GetInstallationUrl
     * - codestar-connections:ListInstallationTargets
     * - codestar-connections:StartOAuthHandshake
     */
    toUpdateConnectionInstallation() {
        this.to('codestar-connections:UpdateConnectionInstallation');
        return this;
    }
    /**
     * Grants permission to use a Connection resource to call provider actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifFullRepositoryId()
     * - .ifProviderAction()
     * - .ifProviderPermissionsRequired()
     */
    toUseConnection() {
        this.to('codestar-connections:UseConnection');
        return this;
    }
    /**
     * Adds a resource of type Connection to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections.html/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnection(connectionId, account, region, partition) {
        var arn = 'arn:${Partition}:codestar-connections:${Region}:${Account}:connection/${ConnectionId}';
        arn = arn.replace('${ConnectionId}', connectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Host to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections.html/API_Host.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHost(hostId, account, region, partition) {
        var arn = 'arn:${Partition}:codestar-connections:${Region}:${Account}:host/${HostId}';
        arn = arn.replace('${HostId}', hostId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the branch name that is passed in the request. Applies only to UseConnection requests for access to a specific repository branch
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBranchName(value, operator) {
        return this.if(`codestar-connections:BranchName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the repository that is passed in the request. Applies only to UseConnection requests for access to a specific repository
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFullRepositoryId(value, operator) {
        return this.if(`codestar-connections:FullRepositoryId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the host resource associated with the connection used in the request
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-hosts
     *
     * Applies to actions:
     * - .toRegisterAppCode()
     * - .toStartAppRegistrationHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHostArn(value, operator) {
        return this.if(`codestar-connections:HostArn`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the third-party ID (such as the Bitbucket App installation ID for CodeStar Connections) that is used to update a Connection. Allows you to restrict which third-party App installations can be used to make a Connection
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-permissions-actions-handshake
     *
     * Applies to actions:
     * - .toUpdateConnectionInstallation()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstallationId(value, operator) {
        return this.if(`codestar-connections:InstallationId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the owner of the third-party repository. Applies only to UseConnection requests for access to repositories owned by a specific user
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwnerId(value, operator) {
        return this.if(`codestar-connections:OwnerId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the service to which the principal is allowed to pass a Connection
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-passconnection
     *
     * Applies to actions:
     * - .toPassConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`codestar-connections:PassedToService`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the provider action in a UseConnection request such as ListRepositories. See documentation for all valid values
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use-provider
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderAction(value, operator) {
        return this.if(`codestar-connections:ProviderAction`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the write permissions of a provider action in a UseConnection request. Valid types include read_only and read_write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderPermissionsRequired(value, operator) {
        return this.if(`codestar-connections:ProviderPermissionsRequired`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider passed in the request
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-permissions-actions
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toGetIndividualAccessToken()
     * - .toGetInstallationUrl()
     * - .toStartOAuthHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderType(value, operator) {
        return this.if(`codestar-connections:ProviderType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider used to filter results
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-permissions-actions
     *
     * Applies to actions:
     * - .toListConnections()
     * - .toListHosts()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderTypeFilter(value, operator) {
        return this.if(`codestar-connections:ProviderTypeFilter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the repository name that is passed in the request. Applies only to UseConnection requests for creating new repositories
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepositoryName(value, operator) {
        return this.if(`codestar-connections:RepositoryName`, value, operator || 'StringLike');
    }
}
exports.CodestarConnections = CodestarConnections;
//# sourceMappingURL=data:application/json;base64,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