"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deepracer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [deepracer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepracer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deepracer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [deepracer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepracer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'deepracer';
        this.accessLevelList = {
            "Write": [
                "CloneReinforcementLearningModel",
                "CreateAccountResources",
                "CreateLeaderboardSubmission",
                "CreateReinforcementLearningModel",
                "DeleteAccountResources",
                "DeleteModel",
                "SetAlias",
                "StartEvaluation",
                "StopEvaluation",
                "StopTrainingReinforcementLearningModel",
                "TestRewardFunction"
            ],
            "Read": [
                "GetAccountResources",
                "GetAlias",
                "GetEvaluation",
                "GetLatestUserSubmission",
                "GetLeaderboard",
                "GetModel",
                "GetRankedUserSubmission",
                "GetTrack",
                "GetTrainingJob"
            ],
            "List": [
                "ListEvaluations",
                "ListLeaderboardSubmissions",
                "ListLeaderboards",
                "ListModels",
                "ListTracks",
                "ListTrainingJobs"
            ]
        };
    }
    /**
     * Grants permission to clone existing DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-clone-trained-model
     */
    toCloneReinforcementLearningModel() {
        this.to('deepracer:CloneReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to create resources needed by DeepRacer on behalf of the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toCreateAccountResources() {
        this.to('deepracer:CreateAccountResources');
        return this;
    }
    /**
     * Grants permission to submit DeepRacer models to be evaluated for leaderboards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toCreateLeaderboardSubmission() {
        this.to('deepracer:CreateLeaderboardSubmission');
        return this;
    }
    /**
     * Grants permission to create reinforcement learning models for DeepRacer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toCreateReinforcementLearningModel() {
        this.to('deepracer:CreateReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to delete resources created by DeepRacer on behalf of the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toDeleteAccountResources() {
        this.to('deepracer:DeleteAccountResources');
        return this;
    }
    /**
     * Grants permission to delete DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toDeleteModel() {
        this.to('deepracer:DeleteModel');
        return this;
    }
    /**
     * Grants permission to retrieve the resources created by DeepRacer on behalf of the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetAccountResources() {
        this.to('deepracer:GetAccountResources');
        return this;
    }
    /**
     * Grants permission to retrieve the user's alias for submitting DeepRacer models to leaderboards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetAlias() {
        this.to('deepracer:GetAlias');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models' evaluation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toGetEvaluation() {
        this.to('deepracer:GetEvaluation');
        return this;
    }
    /**
     * Grants permission to retrieve information about how the latest submitted DeepRacer model for a user performed on a leaderboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetLatestUserSubmission() {
        this.to('deepracer:GetLatestUserSubmission');
        return this;
    }
    /**
     * Grants permission to retrieve information about leaderboards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetLeaderboard() {
        this.to('deepracer:GetLeaderboard');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetModel() {
        this.to('deepracer:GetModel');
        return this;
    }
    /**
     * Grants permission to retrieve information about the performance of a user's DeepRacer model that got placed on a leaderboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetRankedUserSubmission() {
        this.to('deepracer:GetRankedUserSubmission');
        return this;
    }
    /**
     * Grants permission to retrieve information about DeepRacer tracks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toGetTrack() {
        this.to('deepracer:GetTrack');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models' training job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetTrainingJob() {
        this.to('deepracer:GetTrainingJob');
        return this;
    }
    /**
     * Grants permission to list DeepRacer models' evaluation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toListEvaluations() {
        this.to('deepracer:ListEvaluations');
        return this;
    }
    /**
     * Grants permission to list all the submissions of DeepRacer models of a user on a leaderboard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboardSubmissions() {
        this.to('deepracer:ListLeaderboardSubmissions');
        return this;
    }
    /**
     * Grants permission to list all the available leaderboards
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboards() {
        this.to('deepracer:ListLeaderboards');
        return this;
    }
    /**
     * Grants permission to list all existing DeepRacer models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toListModels() {
        this.to('deepracer:ListModels');
        return this;
    }
    /**
     * Grants permission to list all DeepRacer tracks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toListTracks() {
        this.to('deepracer:ListTracks');
        return this;
    }
    /**
     * Grants permission to list DeepRacer models' training jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toListTrainingJobs() {
        this.to('deepracer:ListTrainingJobs');
        return this;
    }
    /**
     * Grants permission to set the user's alias for submitting DeepRacer models to leaderboards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toSetAlias() {
        this.to('deepracer:SetAlias');
        return this;
    }
    /**
     * Grants permission to evaluate DeepRacer models in a simulated environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toStartEvaluation() {
        this.to('deepracer:StartEvaluation');
        return this;
    }
    /**
     * Grants permission to stop DeepRacer model evaluations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toStopEvaluation() {
        this.to('deepracer:StopEvaluation');
        return this;
    }
    /**
     * Grants permission to stop training DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toStopTrainingReinforcementLearningModel() {
        this.to('deepracer:StopTrainingReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to test reward functions for correctness
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-train-models-define-reward-function
     */
    toTestRewardFunction() {
        this.to('deepracer:TestRewardFunction');
        return this;
    }
    /**
     * Adds a resource of type reinforcement_learning_model to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReinforcementLearningModel(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:model/reinforcement_learning/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type training_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrainingJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:training_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvaluationJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:evaluation_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type leaderboard_evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLeaderboardEvaluationJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:leaderboard_evaluation_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type track to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrack(resourceId, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}::track/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type leaderboard to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLeaderboard(resourceId, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}::leaderboard/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deepracer = Deepracer;
//# sourceMappingURL=data:application/json;base64,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