"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotevents = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotevents](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotevents.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotevents extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotevents](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotevents.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotevents';
        this.accessLevelList = {
            "Write": [
                "BatchPutMessage",
                "BatchUpdateDetector",
                "CreateDetectorModel",
                "CreateInput",
                "DeleteDetectorModel",
                "DeleteInput",
                "PutLoggingOptions",
                "UpdateDetectorModel",
                "UpdateInput",
                "UpdateInputRouting"
            ],
            "Read": [
                "DescribeDetector",
                "DescribeDetectorModel",
                "DescribeInput",
                "DescribeLoggingOptions",
                "ListTagsForResource"
            ],
            "List": [
                "ListDetectorModelVersions",
                "ListDetectorModels",
                "ListDetectors",
                "ListInputs"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Sends a set of messages to the AWS IoT Events system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchPutMessage.html
     */
    toBatchPutMessage() {
        this.to('iotevents:BatchPutMessage');
        return this;
    }
    /**
     * Update an detector within the AWS IoT Events system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchUpdateDetector.html
     */
    toBatchUpdateDetector() {
        this.to('iotevents:BatchUpdateDetector');
        return this;
    }
    /**
     * Creates a detector model.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateDetectorModel.html
     */
    toCreateDetectorModel() {
        this.to('iotevents:CreateDetectorModel');
        return this;
    }
    /**
     * Creates an input.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateInput.html
     */
    toCreateInput() {
        this.to('iotevents:CreateInput');
        return this;
    }
    /**
     * Deletes a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteDetectorModel.html
     */
    toDeleteDetectorModel() {
        this.to('iotevents:DeleteDetectorModel');
        return this;
    }
    /**
     * Deletes an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteInput.html
     */
    toDeleteInput() {
        this.to('iotevents:DeleteInput');
        return this;
    }
    /**
     * Returns information about the specified detector (instance).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeDetector.html
     */
    toDescribeDetector() {
        this.to('iotevents:DescribeDetector');
        return this;
    }
    /**
     * Describes a detector model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModel.html
     */
    toDescribeDetectorModel() {
        this.to('iotevents:DescribeDetectorModel');
        return this;
    }
    /**
     * Describes an input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeInput.html
     */
    toDescribeInput() {
        this.to('iotevents:DescribeInput');
        return this;
    }
    /**
     * Retrieves the current settings of the AWS IoT Events logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeLoggingOptions.html
     */
    toDescribeLoggingOptions() {
        this.to('iotevents:DescribeLoggingOptions');
        return this;
    }
    /**
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is returned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModelVersions.html
     */
    toListDetectorModelVersions() {
        this.to('iotevents:ListDetectorModelVersions');
        return this;
    }
    /**
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModels.html
     */
    toListDetectorModels() {
        this.to('iotevents:ListDetectorModels');
        return this;
    }
    /**
     * Lists detectors (the instances of a detector model).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html
     */
    toListDetectors() {
        this.to('iotevents:ListDetectors');
        return this;
    }
    /**
     * Lists the inputs you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputs.html
     */
    toListInputs() {
        this.to('iotevents:ListInputs');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('iotevents:ListTagsForResource');
        return this;
    }
    /**
     * Sets or updates the AWS IoT Events logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        this.to('iotevents:PutLoggingOptions');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        this.to('iotevents:TagResource');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('iotevents:UntagResource');
        return this;
    }
    /**
     * Updates a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateDetectorModel.html
     */
    toUpdateDetectorModel() {
        this.to('iotevents:UpdateDetectorModel');
        return this;
    }
    /**
     * Updates an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInput.html
     */
    toUpdateInput() {
        this.to('iotevents:UpdateInput');
        return this;
    }
    /**
     * Updates input routing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInputRouting.html
     */
    toUpdateInputRouting() {
        this.to('iotevents:UpdateInputRouting');
        return this;
    }
    /**
     * Adds a resource of type detectorModel to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param detectorModelName - Identifier for the detectorModelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorModel(detectorModelName, account, region, partition) {
        var arn = 'arn:${Partition}:iotevents:${Region}:${Account}:detectorModel/${DetectorModelName}';
        arn = arn.replace('${DetectorModelName}', detectorModelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param inputName - Identifier for the inputName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInput(inputName, account, region, partition) {
        var arn = 'arn:${Partition}:iotevents:${Region}:${Account}:input/${InputName}';
        arn = arn.replace('${InputName}', inputName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Iotevents = Iotevents;
//# sourceMappingURL=data:application/json;base64,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