"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LicenseManager = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [license-manager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awslicensemanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class LicenseManager extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [license-manager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awslicensemanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'license-manager';
        this.accessLevelList = {
            "Tagging": [
                "CreateLicenseConfiguration",
                "TagResource",
                "UntagResource"
            ],
            "Write": [
                "DeleteLicenseConfiguration",
                "UpdateLicenseConfiguration",
                "UpdateLicenseSpecificationsForResource"
            ],
            "List": [
                "GetLicenseConfiguration",
                "GetServiceSettings",
                "ListAssociationsForLicenseConfiguration",
                "ListLicenseConfigurations",
                "ListLicenseSpecificationsForResource",
                "ListResourceInventory",
                "ListTagsForResource",
                "ListUsageForLicenseConfiguration"
            ],
            "Permissions management": [
                "UpdateServiceSettings"
            ]
        };
    }
    /**
     * Creates a new license configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_CreateLicenseConfiguration.html
     */
    toCreateLicenseConfiguration() {
        this.to('license-manager:CreateLicenseConfiguration');
        return this;
    }
    /**
     * Permanently deletes a license configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_DeleteLicenseConfiguration.html
     */
    toDeleteLicenseConfiguration() {
        this.to('license-manager:DeleteLicenseConfiguration');
        return this;
    }
    /**
     * Gets a license configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_GetLicenseConfiguration.html
     */
    toGetLicenseConfiguration() {
        this.to('license-manager:GetLicenseConfiguration');
        return this;
    }
    /**
     * Gets service settings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_GetServiceSettings.html
     */
    toGetServiceSettings() {
        this.to('license-manager:GetServiceSettings');
        return this;
    }
    /**
     * Lists associations for a selected license configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListAssociationsForLicenseConfiguration.html
     */
    toListAssociationsForLicenseConfiguration() {
        this.to('license-manager:ListAssociationsForLicenseConfiguration');
        return this;
    }
    /**
     * Lists license configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListLicenseConfigurations.html
     */
    toListLicenseConfigurations() {
        this.to('license-manager:ListLicenseConfigurations');
        return this;
    }
    /**
     * Lists license specifications associated with a selected resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListLicenseSpecificationsForResource.html
     */
    toListLicenseSpecificationsForResource() {
        this.to('license-manager:ListLicenseSpecificationsForResource');
        return this;
    }
    /**
     * Lists resource inventory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListResourceInventory.html
     */
    toListResourceInventory() {
        this.to('license-manager:ListResourceInventory');
        return this;
    }
    /**
     * Lists tags for a selected resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('license-manager:ListTagsForResource');
        return this;
    }
    /**
     * Lists usage records for selected license configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListUsageForLicenseConfiguration.html
     */
    toListUsageForLicenseConfiguration() {
        this.to('license-manager:ListUsageForLicenseConfiguration');
        return this;
    }
    /**
     * Tags a selected resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('license-manager:TagResource');
        return this;
    }
    /**
     * Untags a selected resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('license-manager:UntagResource');
        return this;
    }
    /**
     * Updates an existing license configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UpdateLicenseConfiguration.html
     */
    toUpdateLicenseConfiguration() {
        this.to('license-manager:UpdateLicenseConfiguration');
        return this;
    }
    /**
     * Updates license specifications for a selected resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UpdateLicenseSpecificationsForResource.html
     */
    toUpdateLicenseSpecificationsForResource() {
        this.to('license-manager:UpdateLicenseSpecificationsForResource');
        return this;
    }
    /**
     * Updates service settings
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UpdateServiceSettings.html
     */
    toUpdateServiceSettings() {
        this.to('license-manager:UpdateServiceSettings');
        return this;
    }
    /**
     * Adds a resource of type license-configuration to the statement
     *
     * @param licenseConfigurationId - Identifier for the licenseConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onLicenseConfiguration(licenseConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:license-manager:${Region}:${Account}:license-configuration/${LicenseConfigurationId}';
        arn = arn.replace('${LicenseConfigurationId}', licenseConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on tag-value associated with the resource.
     *
     * Applies to resource types:
     * - license-configuration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`license-manager:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.LicenseManager = LicenseManager;
//# sourceMappingURL=data:application/json;base64,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