"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ram = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ram](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ram extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ram](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ram';
        this.accessLevelList = {
            "Write": [
                "AcceptResourceShareInvitation",
                "AssociateResourceShare",
                "AssociateResourceSharePermission",
                "CreateResourceShare",
                "DeleteResourceShare",
                "DisassociateResourceShare",
                "DisassociateResourceSharePermission",
                "EnableSharingWithAwsOrganization",
                "RejectResourceShareInvitation",
                "TagResource",
                "UntagResource",
                "UpdateResourceShare"
            ],
            "Read": [
                "GetPermission",
                "GetResourcePolicies",
                "GetResourceShareAssociations",
                "GetResourceShareInvitations",
                "GetResourceShares",
                "ListPendingInvitationResources"
            ],
            "List": [
                "ListPermissions",
                "ListPrincipals",
                "ListResourceSharePermissions",
                "ListResources"
            ]
        };
    }
    /**
     * Accept the specified resource share invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html
     */
    toAcceptResourceShareInvitation() {
        this.to('ram:AcceptResourceShareInvitation');
        return this;
    }
    /**
     * Associates resource(s) and/or principal(s) to a resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html
     */
    toAssociateResourceShare() {
        this.to('ram:AssociateResourceShare');
        return this;
    }
    /**
     * Associate a Permission with a Resource Share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html
     */
    toAssociateResourceSharePermission() {
        this.to('ram:AssociateResourceSharePermission');
        return this;
    }
    /**
     * Create resource share with provided resource(s) and/or principal(s)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     * - .ifRequestedAllowsExternalPrincipals()
     * - .ifPrincipal()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html
     */
    toCreateResourceShare() {
        this.to('ram:CreateResourceShare');
        return this;
    }
    /**
     * Delete resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html
     */
    toDeleteResourceShare() {
        this.to('ram:DeleteResourceShare');
        return this;
    }
    /**
     * Disassociates resource(s) and/or principal(s) from a resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html
     */
    toDisassociateResourceShare() {
        this.to('ram:DisassociateResourceShare');
        return this;
    }
    /**
     * Disassociate a Permission from a Resource Share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html
     */
    toDisassociateResourceSharePermission() {
        this.to('ram:DisassociateResourceSharePermission');
        return this;
    }
    /**
     * Grants permission to access customer's organization and create a SLR in the customer's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html
     */
    toEnableSharingWithAwsOrganization() {
        this.to('ram:EnableSharingWithAwsOrganization');
        return this;
    }
    /**
     * Gets the contents of an AWS RAM permission
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html
     */
    toGetPermission() {
        this.to('ram:GetPermission');
        return this;
    }
    /**
     * Gets the policies for the specified resources that you own and have shared
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html
     */
    toGetResourcePolicies() {
        this.to('ram:GetResourcePolicies');
        return this;
    }
    /**
     * Get a set of resource share associations from a provided list or with a specified status of the specified type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html
     */
    toGetResourceShareAssociations() {
        this.to('ram:GetResourceShareAssociations');
        return this;
    }
    /**
     * Get resource share invitations by the specified invitation arn or those for the resource share
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html
     */
    toGetResourceShareInvitations() {
        this.to('ram:GetResourceShareInvitations');
        return this;
    }
    /**
     * Get a set of resource shares from a provided list or with a specified status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html
     */
    toGetResourceShares() {
        this.to('ram:GetResourceShares');
        return this;
    }
    /**
     * Lists the resources in a resource share that is shared with you but that the invitation is still pending for
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html
     */
    toListPendingInvitationResources() {
        this.to('ram:ListPendingInvitationResources');
        return this;
    }
    /**
     * Lists the AWS RAM permissions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html
     */
    toListPermissions() {
        this.to('ram:ListPermissions');
        return this;
    }
    /**
     * Lists the principals that you have shared resources with or that have shared resources with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html
     */
    toListPrincipals() {
        this.to('ram:ListPrincipals');
        return this;
    }
    /**
     * List the Permissions associated with a Resource Share
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html
     */
    toListResourceSharePermissions() {
        this.to('ram:ListResourceSharePermissions');
        return this;
    }
    /**
     * Lists the resources that you added to a resource shares or the resources that are shared with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        this.to('ram:ListResources');
        return this;
    }
    /**
     * Reject the specified resource share invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html
     */
    toRejectResourceShareInvitation() {
        this.to('ram:RejectResourceShareInvitation');
        return this;
    }
    /**
     * Tag the specified resources share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('ram:TagResource');
        return this;
    }
    /**
     * Untag the specified resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('ram:UntagResource');
        return this;
    }
    /**
     * Update attributes of the resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifRequestedAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html
     */
    toUpdateResourceShare() {
        this.to('ram:UpdateResourceShare');
        return this;
    }
    /**
     * Adds a resource of type resource-share to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     */
    onResourceShare(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resource-share-invitation to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceShareInvitation(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type permission to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifPermissionArn()
     */
    onPermission(resourcePath, account, partition) {
        var arn = 'arn:${Partition}:ram::${Account}:permission/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Indicates that the action can only be performed on resource shares that allow or deny sharing with external principals. For example, specify true if the action can only be performed on resource shares that allow sharing with external principals. External principals are AWS accounts that are outside of its AWS organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toAssociateResourceSharePermission()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toDisassociateResourceSharePermission()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAllowsExternalPrincipals(value) {
        return this.if(`ram:AllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on a resource using the specified Permission ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceSharePermission()
     * - .toDisassociateResourceSharePermission()
     * - .toGetPermission()
     *
     * Applies to resource types:
     * - permission
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPermissionArn(value, operator) {
        return this.if(`ram:PermissionArn`, value, operator || 'ArnLike');
    }
    /**
     * Principals with the specified format can be associated to or disassociated from a resource share
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`ram:Principal`, value, operator || 'StringLike');
    }
    /**
     * The request must have the specified value for 'allowExternalPrincipals'. External principals are AWS accounts that are outside of its AWS Organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toCreateResourceShare()
     * - .toUpdateResourceShare()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestedAllowsExternalPrincipals(value) {
        return this.if(`ram:RequestedAllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on the specified resource type
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestedResourceType(value, operator) {
        return this.if(`ram:RequestedResourceType`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on a resource with the specified ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifResourceArn(value, operator) {
        return this.if(`ram:ResourceArn`, value, operator || 'ArnLike');
    }
    /**
     * Indicates that the action can only be performed on a resource share with the specified name.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toAssociateResourceSharePermission()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toDisassociateResourceSharePermission()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceShareName(value, operator) {
        return this.if(`ram:ResourceShareName`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on resource shares owned by a specific account. For example, you can use this condition key to specify which resource share invitations can be accepted or rejected based on the resource share owner’s account ID.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAcceptResourceShareInvitation()
     * - .toRejectResourceShareInvitation()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifShareOwnerAccountId(value, operator) {
        return this.if(`ram:ShareOwnerAccountId`, value, operator || 'StringLike');
    }
}
exports.Ram = Ram;
//# sourceMappingURL=data:application/json;base64,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