"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Robomaker = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [robomaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsrobomaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Robomaker extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [robomaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsrobomaker.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'robomaker';
        this.accessLevelList = {
            "Write": [
                "BatchDeleteWorlds",
                "CancelDeploymentJob",
                "CancelSimulationJob",
                "CancelSimulationJobBatch",
                "CancelWorldExportJob",
                "CancelWorldGenerationJob",
                "CreateDeploymentJob",
                "CreateFleet",
                "CreateRobot",
                "CreateRobotApplication",
                "CreateRobotApplicationVersion",
                "CreateSimulationApplication",
                "CreateSimulationApplicationVersion",
                "CreateSimulationJob",
                "CreateWorldExportJob",
                "CreateWorldGenerationJob",
                "CreateWorldTemplate",
                "DeleteFleet",
                "DeleteRobot",
                "DeleteRobotApplication",
                "DeleteSimulationApplication",
                "DeleteWorldTemplate",
                "DeregisterRobot",
                "RegisterRobot",
                "RestartSimulationJob",
                "StartSimulationJobBatch",
                "SyncDeploymentJob",
                "TagResource",
                "UntagResource",
                "UpdateRobotApplication",
                "UpdateRobotDeployment",
                "UpdateSimulationApplication",
                "UpdateWorldTemplate"
            ],
            "Read": [
                "BatchDescribeSimulationJob",
                "DescribeDeploymentJob",
                "DescribeFleet",
                "DescribeRobot",
                "DescribeRobotApplication",
                "DescribeSimulationApplication",
                "DescribeSimulationJob",
                "DescribeSimulationJobBatch",
                "DescribeWorld",
                "DescribeWorldExportJob",
                "DescribeWorldGenerationJob",
                "DescribeWorldTemplate",
                "GetWorldTemplateBody"
            ],
            "List": [
                "ListDeploymentJobs",
                "ListFleets",
                "ListRobotApplications",
                "ListRobots",
                "ListSimulationApplications",
                "ListSimulationJobBatches",
                "ListSimulationJobs",
                "ListSupportedAvailabilityZones",
                "ListTagsForResource",
                "ListWorldExportJobs",
                "ListWorldGenerationJobs",
                "ListWorldTemplates",
                "ListWorlds"
            ]
        };
    }
    /**
     * Delete one or more worlds in a batch operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDeleteWorlds.html
     */
    toBatchDeleteWorlds() {
        this.to('robomaker:BatchDeleteWorlds');
        return this;
    }
    /**
     * Describe multiple simulation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDescribeSimulationJob.html
     */
    toBatchDescribeSimulationJob() {
        this.to('robomaker:BatchDescribeSimulationJob');
        return this;
    }
    /**
     * Cancel a deployment job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelDeploymentJob.html
     */
    toCancelDeploymentJob() {
        this.to('robomaker:CancelDeploymentJob');
        return this;
    }
    /**
     * Cancel a simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJob.html
     */
    toCancelSimulationJob() {
        this.to('robomaker:CancelSimulationJob');
        return this;
    }
    /**
     * Cancel a simulation job batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJobBatch.html
     */
    toCancelSimulationJobBatch() {
        this.to('robomaker:CancelSimulationJobBatch');
        return this;
    }
    /**
     * Cancel a world export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelWorldExportJob.html
     */
    toCancelWorldExportJob() {
        this.to('robomaker:CancelWorldExportJob');
        return this;
    }
    /**
     * Cancel a world generation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelWorldGenerationJob.html
     */
    toCancelWorldGenerationJob() {
        this.to('robomaker:CancelWorldGenerationJob');
        return this;
    }
    /**
     * Create a deployment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateDeploymentJob.html
     */
    toCreateDeploymentJob() {
        this.to('robomaker:CreateDeploymentJob');
        return this;
    }
    /**
     * Create a deployment fleet that represents a logical group of robots running the same robot application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateFleet.html
     */
    toCreateFleet() {
        this.to('robomaker:CreateFleet');
        return this;
    }
    /**
     * Create a robot that can be registered to a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobot.html
     */
    toCreateRobot() {
        this.to('robomaker:CreateRobot');
        return this;
    }
    /**
     * Create a robot application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplication.html
     */
    toCreateRobotApplication() {
        this.to('robomaker:CreateRobotApplication');
        return this;
    }
    /**
     * Create a snapshot of a robot application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplicationVersion.html
     */
    toCreateRobotApplicationVersion() {
        this.to('robomaker:CreateRobotApplicationVersion');
        return this;
    }
    /**
     * Create a simulation application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplication.html
     */
    toCreateSimulationApplication() {
        this.to('robomaker:CreateSimulationApplication');
        return this;
    }
    /**
     * Create a snapshot of a simulation application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplicationVersion.html
     */
    toCreateSimulationApplicationVersion() {
        this.to('robomaker:CreateSimulationApplicationVersion');
        return this;
    }
    /**
     * Create a simulation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationJob.html
     */
    toCreateSimulationJob() {
        this.to('robomaker:CreateSimulationJob');
        return this;
    }
    /**
     * Create a world export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateWorldExportJob.html
     */
    toCreateWorldExportJob() {
        this.to('robomaker:CreateWorldExportJob');
        return this;
    }
    /**
     * Create a world generation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateWorldGenerationJob.html
     */
    toCreateWorldGenerationJob() {
        this.to('robomaker:CreateWorldGenerationJob');
        return this;
    }
    /**
     * Create a world template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateWorldTemplate.html
     */
    toCreateWorldTemplate() {
        this.to('robomaker:CreateWorldTemplate');
        return this;
    }
    /**
     * Delete a deployment fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteFleet.html
     */
    toDeleteFleet() {
        this.to('robomaker:DeleteFleet');
        return this;
    }
    /**
     * Delete a robot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobot.html
     */
    toDeleteRobot() {
        this.to('robomaker:DeleteRobot');
        return this;
    }
    /**
     * Delete a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobotApplication.html
     */
    toDeleteRobotApplication() {
        this.to('robomaker:DeleteRobotApplication');
        return this;
    }
    /**
     * Delete a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteSimulationApplication.html
     */
    toDeleteSimulationApplication() {
        this.to('robomaker:DeleteSimulationApplication');
        return this;
    }
    /**
     * Delete a world template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteWorldTemplate.html
     */
    toDeleteWorldTemplate() {
        this.to('robomaker:DeleteWorldTemplate');
        return this;
    }
    /**
     * Deregister a robot from a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeregisterRobot.html
     */
    toDeregisterRobot() {
        this.to('robomaker:DeregisterRobot');
        return this;
    }
    /**
     * Describe a deployment job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeDeploymentJob.html
     */
    toDescribeDeploymentJob() {
        this.to('robomaker:DescribeDeploymentJob');
        return this;
    }
    /**
     * Describe a deployment fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeFleet.html
     */
    toDescribeFleet() {
        this.to('robomaker:DescribeFleet');
        return this;
    }
    /**
     * Describe a robot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobot.html
     */
    toDescribeRobot() {
        this.to('robomaker:DescribeRobot');
        return this;
    }
    /**
     * Describe a robot application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobotApplication.html
     */
    toDescribeRobotApplication() {
        this.to('robomaker:DescribeRobotApplication');
        return this;
    }
    /**
     * Describe a simulation application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationApplication.html
     */
    toDescribeSimulationApplication() {
        this.to('robomaker:DescribeSimulationApplication');
        return this;
    }
    /**
     * Describe a simulation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJob.html
     */
    toDescribeSimulationJob() {
        this.to('robomaker:DescribeSimulationJob');
        return this;
    }
    /**
     * Describe a simulation job batch
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJobBatch.html
     */
    toDescribeSimulationJobBatch() {
        this.to('robomaker:DescribeSimulationJobBatch');
        return this;
    }
    /**
     * Describe a world
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorld.html
     */
    toDescribeWorld() {
        this.to('robomaker:DescribeWorld');
        return this;
    }
    /**
     * Describe a world export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorldExportJob.html
     */
    toDescribeWorldExportJob() {
        this.to('robomaker:DescribeWorldExportJob');
        return this;
    }
    /**
     * Describe a world generation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorldGenerationJob.html
     */
    toDescribeWorldGenerationJob() {
        this.to('robomaker:DescribeWorldGenerationJob');
        return this;
    }
    /**
     * Describe a world template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorldTemplate.html
     */
    toDescribeWorldTemplate() {
        this.to('robomaker:DescribeWorldTemplate');
        return this;
    }
    /**
     * Get the body of a world template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_GetWorldTemplateBody.html
     */
    toGetWorldTemplateBody() {
        this.to('robomaker:GetWorldTemplateBody');
        return this;
    }
    /**
     * List deployment jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListDeploymentJobs.html
     */
    toListDeploymentJobs() {
        this.to('robomaker:ListDeploymentJobs');
        return this;
    }
    /**
     * List fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListFleets.html
     */
    toListFleets() {
        this.to('robomaker:ListFleets');
        return this;
    }
    /**
     * List robot applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobotApplications.html
     */
    toListRobotApplications() {
        this.to('robomaker:ListRobotApplications');
        return this;
    }
    /**
     * List robots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobots.html
     */
    toListRobots() {
        this.to('robomaker:ListRobots');
        return this;
    }
    /**
     * List simulation applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationApplications.html
     */
    toListSimulationApplications() {
        this.to('robomaker:ListSimulationApplications');
        return this;
    }
    /**
     * List simulation job batches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobBatches.html
     */
    toListSimulationJobBatches() {
        this.to('robomaker:ListSimulationJobBatches');
        return this;
    }
    /**
     * List simulation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobs.html
     */
    toListSimulationJobs() {
        this.to('robomaker:ListSimulationJobs');
        return this;
    }
    /**
     * Lists supported availability zones
     *
     * Access Level: List
     */
    toListSupportedAvailabilityZones() {
        this.to('robomaker:ListSupportedAvailabilityZones');
        return this;
    }
    /**
     * List tags for a RoboMaker resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('robomaker:ListTagsForResource');
        return this;
    }
    /**
     * List world export jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorldExportJobs.html
     */
    toListWorldExportJobs() {
        this.to('robomaker:ListWorldExportJobs');
        return this;
    }
    /**
     * List world generation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorldGenerationJobs.html
     */
    toListWorldGenerationJobs() {
        this.to('robomaker:ListWorldGenerationJobs');
        return this;
    }
    /**
     * List world templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorldTemplates.html
     */
    toListWorldTemplates() {
        this.to('robomaker:ListWorldTemplates');
        return this;
    }
    /**
     * List worlds
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorlds.html
     */
    toListWorlds() {
        this.to('robomaker:ListWorlds');
        return this;
    }
    /**
     * Register a robot to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RegisterRobot.html
     */
    toRegisterRobot() {
        this.to('robomaker:RegisterRobot');
        return this;
    }
    /**
     * Restart a running simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RestartSimulationJob.html
     */
    toRestartSimulationJob() {
        this.to('robomaker:RestartSimulationJob');
        return this;
    }
    /**
     * Create a simulation job batch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_StartSimulationJobBatch.html
     */
    toStartSimulationJobBatch() {
        this.to('robomaker:StartSimulationJobBatch');
        return this;
    }
    /**
     * Ensures the most recently deployed robot application is deployed to all robots in the fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_SyncDeploymentJob.html
     */
    toSyncDeploymentJob() {
        this.to('robomaker:SyncDeploymentJob');
        return this;
    }
    /**
     * Add tags to a RoboMaker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('robomaker:TagResource');
        return this;
    }
    /**
     * Remove tags from a RoboMaker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('robomaker:UntagResource');
        return this;
    }
    /**
     * Update a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateRobotApplication.html
     */
    toUpdateRobotApplication() {
        this.to('robomaker:UpdateRobotApplication');
        return this;
    }
    /**
     * Report the deployment status for an individual robot
     *
     * Access Level: Write
     */
    toUpdateRobotDeployment() {
        this.to('robomaker:UpdateRobotDeployment');
        return this;
    }
    /**
     * Update a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateSimulationApplication.html
     */
    toUpdateSimulationApplication() {
        this.to('robomaker:UpdateSimulationApplication');
        return this;
    }
    /**
     * Update a world template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateWorldTemplate.html
     */
    toUpdateWorldTemplate() {
        this.to('robomaker:UpdateWorldTemplate');
        return this;
    }
    /**
     * Adds a resource of type robotApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-robot-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRobotApplication(applicationName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:robot-application/${ApplicationName}/${CreatedOnEpoch}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationApplication(applicationName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-application/${ApplicationName}/${CreatedOnEpoch}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation.html
     *
     * @param simulationJobId - Identifier for the simulationJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationJob(simulationJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-job/${SimulationJobId}';
        arn = arn.replace('${SimulationJobId}', simulationJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationJobBatch to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation-job-batch.html
     *
     * @param simulationJobBatchId - Identifier for the simulationJobBatchId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationJobBatch(simulationJobBatchId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-job-batch/${SimulationJobBatchId}';
        arn = arn.replace('${SimulationJobBatchId}', simulationJobBatchId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/deployment.html
     *
     * @param deploymentJobId - Identifier for the deploymentJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentJob(deploymentJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:deployment-job/${DeploymentJobId}';
        arn = arn.replace('${DeploymentJobId}', deploymentJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type robot to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/fleets.html
     *
     * @param robotName - Identifier for the robotName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRobot(robotName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:robot/${RobotName}/${CreatedOnEpoch}';
        arn = arn.replace('${RobotName}', robotName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentFleet to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param fleetName - Identifier for the fleetName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentFleet(fleetName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:deployment-fleet/${FleetName}/${CreatedOnEpoch}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type worldGenerationJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-generation-jobs.html
     *
     * @param worldGenerationJobId - Identifier for the worldGenerationJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorldGenerationJob(worldGenerationJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:world-generation-job/${WorldGenerationJobId}';
        arn = arn.replace('${WorldGenerationJobId}', worldGenerationJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type worldExportJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-export-jobs.html
     *
     * @param worldExportJobId - Identifier for the worldExportJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorldExportJob(worldExportJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:world-export-job/${WorldExportJobId}';
        arn = arn.replace('${WorldExportJobId}', worldExportJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type worldTemplate to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-simworld-templates.html
     *
     * @param worldTemplateJobId - Identifier for the worldTemplateJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorldTemplate(worldTemplateJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:world-template/${WorldTemplateJobId}';
        arn = arn.replace('${WorldTemplateJobId}', worldTemplateJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type world to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-generated-worlds.html
     *
     * @param worldId - Identifier for the worldId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorld(worldId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:world/${WorldId}';
        arn = arn.replace('${WorldId}', worldId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Robomaker = Robomaker;
//# sourceMappingURL=data:application/json;base64,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