"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsoDirectory = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sso-directory](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsssodirectory.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SsoDirectory extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sso-directory](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsssodirectory.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sso-directory';
        this.accessLevelList = {
            "Write": [
                "AddMemberToGroup",
                "CompleteVirtualMfaDeviceRegistration",
                "CreateAlias",
                "CreateBearerToken",
                "CreateExternalIdPConfigurationForDirectory",
                "CreateGroup",
                "CreateProvisioningTenant",
                "CreateUser",
                "DeleteBearerToken",
                "DeleteExternalIdPConfigurationForDirectory",
                "DeleteGroup",
                "DeleteMfaDeviceForUser",
                "DeleteProvisioningTenant",
                "DeleteUser",
                "DisableExternalIdPConfigurationForDirectory",
                "DisableUser",
                "EnableExternalIdPConfigurationForDirectory",
                "EnableUser",
                "RemoveMemberFromGroup",
                "StartVirtualMfaDeviceRegistration",
                "UpdateExternalIdPConfigurationForDirectory",
                "UpdateGroup",
                "UpdatePassword",
                "UpdateUser",
                "VerifyEmail"
            ],
            "Read": [
                "DescribeDirectory",
                "GetAWSSPConfigurationForDirectory",
                "SearchGroups",
                "SearchUsers"
            ],
            "List": [
                "DescribeGroups",
                "DescribeUsers",
                "ListBearerTokens",
                "ListExternalIdPConfigurationsForDirectory",
                "ListGroupsForUser",
                "ListMembersInGroup",
                "ListMfaDevicesForUser",
                "ListProvisioningTenants"
            ]
        };
    }
    /**
     * Adds member to the group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAddMemberToGroup() {
        this.to('sso-directory:AddMemberToGroup');
        return this;
    }
    /**
     * Completes the creation process of a virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCompleteVirtualMfaDeviceRegistration() {
        this.to('sso-directory:CompleteVirtualMfaDeviceRegistration');
        return this;
    }
    /**
     * Creates an alias for the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateAlias() {
        this.to('sso-directory:CreateAlias');
        return this;
    }
    /**
     * Creates a bearer token for a given provisioning tenant.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateBearerToken() {
        this.to('sso-directory:CreateBearerToken');
        return this;
    }
    /**
     * Create an External Identity Provider configuration for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:CreateExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Creates a group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateGroup() {
        this.to('sso-directory:CreateGroup');
        return this;
    }
    /**
     * Creates a provisioning tenant for a given directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProvisioningTenant() {
        this.to('sso-directory:CreateProvisioningTenant');
        return this;
    }
    /**
     * Creates a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateUser() {
        this.to('sso-directory:CreateUser');
        return this;
    }
    /**
     * Deletes the bearer token.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteBearerToken() {
        this.to('sso-directory:DeleteBearerToken');
        return this;
    }
    /**
     * Delete an External Identity Provider configuration associated with the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:DeleteExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Deletes a group from the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteGroup() {
        this.to('sso-directory:DeleteGroup');
        return this;
    }
    /**
     * Deletes a MFA device by device name for a given user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteMfaDeviceForUser() {
        this.to('sso-directory:DeleteMfaDeviceForUser');
        return this;
    }
    /**
     * Deletes the provisioning tenant.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProvisioningTenant() {
        this.to('sso-directory:DeleteProvisioningTenant');
        return this;
    }
    /**
     * Deletes a user from the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteUser() {
        this.to('sso-directory:DeleteUser');
        return this;
    }
    /**
     * Retrieve information about the directory that AWS SSO provides by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeDirectory() {
        this.to('sso-directory:DescribeDirectory');
        return this;
    }
    /**
     * Retrieves information about group from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeGroups() {
        this.to('sso-directory:DescribeGroups');
        return this;
    }
    /**
     * Retrieves information about user from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeUsers() {
        this.to('sso-directory:DescribeUsers');
        return this;
    }
    /**
     * Disable authentication of end users with an External Identity Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisableExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:DisableExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Deactivates user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisableUser() {
        this.to('sso-directory:DisableUser');
        return this;
    }
    /**
     * Enable authentication of end users with an External Identity Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toEnableExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:EnableExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Activates user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toEnableUser() {
        this.to('sso-directory:EnableUser');
        return this;
    }
    /**
     * Retrieve the AWS SSO Service Provider configurations for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetAWSSPConfigurationForDirectory() {
        this.to('sso-directory:GetAWSSPConfigurationForDirectory');
        return this;
    }
    /**
     * Lists bearer tokens for a given provisioning tenant.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListBearerTokens() {
        this.to('sso-directory:ListBearerTokens');
        return this;
    }
    /**
     * List all the External Identity Provider configurations created for the directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListExternalIdPConfigurationsForDirectory() {
        this.to('sso-directory:ListExternalIdPConfigurationsForDirectory');
        return this;
    }
    /**
     * Lists groups for a user from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListGroupsForUser() {
        this.to('sso-directory:ListGroupsForUser');
        return this;
    }
    /**
     * Retrieves all members that are part of the group in the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListMembersInGroup() {
        this.to('sso-directory:ListMembersInGroup');
        return this;
    }
    /**
     * Lists all active MFA devices and their MFA device metadata for a user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListMfaDevicesForUser() {
        this.to('sso-directory:ListMfaDevicesForUser');
        return this;
    }
    /**
     * Lists provisioning tenants for a given directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProvisioningTenants() {
        this.to('sso-directory:ListProvisioningTenants');
        return this;
    }
    /**
     * Removes member that are part of the group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toRemoveMemberFromGroup() {
        this.to('sso-directory:RemoveMemberFromGroup');
        return this;
    }
    /**
     * Search for groups within the associated directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchGroups() {
        this.to('sso-directory:SearchGroups');
        return this;
    }
    /**
     * Search for users within the associated directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchUsers() {
        this.to('sso-directory:SearchUsers');
        return this;
    }
    /**
     * Begins the creation process of virtual mfa device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartVirtualMfaDeviceRegistration() {
        this.to('sso-directory:StartVirtualMfaDeviceRegistration');
        return this;
    }
    /**
     * Update an External Identity Provider configuration associated with the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:UpdateExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Updates information about group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateGroup() {
        this.to('sso-directory:UpdateGroup');
        return this;
    }
    /**
     * Updates password by sending password reset link via email or generating one time password for a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdatePassword() {
        this.to('sso-directory:UpdatePassword');
        return this;
    }
    /**
     * Updates user information in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateUser() {
        this.to('sso-directory:UpdateUser');
        return this;
    }
    /**
     * Verify email address of an User
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toVerifyEmail() {
        this.to('sso-directory:VerifyEmail');
        return this;
    }
}
exports.SsoDirectory = SsoDirectory;
//# sourceMappingURL=data:application/json;base64,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