"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iam = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iam](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_identityandaccessmanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iam extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iam](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_identityandaccessmanagement.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iam';
        this.accessLevelList = {
            "Write": [
                "AddClientIDToOpenIDConnectProvider",
                "AddRoleToInstanceProfile",
                "AddUserToGroup",
                "ChangePassword",
                "CreateAccessKey",
                "CreateAccountAlias",
                "CreateGroup",
                "CreateInstanceProfile",
                "CreateLoginProfile",
                "CreateOpenIDConnectProvider",
                "CreateRole",
                "CreateSAMLProvider",
                "CreateServiceLinkedRole",
                "CreateServiceSpecificCredential",
                "CreateUser",
                "CreateVirtualMFADevice",
                "DeactivateMFADevice",
                "DeleteAccessKey",
                "DeleteAccountAlias",
                "DeleteGroup",
                "DeleteInstanceProfile",
                "DeleteLoginProfile",
                "DeleteOpenIDConnectProvider",
                "DeleteRole",
                "DeleteSAMLProvider",
                "DeleteSSHPublicKey",
                "DeleteServerCertificate",
                "DeleteServiceLinkedRole",
                "DeleteServiceSpecificCredential",
                "DeleteSigningCertificate",
                "DeleteUser",
                "DeleteVirtualMFADevice",
                "EnableMFADevice",
                "PassRole",
                "RemoveClientIDFromOpenIDConnectProvider",
                "RemoveRoleFromInstanceProfile",
                "RemoveUserFromGroup",
                "ResetServiceSpecificCredential",
                "ResyncMFADevice",
                "SetSecurityTokenServicePreferences",
                "UpdateAccessKey",
                "UpdateAccountPasswordPolicy",
                "UpdateGroup",
                "UpdateLoginProfile",
                "UpdateOpenIDConnectProviderThumbprint",
                "UpdateRole",
                "UpdateRoleDescription",
                "UpdateSAMLProvider",
                "UpdateSSHPublicKey",
                "UpdateServerCertificate",
                "UpdateServiceSpecificCredential",
                "UpdateSigningCertificate",
                "UpdateUser",
                "UploadSSHPublicKey",
                "UploadServerCertificate",
                "UploadSigningCertificate"
            ],
            "Permissions management": [
                "AttachGroupPolicy",
                "AttachRolePolicy",
                "AttachUserPolicy",
                "CreatePolicy",
                "CreatePolicyVersion",
                "DeleteAccountPasswordPolicy",
                "DeleteGroupPolicy",
                "DeletePolicy",
                "DeletePolicyVersion",
                "DeleteRolePermissionsBoundary",
                "DeleteRolePolicy",
                "DeleteUserPermissionsBoundary",
                "DeleteUserPolicy",
                "DetachGroupPolicy",
                "DetachRolePolicy",
                "DetachUserPolicy",
                "PutGroupPolicy",
                "PutRolePermissionsBoundary",
                "PutRolePolicy",
                "PutUserPermissionsBoundary",
                "PutUserPolicy",
                "SetDefaultPolicyVersion",
                "UpdateAssumeRolePolicy"
            ],
            "Read": [
                "GenerateCredentialReport",
                "GenerateOrganizationsAccessReport",
                "GenerateServiceLastAccessedDetails",
                "GetAccessKeyLastUsed",
                "GetAccountAuthorizationDetails",
                "GetAccountPasswordPolicy",
                "GetContextKeysForCustomPolicy",
                "GetContextKeysForPrincipalPolicy",
                "GetCredentialReport",
                "GetGroup",
                "GetGroupPolicy",
                "GetInstanceProfile",
                "GetOpenIDConnectProvider",
                "GetOrganizationsAccessReport",
                "GetPolicy",
                "GetPolicyVersion",
                "GetRole",
                "GetRolePolicy",
                "GetSAMLProvider",
                "GetSSHPublicKey",
                "GetServerCertificate",
                "GetServiceLastAccessedDetails",
                "GetServiceLastAccessedDetailsWithEntities",
                "GetServiceLinkedRoleDeletionStatus",
                "GetUser",
                "GetUserPolicy",
                "SimulateCustomPolicy",
                "SimulatePrincipalPolicy"
            ],
            "List": [
                "GetAccountSummary",
                "GetLoginProfile",
                "ListAccessKeys",
                "ListAccountAliases",
                "ListAttachedGroupPolicies",
                "ListAttachedRolePolicies",
                "ListAttachedUserPolicies",
                "ListEntitiesForPolicy",
                "ListGroupPolicies",
                "ListGroups",
                "ListGroupsForUser",
                "ListInstanceProfiles",
                "ListInstanceProfilesForRole",
                "ListMFADevices",
                "ListOpenIDConnectProviders",
                "ListPolicies",
                "ListPoliciesGrantingServiceAccess",
                "ListPolicyVersions",
                "ListRolePolicies",
                "ListRoleTags",
                "ListRoles",
                "ListSAMLProviders",
                "ListSSHPublicKeys",
                "ListServerCertificates",
                "ListServiceSpecificCredentials",
                "ListSigningCertificates",
                "ListUserPolicies",
                "ListUserTags",
                "ListUsers",
                "ListVirtualMFADevices"
            ],
            "Tagging": [
                "TagRole",
                "TagUser",
                "UntagRole",
                "UntagUser"
            ]
        };
    }
    /**
     * Grants permission to add a new client ID (audience) to the list of registered IDs for the specified IAM OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddClientIDToOpenIDConnectProvider.html
     */
    toAddClientIDToOpenIDConnectProvider() {
        this.to('iam:AddClientIDToOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to add an IAM role to the specified instance profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddRoleToInstanceProfile.html
     */
    toAddRoleToInstanceProfile() {
        this.to('iam:AddRoleToInstanceProfile');
        return this;
    }
    /**
     * Grants permission to add an IAM user to the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddUserToGroup.html
     */
    toAddUserToGroup() {
        this.to('iam:AddUserToGroup');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM group
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachGroupPolicy.html
     */
    toAttachGroupPolicy() {
        this.to('iam:AttachGroupPolicy');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachRolePolicy.html
     */
    toAttachRolePolicy() {
        this.to('iam:AttachRolePolicy');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachUserPolicy.html
     */
    toAttachUserPolicy() {
        this.to('iam:AttachUserPolicy');
        return this;
    }
    /**
     * Grants permission for an IAM user to to change their own password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ChangePassword.html
     */
    toChangePassword() {
        this.to('iam:ChangePassword');
        return this;
    }
    /**
     * Grants permission to create access key and secret access key for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccessKey.html
     */
    toCreateAccessKey() {
        this.to('iam:CreateAccessKey');
        return this;
    }
    /**
     * Grants permission to create an alias for your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccountAlias.html
     */
    toCreateAccountAlias() {
        this.to('iam:CreateAccountAlias');
        return this;
    }
    /**
     * Grants permission to create a new group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('iam:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a new instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateInstanceProfile.html
     */
    toCreateInstanceProfile() {
        this.to('iam:CreateInstanceProfile');
        return this;
    }
    /**
     * Grants permission to create a password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateLoginProfile.html
     */
    toCreateLoginProfile() {
        this.to('iam:CreateLoginProfile');
        return this;
    }
    /**
     * Grants permission to create an IAM resource that describes an identity provider (IdP) that supports OpenID Connect (OIDC)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateOpenIDConnectProvider.html
     */
    toCreateOpenIDConnectProvider() {
        this.to('iam:CreateOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to create a new managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        this.to('iam:CreatePolicy');
        return this;
    }
    /**
     * Grants permission to create a new version of the specified managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicyVersion.html
     */
    toCreatePolicyVersion() {
        this.to('iam:CreatePolicyVersion');
        return this;
    }
    /**
     * Grants permission to create a new role
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html
     */
    toCreateRole() {
        this.to('iam:CreateRole');
        return this;
    }
    /**
     * Grants permission to create an IAM resource that describes an identity provider (IdP) that supports SAML 2.0
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateSAMLProvider.html
     */
    toCreateSAMLProvider() {
        this.to('iam:CreateSAMLProvider');
        return this;
    }
    /**
     * Grants permission to create an IAM role that allows an AWS service to perform actions on your behalf
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAWSServiceName()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceLinkedRole.html
     */
    toCreateServiceLinkedRole() {
        this.to('iam:CreateServiceLinkedRole');
        return this;
    }
    /**
     * Grants permission to create a new service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceSpecificCredential.html
     */
    toCreateServiceSpecificCredential() {
        this.to('iam:CreateServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to create a new IAM user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('iam:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a new virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateVirtualMFADevice.html
     */
    toCreateVirtualMFADevice() {
        this.to('iam:CreateVirtualMFADevice');
        return this;
    }
    /**
     * Grants permission to deactivate the specified MFA device and remove its association with the IAM user for which it was originally enabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeactivateMFADevice.html
     */
    toDeactivateMFADevice() {
        this.to('iam:DeactivateMFADevice');
        return this;
    }
    /**
     * Grants permission to delete the access key pair that is associated with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccessKey.html
     */
    toDeleteAccessKey() {
        this.to('iam:DeleteAccessKey');
        return this;
    }
    /**
     * Grants permission to delete the specified AWS account alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountAlias.html
     */
    toDeleteAccountAlias() {
        this.to('iam:DeleteAccountAlias');
        return this;
    }
    /**
     * Grants permission to delete the password policy for the AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountPasswordPolicy.html
     */
    toDeleteAccountPasswordPolicy() {
        this.to('iam:DeleteAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('iam:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from its group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroupPolicy.html
     */
    toDeleteGroupPolicy() {
        this.to('iam:DeleteGroupPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteInstanceProfile.html
     */
    toDeleteInstanceProfile() {
        this.to('iam:DeleteInstanceProfile');
        return this;
    }
    /**
     * Grants permission to delete the password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteLoginProfile.html
     */
    toDeleteLoginProfile() {
        this.to('iam:DeleteLoginProfile');
        return this;
    }
    /**
     * Grants permission to delete an OpenID Connect identity provider (IdP) resource object in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteOpenIDConnectProvider.html
     */
    toDeleteOpenIDConnectProvider() {
        this.to('iam:DeleteOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to delete the specified managed policy and remove it from any IAM entities (users, groups, or roles) to which it is attached
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.to('iam:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to delete a version from the specified managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicyVersion.html
     */
    toDeletePolicyVersion() {
        this.to('iam:DeletePolicyVersion');
        return this;
    }
    /**
     * Grants permission to delete the specified role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRole.html
     */
    toDeleteRole() {
        this.to('iam:DeleteRole');
        return this;
    }
    /**
     * Grants permission to remove the permissions boundary from a role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePermissionsBoundary.html
     */
    toDeleteRolePermissionsBoundary() {
        this.to('iam:DeleteRolePermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from the specified role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePolicy.html
     */
    toDeleteRolePolicy() {
        this.to('iam:DeleteRolePolicy');
        return this;
    }
    /**
     * Grants permission to delete a SAML provider resource in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSAMLProvider.html
     */
    toDeleteSAMLProvider() {
        this.to('iam:DeleteSAMLProvider');
        return this;
    }
    /**
     * Grants permission to delete the specified SSH public key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSSHPublicKey.html
     */
    toDeleteSSHPublicKey() {
        this.to('iam:DeleteSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to delete the specified server certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServerCertificate.html
     */
    toDeleteServerCertificate() {
        this.to('iam:DeleteServerCertificate');
        return this;
    }
    /**
     * Grants permission to delete an IAM role that is linked to a specific AWS service, if the service is no longer using it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceLinkedRole.html
     */
    toDeleteServiceLinkedRole() {
        this.to('iam:DeleteServiceLinkedRole');
        return this;
    }
    /**
     * Grants permission to delete the specified service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceSpecificCredential.html
     */
    toDeleteServiceSpecificCredential() {
        this.to('iam:DeleteServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to delete a signing certificate that is associated with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSigningCertificate.html
     */
    toDeleteSigningCertificate() {
        this.to('iam:DeleteSigningCertificate');
        return this;
    }
    /**
     * Grants permission to delete the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('iam:DeleteUser');
        return this;
    }
    /**
     * Grants permission to remove the permissions boundary from the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPermissionsBoundary.html
     */
    toDeleteUserPermissionsBoundary() {
        this.to('iam:DeleteUserPermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from an IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPolicy.html
     */
    toDeleteUserPolicy() {
        this.to('iam:DeleteUserPolicy');
        return this;
    }
    /**
     * Grants permission to delete a virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteVirtualMFADevice.html
     */
    toDeleteVirtualMFADevice() {
        this.to('iam:DeleteVirtualMFADevice');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified IAM group
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachGroupPolicy.html
     */
    toDetachGroupPolicy() {
        this.to('iam:DetachGroupPolicy');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachRolePolicy.html
     */
    toDetachRolePolicy() {
        this.to('iam:DetachRolePolicy');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachUserPolicy.html
     */
    toDetachUserPolicy() {
        this.to('iam:DetachUserPolicy');
        return this;
    }
    /**
     * Grants permission to enable an MFA device and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_EnableMFADevice.html
     */
    toEnableMFADevice() {
        this.to('iam:EnableMFADevice');
        return this;
    }
    /**
     * Grants permission to generate a credential report for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateCredentialReport.html
     */
    toGenerateCredentialReport() {
        this.to('iam:GenerateCredentialReport');
        return this;
    }
    /**
     * Grants permission to generate an access report for an AWS Organizations entity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifOrganizationsPolicyId()
     *
     * Dependent actions:
     * - organizations:DescribePolicy
     * - organizations:ListChildren
     * - organizations:ListParents
     * - organizations:ListPoliciesForTarget
     * - organizations:ListRoots
     * - organizations:ListTargetsForPolicy
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateOrganizationsAccessReport.html
     */
    toGenerateOrganizationsAccessReport() {
        this.to('iam:GenerateOrganizationsAccessReport');
        return this;
    }
    /**
     * Grants permission to generate a service last accessed data report for an IAM resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateServiceLastAccessedDetails.html
     */
    toGenerateServiceLastAccessedDetails() {
        this.to('iam:GenerateServiceLastAccessedDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about when the specified access key was last used
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccessKeyLastUsed.html
     */
    toGetAccessKeyLastUsed() {
        this.to('iam:GetAccessKeyLastUsed');
        return this;
    }
    /**
     * Grants permission to retrieve information about all IAM users, groups, roles, and policies in your AWS account, including their relationships to one another
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountAuthorizationDetails.html
     */
    toGetAccountAuthorizationDetails() {
        this.to('iam:GetAccountAuthorizationDetails');
        return this;
    }
    /**
     * Grants permission to retrieve the password policy for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountPasswordPolicy.html
     */
    toGetAccountPasswordPolicy() {
        this.to('iam:GetAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about IAM entity usage and IAM quotas in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountSummary.html
     */
    toGetAccountSummary() {
        this.to('iam:GetAccountSummary');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all of the context keys that are referenced in the specified policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForCustomPolicy.html
     */
    toGetContextKeysForCustomPolicy() {
        this.to('iam:GetContextKeysForCustomPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all context keys that are referenced in all IAM policies that are attached to the specified IAM identity (user, group, or role)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForPrincipalPolicy.html
     */
    toGetContextKeysForPrincipalPolicy() {
        this.to('iam:GetContextKeysForPrincipalPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a credential report for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetCredentialReport.html
     */
    toGetCredentialReport() {
        this.to('iam:GetCredentialReport');
        return this;
    }
    /**
     * Grants permission to retrieve a list of IAM users in the specified IAM group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroup.html
     */
    toGetGroup() {
        this.to('iam:GetGroup');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded in the specified IAM group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroupPolicy.html
     */
    toGetGroupPolicy() {
        this.to('iam:GetGroupPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified instance profile, including the instance profile's path, GUID, ARN, and role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetInstanceProfile.html
     */
    toGetInstanceProfile() {
        this.to('iam:GetInstanceProfile');
        return this;
    }
    /**
     * Grants permission to retrieve the user name and password creation date for the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetLoginProfile.html
     */
    toGetLoginProfile() {
        this.to('iam:GetLoginProfile');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified OpenID Connect (OIDC) provider resource in IAM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOpenIDConnectProvider.html
     */
    toGetOpenIDConnectProvider() {
        this.to('iam:GetOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to retrieve an AWS Organizations access report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOrganizationsAccessReport.html
     */
    toGetOrganizationsAccessReport() {
        this.to('iam:GetOrganizationsAccessReport');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified managed policy, including the policy's default version and the total number of identities to which the policy is attached
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        this.to('iam:GetPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about a version of the specified managed policy, including the policy document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicyVersion.html
     */
    toGetPolicyVersion() {
        this.to('iam:GetPolicyVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified role, including the role's path, GUID, ARN, and the role's trust policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRole.html
     */
    toGetRole() {
        this.to('iam:GetRole');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded with the specified IAM role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRolePolicy.html
     */
    toGetRolePolicy() {
        this.to('iam:GetRolePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the SAML provider metadocument that was uploaded when the IAM SAML provider resource was created or updated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSAMLProvider.html
     */
    toGetSAMLProvider() {
        this.to('iam:GetSAMLProvider');
        return this;
    }
    /**
     * Grants permission to retrieve the specified SSH public key, including metadata about the key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSSHPublicKey.html
     */
    toGetSSHPublicKey() {
        this.to('iam:GetSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified server certificate stored in IAM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServerCertificate.html
     */
    toGetServerCertificate() {
        this.to('iam:GetServerCertificate');
        return this;
    }
    /**
     * Grants permission to retrieve information about the service last accessed data report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetails.html
     */
    toGetServiceLastAccessedDetails() {
        this.to('iam:GetServiceLastAccessedDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about the entities from the service last accessed data report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetailsWithEntities.html
     */
    toGetServiceLastAccessedDetailsWithEntities() {
        this.to('iam:GetServiceLastAccessedDetailsWithEntities');
        return this;
    }
    /**
     * Grants permission to retrieve an IAM service-linked role deletion status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLinkedRoleDeletionStatus.html
     */
    toGetServiceLinkedRoleDeletionStatus() {
        this.to('iam:GetServiceLinkedRoleDeletionStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified IAM user, including the user's creation date, path, unique ID, and ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        this.to('iam:GetUser');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded in the specified IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUserPolicy.html
     */
    toGetUserPolicy() {
        this.to('iam:GetUserPolicy');
        return this;
    }
    /**
     * Grants permission to list information about the access key IDs that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccessKeys.html
     */
    toListAccessKeys() {
        this.to('iam:ListAccessKeys');
        return this;
    }
    /**
     * Grants permission to list the account alias that is associated with the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccountAliases.html
     */
    toListAccountAliases() {
        this.to('iam:ListAccountAliases');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedGroupPolicies.html
     */
    toListAttachedGroupPolicies() {
        this.to('iam:ListAttachedGroupPolicies');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedRolePolicies.html
     */
    toListAttachedRolePolicies() {
        this.to('iam:ListAttachedRolePolicies');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedUserPolicies.html
     */
    toListAttachedUserPolicies() {
        this.to('iam:ListAttachedUserPolicies');
        return this;
    }
    /**
     * Grants permission to list all IAM identities to which the specified managed policy is attached
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListEntitiesForPolicy.html
     */
    toListEntitiesForPolicy() {
        this.to('iam:ListEntitiesForPolicy');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupPolicies.html
     */
    toListGroupPolicies() {
        this.to('iam:ListGroupPolicies');
        return this;
    }
    /**
     * Grants permission to list the IAM groups that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.to('iam:ListGroups');
        return this;
    }
    /**
     * Grants permission to list the IAM groups that the specified IAM user belongs to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupsForUser.html
     */
    toListGroupsForUser() {
        this.to('iam:ListGroupsForUser');
        return this;
    }
    /**
     * Grants permission to list the instance profiles that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfiles.html
     */
    toListInstanceProfiles() {
        this.to('iam:ListInstanceProfiles');
        return this;
    }
    /**
     * Grants permission to list the instance profiles that have the specified associated IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfilesForRole.html
     */
    toListInstanceProfilesForRole() {
        this.to('iam:ListInstanceProfilesForRole');
        return this;
    }
    /**
     * Grants permission to list the MFA devices for an IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListMFADevices.html
     */
    toListMFADevices() {
        this.to('iam:ListMFADevices');
        return this;
    }
    /**
     * Grants permission to list information about the IAM OpenID Connect (OIDC) provider resource objects that are defined in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListOpenIDConnectProviders.html
     */
    toListOpenIDConnectProviders() {
        this.to('iam:ListOpenIDConnectProviders');
        return this;
    }
    /**
     * Grants permission to list all managed policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        this.to('iam:ListPolicies');
        return this;
    }
    /**
     * Grants permission to list information about the policies that grant an entity access to a specific service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPoliciesGrantingServiceAccess.html
     */
    toListPoliciesGrantingServiceAccess() {
        this.to('iam:ListPoliciesGrantingServiceAccess');
        return this;
    }
    /**
     * Grants permission to list information about the versions of the specified managed policy, including the version that is currently set as the policy's default version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicyVersions.html
     */
    toListPolicyVersions() {
        this.to('iam:ListPolicyVersions');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRolePolicies.html
     */
    toListRolePolicies() {
        this.to('iam:ListRolePolicies');
        return this;
    }
    /**
     * Grants permission to list the tags that are attached to the specified IAM role.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoleTags.html
     */
    toListRoleTags() {
        this.to('iam:ListRoleTags');
        return this;
    }
    /**
     * Grants permission to list the IAM roles that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoles.html
     */
    toListRoles() {
        this.to('iam:ListRoles');
        return this;
    }
    /**
     * Grants permission to list the SAML provider resources in IAM
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSAMLProviders.html
     */
    toListSAMLProviders() {
        this.to('iam:ListSAMLProviders');
        return this;
    }
    /**
     * Grants permission to list information about the SSH public keys that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSSHPublicKeys.html
     */
    toListSSHPublicKeys() {
        this.to('iam:ListSSHPublicKeys');
        return this;
    }
    /**
     * Grants permission to list the server certificates that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServerCertificates.html
     */
    toListServerCertificates() {
        this.to('iam:ListServerCertificates');
        return this;
    }
    /**
     * Grants permission to list the service-specific credentials that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServiceSpecificCredentials.html
     */
    toListServiceSpecificCredentials() {
        this.to('iam:ListServiceSpecificCredentials');
        return this;
    }
    /**
     * Grants permission to list information about the signing certificates that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSigningCertificates.html
     */
    toListSigningCertificates() {
        this.to('iam:ListSigningCertificates');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserPolicies.html
     */
    toListUserPolicies() {
        this.to('iam:ListUserPolicies');
        return this;
    }
    /**
     * Grants permission to list the tags that are attached to the specified IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserTags.html
     */
    toListUserTags() {
        this.to('iam:ListUserTags');
        return this;
    }
    /**
     * Grants permission to list the IAM users that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('iam:ListUsers');
        return this;
    }
    /**
     * Grants permission to list virtual MFA devices by assignment status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListVirtualMFADevices.html
     */
    toListVirtualMFADevices() {
        this.to('iam:ListVirtualMFADevices');
        return this;
    }
    /**
     * Grants permission to pass a role to a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAssociatedResourceArn()
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html
     */
    toPassRole() {
        this.to('iam:PassRole');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutGroupPolicy.html
     */
    toPutGroupPolicy() {
        this.to('iam:PutGroupPolicy');
        return this;
    }
    /**
     * Grants permission to set a managed policy as a permissions boundary for a role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePermissionsBoundary.html
     */
    toPutRolePermissionsBoundary() {
        this.to('iam:PutRolePermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePolicy.html
     */
    toPutRolePolicy() {
        this.to('iam:PutRolePolicy');
        return this;
    }
    /**
     * Grants permission to set a managed policy as a permissions boundary for an IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPermissionsBoundary.html
     */
    toPutUserPermissionsBoundary() {
        this.to('iam:PutUserPermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPolicy.html
     */
    toPutUserPolicy() {
        this.to('iam:PutUserPolicy');
        return this;
    }
    /**
     * Grants permission to remove the client ID (audience) from the list of client IDs in the specified IAM OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveClientIDFromOpenIDConnectProvider.html
     */
    toRemoveClientIDFromOpenIDConnectProvider() {
        this.to('iam:RemoveClientIDFromOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to remove an IAM role from the specified EC2 instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveRoleFromInstanceProfile.html
     */
    toRemoveRoleFromInstanceProfile() {
        this.to('iam:RemoveRoleFromInstanceProfile');
        return this;
    }
    /**
     * Grants permission to remove an IAM user from the specified group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveUserFromGroup.html
     */
    toRemoveUserFromGroup() {
        this.to('iam:RemoveUserFromGroup');
        return this;
    }
    /**
     * Grants permission to reset the password for an existing service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResetServiceSpecificCredential.html
     */
    toResetServiceSpecificCredential() {
        this.to('iam:ResetServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to synchronize the specified MFA device with its IAM entity (user or role)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResyncMFADevice.html
     */
    toResyncMFADevice() {
        this.to('iam:ResyncMFADevice');
        return this;
    }
    /**
     * Grants permission to set the version of the specified policy as the policy's default version
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetDefaultPolicyVersion.html
     */
    toSetDefaultPolicyVersion() {
        this.to('iam:SetDefaultPolicyVersion');
        return this;
    }
    /**
     * Grants permission to set the STS global endpoint token version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetSecurityTokenServicePreferences.html
     */
    toSetSecurityTokenServicePreferences() {
        this.to('iam:SetSecurityTokenServicePreferences');
        return this;
    }
    /**
     * Grants permission to simulate whether an identity-based policy or resource-based policy provides permissions for specific API operations and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulateCustomPolicy.html
     */
    toSimulateCustomPolicy() {
        this.to('iam:SimulateCustomPolicy');
        return this;
    }
    /**
     * Grants permission to simulate whether an identity-based policy that is attached to a specified IAM entity (user or role) provides permissions for specific API operations and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulatePrincipalPolicy.html
     */
    toSimulatePrincipalPolicy() {
        this.to('iam:SimulatePrincipalPolicy');
        return this;
    }
    /**
     * Grants permission to add tags to an IAM role.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagRole.html
     */
    toTagRole() {
        this.to('iam:TagRole');
        return this;
    }
    /**
     * Grants permission to add tags to an IAM user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagUser.html
     */
    toTagUser() {
        this.to('iam:TagUser');
        return this;
    }
    /**
     * Grants permission to remove the specified tags from the role.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagRole.html
     */
    toUntagRole() {
        this.to('iam:UntagRole');
        return this;
    }
    /**
     * Grants permission to remove the specified tags from the user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagUser.html
     */
    toUntagUser() {
        this.to('iam:UntagUser');
        return this;
    }
    /**
     * Grants permission to update the status of the specified access key as Active or Inactive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccessKey.html
     */
    toUpdateAccessKey() {
        this.to('iam:UpdateAccessKey');
        return this;
    }
    /**
     * Grants permission to update the password policy settings for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccountPasswordPolicy.html
     */
    toUpdateAccountPasswordPolicy() {
        this.to('iam:UpdateAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to update the policy that grants an IAM entity permission to assume a role
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAssumeRolePolicy.html
     */
    toUpdateAssumeRolePolicy() {
        this.to('iam:UpdateAssumeRolePolicy');
        return this;
    }
    /**
     * Grants permission to update the name or path of the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.to('iam:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to change the password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateLoginProfile.html
     */
    toUpdateLoginProfile() {
        this.to('iam:UpdateLoginProfile');
        return this;
    }
    /**
     * Grants permission to update the entire list of server certificate thumbprints that are associated with an OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateOpenIDConnectProviderThumbprint.html
     */
    toUpdateOpenIDConnectProviderThumbprint() {
        this.to('iam:UpdateOpenIDConnectProviderThumbprint');
        return this;
    }
    /**
     * Grants permission to update the description or maximum session duration setting of a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRole.html
     */
    toUpdateRole() {
        this.to('iam:UpdateRole');
        return this;
    }
    /**
     * Grants permission to update only the description of a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRoleDescription.html
     */
    toUpdateRoleDescription() {
        this.to('iam:UpdateRoleDescription');
        return this;
    }
    /**
     * Grants permission to update the metadata document for an existing SAML provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSAMLProvider.html
     */
    toUpdateSAMLProvider() {
        this.to('iam:UpdateSAMLProvider');
        return this;
    }
    /**
     * Grants permission to update the status of an IAM user's SSH public key to active or inactive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSSHPublicKey.html
     */
    toUpdateSSHPublicKey() {
        this.to('iam:UpdateSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to update the name or the path of the specified server certificate stored in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServerCertificate.html
     */
    toUpdateServerCertificate() {
        this.to('iam:UpdateServerCertificate');
        return this;
    }
    /**
     * Grants permission to update the status of a service-specific credential to active or inactive for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServiceSpecificCredential.html
     */
    toUpdateServiceSpecificCredential() {
        this.to('iam:UpdateServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to update the status of the specified user signing certificate to active or disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSigningCertificate.html
     */
    toUpdateSigningCertificate() {
        this.to('iam:UpdateSigningCertificate');
        return this;
    }
    /**
     * Grants permission to update the name or the path of the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        this.to('iam:UpdateUser');
        return this;
    }
    /**
     * Grants permission to upload an SSH public key and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSSHPublicKey.html
     */
    toUploadSSHPublicKey() {
        this.to('iam:UploadSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to upload a server certificate entity for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadServerCertificate.html
     */
    toUploadServerCertificate() {
        this.to('iam:UploadServerCertificate');
        return this;
    }
    /**
     * Grants permission to upload an X.509 signing certificate and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSigningCertificate.html
     */
    toUploadSigningCertificate() {
        this.to('iam:UploadSigningCertificate');
        return this;
    }
    /**
     * Adds a resource of type access-report to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html
     *
     * @param entityPath - Identifier for the entityPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccessReport(entityPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:access-report/${EntityPath}';
        arn = arn.replace('${EntityPath}', entityPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assumed-role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_use-resources.html
     *
     * @param roleName - Identifier for the roleName.
     * @param roleSessionName - Identifier for the roleSessionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssumedRole(roleName, roleSessionName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:assumed-role/${RoleName}/${RoleSessionName}';
        arn = arn.replace('${RoleName}', roleName);
        arn = arn.replace('${RoleSessionName}', roleSessionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type federated-user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html
     *
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFederatedUser(userName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:federated-user/${UserName}';
        arn = arn.replace('${UserName}', userName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html
     *
     * @param groupNameWithPath - Identifier for the groupNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(groupNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:group/${GroupNameWithPath}';
        arn = arn.replace('${GroupNameWithPath}', groupNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance-profile to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_switch-role-ec2_instance-profiles.html
     *
     * @param instanceProfileNameWithPath - Identifier for the instanceProfileNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstanceProfile(instanceProfileNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:instance-profile/${InstanceProfileNameWithPath}';
        arn = arn.replace('${InstanceProfileNameWithPath}', instanceProfileNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mfa to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html
     *
     * @param mfaTokenIdWithPath - Identifier for the mfaTokenIdWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMfa(mfaTokenIdWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:mfa/${MfaTokenIdWithPath}';
        arn = arn.replace('${MfaTokenIdWithPath}', mfaTokenIdWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type oidc-provider to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     *
     * @param oidcProviderName - Identifier for the oidcProviderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOidcProvider(oidcProviderName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:oidc-provider/${OidcProviderName}';
        arn = arn.replace('${OidcProviderName}', oidcProviderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_managed-vs-inline.html
     *
     * @param policyNameWithPath - Identifier for the policyNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(policyNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:policy/${PolicyNameWithPath}';
        arn = arn.replace('${PolicyNameWithPath}', policyNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html
     *
     * @param roleNameWithPath - Identifier for the roleNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onRole(roleNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}';
        arn = arn.replace('${RoleNameWithPath}', roleNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type saml-provider to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html
     *
     * @param samlProviderName - Identifier for the samlProviderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSamlProvider(samlProviderName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:saml-provider/${SamlProviderName}';
        arn = arn.replace('${SamlProviderName}', samlProviderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type server-certificate to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html
     *
     * @param certificateNameWithPath - Identifier for the certificateNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onServerCertificate(certificateNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:server-certificate/${CertificateNameWithPath}';
        arn = arn.replace('${CertificateNameWithPath}', certificateNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sms-mfa to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa_enable_sms.html
     *
     * @param mfaTokenIdWithPath - Identifier for the mfaTokenIdWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSmsMfa(mfaTokenIdWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:sms-mfa/${MfaTokenIdWithPath}';
        arn = arn.replace('${MfaTokenIdWithPath}', mfaTokenIdWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     *
     * @param userNameWithPath - Identifier for the userNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onUser(userNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:user/${UserNameWithPath}';
        arn = arn.replace('${UserNameWithPath}', userNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the AWS service to which this role is attached
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_AWSServiceName
     *
     * Applies to actions:
     * - .toCreateServiceLinkedRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSServiceName(value, operator) {
        return this.if(`iam:AWSServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters by the resource that the role will be used on behalf of
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_AssociatedResourceArn
     *
     * Applies to actions:
     * - .toPassRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifAssociatedResourceArn(value, operator) {
        return this.if(`iam:AssociatedResourceArn`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by the ID of an AWS Organizations policy
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_OrganizationsPolicyId
     *
     * Applies to actions:
     * - .toGenerateOrganizationsAccessReport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOrganizationsPolicyId(value, operator) {
        return this.if(`iam:OrganizationsPolicyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service to which this role is passed
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PassedToService
     *
     * Applies to actions:
     * - .toPassRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`iam:PassedToService`, value, operator || 'StringLike');
    }
    /**
     * Filters access if the specified policy is set as the permissions boundary on the IAM entity (user or role)
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PermissionsBoundary
     *
     * Applies to actions:
     * - .toAttachRolePolicy()
     * - .toAttachUserPolicy()
     * - .toCreateRole()
     * - .toCreateUser()
     * - .toDeleteRolePermissionsBoundary()
     * - .toDeleteRolePolicy()
     * - .toDeleteUserPermissionsBoundary()
     * - .toDeleteUserPolicy()
     * - .toDetachRolePolicy()
     * - .toDetachUserPolicy()
     * - .toPutRolePermissionsBoundary()
     * - .toPutRolePolicy()
     * - .toPutUserPermissionsBoundary()
     * - .toPutUserPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissionsBoundary(value, operator) {
        return this.if(`iam:PermissionsBoundary`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an IAM policy
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PolicyARN
     *
     * Applies to actions:
     * - .toAttachGroupPolicy()
     * - .toAttachRolePolicy()
     * - .toAttachUserPolicy()
     * - .toDetachGroupPolicy()
     * - .toDetachRolePolicy()
     * - .toDetachUserPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPolicyARN(value, operator) {
        return this.if(`iam:PolicyARN`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by the tags attached to an IAM entity (user or role).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_ResourceTag
     *
     * Applies to resource types:
     * - role
     * - user
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iam:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Iam = Iam;
//# sourceMappingURL=data:application/json;base64,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